/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * run.c
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include "mdapi/mdengine.h"
#include "force/force.h"
#include "timer/timer.h"
#include "deven/engine.h"
#undef DEBUG_WATCH
#include "debug/debug.h"


/* prototypes for internal functions */
static int reductions(MD_Front *, StepSystem *);


int deven_output_from_step(void *vfront, const char *msg)
{
  /* any MDAPI support for text output? */

  /* otherwise */
  printf("#STEP: %s", msg);  /* assume msg ends with newline */

  return 0;
}


int deven_results_from_step(void *vfront, StepSystem *sys)
{
  MD_Front *front = (MD_Front *) vfront;
  Engine *eng = (Engine *) MD_engine_data(front);

  /* increment step counter with interface */
  if (sys->stepnum > eng->last_stepnum) {
    MD_incrstep(front, sys->stepnum - eng->last_stepnum);
  }

  /* see if callbacks need to be processed for this step */
  if (MD_ready_callback(front)) {

    /* reductions might be needed by callback */
    if (reductions(front, sys)) return MD_FAIL;

    /* execute callbacks */
    if (MD_exec_callback(front) || MD_wait_callback(front)) return MD_FAIL;
  }

  eng->last_stepnum = sys->stepnum;

  return 0;
}


int32 deven_run(MD_Front *front, int32 numsteps, int32 flags)
{
  Engine *eng = MD_engine_data(front);
  Step *step = eng->step;
  StepParam *step_param = &(eng->step_param);
#if 0
  StepSystem *step_system = &(eng->step_system);
  int32 n;
#endif

  /* start timer if requested */
  if (eng->param.outputTiming > 0 && eng->timer == NULL) {
    eng->timer = timer_create();
  }

  /* update internal data and force library */
  if (flags & MD_UPDATE) {

    /* call update routine */
    TEXT("calling update");
    if (deven_update(front)) return MD_FAIL;
    TEXT("update done");
  }

  /* reset engine's last step number (in case front end changed it) */
  eng->last_stepnum = MD_stepnum(front);

  /*** take advantage of fact that step object points to step param ***/
  /*** XXX this is a kludge! ***/
  step_param->timestep = eng->timestep;
  step_param->init_stepnum = eng->last_stepnum;

  if (STEP_FAILURE==step_compute(step, numsteps)) {
    printf("#STEP ERROR:  %s\n", step_errmsg(step));
    return MD_FAIL;
  }

#if 0
  /* see if callbacks need to be processed before stepping */
  if (MD_ready_callback(front)) {

    TEXT("calling step_compute");
    /* make sure that forces are up-to-date */
    if (step_compute(step, step_system, 0)) {
      return MD_FAIL;
    }
    TEXT("done calling step_compute");

    /* reductions might be needed by callback */
    if (reductions(front)) return MD_FAIL;

    /* execute callbacks */
    if (MD_exec_callback(front) || MD_wait_callback(front)) return MD_FAIL;
  }

  /* call step library for numsteps */
  for (n = 0;  n < numsteps;  n++) {

    /* take single steps so we can monitor callbacks */
    if (step_compute(step, step_system, 1)) {
      return MD_FAIL;
    }

    /* increment step counter with interface */
    MD_incrstep(front);

    /* see if callbacks need to be processed for this step */
    if (MD_ready_callback(front)) {

      /* reductions might be needed by callback */
      if (reductions(front)) return MD_FAIL;

      /* execute callbacks */
      if (MD_exec_callback(front) || MD_wait_callback(front)) return MD_FAIL;
    }
  }

  /* compute reductions before returning */
  if (reductions(front)) return MD_FAIL;
#endif

  return 0;
}


int reductions(MD_Front *front, StepSystem *step_system)
{
  Engine *e = MD_engine_data(front);
  Result *result = &(e->result);
  Param *param = &(e->param);
  /*
  Step *step = e->step;
  */
  ForceResult *fr = &(e->f_result);
  double *kv = step_system->kinetic_virial;
  double *fv = fr->virial;
  int32 this_stepnum = MD_stepnum(front);  /* get current stepnum */
  int32 delta_stepnum = this_stepnum - e->last_stepnum;
  int32 i;

  /* copy potential energies from force evaluation */
  result->pe = fr->u_total;
  result->bond = fr->u_bond;
  result->angle = fr->u_angle;
  result->dihed = fr->u_dihed;
  result->impr = fr->u_impr;
  result->elec = fr->u_elec;
  result->vdw = fr->u_vdw;
  result->bound = fr->u_bres;

  FLT(result->pe);
  FLT(result->bond);
  FLT(result->angle);
  FLT(result->dihed);
  FLT(result->impr);
  FLT(result->elec);
  FLT(result->vdw);
  FLT(result->bound);

#if 0
  /* call step library for kinetic energy and temperature */
  if (step_find_reductions(step, step_system)) {
    return MD_FAIL;
  }
#endif

  result->ke = step_system->kinetic_energy;
  result->energy = result->ke + result->pe;
  result->temp = step_system->temperature;
  result->linmo = step_system->linear_momentum;
  result->angmo = step_system->angular_momentum;

  /* copy shadow energies */
  result->shadow[STEP_SHADOW_4] = step_system->shadow_energy[STEP_SHADOW_4];
  result->shadow[STEP_SHADOW_8] = step_system->shadow_energy[STEP_SHADOW_8];
  result->shadow[STEP_SHADOW_12] = step_system->shadow_energy[STEP_SHADOW_12];
  result->shadow[STEP_SHADOW_16] = step_system->shadow_energy[STEP_SHADOW_16];
  result->shadow[STEP_SHADOW_20] = step_system->shadow_energy[STEP_SHADOW_20];
  result->shadow[STEP_SHADOW_24] = step_system->shadow_energy[STEP_SHADOW_24];

  /* copy Nose-Hoover reductions */
  result->nhexten = step_system->nh_extended_energy;
  result->nhfric = step_system->nh_friction_coef;
  result->nhlogs = step_system->nh_log_s;

  /* copy thermalized Drude oscillator reductions */
  result->druexten = step_system->drude_extended_energy;
  result->drucome = step_system->drude_com_energy;
  result->drubonde = step_system->drude_bond_energy;
  result->drucomt = step_system->drude_com_temperature;
  result->drubondt = step_system->drude_bond_temperature;

  /* volume of periodic domain */
  if (e->is_domain_periodic == FORCE_PERIODIC) {
#define PRESSUREFACTOR 6.95E4
    double scale;

    result->vol = force_get_volume(e->force); 

    /*
     * sum kinetic and force contributions to virial
     *
     * unit conversion:  scaling factor converts kcal/mol/A^3 to bar
     */
    scale = PRESSUREFACTOR / result->vol;
    for (i = 0;  i < 9;  i++) {
      result->virial[i] = scale * (kv[i] + fv[i]);
    }

    /* scalar pressure value is 1/3 trace of virial tensor */
    result->press = (1./3) * (result->virial[0] + result->virial[4]
        + result->virial[8]);
  }

  /*
   * kludge:  honor particular output options directly
   *   (treat step number 0 as special output case)
   */
  if (param->outputMomenta > 0) {
    e->cntsteps_outputMomenta += delta_stepnum;
    if (e->cntsteps_outputMomenta >= param->outputMomenta
        || this_stepnum == 0) {
      MD_Dvec *lm = &(result->linmo);
      MD_Dvec *am = &(result->angmo);
      printf("# linear momentum: %g %g %g\n", lm->x, lm->y, lm->z);
      printf("# angular momentum: %g %g %g\n", am->x, am->y, am->z);
      e->cntsteps_outputMomenta = 0;
    }
  }

  if (param->outputPressure > 0) {
    e->cntsteps_outputPressure += delta_stepnum;
    if (e->cntsteps_outputPressure >= param->outputPressure
        || this_stepnum == 0) {
      double *v = result->virial;
      printf("# virial tensor: %g %g %g  %g %g %g  %g %g %g\n",
          v[0], v[1], v[2], v[3], v[4], v[5], v[6], v[7], v[8]);
      e->cntsteps_outputPressure = 0;
    }
  }

  if (param->outputTiming > 0) {
    e->cntsteps_outputTiming += delta_stepnum;
    if (e->cntsteps_outputTiming >= param->outputTiming
        || this_stepnum == 0) {
      double delta_walltime = timer_click(e->timer);
      printf("# ellapsed wall-clock time for simulation: %s\n",
          timer_msg_total(e->timer));
      if (e->cntsteps_outputTiming > 0) {
        printf("# average wall-clock time per step: %g seconds\n",
            delta_walltime / e->cntsteps_outputTiming);
      }
      e->cntsteps_outputTiming = 0;
    }
  }

  /* save current stepnum for next reduction call */
  e->last_stepnum = this_stepnum;

  return 0;
}
