/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * deven.c
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <sys/time.h>
#include "mdapi/mdengine.h"
#include "force/force.h"
#include "mgrid/mgrid.h"
#include "deven/deven.h"
#include "deven/engine.h"
#include "debug/debug.h"

#define NELEMS(x)  (sizeof(x) / sizeof(x[0]))


/* needed to define type Result to interface */
static const MD_Member resultMemberList[] = {
  { MD_DOUBLE, 1, "energy" },
  { MD_DOUBLE, 1, "ke" },
  { MD_DOUBLE, 1, "pe" },
  { MD_DOUBLE, 1, "bond" },
  { MD_DOUBLE, 1, "angle" },
  { MD_DOUBLE, 1, "dihed" },
  { MD_DOUBLE, 1, "impr" },
  { MD_DOUBLE, 1, "elec" },
  { MD_DOUBLE, 1, "vdw" },
  { MD_DOUBLE, 1, "bound" },
  { MD_DOUBLE, 1, "temp" },
  { MD_DOUBLE, 1, "vol" },
  { MD_DOUBLE, 1, "press" },
  { MD_DVEC, 1, "vir_x" },
  { MD_DVEC, 1, "vir_y" },
  { MD_DVEC, 1, "vir_z" },
  { MD_DVEC, 1, "linmo" },
  { MD_DVEC, 1, "angmo" },
  { MD_DVEC, 1, "fcorr" },
  { MD_DVEC, 1, "wcomd" },
  { MD_DOUBLE, 1, "shadow4" },
  { MD_DOUBLE, 1, "shadow8" },
  { MD_DOUBLE, 1, "shadow12" },
  { MD_DOUBLE, 1, "shadow16" },
  { MD_DOUBLE, 1, "shadow20" },
  { MD_DOUBLE, 1, "shadow24" },
  { MD_DOUBLE, 1, "nhexten" },
  { MD_DOUBLE, 1, "nhfric" },
  { MD_DOUBLE, 1, "nhlogs" },
  { MD_DOUBLE, 1, "druexten" },
  { MD_DOUBLE, 1, "drucome" },
  { MD_DOUBLE, 1, "drubonde" },
  { MD_DOUBLE, 1, "drucomt" },
  { MD_DOUBLE, 1, "drubondt" },
};


/* needed to define type Param to interface */
static const MD_Member paramMemberList[] = {
  { MD_DOUBLE, 1, "cutoff" },
  { MD_DOUBLE, 1, "switchdist" },
  { MD_CHAR, 4, "switching" },
  { MD_CHAR, 12, "exclude" },
  { MD_DOUBLE, 1, "dielectric" },
  { MD_DOUBLE, 1, "1-4scaling" },    /* for "scaling" member of Param */
  { MD_CHAR, 4, "fulldirect" },
  { MD_CHAR, 4, "sphericalbc" },
  { MD_DVEC, 1, "sphericalbccenter" },
  { MD_DOUBLE, 1, "sphericalbcr1" },
  { MD_DOUBLE, 1, "sphericalbcr2" },
  { MD_DOUBLE, 1, "sphericalbck1" },
  { MD_DOUBLE, 1, "sphericalbck2" },
  { MD_INT32, 1, "sphericalbcexp1" },
  { MD_INT32, 1, "sphericalbcexp2" },
  { MD_CHAR, 4, "cylindricalbc" },
  { MD_CHAR, 4, "cylindricalbcaxis" },
  { MD_DVEC, 1, "cylindricalbccenter" },
  { MD_DOUBLE, 1, "cylindricalbcr1" },
  { MD_DOUBLE, 1, "cylindricalbcr2" },
  { MD_DOUBLE, 1, "cylindricalbcl1" },
  { MD_DOUBLE, 1, "cylindricalbcl2" },
  { MD_DOUBLE, 1, "cylindricalbck1" },
  { MD_DOUBLE, 1, "cylindricalbck2" },
  { MD_INT32, 1, "cylindricalbcexp1" },
  { MD_INT32, 1, "cylindricalbcexp2" },
  { MD_DVEC, 1, "cellbasisvector1" },
  { MD_DVEC, 1, "cellbasisvector2" },
  { MD_DVEC, 1, "cellbasisvector3" },
  { MD_DVEC, 1, "cellorigin" },
/*** mgrid params ***/
  { MD_CHAR, 8, "mgrid" },
  { MD_DOUBLE, 1, "mgridlength" },
  { MD_DOUBLE, 1, "mgridcutoff" },
  { MD_DOUBLE, 1, "mgridspacing" },
  { MD_INT32, 1, "mgridnspacings" },
  { MD_INT32, 1, "mgridnlevels" },
  { MD_CHAR, 12, "mgridapprox" },
  { MD_CHAR, 12, "mgridsplit" },
/*** pmetest params ***/
  { MD_CHAR, 8, "pme" },
  { MD_DOUBLE, 1, "pmecutoff" },
  { MD_DOUBLE, 1, "pmetolerance" },
  { MD_INT32, 1, "pmeinterporder" },
  { MD_INT32, 1, "pmegridsizex" },
  { MD_INT32, 1, "pmegridsizey" },
  { MD_INT32, 1, "pmegridsizez" },
/*** correction for constant linear momentum ***/
  { MD_CHAR, 12, "correctlinmo" },
/*** kludge ***/
  { MD_INT32, 1, "outputmomenta" },
  { MD_INT32, 1, "outputpressure" },
  { MD_INT32, 1, "outputtiming" },
/*** shadow Hamiltnonian ***/
  { MD_CHAR, 8, "shadow" },
/*** energy minimization ***/
  { MD_CHAR, 4, "cgmin" },
  { MD_INT32, 1, "cgmineval" },
  { MD_DOUBLE, 1, "cgmindis" },
  { MD_DOUBLE, 1, "cgmintol" },
/*** buckingham potential ***/
  { MD_CHAR, 4, "buckingham" },
  { MD_CHAR, 4, "bucknodispersion" },
  { MD_CHAR, 8, "buckparam" },  /* choose "fb" (default), "ttam", or "bks" */
  { MD_CHAR, 8, "bucksafe" },
/*** coupling to temperature bath ***/
  { MD_CHAR, 8, "temperaturebath" },
  { MD_DOUBLE, 1, "targettemperature" },
  { MD_DOUBLE, 1, "relaxationtime" },
/*** explicit Nose-Hoover ***/
  { MD_CHAR, 8, "nosehoover" },
  { MD_DOUBLE, 1, "nhtemperature" },
  { MD_DOUBLE, 1, "nhtimescale" },
/*** for harmonic constraints ***/
  { MD_DOUBLE, 1, "constraintscaling" },
  { MD_CHAR, 4, "selectconstraints" },
  { MD_CHAR, 4, "selectconstrx" },
  { MD_CHAR, 4, "selectconstry" },
  { MD_CHAR, 4, "selectconstrz" },
/*** thermalized Drude oscillators ***/
  { MD_CHAR, 8, "drude" },
  { MD_CHAR, 4, "useroux" },
  { MD_CHAR, 4, "usechen" },
  { MD_DOUBLE, 1, "drucomtemp" },
  { MD_DOUBLE, 1, "drubondtemp" },
  { MD_DOUBLE, 1, "drucomtimescale" },
  { MD_DOUBLE, 1, "drubondtimescale" },
  { MD_INT32, 1, "drumultistep" },
/*** rigid waters ***/
  { MD_CHAR, 8, "rigidbonds" },
  { MD_CHAR, 4, "usesettle" },
};


/* default values for params */
static const Param defaultParam = {
  /* cutoff = */ 0.0,
  /* switchDist = */ 0.0,
  /* switching = */ "off",
  /* exclude = */ "",
  /* dielectric = */ 1.0,
  /* 1-4scaling = */ 1.0,
  /* fullDirect = */ "no",
  /* sphericalBC = */ "off",
  /* sphericalBCCenter = */ { 0.0, 0.0, 0.0 },
  /* sphericalBCr1 = */ 0.0,
  /* sphericalBCr2 = */ 0.0,
  /* sphericalBCk1 = */ 0.0,
  /* sphericalBCk2 = */ 0.0,
  /* sphericalBCexp1 = */ 2,
  /* sphericalBCexp2 = */ 2,
  /* cylindricalBC = */ "off",
  /* cylindricalBCAxis = */ "",
  /* cylindricalBCCenter = */ { 0.0, 0.0, 0.0 },
  /* cylindricalBCr1 = */ 0.0,
  /* cylindricalBCr2 = */ 0.0,
  /* cylindricalBCl1 = */ 0.0,
  /* cylindricalBCl2 = */ 0.0,
  /* cylindricalBCk1 = */ 0.0,
  /* cylindricalBCk2 = */ 0.0,
  /* cylindricalBCexp1 = */ 2,
  /* cylindricalBCexp2 = */ 2,
  /* cellBasisVector1 = */ { 0.0, 0.0, 0.0 },
  /* cellBasisVector2 = */ { 0.0, 0.0, 0.0 },
  /* cellBasisVector3 = */ { 0.0, 0.0, 0.0 },
  /* cellOrigin = */ { 0.0, 0.0, 0.0 },
/*** mgrid params ***/
  /* mgrid = */ "off",
  /* mgridLength = */ 0.0,
  /* mgridCutoff = */ 0.0,
  /* mgridSpacing = */ 0.0,
  /* mgridNspacings = */ 0,
  /* mgridNlevels = */ 0,
  /* mgridApprox = */ "cubic",
  /* mgridSplit = */ "taylor2",
/*** pmetest params ***/
  /* pmetest = */ "off",
  /* pmetestCutoff = */ 0.0,
  /* pmetestTolerance = */ 1e-6,  /* NAMD's default */
  /* pmetestInterpOrder = */ 4,   /* NAMD's default */
  /* pmetestGridSizeX = */ 0,
  /* pmetestGridSizeY = */ 0,
  /* pmetestGridSizeZ = */ 0,
/*** correction for constant linear momentum ***/
  /* correctLinmo = */ "",  /* other values are "standard" and "conserve" */
/*** kludge ***/
  /*
   * number of steps between output of these quantities, but restricted
   * to appearing only at multiples of "reductions" step interval
   * (so occurs at an immediately following reduction step)
   *
   * deven prints out these values itself (simpler, but a bad thing)
   */
  /* outputMomenta = */ 0,
  /* outputPressure = */ 0,
  /* outputTiming = */ 0,
/*** shadow Hamiltonian ***/
  /* shadow = */ "off",
/*** energy minimization ***/
  /* cgmin = */ "off",
  /* cgmineval = */ 50,   /* max number of force evals before giving up */
  /* cgmindis = */ 1.0,   /* max bracketing distance */
  /* cgmintol = */ 1e-2,  /* tolerance for line search */
/*** buckingham potential ***/
  /* buckingham = */ "off",
  /* bucknodispersion = */ "off",
  /* buckparam = */ "fb",
  /* bucksafe = */ "off",
/*** coupling to temperature bath ***/
  /* temperaturebath = */ "off",
  /* targettemperature = */ 0,
  /* relaxationtime = */ 400.0,  /* in fs, prescribed by Leach for dt=1fs */
/*** explicit Nose-Hoover ***/
  /* nosehoover = */ "off",
  /* nhtemperature = */ 0,
  /* nhtimescale = */ 50.0,  /* in fs, period of harmonic oscillator */
/*** for harmonic constraints ***/
  /* constraintscaling = */ 1.0,
  /* selectConstraints = */ "off",
  /* selectConstrX = */ "off",
  /* selectConstrY = */ "off",
  /* selectConstrZ = */ "off",
/*** thermalized Drude oscillators ***/
  /* drude = */ "off",
  /* useRoux = */ "off",
  /* useChen = */ "off",
  /* drucomtemp = */ 0,
  /* drubondtemp = */ 0,
  /* drucomtimescale = */ 100.0,  /* in fs, period for COM thermostat */
  /* drubondtimescale = */ 7.07,  /* in fs, period for bond thermostat */
  /* drumultisteps = */ 20,       /* number of multisteps for ext vars */
/*** rigid waters ***/
  /* rigidBonds = */ "none",
  /* useSettle = */ "no",
};


/* needed to define type Constraint to interface */
static const MD_Member constraintMemberList[] = {
  { MD_DVEC, 1, "refpos" },
  { MD_DOUBLE, 1, "k" },
  { MD_INT32, 1, "expo" },
  { MD_INT32, 1, "index" },
};


int32 deven_init(MD_Front *f, int32 flags)
{
  /* pointer to internal data */
  Engine *eng = NULL;

  /* access permissions for engine data arrays */
  const int32 aw = MD_RESIZE | MD_WRITE | MD_NOTIFY;
  const int32 arwc = MD_RESIZE | MD_READ | MD_WRITE | MD_CBREAD | MD_NOTIFY;
  const int32 arc = MD_RESIZE | MD_READ | MD_CBREAD | MD_NOTIFY;
  /* const int32 arc = MD_RESIZE | MD_READ | MD_NOTIFY; */
  const int32 aerc = MD_ERESIZE | MD_READ | MD_CBREAD;
  const int32 ae = MD_ERESIZE;

  /* attributes static engine data arrays */
  /* for result and param, change 0 to type number after definition */
  MD_Attrib attr_result = { 0, 1, 1, MD_READ | MD_CBREAD };
  MD_Attrib attr_param = { 0, 1, 1, MD_READ | MD_WRITE | MD_NOTIFY };
  MD_Attrib attr_timestep = { MD_DOUBLE, 1, 1, MD_READ | MD_WRITE };
  MD_Attrib attr_init_temp
    = { MD_DOUBLE, 1, 1, MD_READ | MD_WRITE | MD_NOTIFY };
  MD_Attrib attr_com_motion = { MD_CHAR, 0, 4, MD_READ | MD_WRITE | MD_SETLEN };
  MD_Attrib attr_seed = { MD_INT32, 1, 1, MD_READ | MD_WRITE };
  MD_Attrib attr_ndegfreedom = { MD_INT32, 1, 1, MD_READ };

  /* get type id number for "Constraint" */
  int32 type_constraint;

  /* need clock time to init seed for random number generator */
  struct timeval tv;

  /* allocate internal data structure */
  if ((eng = (Engine *) calloc(1, sizeof(Engine))) == NULL) {
    return MD_error(f, MD_ERR_MEMALLOC);
  }

  /* setup engine and internal data with interface */
  if (MD_setup_engine(f, eng)) return MD_FAIL;

  /* define Result and Param types with interface */
  attr_result.type = MD_new_type(f, "Result", resultMemberList,
      NELEMS(resultMemberList), sizeof(Result));
  if (attr_result.type == MD_FAIL) return MD_FAIL;

  attr_param.type = MD_new_type(f, "Param", paramMemberList,
      NELEMS(paramMemberList), sizeof(Param));
  if (attr_param.type == MD_FAIL) return MD_FAIL;

  /* define Constraint type with interface */
  type_constraint = MD_new_type(f, "Constraint", constraintMemberList,
      NELEMS(constraintMemberList), sizeof(DevenConstraint));
  if (type_constraint == MD_FAIL) return MD_FAIL;

  /* define errors with interface */
  eng->err_param = MD_new_error(f, "Invalid engine parameters", 0);
  if (eng->err_param == MD_FAIL) return MD_FAIL;

  eng->err_unsupported = MD_new_error(f, "Unsupported engine feature", 0);
  if (eng->err_unsupported == MD_FAIL) return MD_FAIL;

  eng->err_force_init = MD_new_error(f, "Force initialization failed", -1);
  if (eng->err_force_init == MD_FAIL) return MD_FAIL;

  eng->err_force = MD_new_error(f, "Force computation failed", -1);
  if (eng->err_force == MD_FAIL) return MD_FAIL;

  /* set engine params to default values */
  eng->param = defaultParam;

  /* set default values for additional sim params */
  eng->timestep = 1.0;
  eng->init_temp = 0.0;
  strcpy(eng->com_motion, "no");
  attr_com_motion.len = 3;  /* strlen("no") + nil-terminator */

  /* choose random seed using clock */
  if (gettimeofday(&tv, NULL)) return MD_FAIL;
  eng->seed = tv.tv_sec;

  /* establish engine data arrays with interface */
  if (/* force field params and topology arrays allow write and resize */
      (eng->atomprm = MD_engdata(f, "atomprm", MD_ATOMPRM, aw)) == NULL
      || (eng->bondprm = MD_engdata(f, "bondprm", MD_BONDPRM, aw)) == NULL
      || (eng->angleprm = MD_engdata(f, "angleprm", MD_ANGLEPRM, aw)) == NULL
      || (eng->dihedprm = MD_engdata(f, "dihedprm", MD_TORSPRM, aw)) == NULL
      || (eng->imprprm = MD_engdata(f, "imprprm", MD_TORSPRM, aw)) == NULL
      || (eng->nbfixprm = MD_engdata(f, "nbfixprm", MD_NBFIXPRM, aw)) == NULL
      || (eng->atom = MD_engdata(f, "atom", MD_ATOM, aw)) == NULL
      || (eng->bond = MD_engdata(f, "bond", MD_BOND, aw)) == NULL
      || (eng->angle = MD_engdata(f, "angle", MD_ANGLE, aw)) == NULL
      || (eng->dihed = MD_engdata(f, "dihed", MD_TORS, aw)) == NULL
      || (eng->impr = MD_engdata(f, "impr", MD_TORS, aw)) == NULL
      || (eng->excl = MD_engdata(f, "excl", MD_EXCL, aw)) == NULL
      /* pos and vel allow read, write, and callback read */
      || (eng->pos = MD_engdata(f, "pos", MD_DVEC, arwc)) == NULL
      || (eng->vel = MD_engdata(f, "vel", MD_DVEC, arwc)) == NULL
      /* engine resized force array allows read and callback read */
      || (eng->force_engdata = MD_engdata(f, "force", MD_DVEC, aerc)) == NULL
      /* result and param point to pre-allocated memory */
      || (eng->result_engdata = MD_engdata_buffer(f, "result", attr_result,
          &(eng->result))) == NULL
      || (eng->param_engdata = MD_engdata_buffer(f, "param", attr_param,
          &(eng->param))) == NULL
      /* additional sim params */
      || (eng->timestep_engdata = MD_engdata_buffer(f, "timestep",
          attr_timestep, &(eng->timestep))) == NULL
      || (eng->init_temp_engdata = MD_engdata_buffer(f, "temperature",
          attr_init_temp, &(eng->init_temp))) == NULL
      || (eng->com_motion_engdata = MD_engdata_buffer(f, "commotion",
          attr_com_motion, &(eng->com_motion))) == NULL
      || (eng->seed_engdata = MD_engdata_buffer(f, "seed",
          attr_seed, &(eng->seed))) == NULL
      /* additional results */
      || (eng->numdegreesfreedom = MD_engdata_buffer(f, "numdegreesfreedom",
          attr_ndegfreedom, &(eng->ndegfreedom))) == NULL
    /*** data arrays for mgrid - permit only engine resize ***/
      || (eng->poswrap = MD_engdata(f, "poswrap", MD_DVEC, ae)) == NULL
      || (eng->f_elec = MD_engdata(f, "f_elec", MD_DVEC, ae)) == NULL
      || (eng->charge = MD_engdata(f, "charge", MD_DOUBLE, ae)) == NULL
    /*** for debugging electrostatics ***/
   /* || (eng->force_elec = MD_engdata(f, "force_elec", MD_DVEC, ae)) == NULL */
    /*** data arrays for computing potentials ***/
      || (eng->u_bond = MD_engdata(f, "u_bond",  MD_DOUBLE, arc)) == NULL
      || (eng->u_angle = MD_engdata(f, "u_angle", MD_DOUBLE, arc)) == NULL
      || (eng->u_dihed = MD_engdata(f, "u_dihed", MD_DOUBLE, arc)) == NULL
      || (eng->u_impr = MD_engdata(f, "u_impr", MD_DOUBLE, arc)) == NULL
      || (eng->u_elec = MD_engdata(f, "u_elec", MD_DOUBLE, arc)) == NULL
      || (eng->u_vdw = MD_engdata(f, "u_vdw", MD_DOUBLE, arc)) == NULL
      || (eng->u_bound = MD_engdata(f, "u_bound", MD_DOUBLE, arc)) == NULL
      || (eng->pot_elec = MD_engdata(f, "pot_elec", MD_DOUBLE, arc)) == NULL
    /*** data array for fixed atoms, allow write and resize ***/
      || (eng->fixedatomslist =
          MD_engdata(f, "fixedatomslist", MD_INT32, aw)) == NULL
    /*** data array for constraints, allow write and resize ***/
      || (eng->constraintlist = MD_engdata(f, "constraintlist", type_constraint,
          aw)) == NULL
      ) {
    return MD_FAIL;
  }

  /* setup run routine with interface */
  if (MD_setup_run(f, deven_run)) return MD_FAIL;

  /* init mgrid library */
  if (mgrid_init(&(eng->mgrid))) return MD_FAIL;

  /*** force library is initialized later ***/

  /* create Step object */
  if (NULL==(eng->step = new_Step())) return MD_FAIL;

  /*** pmetest library is initialized later ***/

  printf("# deven started successfully\n");

  return 0;
}


void deven_done(MD_Front *f)
{
  Engine *eng = MD_engine_data(f);

  /* destroy timer object if it was ever created */
  if (eng->timer) {
    timer_destroy(eng->timer);
  }

  free(eng->scaled_mass);
  free(eng->init_pos);

  /* done with pmetest library */
  if (eng->pmetest) {
    pmetest_destroy(eng->pmetest);
  }

  /* done with mgrid library */
  mgrid_done(&(eng->mgrid));

  /* done with force library */
  force_destroy(eng->force);

  /* done with step library */
  step_cleanup(eng->step);
  delete_Step(eng->step);

  /* free memory allocated during init */
  free(eng);

  printf("# deven stopped successfully\n");
}
