/* ------------------------------------------- */
/*
 *   Author: James Phillips
 *           Department of Physics
 *           Marquette University
 *           Milwaukee, WI 53233
 *
 *   Purpose: Explorer module which reads the
 *            CIA world map database into a
 *            pyramid for display with the
 *            Explorer visualization system.
 *
 *   Written for the David A. Yuen research group
 *   at the Minnesota Supercomputer Institute and
 *   Department of Geology and Geophysics, 
 *   University of Minnesota.  Summer, 1992.
 *
 */
/* ------------------------------------------- */

#include <cx/DataTypes.h>
#include <cx/DataAccess.h>
#include <cx/UserFuncs.h>
#include <string.h>
#include <math.h>
#include "util.h"
#define max(x,y) (((x)>(y))?(x):(y))
#define min(x,y) (((x)<(y))?(x):(y))
#define moduleAlert cxModAlert

#define NCONTINENTS 5

typedef struct SegDict {
    unsigned long	segid;
    long		maxlat;
    long		minlat;
    long		maxlong;
    long		minlong;
    size_t		absaddr;
    size_t		nbytes;
    long		rank;
} SegDict;

typedef struct PolyLine {
   struct PolyLine *prev;
   struct PolyLine *next;
   long      npoints;
   long     *line;
} PolyLine;

int ReadInMap(FILE *mfile, long xlo, long xhi, 
	     long ylo, long yhi, long xshift, long precision,
             int *linesOn, PolyLine *mapRec[]);

int ReadInPlates(FILE *mfile, long xlo, long xhi, 
	     long ylo, long yhi, long xshift, long precision,
             int *linesOn, PolyLine *mapRec[]);

int GenLatt(long xlo, long xhi, 
	     long ylo, long yhi,
             long parLevel,
             PolyLine **mapRec);

int GenLong(long xlo, long xhi, 
	     long ylo, long yhi,
             long meridLevel,
             PolyLine **mapRec);

void decimate(PolyLine *lineRec, long precision);

void mapMemRelease(PolyLine *mapRec[5][20]);

#ifdef __cplusplus
	extern "C" {
#endif

/*  Main function called by module */

void makemap  (
	 unsigned char   *mapDir, 
	 long   	  bdyLevel, 
	 long   	  cilLevel, 
	 long   	  reefLevel, 
	 long   	  parLevel, 
	 long   	  iceLevel, 
	 long   	  meridLevel, 
	 long   	  riverLevel, 
	 long   	  canalLevel, 
         cxPyramid    * * pyrout,
         long             precision,
         cxParameter    * xmin,
         cxParameter    * xmax,
         cxParameter    * ymin,
         cxParameter    * ymax,
         long             pbyLevel,
         long             plateLevel)
{

   int g,h,i,j,k,flag,outPort,inPort;
   long maxplines,maxpoints,npoints,nsegments,m,n,p;
   double xlo,xhi,ylo,yhi,oxlo,oxhi,orxlo,orxhi,xshift;
   char mapFile[4][NCONTINENTS][80];
   char platesFile[80];
   int linesOn[5][20];
   long dims[2];
   FILE	*mfile;
   cxPyramid *map;
   cxLattice *segments;
   cxLattice *points;
   float *coords;
   float *colors;
   cxConnection *segmGrp;
   PolyLine *mapRec[5][20];
   float mapColor[5][20];
   PolyLine *this;
   PolyLine *last;
   long *line;
   long *cons;
   long *grps;
   long *preptr;
   int errflag;
   long *elements;
   long *connections;

   /* Zero output port. */

   *pyrout = NULL;

   /* Check if user has typed in a map directory. */

   if (!strcmp(mapDir,""))
   {
      return;
   }

   /* Get data for cropping. */

   xlo = -180*3600;
   xhi = 180*3600;
   ylo = -90*3600;
   yhi = 90*3600;

   if( xmin ) xlo = (long)(cxParamDblGet(xmin)*3600.0);
   if( xmax ) xhi = (long)(cxParamDblGet(xmax)*3600.0);
   if( ymin ) ylo = (long)(cxParamDblGet(ymin)*3600.0);
   if( ymax ) yhi = (long)(cxParamDblGet(ymax)*3600.0);

   if( xlo>xhi || ylo>yhi ) return;

   if( ylo < (-90*3600) ) {
      cxModAlert("Minimum lattitude is -90.");
      return;
   }
   if( yhi > (90*3600) ) {
      cxModAlert("Maximum lattitude is 90.");
      return;
   }

   printf("Cropping: %f, %f, %f, %f\n",(xlo)/3600.0,(xhi)/3600.0,ylo/3600.0,yhi/3600.0);

   /* Check data for decimation function. */

   if ( precision < 1 ) {
      cxModAlert("Precision is in seconds and must be given.");
      return;
   }

   /* Build pathnames for all maps. */

   for(i=0;i<4;i++)
   for(j=0;j<NCONTINENTS;j++)
   {
      strcpy(mapFile[i][j],mapDir);
   }
   for(i=0;i<4;i++)
   {
      strcat(mapFile[i][0],"/africa.Map");
      strcat(mapFile[i][1],"/asia.Map");
      strcat(mapFile[i][2],"/europe.Map");
      strcat(mapFile[i][3],"/namer.Map");
      strcat(mapFile[i][4],"/samer.Map");
   }
   for(j=0;j<NCONTINENTS;j++)
   {
      strcat(mapFile[0][j],"/bdy.cbd");
      strcat(mapFile[1][j],"/pby.cbd");
      strcat(mapFile[2][j],"/cil.cbd");
      strcat(mapFile[3][j],"/riv.cbd");
   }

   strcpy(platesFile,mapDir);
   strcat(platesFile,"/USGS_plates.txt");

   /* Use parameters to set which features are needed. */

   for(i=0;i<5;i++)for(j=0;j<20;j++) linesOn[i][j] = 0;

   switch((int)bdyLevel) {
      case 3:  
                  linesOn[0][3] = 1;
      case 2:  
                  linesOn[0][2] = 1;
      case 1:  
                  linesOn[0][1] = 1;
   }
   switch((int)pbyLevel) {
      case 3:  
                  linesOn[1][3] = 1;
      case 2:  
                  linesOn[1][2] = 1;
      case 1:  
                  linesOn[1][1] = 1;
   }   
   switch((int)cilLevel) {
      case 6:  
                  linesOn[2][7] = 1;
      case 5:  
                  linesOn[2][6] = 1;
      case 4:  
                  linesOn[2][4] = 1;
      case 3:  
                  linesOn[2][3] = 1;
      case 2:  
                  linesOn[2][2] = 1;
      case 1:  
                  linesOn[2][1] = 1;
   }
   switch((int)reefLevel) {
      case 3:  
                  linesOn[2][8] = 1;
                  linesOn[2][9] = 1;
                  linesOn[2][10] = 1;
                  break;
      case 2:  
                  linesOn[2][9] = 1;
                  linesOn[2][10] = 1;
                  break;
      case 1:  
                  linesOn[2][8] = 1;
                  break;
   }
   switch((int)iceLevel) {
      case 3:  
                 linesOn[2][15] = 1;
                 linesOn[2][14] = 1;
                 linesOn[2][13] = 1;
      case 2:  
                 linesOn[2][14] = 1;
                 linesOn[2][13] = 1;
      case 1:  
                 linesOn[2][15] = 1;
   }
  switch((int)riverLevel) {
      case 8:  
                  linesOn[3][8] = 1;
      case 7:  
                  linesOn[3][7] = 1;
      case 6:  
                  linesOn[3][6] = 1;
      case 5:  
                  linesOn[3][5] = 1;
      case 4:  
                  linesOn[3][4] = 1;
      case 3:  
                  linesOn[3][3] = 1;
      case 2:  
                  linesOn[3][2] = 1;
      case 1:  
                  linesOn[3][1] = 1;
   }
   switch((int)canalLevel) {
      case 3:  
                 linesOn[3][12] = 1;
      case 2:  
                 linesOn[3][11] = 1;
      case 1:  
                 linesOn[3][10] = 1;
   }
   switch((int)plateLevel) {
      case 1:  
                 linesOn[4][1] = 1;
   }

   /* Set numbers for colormap. */

   for(i=0;i<5;i++)
      for(j=0;j<20;j++)
         mapColor[i][j] = (float)(20*i + j);

   /* Zero the matrix of PolyLine structures. */
   
   for(i=0;i<5;i++)
      for(j=0;j<20;j++)
         mapRec[i][j] = NULL;

   /* Loop over longitude. */

   orxlo = oxlo = xlo;
   orxhi = oxhi = xhi;
   xshift = 0;
   
   while(oxlo>(540*3600)) {
      oxlo -= 360*3600;
      oxhi -= 360*3600;
      xshift += 360*3600;
   }
   while(oxlo<(180*3600)) {
      oxlo += 360*3600;
      oxhi += 360*3600;
      xshift -= 360*3600;
   }

   while(oxhi>(180*3600)) {
      oxlo -= 360*3600;
      oxhi -= 360*3600;
      xshift += 360*3600;
      xlo = max(oxlo,-180*3600);
      xhi = min(oxhi,180*3600);

   /* Read in USGS plate boundary map. */

   if(linesOn[4][1]) {
      mfile = fopen(platesFile,"r");
      if (! mfile) {
         moduleAlert("Can't open plates file!\nPlace file USGS_plates.txt in CIA Map directory and try again.");
         return;
      }
      printf("%s%s.\n","Now reading file ",platesFile);
      errflag =
         ReadInPlates(mfile,xlo,xhi,ylo,yhi,xshift,precision,linesOn[4],mapRec[4]);
      fclose(mfile);
      if ( errflag ) {
         mapMemRelease(mapRec);
         return;
      }
   }

   /* Read in  CIA map files. */

   for(i=0;i<4;i++) {
      flag = 0;
      for(k=0;k<20;k++) { flag += linesOn[i][k]; }
      if ( flag ) for(j=(i!=1?0:3);j<(i!=1?NCONTINENTS:4);j++)
      {
         mfile = fopen(mapFile[i][j],"r");
         if (! mfile) {
            moduleAlert("Can't open map file!");
            mapMemRelease(mapRec);
            return;
         }
         printf("%s%s.\n","Now reading file ",mapFile[i][j]);
         errflag = 
            ReadInMap(mfile,xlo,xhi,ylo,yhi,xshift,precision,linesOn[i],mapRec[i]);
         fclose(mfile);
         if ( errflag ) {
            mapMemRelease(mapRec);
            return;
         }
      }
   }

   }

/* End loop. */

   /* Add parallels and meridians. */

   if(parLevel) GenLatt(orxlo,orxhi,ylo,yhi,
                  parLevel,&mapRec[4][19]);
   if(meridLevel) GenLong(orxlo,orxhi,ylo,yhi,
                  meridLevel,&mapRec[4][19]);

   printf("Done reading map files.\n");

   /* Count segments, points, PolyLines, etc. */

   maxplines = 0;
   maxpoints = 0;
   npoints = 0;
   nsegments = 0;
   for(i=0;i<5;i++)
      for(j=0;j<20;j++) {
         m = 0;
         this = mapRec[i][j];
         while ( this ) {
            m++;
            npoints += this->npoints;
            nsegments += this->npoints - 1;
            maxpoints = max(maxpoints,this->npoints);
            this = this->next;
         }
         maxplines = max(maxplines,m);
      }

   /* Check for no output. */
   if ( ! npoints ) return;

   printf("%ld points read in.\n",npoints);
   printf("%ld segments read in.\n",npoints-nsegments);

   /* Allocate lattice with coordinates and data for base. */
   dims[0] = npoints;
   points = cxLatNew(1,dims,1,cx_prim_float,3,cx_coord_curvilinear);
   /* Get pointer to coordinate and data arrays. */
   cxLatPtrGet(points,NULL,&colors,NULL,&coords);
   /* Allocate connections for 1st. layer. */
   segmGrp = cxConnNew(nsegments,2*nsegments);
   /* Check if any of the above failed. */
   if (cxDataAllocErrorGet()) {
      moduleAlert("Out of shared memory!");
      mapMemRelease(mapRec);
      cxDataRefDec(points);
      cxDataRefDec(segmGrp);
      return;
   }
   /* Get elements and connection lists. */
   cxConnPtrGet(segmGrp,NULL,NULL,&elements,&connections);
   elements[nsegments] = 2*nsegments;

   /* Transfer PolyLines to Explorer data types. */

   n = 0;
   m = -2;
   for(i=0;i<5;i++)
      for(j=0;j<20;j++) {
         last = NULL;
         this = mapRec[i][j];
         while ( this ) {
            line = this->line;
            for(k=0;k<(this->npoints);k++) {
               coords[0] = ((float)line[0]) / 3600.0;
               coords[1] = ((float)line[1]) / 3600.0;
               coords[2] = 0.0;
               colors[0] = mapColor[i][j];
               coords += 3;
               line += 2;
               colors++;
               m++;
               if ( k > 0 ) {
                  elements[n] = 2*n;
                  connections[2*n] = m;
                  connections[2*n+1] = m+1;
                  n++;
               }
            }
            free(this->line);
            last = this;
            this = this->next;
         }
         while ( last ) {
            free(last->next);
            this = last;
            last = this->prev;
         }
         free(this);
      }

   /* Allocate pyramid. */
   map = cxPyrNew(1);
   if (cxDataAllocErrorGet()) {
      moduleAlert("Out of shared memory!");
      cxDataRefDec(points);
      cxDataRefDec(segmGrp);
      return;
   }
   /* Set base lattice. */
   cxPyrSet(map,points);
   /* Allocate empty lattice for 1st. layer. */
   dims[0] = nsegments;
   segments = cxLatRootNew(1,dims);
   if (cxDataAllocErrorGet()) {
      moduleAlert("Out of shared memory!");
      cxDataRefDec(segmGrp);
      cxDataRefDec(map);
      return;
   }
   /* Set 1st. layer connections and lattice. */
   cxPyrLayerSet(map,1,segmGrp,segments);
   if (cxDataAllocErrorGet()) {
      moduleAlert("Out of shared memory!");
      cxDataRefDec(segmGrp);
      cxDataRefDec(map);
      cxDataRefDec(segments);
      return;
   }

/* Send pyramid to output port and return. */

   *pyrout = map;
   return;

}

#ifdef __cplusplus
}
#endif


/* ------------------------------------------- */

/* Portions adapted from the following source. */

/* ------------------------------------------- */

/*
 * Author:	Kenneth Chin-Purcell, AHPCRC
 *		Copyright 1990, 1991  Minnesota Supercomputer Center, Inc.
 *
 * Purpose:	extract useful line lists from CIA database
 *
 * RESTRICTED RIGHTS LEGEND
 *
 * Use, duplication, or disclosure of this software and its documentation
 * by the Government is subject to restrictions as set forth in subdivision
 * { (b) (3) (ii) } of the Rights in Technical Data and Computer Software
 * clause at 52.227-7013.
 */



static unsigned long ReadMapWord(FILE *file)
{
    unsigned long	l;

    Verify(fread(&l, 4, 1, file) == 1, "Short ReadMapWord");
    return ((0x000000FF & l) << 24 | (0x0000FF00 & l) << 8  |
	    (0x00FF0000 & l) >> 8  | (0xFF000000 & l) >> 24);
}

static unsigned short ReadMapShort(FILE *file)
{
    unsigned short	s;

    Verify(fread(&s, 2, 1, file) == 1, "Short ReadMapShort");
    return ((0x00FF & s) << 8 | (0xFF00 & s) >> 8);
}

static long ReadMapLong(FILE *file)
{
    unsigned long	l;

    l = ReadMapWord(file);
    if (l & 0x80000000)
	return -(~l);
    else
	return l;
}

static void NextPoint(FILE *mfile, long *ixp, long *iyp)
{
            long ix,iy;
            unsigned short ls;

	    ls = ReadMapShort(mfile);
	    if (ls & 0x4000) {

		iy = ls & 0xff;
		if (iy & 0x80)
		    iy -= 0x100;	/* extend sign */
		ix = ls >> 8;
		if (ix & 0x80)
		    ix -= 0x100;	/* extend sign */
		else
		    ix -= 0x40;

	    } else {

		if (ls & 0x8000)
		    ix = (long)(ls | 0x4000) - 0x10000;
		else
		    ix = ls;
		ix = 0x10000 * ix + ReadMapShort(mfile);

		ls = ReadMapShort(mfile);
		if (ls & 0x8000)
		    iy = (long)(ls | 0x4000) - 0x10000;
		else
		    iy = ls;
		iy = 0x10000 * iy + ReadMapShort(mfile);
	    }
            *ixp = ix;
            *iyp = iy;
            return;
}



#define CBD_MAGIC 0x20770002

int ReadInMap(FILE *mfile, long xlo, long xhi, 
              long ylo, long yhi, long xshift, long precision,
              int *linesOn, PolyLine *mapRec[])
{
    int		i;
    unsigned long	magic;		/* Magic number */
    size_t		dictaddr;	/* Offset of segment dictionary */
    size_t		segcount;	/* Number of segments in file */
    size_t		segsize;	/* Size of segment dictionary */
    size_t		segmax;		/* Size of max segment's strokes */
    SegDict		*mdict;
    long		id;
    size_t		nstrokes;
    long                xpos,xold;
    long                ypos,yold;
    long                xtmp,ytmp;

    rewind(mfile);
    magic    = ReadMapWord(mfile);
    dictaddr = ReadMapWord(mfile);
    segcount = ReadMapWord(mfile);
    segsize  = ReadMapWord(mfile);
    segmax   = ReadMapWord(mfile);
    
    if ( ! ( magic == CBD_MAGIC ) ) {
       moduleAlert("File has bad magic number!");
       return 2;
    }
    
    if ( ! ( mdict = CallocType(SegDict, segcount) ) ) {
       moduleAlert("Out of memory!");
       return 1;
    }
    
    if ( ! ( fseek(mfile, dictaddr, SEEK_SET) == 0 ) ) {
       moduleAlert("Bad seek!");
       Free(mdict);
       return 2;
    }
    
    for (i = 0; i < segcount; ++i) {
	SegDict	*dict = mdict + i;

	dict->segid   = ReadMapWord(mfile);
	dict->maxlat  = ReadMapLong(mfile);
	dict->minlat  = ReadMapLong(mfile);
	dict->maxlong = ReadMapLong(mfile);
	dict->minlong = ReadMapLong(mfile);
	dict->absaddr = ReadMapWord(mfile);
	dict->nbytes  = ReadMapShort(mfile);
	dict->rank    = ReadMapShort(mfile);
    }
    
    for (i = 0; i < segcount; ++i) {
	SegDict		*dict = mdict + i;
	long		ix, iy;
	int		fits;
	int		j = 0;
        long        npoints;
        long       *line;
        PolyLine   *this;


	fits = (dict->minlat  < yhi  &&  dict->maxlat  > ylo  &&
		dict->minlong < xhi  &&  dict->maxlong > xlo);

	if (fits && linesOn[dict->rank]) {
	
            if ( ! ( fseek(mfile, dict->absaddr, SEEK_SET) == 0 ) ) {
               moduleAlert("Bad seek!");
               Free(mdict);
               return 2;
            }
	
	    xpos     = ReadMapLong(mfile);
	    ypos     = ReadMapLong(mfile);
	    id       = ReadMapWord(mfile);
	    nstrokes = ReadMapShort(mfile);
	    (void)     ReadMapShort(mfile);
    
            /* Get a new PolyLine ready for this segment. */

            this = mapRec[dict->rank];
            if ( ! this ) {
               if ( ! ( this = mapRec[dict->rank] = MallocType(PolyLine) ) ) {
                  moduleAlert("Out of memory!");
                  return 1;
               }
               this->next = NULL;
               this->prev = NULL;
               if ( ! ( line = this->line = CallocType(long,2*(nstrokes+1)) ) ) {
                  moduleAlert("Out of memory!");
                  return 1;
               }
            } else {
               while ( this->next ) {
                  this = this->next;
               }
               if ( ! ( this->next = MallocType(PolyLine) ) ) {
                  moduleAlert("Out of memory!");
                  return 1;
               }
               this->next->prev = this;
               this = this->next;
               this->next = NULL;
               if ( ! ( line = this->line = CallocType(long,2*(nstrokes+1)) ) ) {
                  moduleAlert("Out of memory!");
                  return 1;
               }
            }
            npoints = 0;

            /* Build PolyLine(s) from this segment. */

	    while (j <= nstrokes) {

                /* As long as the point fits, add it. */

		while (ypos <= yhi  &&  ypos >= ylo  &&
		       xpos <= xhi  &&  xpos >= xlo  &&
		       j <= nstrokes) {
                   line[0] = ( xold = xpos ) + xshift;
                   line[1] = yold = ypos;
                   line += 2;
                   npoints++;
                   NextPoint(mfile,&ix,&iy);
                   xpos += ix;
                   ypos += iy;
                   if ( xpos > 90*3600 && xold < -90*3600 ) { xpos -= 360*3600; }
                   if ( xpos < -90*3600 && xold > 90*3600 ) { xpos += 360*3600; }
		   ++j;
		}

                /* Still more points, so one didn't fit. */
                /* At least one point on line so far. */

		if (npoints > 0 && j <= nstrokes) {
                   xtmp = xpos;
                   ytmp = ypos;
                   /* Chop the end of this line to fit. */
                   if (ypos > yhi) {
                      xpos = xold + (float)(xpos-xold)*(float)(yhi-yold)/(float)(ypos-yold);
                      ypos = yhi; }
                   if (ypos < ylo) {
                      xpos = xold + (float)(xpos-xold)*(float)(yold-ylo)/(float)(yold-ypos);
                      ypos = ylo; }
                   if (xpos > xhi) {
                      ypos = yold + (float)(ypos-yold)*(float)(xhi-xold)/(float)(xpos-xold);
                      xpos = xhi; }
                   if (xpos < xlo) {
                      ypos = yold + (float)(ypos-yold)*(float)(xold-xlo)/(float)(xold-xpos);
                      xpos = xlo; }
                   /* Add it to the line. */
                   line[0] = xpos + xshift;
                   line[1] = ypos;
                   xpos = xtmp;
                   ypos = ytmp;
                   if ( xpos > 180*3600 ) { xpos -= 360*3600; }
                   if ( xpos < -180*3600 ) { xpos += 360*3600; }
                   npoints++;
                   /* Finish off the line. */
                   this->npoints = npoints;
                   decimate(this,precision);
                   /* Start a new PolyLine. */
                   if ( ! ( this->next = MallocType(PolyLine) ) ) {
                      moduleAlert("Out of memory!");
                      return 1;
                   }
                   this->next->prev = this;
                   this = this->next;
                   this->next = NULL;
                   if ( ! ( line = this->line = CallocType(long,2*(nstrokes+1)) ) ) {
                      moduleAlert("Out of memory!");
                      return 1;
                   }
                   npoints = 0;
		}

                /* Go through points until one fits. */

		while ((j <= nstrokes) &&
		       (ypos > yhi  ||  ypos < ylo  ||
			xpos > xhi  ||  xpos < xlo)) {
                   NextPoint(mfile,&ix,&iy);
                   xold = xpos;
                   yold = ypos;
                   xpos += ix;
                   ypos += iy;
	           ++j;
		}

                /* Still more points, so one fit. */

		if (j <= nstrokes) {
                   if ( xpos > 90*3600 && xold < -90*3600 ) { xold += 360*3600; }
                   if ( xpos < -90*3600 && xold > 90*3600 ) { xold -= 360*3600; }
                   /* Chop the head of this line to fit. */
                   if (yold > yhi) {
                      xold = xold - (float)(xpos-xold)*(float)(yhi-yold)/(float)(yold-ypos);
                      yold = yhi; }
                   if (yold < ylo) {
                      xold = xold - (float)(xpos-xold)*(float)(yold-ylo)/(float)(ypos-yold);
                      yold = ylo; }
                   if (xold > xhi) {
                      yold = yold - (float)(ypos-yold)*(float)(xhi-xold)/(float)(xold-xpos);
                      xold = xhi; }
                   if (xold < xlo) {
                      yold = yold - (float)(ypos-yold)*(float)(xold-xlo)/(float)(xpos-xold);
                      xold = xlo; }
                   /* Add it to the line. */
                   line[0] = xold + xshift;
                   line[1] = yold;
                   line += 2;
                   npoints++;
		}

                /* Out of points, finish the line or destroy it. */

                if (j > nstrokes && npoints > 1) {
                   this->npoints = npoints;
                   decimate(this,precision);
                }
                else if (j > nstrokes) {
                   Free(this->line);
                   if ( this->prev ) this->prev->next = NULL;
                   else mapRec[dict->rank] = NULL;
                   Free(this);
                }

           }

        }
    }
    Free(mdict);
    return 0;
}

int ReadInPlates(FILE *mfile, long xlo, long xhi, 
              long ylo, long yhi, long xshift, long precision,
              int *linesOn, PolyLine *mapRec[])
{
    int		        err,i,j;
    int		        nstrokes;
    long                npoints;
    long                xpos,xold;
    long                ypos,yold;
    long                xtmp,ytmp;
    float               xfl,yfl;
    long       *line;
    PolyLine   *this;

    rewind(mfile);
    
    err = fscanf(mfile,"%f%f",&xfl,&yfl);

    while ( err == 2 ) {

        nstrokes = (int)yfl - 1;
	
	err = fscanf(mfile,"%f%f",&xfl,&yfl);
        xpos     = (long)(xfl*3600);
        ypos     = (long)(yfl*3600);

	j = 0;

            /* Get a new PolyLine ready for this segment. */

            this = mapRec[1];
            if ( ! this ) {
               if ( ! ( this = mapRec[1] = MallocType(PolyLine) ) ) {
                  moduleAlert("Out of memory!");
                  return 1;
               }
               this->next = NULL;
               this->prev = NULL;
               if ( ! ( line = this->line = CallocType(long,2*(nstrokes+1)) ) ) {
                  moduleAlert("Out of memory!");
                  return 1;
               }
            } else {
               while ( this->next ) {
                  this = this->next;
               }
               if ( ! ( this->next = MallocType(PolyLine) ) ) {
                  moduleAlert("Out of memory!");
                  return 1;
               }
               this->next->prev = this;
               this = this->next;
               this->next = NULL;
               if ( ! ( line = this->line = CallocType(long,2*(nstrokes+1)) ) ) {
                  moduleAlert("Out of memory!");
                  return 1;
               }
            }
            npoints = 0;

            /* Build PolyLine(s) from this segment. */

	    while (j <= nstrokes) {

                /* As long as the point fits, add it. */

		while (ypos <= yhi  &&  ypos >= ylo  &&
		       xpos <= xhi  &&  xpos >= xlo  &&
		       j <= nstrokes) {
                   line[0] = ( xold = xpos ) + xshift;
                   line[1] = yold = ypos;
                   line += 2;
                   npoints++;
	           err = fscanf(mfile,"%f%f",&xfl,&yfl);
                   xpos     = (long)(xfl*3600);
                   ypos     = (long)(yfl*3600);
                   if ( xpos > 90*3600 && xold < -90*3600 ) { xpos -= 360*3600; }
                   if ( xpos < -90*3600 && xold > 90*3600 ) { xpos += 360*3600; }
		   ++j;
		}

                /* Still more points, so one didn't fit. */
                /* At least one point on line so far. */

		if (npoints > 0 && j <= nstrokes) {
                   xtmp = xpos;
                   ytmp = ypos;
                   /* Chop the end of this line to fit. */
                   if (ypos > yhi) {
                      xpos = xold + (float)(xpos-xold)*(float)(yhi-yold)/(float)(ypos-yold);
                      ypos = yhi; }
                   if (ypos < ylo) {
                      xpos = xold + (float)(xpos-xold)*(float)(yold-ylo)/(float)(yold-ypos);
                      ypos = ylo; }
                   if (xpos > xhi) {
                      ypos = yold + (float)(ypos-yold)*(float)(xhi-xold)/(float)(xpos-xold);
                      xpos = xhi; }
                   if (xpos < xlo) {
                      ypos = yold + (float)(ypos-yold)*(float)(xold-xlo)/(float)(xold-xpos);
                      xpos = xlo; }
                   /* Add it to the line. */
                   line[0] = xpos + xshift;
                   line[1] = ypos;
                   xpos = xtmp;
                   ypos = ytmp;
                   if ( xpos > 180*3600 ) { xpos -= 360*3600; }
                   if ( xpos < -180*3600 ) { xpos += 360*3600; }
                   npoints++;
                   /* Finish off the line. */
                   this->npoints = npoints;
                   decimate(this,precision);
                   /* Start a new PolyLine. */
                   if ( ! ( this->next = MallocType(PolyLine) ) ) {
                      moduleAlert("Out of memory!");
                      return 1;
                   }
                   this->next->prev = this;
                   this = this->next;
                   this->next = NULL;
                   if ( ! ( line = this->line = CallocType(long,2*(nstrokes+1)) ) ) {
                      moduleAlert("Out of memory!");
                      return 1;
                   }
                   npoints = 0;
		}

                /* Go through points until one fits. */

		while ((j <= nstrokes) &&
		       (ypos > yhi  ||  ypos < ylo  ||
			xpos > xhi  ||  xpos < xlo)) {
                   xold = xpos;
                   yold = ypos;
              	   err = fscanf(mfile,"%f%f",&xfl,&yfl);
                   xpos     = (long)(xfl*3600);
                   ypos     = (long)(yfl*3600);
	           ++j;
		}

                /* Still more points, so one fit. */

		if (j <= nstrokes) {
                   /* Chop the head of this line to fit. */
                   if ( xpos > 90*3600 && xold < -90*3600 ) { xold += 360*3600; }
                   if ( xpos < -90*3600 && xold > 90*3600 ) { xold -= 360*3600; }
                   if (yold > yhi) {
                      xold = xold - (float)(xpos-xold)*(float)(yhi-yold)/(float)(yold-ypos);
                      yold = yhi; }
                   if (yold < ylo) {
                      xold = xold - (float)(xpos-xold)*(float)(yold-ylo)/(float)(ypos-yold);
                      yold = ylo; }
                   if (xold > xhi) {
                      yold = yold - (float)(ypos-yold)*(float)(xhi-xold)/(float)(xold-xpos);
                      xold = xhi; }
                   if (xold < xlo) {
                      yold = yold - (float)(ypos-yold)*(float)(xold-xlo)/(float)(xpos-xold);
                      xold = xlo; }
                   /* Add it to the line. */
                   line[0] = xold + xshift;
                   line[1] = yold;
                   line += 2;
                   npoints++;
		}

                /* Out of points, finish the line or destroy it. */

                if (j > nstrokes && npoints > 1) {
                   this->npoints = npoints;
                   decimate(this,precision);
                }
                else if (j > nstrokes) {
                   Free(this->line);
                   if ( this->prev ) this->prev->next = NULL;
                   else mapRec[1] = NULL;
                   Free(this);
                }

           }
    }
    return 0;
}

#define square(x) ((x)*(x))

void decimate(PolyLine *lineRec, long precision)
{

   long *line;
   long *newline;
   long i,npoints,start,needed,last;
   line = lineRec->line;
   npoints = lineRec->npoints;
   needed = 1;
   start = 0;

   for(i=1;i<npoints;i++) {
      if( square(line[2*i] - line[2*start]) +
          square(line[2*i+1]-line[2*start+1]) > square(precision) ) {
         if( start != i-1 ) i--;
         line[2*needed] = line[2*i];
         line[2*needed+1] = line[2*i+1];
         needed++;
         start = i;
      }
   }

   if( start != npoints-1 ) {
            line[2*needed] = line[2*(npoints-1)];
            line[2*needed+1] = line[2*(npoints-1)+1];
            needed++;
   }

   lineRec->npoints = needed;
   if ( ! ( newline = ReallocType(line,long,2*needed) ) ) return;
   lineRec->line = newline;
   return;
}

int GenLatt(long xlo, long xhi, 
	     long ylo, long yhi,
             long parLevel,
             PolyLine **mapRec)
{

   long positions[181];
   long pos;
   long *line;
   long i,j,k,npos,npoints,nstrokes;
   PolyLine *this;

   /* Decide which lines are needed. */

   switch( (int)parLevel ) {
      case 1: /* Equator */
         npos = 0;
         for(pos= -90;pos<=90;pos+=90*3600)
            if(ylo<=pos&&yhi>=pos) {
               positions[npos] = pos;
               npos ++;
            }
         break;
      case 2: /* Tropics */
         npos = 0;
         for(pos= -90*3600;pos<=90*3600;pos+=45*3600)
            if(ylo<=pos&&yhi>=pos) {
               positions[npos] = pos;
               npos ++;
            }
         break;
      case 3: /* Every 20 */
         npos = 0;
            if(ylo<=-90&&yhi>=-90) {
               positions[npos] = -90;
               npos ++;
            }
         for(pos= -80*3600;pos<=80*3600;pos+=20*3600)
            if(ylo<=pos&&yhi>=pos) {
               positions[npos] = pos;
               npos ++;
            }
            if(ylo<=90&&yhi>=90) {
               positions[npos] = 90;
               npos ++;
            }
         break;
      case 4: /* Every 10 */
         npos = 0;
         for(pos= -90*3600;pos<=90*3600;pos+=10*3600)
            if(ylo<=pos&&yhi>=pos) {
               positions[npos] = pos;
               npos ++;
            }
         break;
      case 5: /* Every 5 */
         npos = 0;
         for(pos= -90*3600;pos<=90*3600;pos+=5*3600)
            if(ylo<=pos&&yhi>=pos) {
               positions[npos] = pos;
               npos ++;
            }
         break;
      case 6: /* Every 1 */
         npos = 0;
         for(pos= -90*3600;pos<=90*3600;pos+=1*3600)
            if(ylo<=pos&&yhi>=pos) {
               positions[npos] = pos;
               npos ++;
            }
         break;
   }

   /* Figure out how many segments each line will have. */

   nstrokes = 0;
   for(pos=xlo;pos<xhi;pos+=3600) nstrokes++;

   /* Fill in lines. */

   this = *mapRec;

   for(i=0;i<npos;i++) {

            if ( ! this ) {
               if ( ! ( this = *mapRec = MallocType(PolyLine) ) ) {
                  moduleAlert("Out of memory!");
                  return 1;
               }
               this->next = NULL;
               this->prev = NULL;
               if ( ! ( line = this->line = CallocType(long,2*(nstrokes+1)) ) ) {
                  moduleAlert("Out of memory!");
                  return 1;
               }
            } else {
               while ( this->next ) {
                  this = this->next;
               }
               if ( ! ( this->next = MallocType(PolyLine) ) ) {
                  moduleAlert("Out of memory!");
                  return 1;
               }
               this->next->prev = this;
               this = this->next;
               this->next = NULL;
               if ( ! ( line = this->line = CallocType(long,2*(nstrokes+1)) ) ) {
                  moduleAlert("Out of memory!");
                  return 1;
               }
            }
            npoints = 0;

            for(pos=xlo;pos<xhi;pos+=3600) {
               line[0] = pos;
               line[1] = positions[i];
               line += 2;
               npoints++;
            }
            line[0] = xhi;
            line[1] = positions[i];
            npoints++;
            this->npoints = npoints;
   }

   return 0;

}

int GenLong(long xlo, long xhi, 
	     long ylo, long yhi,
             long parLevel,
             PolyLine **mapRec)
{

   long *positions;
   long pos;
   long spac;
   long *line;
   long i,j,k,npos,npoints,nstrokes;
   PolyLine *this;

   /* Decide which lines are needed. */

   switch( (int)parLevel ) {
      case 1: /* Prime */
         spac = 360*3600;
         break;
      case 2: /* Every 90 */
         spac = 90*3600;
         break;
      case 3: /* Every 45 */
         spac = 45*3600;
         break;
      case 4: /* Every 20 */
         spac = 20*3600;
         break;
      case 5: /* Every 10 */
         spac = 10*3600;
         break;
      case 6: /* Every 5 */
         spac = 5*3600;
         break;
      case 7: /* Every 1 */
         spac = 1*3600;
         break;
   }

   if ( ! ( positions = (long*)malloc(sizeof(long)*(1+floor(xhi-xlo)/spac)) ) ) {
      moduleAlert("Out of memory!");
      return 1;
   }

   npos = 0;
   for(pos=spac*ceil(xlo/spac);pos<=xhi;pos+=spac) {
      positions[npos] = pos;
      npos ++;
   }

   /* Figure out how many segments each line will have. */

   nstrokes = 0;
   for(pos=ylo;pos<yhi;pos+=3600) nstrokes++;

   /* Fill in lines. */

   this = *mapRec;

   for(i=0;i<npos;i++) {

            if ( ! this ) {
               if ( ! ( this = *mapRec = MallocType(PolyLine) ) ) {
                  moduleAlert("Out of memory!");
                  free(positions);
                  return 1;
               }
               this->next = NULL;
               this->prev = NULL;
               if ( ! ( line = this->line = CallocType(long,2*(nstrokes+1)) ) ) {
                  moduleAlert("Out of memory!");
                  free(positions);
                  return 1;
               }
            } else {
               while ( this->next ) {
                  this = this->next;
               }
               if ( ! ( this->next = MallocType(PolyLine) ) ) {
                  moduleAlert("Out of memory!");
                  free(positions);
                  return 1;
               }
               this->next->prev = this;
               this = this->next;
               this->next = NULL;
               if ( ! ( line = this->line = CallocType(long,2*(nstrokes+1)) ) ) {
                  moduleAlert("Out of memory!");
                  free(positions);
                  return 1;
               }
            }
            npoints = 0;

            for(pos=ylo;pos<yhi;pos+=3600) {
               line[0] = positions[i];
               line[1] = pos;
               line += 2;
               npoints++;
            }
            line[0] = positions[i];
            line[1] = yhi;
            npoints++;
            this->npoints = npoints;
   }

   free(positions);
   return 0;

}

void mapMemRelease(PolyLine *mapRec[5][20])
{
   int i,j;
   PolyLine *last;
   PolyLine *this;

   for(i=0;i<5;i++)
      for(j=0;j<20;j++) {
         last = NULL;
         this = mapRec[i][j];
         while ( this ) {
            free(this->line);
            last = this;
            this = this->next;
         }
         while ( last ) {
            free(last->next);
            this = last;
            last = this->prev;
         }
         free(this);
   }
   return;
}

/* Assorted utility stuff. */

/* 
 * Author:	Kenneth Chin-Purcell, AHPCRC
 *		Copyright 1991, Minnesota Supercomputer Center, Inc.
 *
 * Purpose:	General utilities that use basic C types and UNIX functions.
 *
 * RESTRICTED RIGHTS LEGEND
 *
 * Use, duplication, or disclosure of this software and its documentation
 * by the Government is subject to restrictions as set forth in subdivision
 * { (b) (3) (ii) } of the Rights in Technical Data and Computer Software
 * clause at 52.227-7013.
 *
 */

#include "util.h"

#include <math.h>
#include <sys/types.h>
#include <sys/stat.h>

extern void	abort(void);
extern void	exit(int); 
extern long	atol(char *); 


/* Print a string to standard eror.
 */
void Error(char *errstr)
{
    (void) fprintf(stderr, "%s\n", errstr);
}


/* Terminate program after printing an error message.
 * Use via the macros Verify and MemCheck.
 */
void BailOut(char *errstr, char *fname, int lineno)
{
    char	yesno[256];

    yesno[0] = 0;

    (void) fprintf(stderr, "Error: %s, at %s:%d\n", errstr, fname, lineno);
    (void) fprintf(stderr, "Core dump [y/n] ? ");
    (void) fscanf(stdin, "%s", yesno);
    if ((unsigned int)yesno[0] == 'y')
	(void) abort();
    exit(1);
}


/* Find the size of a file in bytes,
 */
long FindFileSize(char *fname)
{
    struct stat	sbuf;

    if (stat(fname, &sbuf) != 0) {
	Error("Can't stat file");
	return 0;
    }
    return sbuf.st_size;
}


/* Depending on the file length, read a 768 bytes
 * or 1024 ascii integers to create a color map.
 * Bytes ordered	rgbrgbrgb...
 * Integers ordered	index red grn blu index red grn blu ...
 */
void ReadCmap(short cmap[256][3], char *cmapName)
{
    int		ix, r, g, b;
    FILE	*cmapFile;

    cmapFile = fopen(cmapName, "r");
    Verify(cmapFile, "can't open color map file");
    
    if (FindFileSize(cmapName) == 768) {
	for (ix = 0; ix < 256; ix++) {
	    cmap[ix][0] = fgetc(cmapFile);
	    cmap[ix][1] = fgetc(cmapFile);
	    cmap[ix][2] = fgetc(cmapFile);
	}
    } else {
	while (fscanf(cmapFile, "%d%d%d%d", &ix, &r, &g, &b) == 4) {
	    if (ix < 0  ||  ix >= 256) {
		Error("Color map index out of range");
		continue;
	    }
	    cmap[ix][0] = r & 0xff;
	    cmap[ix][1] = g & 0xff;
	    cmap[ix][2] = b & 0xff;
	}
    }
    (void) fclose(cmapFile);
}


/* Create a new string from an old one, tacking on extra bytes
 */
char *NewString(char *old, int extra)
{
    int		slen;
    char	*s;

    if (old)
	slen = strlen(old) + 1 + extra;
    else
	slen = MIN(extra, 1);

    MemCheck(s = CallocType(char, slen));

    if (old)
	(void) strcpy(s, old);
    else
	s[0] = '\0';

    return s;
}


/* Return a comma separated string representation of a long.
 */
void CommaLong(char *s, long i)
{
    long	ai = ABS(i);

    if (ai < 10000)
	(void) sprintf(s, "%ld", i);
    else if (ai < 1000000)
	(void) sprintf(s, "%ld,%03ld", i / 1000, ai % 1000);
    else
	(void) sprintf(s, "%ld,%03ld,%03ld", 
		       i / 1000000, (ai % 1000000) / 1000, ai % 1000);
}


/* Find the largest even divisor of an integer.
 */
long Factor(long a)
{
    long	i;

    for (i = sqrt((double)a); i; i--)
	if (a % i == 0)
	    break;
    return i;
}


char *ReadString(FILE *file)
{
    static char	buf[1024];
    int		nread;

    do {
	nread = fscanf(file, "%s", buf);
	if (nread < 1)
	    return NULL;
	if (buf[0] == '#') {
	    (void) fgets(buf, 1024, file);
	    nread = 0;
	}
    } while (nread < 1);

    return buf;
}


long ReadLong(FILE *file)
{
    char	*s = ReadString(file);

    if (s)
	return atol(s);
    return 0;
}


float ReadFloat(FILE *file)
{
    char	*s = ReadString(file);

    if (s)
	return atof(s);
    return 0.0;
}


char *ReadToken(FILE *file)
{
    char	*s = ReadString(file);

    if (s)
	return s;
    return "";
}


int PositionAtString(char *s, FILE *file)
{
    char	*token;

    rewind(file);
    do {
	token = ReadString(file);
	if (!token)	/* String s not found */
	    return EOF;
    } while (!MATCH(token, s));
    return 0;
}


long *MakeLongArray(long num)
{
    long        *larr = CallocType(long, num);
    MemCheck(larr);
    return larr;
}


float *MakeFloatArray(long num)
{
    float       *farr = CallocType(float, num);
    MemCheck(farr);
    return farr;
}


void ReadFloatArray(float *farr, long num, FILE *file)
{
    int         i;
    for (i = 0; i < num; i++)
        farr[i] = ReadFloat(file);
}


void ReadLongArray(long *larr, long num, FILE *file)
{
    int         i;
    for (i = 0; i < num; i++)
        larr[i] = ReadLong(file);
}


