/*  A 1D circular overdamped langevin dynamics
    Verlet algorithm used for the dynamics
    Last updated on 3/2/2013 by Mahmoud Moradi
*/

#include <stdio.h>
#include <cmath>
#include <time.h>
#include <cassert>
#include <iostream>

using namespace std;
#define boltzman 1.0

// first I define a function to generate a white noise
double randn(double mu, double sigma) {
        static bool deviateAvailable=false;
        static float storedDeviate;
        double polar, rsquared, var1, var2;
       
        //        If no deviate has been stored, the polar Box-Muller transformation is
        //        performed, producing two independent normally-distributed random
        //        deviates.  One is stored for the next round, and one is returned.
        if (!deviateAvailable) {
               
                //        choose pairs of uniformly distributed deviates, discarding those
                //        that don't fall within the unit circle
                do {
                        var1=2.0*( double(rand())/double(RAND_MAX) ) - 1.0;
                        var2=2.0*( double(rand())/double(RAND_MAX) ) - 1.0;
                        rsquared=var1*var1+var2*var2;
                } while ( rsquared>=1.0 || rsquared == 0.0);
               
                //        calculate polar tranformation for each deviate
                polar=sqrt(-2.0*log(rsquared)/rsquared);
               
                //        store first deviate and set flag
                storedDeviate=var1*polar;
                deviateAvailable=true;
               
                //        return second deviate
                return var2*polar*sigma + mu;
        }
       
        //        If a deviate is available from a previous call to this function, it is
        //        returned, and the flag is set to false.
        else {
                deviateAvailable=false;
                return storedDeviate*sigma + mu;
        }
}

// define a function for internal force = - d/dx U(x)
double f(double x) {
    return 0;
}

double colvar(double x, double l) {
    return (x<=l)?(1-x/l):((x-l)/(1-l));
}

double colvar_x(double x, double l) {
    return (x<=l)?(-1/l):(1/(1-l));
}

// my driven time-dependent force is defined here

int main(int argc,char *argv[]) {

    // general parameters    
    double dt = 1e-5;  	// timestep
    double rt = 1;     	// runtime
    double eq = 1;     	// equilibration runtime
    double T = 1;   	// temperature
    double damping = 1; // damping
    int N = 1;         	// number of iterations
    int R = 1;		// report the trajectory every R step

    // circular condition
    double min = 0;    // lower wall for x
    double max = 1;    // upper wall for x
    double x0 = (min+max)/2;   	// initial value of the coordinate
    int p = 0; // pathway 0 or 1

    //  driven dynamics
    double harmonic = 0;// driven harmonic constant
    double l = 0.5;     // colvar parameter
    double l0 = 0;     	// initial value of the colavar
    double l1 = 1;     	// final value of the colvar
    string protocol = "unidirectional"; // protocol uni- or bidirectional

    // reading the parameters
    // only the keywords and their immediately followed values are parsed.
    string key,value;
    while(!std::cin.eof()) {
	std::cin >> key;
	// genral parameters
	if (key=="/iteration") {
	    std::cin >> N;
	} else if (key=="/timestep") {
	    std::cin >> dt;
	} else if (key=="/runtime") {
	    std::cin >> rt;
	} else if (key=="/temperature") {
	    std::cin >> T;
	} else if (key=="/damping") {
	    std::cin >> damping;
	} else if (key=="/report") {
	    std::cin >> R;
	} else if (key=="/equilibration") {
	    std::cin >> eq;
	} else if (key=="/initial") {
	    std::cin >> x0;
	} else if (key=="/min") {
	    std::cin >> min;
	} else if (key=="/max") {
	    std::cin >> max;
	} else if (key=="/harmonic") {
	    std::cin >> harmonic;
	} else if (key=="/protocol") {
	    std::cin >> protocol;
	} else if (key=="/colvar") {
	    std::cin >> l >> l0 >> l1;
	}
    }

    double wfr = (l1-l0)/rt;
    double beta = 1/(boltzman*T);
    double g = sqrt(2/(beta*dt*damping));

    // running the verlet algorithm
    srand ( time(NULL) );
    double t,x,v;
    double F;
    double Fdr;
    double W;
    double L0,L1;
    double w;
    
    x=x0;
    for (int i=1;i<=N;i++) {
	if (protocol=="bidirectional") {
	    if (i%2) {
		L0=l0;
		w=wfr;
    		std::cout << "#F\n";
	    } else {
		L0=l1;
		w=-wfr;
    		std::cout << "#R\n";
	    }
	} else {
	    L0=l0;
	    w=wfr;
	    x=x0;
	}
        t=0;
        while (t<eq) {
	    t+=dt;
	    // my driving force is
	    F = - harmonic * ( colvar(x,l) - L0 );
	    Fdr = F * colvar_x(x,l);
	    v = g * randn(0,1) + (f(x) + Fdr) / damping;
	    x += v * dt;
	    //circular condition
	    if (x<min) x+=max-min;
	    if (x>=max) x-=max-min;
	}
        t=0;
        W=0;
	std::cout << t << " " << x << " " << colvar(x,l) << " " << W << "\n";
        while (t<rt) {
	    t += dt;
	    // driving force
	    F = - harmonic * ( colvar(x,l) - (L0+w*t) );
	    Fdr = F * colvar_x(x,l);
	    v = g * randn(0,1) + (f(x) + Fdr) / damping;
	    x += v * dt;
	    //circular condition
	    if (x<min) x+=max-min;
	    if (x>=max) x-=max-min;
	    
	    // detemine pathway
	    if (int(t/dt)==int(rt/dt)/2) p=(x<=l)?0:1;

	    // measure work
	    W += F * w * dt;
	    // Reporting
	    if (R>=1 && int(t/dt+0.5)%R==0) {
    		std::cout << t << " " << x << " " << colvar(x,l) <<  " " << W << "\n";
	    }
    	}
    	std::cerr << (1+p) << "\n";
    }
}
