/*
    Lowest Free Energy Path (LFEP) Algorithm (Mahmoud Moradi, 8/16/2013)
    Finds the minimum free energy path between the minima in a 2D free energy map.
    The algorithm is based on:
    B. Ensing, A. Laio, M. Parrinello, M. L. Klein, "A recipe for the computation
    of the free energy barrier and the lowest free energy path of concerted reactions."
    J. Phys. Chem. B 2005, 109, 6676-6687
*/

#include <stdio.h>
#include <ctype.h>
#include <cmath>
#include <cassert>
#include <iostream>
#include <iomanip>
#include <fstream>

#define PI 3.14159265

using namespace std;

namespace {

    // Binning variables
    int NX = 64; // number of X bins
    int NY = 64; // number of Y bins
    double Xmin = 0; // range of the coordinates
    double Xmax = 1;
    double Ymin = 0;
    double Ymax = 1;
    double LX; // Xmax-Xmin
    double LY; // Ymax-Ymin
    double dx; // X bin size
    double dy; // Y bin size

    // Minimum-finding-algorithm variables
    double sigma_x = 1.0;
    double sigma_y = 1.0;
    int Nm = 0; // number of minima
    double* minima_x = new double[100];
    double* minima_y = new double[100]; // initial guess for the positions of the minima

    // LFEP algorithm variables
    float step_small = 2;
    float step_large = 2;
    float angle_step = PI/18;
    float angle_range = PI/2;

    // Smoothing-algorithm variables
    double Wp = 10.0; // Weight of the end/mid-points (relative to other points)
    double dt = 0.001; // number of data points on the smoothed path inverse

    // Output filenames
    string minimafile = "minima.txt"; // filename to report minima and their free energies
    string pathfile = "path.txt"; // filename to report the LFE path and the free energies
    string smoothpathfile = "smoothpath.txt"; // filename to report the smoothed LFE path and the free energies

    // Free energy array
    double* fe;

    // golden section search algorithm
    int golden(int n, double* q) {
	int i_0 = 0;
	int i_n = n-1;
	int i_m = n/2;
	int turn = 0;
	int i;

	while ( (i_m>i_0) && (i_m<i_n) ) {
	    if ((turn==0)&&(i_m>i_0)) {
		i = (i_m + i_0) / 2;
		if (q[i] < q[i_m]) {
		    i_n = i_m;
		    i_m = i;
		} else {
		    i_0 = i;
		}
		turn = 1;
	    } else if ((turn==1)&&(i_n>i_m)) {
		i = (i_n + i_m) / 2;
		if (q[i] < q[i_m]) {
		    i_0 = i_m;
		    i_m = i;
		} else {
		    i_n = i;
		}
		turn = 0;
	    }
	}
	return i_m;
    }
    
    // finding the minimum in a 2D space [in the (+/-sigma_x,+/-sigma_y) range around initial guess]
    int* minimum_in_range(int n, int m, double* q) {
	int* j_m = new int [n];
	int* m_ij = new int[2];
	double* qy = new double [m];
	double* qx = new double [n];
    
	for (int i=0; i<n; i++) {
	    for (int j=0; j<m; j++) qy[j] = q[i+n*j];
	    j_m[i] = golden(m, qy);
	    qx[i] = q[i+n*j_m[i]];
	}
	m_ij[0] = golden(n, qx);
	m_ij[1] = j_m[m_ij[0]];

	return m_ij;
    }

    // Rational Bezier Spline Curve algorithm to draw a smooth pathway
    // It gets the results of LFE algorithm to build a smooth curve

    void drawspline(int n, double* x1, double* x2, int m, int* poles, string smoothpathfile) {
	double xx1, xx2, xx1_, xx2_, t;
	double* Px1 = new double[n];
	double* Px2 = new double[n];
	double* w = new double[n];
	double* s = new double[n];
	xx1 = x1[0];
	xx2 = x2[0];
    
	int counter = 0;
	for (int i = 0; i < n; i++) { 
    	    if (counter < m)
    		if (i == poles[counter]) {
    		    w[i] = Wp;
		    counter++;
    		} else w[i] = 1.0;
	    }

	const char* SMOOTHPATHFILE = smoothpathfile.c_str();
	ofstream pathspline (SMOOTHPATHFILE);
	
	pathspline <<  x1[0] << " " << x2[0] << " " << fe[int(round((x1[0]-Xmin)/dx))*NY+int(round((x2[0]-Ymin)/dy))] << "\n";

	t = 0;
	int strt = 0;
	while (t < 1.0+dt) {
    	    xx1_ = xx1;
	    xx2_ = xx2;

	    for (int i = 0; i < n; i++) { 
		Px1[i] = w[i]*x1[i];
		Px2[i] = w[i]*x2[i];
		s[i] = w[i];
	    }

	    for (int j = n-1; j > 0; j--)
    		for (int i = 0; i < j; i++) {
        	    Px1[i] = (1-t)*Px1[i] + t*Px1[i+1];
        	    Px2[i] = (1-t)*Px2[i] + t*Px2[i+1];
        	    s[i] = (1-t)*s[i] + t * s[i+1];
        	}
	    xx1 = Px1[0]/s[0];
    	    xx2 = Px2[0]/s[0];
	    pathspline <<  xx1 << " " << xx2 << " " << fe[int(round((xx1-Xmin)/dx))*NY+int(round((xx2-Ymin)/dy))] << "\n";
	    t += dt;
	}
        
	pathspline <<  x1[n-1] << " " << x2[n-1] << " " << fe[int(round((x1[n-1]-Xmin)/dx))*NY+int(round((x2[n-1]-Ymin)/dy))] << "\n";
	pathspline.close();
}

    void readparm() {

	// reading parameters from standard input
	// '()' gives the default value
	// the parameters will be written in the standard error

	// reading parameters from standard input
	string key,value;

	std::cerr << "Reading the parameters ...\n";
        while(!std::cin.eof()) {
	    std::cin >> key;
	    // BINS [ NX , NY ] ([ 64 , 64 ])
	    if (key=="BINS") {
		std::cin.ignore(100,'[');
		std::cin >> NX;
		std::cin.ignore(100,',');
		std::cin >> NY;
		std::cerr << "Number of Bins = " << NX << " x " << NY << "\n";
	    // RANGE = [ Xmin, Xmax ], [ Ymin, Ymax ]
	    } else if (key=="RANGE") {
		std::cin.ignore(100,'[');
		std::cin >> Xmin;
		std::cin.ignore(100,',');
		std::cin >> Xmax;
		std::cin.ignore(100,'[');
		std::cin >> Ymin;
		std::cin.ignore(100,',');
		std::cin >> Ymax;
		std::cerr << "Range = [" << Xmin << ", " << Xmax << "] , ["<< Ymin << ", "<< Ymax<< "]\n";
	    // PATHFILE (path.txt)
	    } else if (key=="PATHFILE") {
		std::cin >> pathfile;
		std::cerr << "Path Filename = " << pathfile << "\n";
	    // SMOOTHPATHFILE (smoothpath.txt)
	    } else if (key=="SMOOTHPATHFILE") {
		std::cin >> smoothpathfile;
		std::cerr << "Smoothed Path Filename = " << smoothpathfile << "\n";
	    // MINIMAFILE (minima.txt)
	    } else if (key=="MINIMAFILE") {
		std::cin >> minimafile;
		std::cerr << "Minima Filename = " << minimafile << "\n";
	    // Path Smoothing Algorithm Parameters
	    // WEIGHT Wp (10.0)
	    } else if (key=="WEIGHT") {
		std::cin >> Wp;
		std::cerr << "B-Spline's Parameter Weights = " << Wp << "\n";
	    // SPLINE (0.001)
	    } else if (key=="SPLINE") {
		std::cin >> dt;
		std::cerr << "B-Spline's Parameter (dt) = " << dt << "\n";
	    // LFEP Algorithm Parameters
	    // SIGMA sigma_x sigma_y (1 1) [search for the minimum in the (+/-sigma_x,+/-sigma_y) range around initial guess]
	    // STEPS step_small step_large (2 2) [the two are usually the same but if you have a very detailed map you can use a larger number for the large step. It will search based on the large step but move based on the small step.]
	    // ANGLES angle_step angle_range (10 90) [on a circle centered around the current point it searches +/-range around the angle towards the next target, it doesn't look for all the points, it only takes angle steps]
	    } else if (key=="SIGMA") {
		std::cin >> sigma_x >> sigma_y;
		std::cerr << "Sigma_x = " << sigma_x << "\n";
		std::cerr << "Sigma_y = " << sigma_y << "\n";
	    } else if (key=="STEPS") {
		std::cin >> step_small >> step_large;
		std::cerr << "small step = " << step_small << "\n";
		std::cerr << "large step = " << step_large << "\n";
	    } else if (key=="ANGLES") {
		std::cin >> angle_step >> angle_range;
		std::cerr << "angle step = " << angle_step << "\n";
		std::cerr << "angle range = " << angle_range << "\n";
		angle_step *= PI/180;
		angle_range *= PI/180;
	    }
	}

	LX = Xmax - Xmin;
	LY = Ymax - Ymin;

	dx = LX/(NX-1);
	dy = LY/(NY-1);

    }

}

// Main

int main(int argc,char *argv[]) {

    // Reading the arguments (file names)
    char* datafile = ""; // data file (data points)
    char* minfile = ""; // minima file (guesses)
    int argi = 0;
    std::cerr << "Parsing the arguments ...\n";
    while (++argi < argc) {
        std::string argstr(argv[argi]);
        if ( argstr == "--freeenergy" || argstr == "-f" ) {
            datafile = argv[++argi];
	} else if ( argstr == "--minima" || argstr == "-m" ) {
            minfile = argv[++argi];
	}
    }
    
    // Reading the parameters
    readparm();

    //
    // Reading the free energies
    //

    fe = new double[NX*NY];

    if (datafile!="") {
	std::cerr << "Reading the free energy file ...\n";
      	ifstream fefile (datafile);
      	double xx, yy, zz;
      	int i,j;
	while (!fefile.eof()) {
	    fefile >> xx >> yy >> zz;
	    if (xx > Xmin && yy > Ymin && xx < Xmax && yy < Ymax) {
		i = int(round((xx - Xmin)/dx));
		j = int(round((yy - Ymin)/dy));
		fe[i*NY+j] += zz;
	    }
	}
	fefile.close();
    }

    if (minfile!="") {
	std::cerr << "Reading the minima file ...\n";
      	ifstream mfile (minfile);
      	double xx, yy;
	while (!mfile.eof()) {
	    mfile >> xx >> yy;
	    if (xx > Xmin && yy > Ymin && xx < Xmax && yy < Ymax) {
		minima_x[Nm] = xx;
		minima_y[Nm] = yy;
		Nm++;
		std::cerr << "point ( " << xx << " , " << yy << " ) is an initial guess for minimum " << Nm << "\n";
	    } else {
		std::cerr << "point ( " << xx << " , " << yy << " ) is not in [ (" << Xmin << " , " << Ymin << " ) : ( " << Xmax << " , " << Ymax << " ) ] range.\n";
		exit(0);
	    }
	}
	mfile.close();
    }

// Finding the minima

    double* Mx = new double[Nm];
    double* My = new double[Nm];
    int* Mi = new int[Nm];
    int* Mj = new int[Nm];

    if (Nm >= 1) {

	std::cerr << "Finding the minima ...\n";
	
	int N_i = 2*int(sigma_x/dx)+1;
	int N_j = 2*int(sigma_y/dy)+1;

	int m_i0, m_j0;

    	double* zk = new double[N_i*N_j];

	int* minim = new int[2];
	
	const char* MINIMAFILE = minimafile.c_str();
	ofstream minima_list (MINIMAFILE);

	for (int k = 0; k < Nm; k++) {
	    m_i0 = int((minima_x[k]-Xmin)/dx) - N_i/2;
	    m_j0 = int((minima_y[k]-Ymin)/dy) - N_j/2;
	
	    for (int i=0; i<N_i; i++)
		for (int j=0; j<N_j; j++)
		    if (m_i0+i>=0&&m_i0+i<NX&&m_j0+j>=0&&m_j0+j<NY) {
			zk[i+N_i*j] = fe[(m_i0+i)*NY+m_j0+j];
		    } else {
			zk[i+N_i*j] = 1e8;
		    }

            minim = minimum_in_range(N_i, N_j, zk);

            Mi[k] = m_i0 + minim[0];
            Mj[k] = m_j0 + minim[1];
	
	    Mx[k] = Xmin + dx * Mi[k];
	    My[k] = Ymin + dy * Mj[k];

	    minima_list << Mx[k] << "   " << My[k] << "   " << fe[Mi[k]*NY+Mj[k]] <<"\n";
	}
	minima_list.close();
    }

// Finding the LFE pathway between two minima

    std::cerr << "Finding the LFE path!\n";

    double r_, r_s, sin_, cos_, sin_k, cos_k, alpha_k;
    int is_k, js_k, i_k, j_k, k_m;
    
    int N_q = 2*int(angle_range/angle_step) + 1;

    double* q_= new double[N_q];
    int* i_ = new int[N_q];
    int* j_ = new int[N_q];

    int* is_ = new int[N_q];
    int* js_ = new int[N_q];
	
    int* path_i = new int[NX*1000];
    int* path_j = new int[NY*1000];
    int* midpoints = new int[Nm];

    int path_counter = 0;
    int mp_counter = 0;

    path_i[0] = Mi[0];
    path_j[0] = Mj[0];
    midpoints[mp_counter] = path_counter;
    mp_counter++;
    path_counter++;
	
    for (int ip = 1; ip < Nm; ip++) {
	    
        int m2_i = Mi[ip];
        int m2_j = Mj[ip];
        int m_i = Mi[ip-1];
        int m_j = Mj[ip-1];
	
        int Tm_i, Tm_j, Tm_i_, Tm_j_;
  
        int m_i_ = 0;
        int m_j_ = 0;

        r_ = sqrt((m2_i-m_i)*(m2_i-m_i) + (m2_j-m_j)*(m2_j-m_j));

        while ( (r_>step_small)&&(m_i<NX)&&(m_i>0)&&(m_j<NY)&&(m_j>0) ) {
	    sin_ = (m2_j-m_j)/r_;  // the angle of the line from pi to B
	    cos_ = (m2_i-m_i)/r_;

	    for (int k=0; k<N_q; k++) {
	        alpha_k = k * angle_step - angle_range; // the angular steps
	        sin_k = sin(alpha_k);
	        cos_k = cos(alpha_k);
	        i_k = m_i + int((cos_ * cos_k - sin_ * sin_k)*step_large);
	        j_k = m_j + int((sin_ * cos_k + cos_ * sin_k)*step_large);
	        is_k = m_i + int((cos_ * cos_k - sin_ * sin_k)*step_small);
	        js_k = m_j + int((sin_ * cos_k + cos_ * sin_k)*step_small);
	        q_[k] = fe[i_k*NY+j_k];
	        i_[k] = i_k;
	        j_[k] = j_k;
	        is_[k] = is_k;
	        js_[k] = js_k;
	    }
	
	    k_m = golden(N_q, q_);

	    int ij = 0;
	    while (ij < path_counter && (is_[k_m]!=path_i[ij] || js_[k_m]!=path_j[ij])) ij++;
		
    	    if (ij < path_counter) {
    	        Tm_i_ = m_i;
	        Tm_j_ = m_j;
	        Tm_i = 2*m_i-m_i_;
	        Tm_j = 2*m_j-m_j_;
	        int ij = 0;
	        while (ij < path_counter && (Tm_i!=path_i[ij] || Tm_j!=path_j[ij])) ij++;
		if (ij < path_counter) {
		    m_i = m_i+(m_i-m_i_)/2;
		    m_j = m_j+(m_j-m_j_)/2;
		} else {
		    m_i = Tm_i;
		    m_j = Tm_j;
		}
		m_i_ = Tm_i_;
		m_j_ = Tm_j_;
	    } else {
	        m_i_ = m_i;
    	        m_j_ = m_j;
	        m_i = is_[k_m];
	        m_j = js_[k_m];
	    }
		
	    path_i[path_counter] = m_i;
	    path_j[path_counter] = m_j;
	    path_counter++;
		
	    r_ = sqrt((m2_i-m_i)*(m2_i-m_i) + (m2_j-m_j)*(m2_j-m_j));
	}
	    
	path_i[path_counter] = m2_i;
	path_j[path_counter] = m2_j;
	midpoints[mp_counter] = path_counter;
	mp_counter++;
	path_counter++;
	    
    }
    	
    double* pathX = new double[path_counter];
    double* pathY = new double[path_counter];

    const char* PATHFILE = pathfile.c_str();
    ofstream path (PATHFILE);
    for (int k = 0; k < path_counter; k++) {
        pathX[k] = Xmin + dx * path_i[k];
        pathY[k] = Ymin + dy * path_j[k];
        path << pathX[k] << "   " << pathY[k] << "   " << fe[path_i[k]*NY+path_j[k]] << "\n";
    }
    path.close();

    std::cerr << "Smoothing the path!\n";

    drawspline(path_counter, pathX, pathY, Nm, midpoints, smoothpathfile);

    std::cerr << "Done!\n";

}
