#!/bin/bash

# Example: a distance-based harmonic potential with 15 replicas/windows

N=15 # number of replicas

# first step: preparing the data (potentials from the colvar trajectories)
ROOT="." # where the output of my runs are
x0="5.5 3.0 1.7 0.5 -1.0 -2.2 -2.6 -3.2 -3.6 -4.2 -4.7 -5.1 -5.8 -6.7 -8.7" # these are my centers
k="0.1 0.1 0.1 0.1 0.2 0.5 0.5 0.5 0.5 0.5 0.5 0.5 0.5 0.2 0.2" # these are my harmonic constants

for l in `seq 2 8` # these are runs that I will consider for analysis (job2,job3,...,job8)
do
for i in `seq 0 $((N-1))` # these are my replicas
do
    cat ${ROOT}/$i/move.job${l}.${i}.sort.colvars.traj | \
    awk -v N=$N -v i=$i -v k="$k" -v x0="$x0" '
    BEGIN{
	split(x0,X0)
	split(k,K)
    }($1 !~ /#/){
	    t=$1
	    x=$2
	    printf "%d %d",i,t
	    for (j=1;j<=N;j++) printf " %f",0.5*K[j]*(x-X0[j])*(x-X0[j])
	    printf "\n"
	}' >> data.txt
done
done

# second step: running the reweighting algorithm
# Bayesian Reweighting with Gibbs Sampler (BRwGS)
#    Based on M. Habeck, "Bayesian Estimation of Free Energies From Equilibrium Simulations", Phys. Rev. Lett. 109:100601 (2012)
#    and C. Bartels, "Analyzing Biased Monte Carlo and Molecular Dynamics Simulations", Chem. Phys. Lett. 331:446 (2000)
#    Coded by Mahmoud Moradi, UIUC (last modified on 11/07/2013)
#    It first uses the C. Bartles deterministic method then it draws multiple distributions using Gibbs sampler
#    Usage:  stdin: data
#            stdout: optimized probabilities
#            stderr: log file
#            arguments:
#            --windows (-w) number of windows (default: 1)
#            --lines (-l) number of data points (lines) read from the stdandard input  (default: 1)
#                        Each line (in stdin) includes "i t u_0 u_1 ... "
#                                  t and i are the time and window index (both integers)
#                                  u_i is the biasing potential associated with the window i
#            --iterations (-i) maximum number of iterations for self-consistent solution (default: 1000)
#            --gibbs (-g) number of Gibbs sampling repeats (default: 0; i.e., no Gibbs sampling)
#            --accuracy (-a) accuracy in kcal/mol for self-consistent solution of C. Bartels method (default: 0.1)
#            --temperature (-t) simulation temperature in Kelvin (default: 300)
#            --restart (-r) a filename to store resulted p's and f's to continue iterations (default: "")
#            --start (s) an existing restart file to continue iterations (default: "")
#    Program generates the {p(i,t)} probability for each data point "I i t p" (in stdout)
#    I is the Gibbs sampler repeat; I=0 repeat: the solution based on C. Bartels method
#    Program also reports the estimated free energies of windows (at the end of each iteration) in stderr

l=`wc data.txt | awk '{print $1}'` # number of data points (you can use only a portion of the data)
./rwgs -w $N -l $l -g 100 -t 310 < data.txt > density.txt 2> density.err
