/*  Bayesian Reweighting with Gibbs Sampler (BRwGS)
    Based on M. Habeck, "Bayesian Estimation of Free Energies From Equilibrium Simulations", Phys. Rev. Lett. 109:100601 (2012)
    and C. Bartels, "Analyzing Biased Monte Carlo and Molecular Dynamics Simulations", Chem. Phys. Lett. 331:446 (2000)
    Created by Mahmoud Moradi, UIUC (last modified on 8/21/2014)
    It first uses the C. Bartles method for a deterministic solution then it draws multiple distributions using Gibbs sampler
    Usage:  stdin: data
            stdout: optimized probabilities
            stderr: log file (including the free energies along the iterative optimization)
            arguments:
            --windows (-w) number of windows (default: 1)
            --lines (-l) maximum number of data points (lines) to be read from the stdandard input (default: 1)
                        Each line (in stdin) includes "i t u_0 u_1 ... "
                                  t and i are the time and window index (both integers)
                                  u_i is the biasing potential associated with the window i
            --iterations (-i) maximum number of iterations for self-consistent solution (default: 1000)
            --gibbs (-g) number of Gibbs sampling repeats (default: 0; i.e., no Gibbs sampling)
            --prior (-p) Dirichlet prior concentration parameter for maximum a posteriri method (default: 1; i.e., uninformative)
            --Prior (-P) Dirichlet prior concentration parameter for Gibbs sampler (default: 0; i.e., maximally concentrated)
            --count (-c) Dirichlet prior concentration parameter to be imported along the potentials
        	    If this flag is used; each line (in stdin) includes "w i t u_0 u_1 ..." in which w is a prior weight for each data point
            --marginal (-m) Marginal likelihood will be reported
            --accuracy (-a) accuracy in kcal/mol for self-consistent solution of C. Bartels method (default: 0.1)
            --temperature (-t) simulation temperature in Kelvin (default: 300)
            --restart (-r) a filename to store resulted p's and f's to continue iterations (default: "")
            --Restart (-R) update the restart file every "Restart" iterations (default: "1000").
            --start (-s) an existing restart file to continue iterations (default: "")
            --frequency (-f) reports the f's every "frequency" iterations (default: "1"; use "0" for none; converged f's will be reported anyway) 
    Program generates the {p(i,t)} probability for each data point "I i t p" (in stdout)
    I is the Gibbs sampler repeat; I=0 repeat: the solution based on C. Bartels method
    Program also reports the estimated free energies of windows in stderr
*/

#include <stdio.h>
#include <cmath>
#include <math.h>
#include <time.h>
#include <limits.h>
#include <cassert>
#include <iostream>
#include <fstream>
#include <string.h>
#ifdef __GXX_EXPERIMENTAL_CXX0X__
#include <random>
#endif

using namespace std;

#define boltzman 0.001987191683

int main(int argc,char *argv[]) {

    // general parameters    
    double temperature = 300;
    int L = 1; // number of replicas
    int N = 1; // number of input data
    int it = 1000; // self-consistency maximum number of iterations
    int IT = 0; // Gibbs-sampling number of iterations
    double df = 0.1; // self-consistency targeted accuracy
    double ps = 1.0; // Dirichlet pseudo-count for MAP prior
    double Ps = 1.0; // Dirichlet pseudo-count for Gibbs prior
    int priorcount = 0; // Dirichlet pseudo-counts being imported (along with potenials)
    int marginal = 0; // Report marginal likelihood
    char* restart = "";
    int Restart = 1000; // update the restart file every Restart iterations
    int Report = 1; // report the f's every "Report" iterations
    char* start = "";
    int argi = 0;
    while (++argi < argc) {
	std::string argstr(argv[argi]);
	if ( argstr == "--windows" || argstr == "-w" )
	    L = atoi(argv[++argi]);
	else if  ( argstr == "--lines" || argstr == "-l" )
	    N = atoi(argv[++argi]);
	else if  ( argstr == "--gibbs" || argstr == "-g" )
	    IT = atoi(argv[++argi]);
	else if  ( argstr == "--iterations" || argstr == "-i" )
	    it = atoi(argv[++argi]);
	else if  ( argstr == "--accuracy" || argstr == "-a" )
	    df = atof(argv[++argi]);
	else if  ( argstr == "--prior" || argstr == "-p" )
	    ps = atof(argv[++argi]);
	else if  ( argstr == "--Prior" || argstr == "-P" )
	    Ps = atof(argv[++argi]);
	else if  ( argstr == "--count" || argstr == "-c" )
	    priorcount = 1;
	else if  ( argstr == "--marginal" || argstr == "-m" )
#ifdef _RANDOM_H
	    marginal = 1;
#else
	    std::cerr << "# Note: Marginal likelihood cannot be calculated with MLE.\n";
#endif
	else if  ( argstr == "--temperature" || argstr == "-t" )
	    temperature = atof(argv[++argi]);
	else if  ( argstr == "--restart" || argstr == "-r" )
	    restart = argv[++argi];
	else if  ( argstr == "--Restart" || argstr == "-R" )
	    Restart = atoi(argv[++argi]);
	else if  ( argstr == "--start" || argstr == "-s" )
	    start = argv[++argi];
	else if  ( argstr == "--frequency" || argstr == "-f" )
	    Report = atoi(argv[++argi]);
	else if  ( argstr == "--help" || argstr == "-h" ) {
	    std::cout << "Bayesian Reweighting with Gibbs Sampler (BRwGS)\n"
		<< "    Based on M. Habeck, \"Bayesian Estimation of Free Energies From Equilibrium Simulations\", Phys. Rev. Lett. 109:100601 (2012)\n"
		<< "    and C. Bartels, \"Analyzing Biased Monte Carlo and Molecular Dynamics Simulations\", Chem. Phys. Lett. 331:446 (2000)\n"
		<< "    Created by Mahmoud Moradi, UIUC (last modified on 8/21/2014)\n"
		<< "    It first uses the C. Bartles method for a deterministic solution then it draws multiple distributions using Gibbs sampler\n"
		<< "Usage:  stdin: data\n"
		<< "        stdout: optimized probabilities (only for the last iteration)\n"
		<< "        stderr: log file including the free energies at the end of each iteration\n"
		<< " -- Options:\n"
		<< "	    --help (-h)\n"
		<< "	    --windows (-w) number of windows (default: 1)\n"
        	<< "	    --lines (-l) number of data points (lines) read from the stdandard input  (default: 1)\n"
                << "		    Each line (in stdin) includes \"i [b] t u_0 u_1 ... \"\n"
                << "        	    t and i are the time and window index (both integers)\n"
                << "        	    u_i is the biasing potential associated with the window i\n"
        	<< "	    --iterations (-i) maximum number of iterations for self-consistent solution (default: 1000)\n"
        	<< "	    --accuracy (-a) accuracy in kcal/mol for self-consistent solution (default: 0.1)\n"
        	<< "	    --temperature (-t) simulation temperature in Kelvin (default: 300)\n"
        	<< "	    --gibbs (-g) number of Gibbs sampling repeats (default: 0; i.e., no Gibbs sampling)\n"
        	<< "	    --prior (-p) Dirichlet prior concentration parameter for maximum a posteriri method (default: 1; i.e., uninformative)\n"
        	<< "	    --Prior (-P) Dirichlet prior concentration parameter for Gibbs sampler (default: 0; i.e., maximally concentrated)\n"
        	<< "	    --count (-c) Dirichlet prior concentration parameter to be imported along the potentials\n"
        	<< "		If this flag is used; each line (in stdin) includes \"w i t u_0 u_1 ...\" in which w is a prior weight for each data point\n"
        	<< "	    --marginal (-m) Marginal likelihood will be reported\n"
        	<< "	    --accuracy (-a) accuracy in kcal/mol for self-consistent solution of C. Bartels method (default: 0.1)\n"
        	<< "	    --restart (-r) a filename to store resulted p's and f's to continue iterations (default: \"\")\n"
        	<< "	    --Restart (-R) update the restart file every \"Restart\" iterations (default: \"1000\").\n"
        	<< "	    --start (-s) an existing restart file to continue iterations (default: \"\")\n"
        	<< "	    --frequency (-f) reports the f's every \"frequency\" iterations (default: \"1\"; use \"0\" for none; converged f's will be reported anyway)\n"
		<< "	    Program generates the {p(i,t)} probability for each data point \"I i t p\" (in stdout)\n"
		<< "	    I is the Gibbs sampler repeat; I=0 repeat: the solution based on C. Bartels method\n"
		<< "	    Program also reports the estimated free energies of windows in stderr\n";
	    exit(0);
	}
    }

    std::cerr << "# Done reading parameters:\n";
    std::cerr << "# Temperature: "<< temperature <<"\n";
    std::cerr << "# Number of windows: "<< L <<"\n";
    std::cerr << "# Maximum number of data points: "<< N <<"\n";
    std::cerr << "# Maximum number of iterations for self-consistent algorithm: "<< it <<"\n";
    std::cerr << "# Accuracy (in kcal/mol) to be targeted for self-consistent algorithm: "<< df <<"\n";
    std::cerr << "# Dirichlet prior concentration parameter for self-consistent algorithm: "<< ps <<"\n";
    if (priorcount) std::cerr << "# Dirichlet prior concentration parameters will be imported along with potentials.\n";
    std::cerr << "# Number of iterations for Gibbs sampling: "<< IT << "\n";
    std::cerr << "# Dirichlet prior concentration parameter for Gibbs sampling: "<< Ps <<"\n";
    if (marginal) std::cerr << "#  Marginal likelihood will be reported.\n";

    if (Restart<=0) Restart=1000;
    std::cerr << "# Restart file \""<< restart <<"\" will be updated every "<< Restart <<" iterations.\n";
    if (Report<0) Report=0;
    std::cerr << "# Free energies (and densities from Gibbs sampling) will be reported every "<< Report <<" iterations.\n";

    double beta = 1.0/(boltzman*temperature);
    df *= boltzman;

    double* p = new double[N]; // weights (at iteration k)
    double* p_ = new double[N]; // weights (at iteration k-1)
    double* w = new double[N]; // prior concentration parameters
    double* C = new double[N]; // For marginal likelihood calculations
    double* c = new double[N*L]; // biases
    double* f = new double[L]; // window proportionality factors
    double* n = new double[L]; // number of points for each window
    int* T = new int[N]; // actual time for each data point
    int* W = new int[N]; // window of each data point

    for (int i=0; i<L; i++) {
	f[i] = 1.0;
	n[i] = 0;
    }

    // reading the data
    double U;
    std::cerr << "# Reading data points starts...\n";
    double totalprior = 0.0;
    double multiprior = 0.0;
    int t=0;
    while(t<N&&!std::cin.eof()) {
	if (priorcount)
	    std::cin >> w[t];
	else
	    w[t]=1.0;
	if (marginal) {
	    totalprior += w[t]*Ps;
#ifdef _RANDOM_H
	    multiprior += log(std::tgamma(w[t]*Ps));
#endif
	}
	std::cin >> W[t];
	std::cin >> T[t];
	n[W[t]]++;
	if (marginal) C[t]=1.0;
	for (int i=0; i<L; i++) {
	    std::cin >> U;
	    c[t*L+i]=exp(-beta*U);
	    if (marginal) C[t]*=c[t*L+i];
	}
	std::cin.ignore(INT_MAX,'\n');
	t++;
    }
    N=t;
    std::cerr << "# Done reading " << N << " data points.\n";
    for (int i = 0; i < L; i++) {
	std::cerr << "# window " << i << " : " << n[i] << " samples\n";
    }

    // initializing the p's and f's from start file
    if (start!="") {
	ifstream init (start);
	if (init) {
	    for (int i=0; i<L; i++) init >> f[i];
	    for (int t=0; t<N; t++) init >> p[t];
	    init.close();
	    std::cerr << "# Done reading the \"" << start << "\" file!\n";
	} else {
	    std::cerr << "Error: Could not open \"" << start << "\" file!\n";
	    exit(EXIT_FAILURE);
	}
    }

    // starting the calculations
    int k = 0; // iteration

    // deterministic solution
    if (it>0) {
	double dfe=df;
	double Pt=0;
	int tt = 0;
	while(k<it && tt<N) {
	    k++;
	    Pt=0;
	    for (int t=0; t<N; t++) {
	        double P = 0.0;
	        for (int i=0; i<L; i++)
	    	    P+=n[i]*f[i]*c[t*L+i];
		p[t]=ps*w[t]/P;
    		Pt+=p[t];
	    }
    	    for (int t=0; t<N; t++)
    		p[t]/=Pt;
	    for (int i=0; i<L; i++) {
	        double F = 0.0;
	        for (int t=0; t<N; t++)
	            F+=p[t]*c[t*L+i];
		f[i]=1.0/F;
	    }
	    // report the f's
    	    if ( k%(Report>0?Report:INT_MAX)==0 ) {
	        std::cerr << "# iteration " << k << "\n";
	        double minf;
	        for (int i=0; i<L; i++)
		    if (i==0) minf=f[i]; else if (minf>f[i]) minf=f[i];
		for (int i=0; i<L; i++)
		    std::cerr << i << " " << log(f[i]/minf)/beta << "\n";
	    }
	    // checkpoint
	    if ( restart!="" && k%Restart==0 ) {
		ofstream rest (restart);
		for (int i=0; i<L; i++) rest << " " << f[i];
		rest << "\n";
		for (int t=0; t<N; t++) rest << p[t] << "\n";
		rest.close();
	    }
	    // check for convergence
	    if (k>0) {
	        int t = 0;
	        while( abs(log(p[t]/p_[t]))<df && t<N ) t++;
	        tt = t;
	    }
	    for (int t=0; t<N; t++) p_[t]=p[t];
	}
	// report the converged values
	for (int t=0; t<N; t++)
    	    std::cout << 0 << " " << W[t] << " "  << T[t] << " " << p[t] << "\n";
	if ( k%(Report>0?Report:INT_MAX)!=0 ) {
    	    std::cerr << "# iteration " << k << " (final)\n";
    	    double minf;
    	    for (int i=0; i<L; i++)
        	if (i==0) minf=f[i]; else if (minf>f[i]) minf=f[i];
    	    for (int i=0; i<L; i++)
        	std::cerr << i << " " << log(f[i]/minf)/beta << "\n";
	}
    }

#ifdef _RANDOM_H
    // for Gibbs sampling
    std::default_random_engine generator (time(NULL));
    // Gibbs sampling
    if (IT>0) k=0;
    while(k<IT) {
	k++;
	double Pt=0;
	for (int t=0; t<N; t++) {
	    double P = 0.0;
	    for (int i=0; i<L; i++) 
	        P+=n[i]*f[i]*c[t*L+i];
	    std::gamma_distribution<double> distribution (1.0+Ps*w[t],1.0/P);
	    p[t]=distribution(generator);
	    Pt+=p[t];
	}
	double mln = 0.0; // marginal likelihood numerator
	double mld = 0.0; // marginal likelihood denominator
	for (int t=0; t<N; t++) {
    	    p[t]/=Pt;
	    if ( k%Report==0 && Report>0 ) std::cout << k << " " << W[t] << " " << T[t] << " " << p[t] << "\n";
	    if (marginal) {
		mln+=w[t]*Ps*log(p[t]);
		mld+=p[t]*C[t];
	    }
    	}
	if (marginal) std::cerr << "# Marginal likelihood: " << (((totalprior-0.5)*log(totalprior)-totalprior)-multiprior)+(mln-log(mld)) << " " << mln << " " << log(mld) << " " << ((totalprior-0.5)*log(totalprior)-totalprior) << " " << multiprior << " " << "\n";
	for (int i=0; i<L; i++) {
	    double F = 0.0;
	    for (int t=0; t<N; t++)
	        F+=p[t]*c[t*L+i];
	        std::gamma_distribution<double> distribution (n[i],1.0/(F*n[i]));
	        f[i]=distribution(generator);
	}
	// report the f's
	if ( k%(Report>0?Report:INT_MAX)==0 ) {
	    double minf;
	    std::cerr << "# Gibbs " << k << "\n";
	    for (int i=0; i<L; i++)
	        if (i==0) minf=f[i]; else if (minf>f[i]) minf=f[i];
	    for (int i=0; i<L; i++)
	        std::cerr << i << " " << log(f[i]/minf)/beta << "\n";
	}
	// checkpoint
	if ( restart!="" && k%Restart==0 ) {
	    ofstream rest (restart);
	    for (int i=0; i<L; i++) rest << " " << f[i];
	    rest << "\n";
	    for (int t=0; t<N; t++) rest << p[t] << "\n";
	    rest.close();
	}
    }
    // report the last values
    if ( IT>0 && ( k%(Report>0?Report:INT_MAX)!=0 || k==0 ) ) {
	for (int t=0; t<N; t++)
    	    std::cout << k << " " << W[t] << " "  << T[t] << " " << p[t] << "\n";
        std::cerr << "# iteration " << k << " (final)\n";
        double minf;
        for (int i=0; i<L; i++)
            if (i==0) minf=f[i]; else if (minf>f[i]) minf=f[i];
        for (int i=0; i<L; i++)
            std::cerr << i << " " << log(f[i]/minf)/beta << "\n";
    }
#endif

    // checkpoint
    if ( restart!="" && (k%Restart!=0||k==0) ) {
        ofstream rest (restart);
        for (int i=0; i<L; i++) rest << " " << f[i];
        rest << "\n";
        for (int t=0; t<N; t++) rest << p[t] << "\n";
        rest.close();
    }

    delete[] c;
    delete[] p;
    delete[] p_;
    delete[] n;
    delete[] f;
    delete[] W;
    delete[] w;
    delete[] C;
    delete[] T;
}
