"""MDTools provides utilities for manipulating molecular data.

Written by James Phillips, University of Illinois.

WWW: http://www.ks.uiuc.edu/~jim/mdtools/

RCS: $Id: md.py,v 0.62 1996/05/17 15:37:02 jim Exp $

Class Hierarchy:
   HomoCoord -> Coord -> Atom
             -> Vector
   AtomGroup -> ASel
             -> Residue
             -> ResidueGroup -> RSel
                             -> Segment
                             -> SegmentGroup -> Molecule
   Trans
   DCD -> DCDWrite
   Data -> NAMDOutput

Utilities:
   help([topic]) - easy access to documentation
   dist(a,b) - distances between Coords or groups
   distsq(a,b) - distance squared between Coords
   angle(a,b,c,[d],[units]) - angle (a,b,c) or dihedral (a,b,c,d)

Localizations:
   xyplotfunction(filename) - plot a file of (x,y) data
   pdbdisplayfunction(filename) - display a pdb file

Constants:
   backbone - names of atoms considered part of the backbone
   angleunits - definitions of 'rad', 'pi', and 'deg'
   angledefault - set to 'deg'
"""

# $Log: md.py,v $
# Revision 0.62  1996/05/17 15:37:02  jim
# Small doc and error checking changes to Data.
#
# Revision 0.61  1996/05/17 15:00:30  jim
# Added flush() to end of DCDWrite.append() because the last
# frame appended to the file wasn't being recorded when the
# DCDWrite object was deleted with del().
#
# Revision 0.60  1996/05/08 16:31:20  jim
# Improved __repr__ for AtomGroup and subclasses.
# Changed DCD.dummymol() to DCD.asel().
# Fixed DCD.aselfree() to work without fixed atoms.
#
# Revision 0.59  1996/05/07 22:17:21  jim
# Started using RCS system.
#
# print "MDTools 0.58 beta (4/12/96) by James Phillips.",
# Added dummy atom creation methods to DCD.
# print "MDTools 0.57 beta (3/27/96) by James Phillips.",
# Enhanced molecule display to generate fewer leftover files.
# print "MDTools 0.56 beta (3/12/96) by James Phillips.",
# Added underscores to internal functions, split localization.
# print "MDTools 0.55 beta (3/11/96) by James Phillips.",
# Improved help system to reduce need for quotes.
# print "MDTools 0.54 beta (3/11/96) by James Phillips.",
# Added help system, fixed little bugs (unit vector from 0 and variance).
# print "MDTools for Python version 0.53 beta (2/12/96) by James Phillips"
# Fixed bug in Data.deviation().
# print "MDTools for Python version 0.52 beta (1/30/96) by James Phillips"
# Fixed old use of buildAtomList in RSel.__init__().
# print "MDTools for Python version 0.51 beta (1/18/96) by James Phillips"
# Allowed negative subscripts in DCD and fixed small bug in NAMDOutput.plot().
# print "MDTools for Python version 0.5 beta (12/18/95) by James Phillips"
# Original beta release.
# print " "+__name__+".help() for more info."

print "MDTools "+"$Revision: 0.62 $"[11:-1]+"$State: Exp $"[8:-1]+"("+"$Date: 1996/05/17 15:37:02 $"[7:-11]+") by James Phillips.  "+__name__+".help() for more info."

import string
import math
import struct
import copy
import types
import tempfile
import os
import sys
import time

#
# Localizations
#

from md_local import xyplotfunction, pdbdisplayfunction, pdbdisplaysend

#
# Constants
#

backbone = ('N','HN','CA','HA','H','C','O','HT1','HT2','HT3','OT1','OT2')

angleunits = {'rad':1.,'pi':math.pi,'deg':math.pi/180.}
angledefault = 'deg'

#
# Help utility
#
_tips = {
'help':"""Function help([topic]): Prints documentation.

The topic may be any object, behavior varies.  Classes and Instances have __doc__ strings printed for that class and all superclasses.  Functions with names in the _tips dictionary have documentation strings printed.  Strings are searched for in the _tips dictionary and evaluated if not found.
""",
'angles':"""Units for angles are defined in angleunits.  Currently defined units are 'rad', 'pi', and 'deg'.  Angles have a default unit set in angledefault, which is normally 'deg'.

See also: angleunits, angledefault, angleconvert, angle
""",
'angleconvert':"""Function angleconvert(a,units,[newunits]): Converts angles.

See also: angleunits, angledefault, angle, 'angles'
""",
'dist':"""Function dist(a,b): Returns the distance between the objects a and b.  If a or b is a group then the closest distance is returned.
""",
'distsq':"""Function distsq(a,b): Returns the square of the distance between a and b more efficiently than squaring the result of dist(a,b).
""",
'angle':"""Function angle(a,b,c): Returns the angle between the coordinates a, b, and c.  angle(a,b,c,d) returns the dihedral angle.  Desired units may be appended to the parameters, as in angle(a,b,c,'rad').  Sorry, no angles between vectors.
""",
'xyplotfunction':"""Function xyplotfunction(file): Localized to plot a file of (x,y) data
""",
'pdbdisplayfunction':"""Function pdbdisplayfunction(file): Localized to display a pdb file
"""
}
def help(name=None):
	if ( name is None ):
		print __doc__
	elif ( type(name) is types.ClassType ):
		if ( len(name.__bases__) ):
			classdesc = name.__name__ + "["
			for c in name.__bases__:
				help(c)
				classdesc = classdesc + c.__name__ + ","
			classdesc = classdesc[:-1] + "]"
		else:
			classdesc = name.__name__
		print "Class",classdesc+":",name.__doc__
	elif ( type(name) is types.InstanceType ):
		help(name.__class__)
	elif ( name in _tips.keys() ):
		print _tips[name]
	elif ( hasattr(name,"__name__") and name.__name__ in _tips.keys() ):
		print _tips[name.__name__]
	else:
		try:
			help(eval(name))
		except:
			print 'Evaluates to',repr(name)+'.'

#
# HomoCoord class hierarcy:
#                                        HomoCoord
#                                         |     |
#                                      Coord   Vector
#                                       |
#                                     Atom
#
#    distsq(a,b)  angle(a,b,c,[d],[u])  angleconvert(angle,old,[new])
#

def _HomoCoord_downcast(x,y,z,W):
	if W == 0:
		return Vector(x,y,z)
	elif abs(W-1) < 0.0001:
		return Coord(x,y,z)
	else:
		return HomoCoord(x,y,z,W)

class HomoCoord:
	"""Homogeneous coordinates distinguish vectors and positions.

As defined in many computer graphics texts, homogeneous coordinates consist of four values: x, y, z, and W.  W is 0 for vectors and 1 for coordinates.  Downcasting to Vector and Coord is done automatically for arithmetic operations on a HomoCoord.

Data: x, y, z, W

Methods:
   a = HomoCoord(x,y,z,W)
   b = HomoCoord(x,y,z,W)
   a + b 
   a - b
   -a
   10. * a
   a * 10.
   a / 10.
   len(a) - returns 4
   a[2] - returns z
"""
	def __init__(self,x,y,z,W):
		self.x = float(x); self.y = float(y); self.z = float(z); self.W = float(W)
	def __repr__(s):
		return 'HomoCoord('+`s.x`+','+`s.y`+','+`s.z`+','+`s.W`+')'
	def __add__(s,o):
		return _HomoCoord_downcast(s.x+o.x,s.y+o.y,s.z+o.z,s.W+o.W)
	def __sub__(s,o):
		return _HomoCoord_downcast(s.x-o.x,s.y-o.y,s.z-o.z,s.W-o.W)
	def __neg__(s):
		return _HomoCoord_downcast(-s.x,-s.y,-s.z,-s.W)
	def __mul__(s,a):
		if type(a) in (types.IntType,types.FloatType,types.LongType):
			return _HomoCoord_downcast(s.x*a,s.y*a,s.z*a,s.W*a)
		else: raise TypeError,'HomoCoord multiplication by non-numeric'
	def __rmul__(s,a):
		if type(a) in (types.IntType,types.FloatType,types.LongType):
			return _HomoCoord_downcast(s.x*a,s.y*a,s.z*a,s.W*a)
		else: raise TypeError,'HomoCoord multiplication by non-numeric'
	def __div__(s,a):
		if type(a) in (types.IntType,types.FloatType,types.LongType):
			return _HomoCoord_downcast(s.x/a,s.y/a,s.z/a,s.W/a)
		else: raise TypeError,'HomoCoord division by non-numeric'
	def __len__(self):
		return 4
	def __getitem__(s,i):
		return (s.x,s.y,s.z,s.W)[i]

class Vector(HomoCoord):
	"""A vector has a length, dot product, and cross product.

A vector is a homogeneous coordinate with W = 0 and additional operations.

Methods:
   a = Vector(x,y,z)
   b = Vector(x,y,z)
   abs(a) - returns |a|
   a * b - dot product
   a % b - cross product
   a.unit() - returns a / |a|
"""
	def __init__(self,x,y,z):
		HomoCoord.__init__(self,x,y,z,0)
	def __abs__(s):
		return math.sqrt(s.x*s.x+s.y*s.y+s.z*s.z)
	def __mul__(s,o):
		if type(o) in (types.IntType,types.FloatType,types.LongType):
			return HomoCoord.__mul__(s,o)
		elif o.W == 0:
			return s.x*o.x+s.y*o.y+s.z*o.z
		else: raise TypeError,'Vector multiplication by non-numeric and non-Vector'
	def __mod__(s,o):
		if o.W == 0:
			return Vector(s.y*o.z-s.z*o.y,s.z*o.x-s.x*o.z,s.x*o.y-s.y*o.x)
		else: raise TypeError,'Vector cross-product with non-Vector'
	def unit(s):
		a = abs(s)
		if ( a ):
			return s / a
		else:
			raise ZeroDivisionError, "can't create unit vector from zero vector"
	def __repr__(s):
		return 'Vector('+`s.x`+','+`s.y`+','+`s.z`+')'

class Coord(HomoCoord):
	"""A coordinate cannot be scaled.

A coordinate is a homogeneous coordinate with W = 1.

Methods:
   a = Coord(x,y,z)
   b = Coord(x,y,z)
   a.set(b) - important for subclasses

See also: dist, distsq, angle
"""
	def __init__(self,x,y,z):
		HomoCoord.__init__(self,x,y,z,1)
	def set(s,o):
		if o.W == 1:
			s.x = o.x; s.y = o.y; s.z = o.z
		else: raise TypeError,'Coord set to non-Coord'
	def __repr__(s):
		return 'Coord('+`s.x`+','+`s.y`+','+`s.z`+')'

def dist(a,b):
	if not ( hasattr(a,'atoms') or hasattr(b,'atoms') ):
		return math.sqrt(math.pow(a[0]-b[0],2)+math.pow(a[1]-b[1],2)+math.pow(a[2]-b[2],2))
	else:
		if hasattr(a,'atoms'):
			al = a.atoms
		else:
			al = (a)
		if hasattr(b,'atoms'): 
			bl = b.atoms
		else:
			bl = (b)
		if len(al) > len(bl): (al,bl) = (bl,al)
		ds = 1000000000.
		for aa in al:
			for ba in bl:
				ds = min(ds,math.pow(aa[0]-ba[0],2)+math.pow(aa[1]-ba[1],2)+math.pow(aa[2]-ba[2],2))
		return math.sqrt(ds)

def distsq(a,b):
	return math.pow(a[0]-b[0],2)+math.pow(a[1]-b[1],2)+math.pow(a[2]-b[2],2)

def angleconvert(angle,old,new=angledefault):
	return angle * ( angleunits[old] / angleunits[new] )

def angle(a,b,c,x1=angledefault,x2=angledefault):
	if type(x1) is types.StringType:
		d = None
		units = x1
	else:
		d = x1
		units = x2
	if d:
		e = ((c-b)%(b-a)).unit()
		f = ((d-c)%(c-b)).unit()
		return angleconvert(math.asin((e%f)*((c-b).unit())),'rad',units)
	else:
		e = (a-b).unit()
		f = (c-b).unit()
		return angleconvert(math.acos(e*f),'rad',units)

class Atom(Coord):
	"""Holds all atom-based information.

Data: mass, charge, type, name, id, q, b, residue

Methods:
   a = Atom()
"""
	def __init__(self):
		Coord.__init__(self,0,0,0)
		self.mass = 1.
		self.charge = 0.
		self.type = '???'
		self.name = '???'
		self.id = 0
		self.q = 0.
		self.b = 0.
		self.residue = None
	def __repr__(s):
		return '< Atom '+`s.name`+' at ('+`s.x`+','+`s.y`+','+`s.z`+') >'

#
# AtomGroup class hierarcy:
#                                        AtomGroup -------------
#                                         |     |              |
#                                    Residue   ResidueGroup   ASel
#                                              |    |     | 
#                                        Segment Molecule RSel
#

class AtomGroup:
	"""A group of atoms.

Data: atoms, [frames]

Methods:
   g = AtomGroup()
   g.atoms.append(a)
   g.tmass() - total mass
   g.tcharge() - total charge
   g.cgeom() - center of geometry
   g.cmass() - center of mass
   g.rgyration() - radius of gyration
   g.saveframe([key]) - save coordinates to internal dictionary
   g.loadframe([key]) - get coordinates from internal dictionary
   g.delframe([key]) - remove coordinates from internal dictionary
   frame = []
   g.putframe(frame) - put coordinates in list
   g.getframe(frame) - get coordinates from list of (x,y,z)
   mol.putframe(frame)
   g.getmolframe(frame) - get coordinates from list for all atoms in molecule
   g.asel(func) - return atom selection based on filter function

See also: Molecule, ASel
"""
	def __init__(self):
		self.atoms = []
	def tmass(self):
		return reduce(lambda t,a: t+a.mass, self.atoms, 0.)
	def tcharge(self):
		return reduce(lambda t,a: t+a.charge, self.atoms, 0.)
	def cgeom(self):
		t = reduce(lambda t,a: t+a, self.atoms, Vector(0,0,0))
		return t / t.W
	def cmass(self):
		t = reduce(lambda t,a: t + a.mass*a, self.atoms, Vector(0,0,0))
		return t / t.W
	def rgyration(self):
		t = reduce(lambda t,a,com = self.cmass():
			t + a.mass*distsq(a,com), self.atoms, 0.)
		return math.sqrt( t / self.tmass() )
	def saveframe(self,key=None):
		if not hasattr(self,'frames'): self.frames = {}
		f = []
		self.frames[key] = f
		self.putframe(f)
	def loadframe(self,key=None):
		if not hasattr(self,'frames') or not len(self.frames):
			raise "no frames saved internally"
		self.getframe(self.frames[key])
	def delframe(self,key=None):
		if not hasattr(self,'frames') or not len(self.frames):
			raise "no frames saved internally"
		del(self.frames[key])
	def getframe(self,frame):
		i = 0
		for a in self.atoms:
			(a.x,a.y,a.z) = frame[i]
			i = i + 1
	def getmolframe(self,frame):
		for a in self.atoms:
			(a.x,a.y,a.z) = frame[a.id-1]
	def putframe(self,frame):
		frame[:] = []
		for a in self.atoms:
			frame.append((a.x,a.y,a.z))
	def asel(self,func):
		return ASel(self,func)
	def __repr__(self):
		return '< '+self.__class__.__name__+' with '\
			+`len(self.atoms)`+' atoms >'

class Residue(AtomGroup):
	"""A group of atoms with extra information.

Data: type, name, id, segment, prev, next

Methods:
   r = Residue()
   r.buildrefs() - assigns residue for atoms
   r[name] - returns atoms by name (like a dictionary)
   r.rotate(angle,[units]) - rotate side chain
   r.phipsi([units]) - returns (phi,psi)

See also: Atom, Molecule, 'angles'
"""
	def __init__(self):
		AtomGroup.__init__(self)
		self.type = '???'
		self.name = '???'
		self.id = 0
		self.segment = None
		self.prev = None
		self.next = None
	def buildrefs(self):
		for a in self.atoms: a.residue = self
	def __getitem__(self,name):
		for a in self.atoms:
			if ( a.name == name ): return a
		raise "No such atom."
	def rotate(self,angle,units=angledefault):
		t = Trans(center=self['CA'],axis=self['CB'],angle=angle,units=units)
		for a in self.atoms:
			if a.name not in backbone: t(a)
	def phipsi(self,units=angledefault):
		try: phi = angle(self.prev['C'],self['N'],self['CA'],self['C'],units)
		except: phi = None
		try: psi = angle(self['N'],self['CA'],self['C'],self.next['N'],units)
		except: psi = None
		return (phi,psi)
	def __repr__(self):
		return '< Residue '+self.name+' with '\
			+`len(self.atoms)`+' atoms >'

class ASel(AtomGroup):
	"""A group of atoms generated from a filter function.

Methods:
   s = ASel(base,func)

See also: RSel
"""
	def __init__(self,base,func):
		AtomGroup.__init__(self)
		self.atoms = filter(func,base.atoms)

class ResidueGroup(AtomGroup):
	"""A group of residues.

Data: residues

Methods:
   g = ResidueGroup()
   g.buildlists() - generate atoms from residues
   g.phipsi([units]) - returns list of all (phi,psi)
   g.rsel(func) - returns residue selection based on filter function

See also: RSel
"""
	def __init__(self):
		AtomGroup.__init__(self)
		self.residues = []
	def buildlists(self):
		self.atoms[:] = []
		for r in self.residues:
			for a in r.atoms: self.atoms.append(a)
	def phipsi(self,units=angledefault):
		return map(lambda r,u=units: r.phipsi(u), self.residues)
	def rsel(self,func):
		return RSel(self,func)
	def __repr__(self):
		return '< '+self.__class__.__name__+' with '\
			+`len(self.residues)`+' residues, and '\
			+`len(self.atoms)`+' atoms >'

class RSel(ResidueGroup):
	"""A group of residues generated from a filter function.

Methods:
   s = RSel(base,func)

See also: ASel
"""
	def __init__(self,base,func):
		ResidueGroup.__init__(self)
		self.residues = filter(func,base.residues)
		self.buildlists()

class Segment(ResidueGroup):
	"""A group of residues with extra information.

Data: name, molecule

Methods:
   s = Segment()
   s.buildrefs() - assigns segment for residues

See also: Residue, Molecule
"""
	def __init__(self):
		ResidueGroup.__init__(self)
		self.name = '???'
		molecule = None
	def buildrefs(self):
		for r in self.residues:
			r.segment = self
			r.buildrefs()
		for i in range(1,len(self.residues)):
			self.residues[i-1].next = self.residues[i]
			self.residues[i].prev = self.residues[i-1]
	def __repr__(self):
		return '< Segment '+self.name+' with '\
			+`len(self.residues)`+' residues, and '\
			+`len(self.atoms)`+' atoms >'

class SegmentGroup(ResidueGroup):
	"""A group of segments.

Data: segments

Methods:
   g = SegmentGroup()
   g.buildlists() - generate residues from segments
"""
	def __init__(self):
		ResidueGroup.__init__(self)
		self.segments = []
	def buildlists(self):
		self.residues[:] = []
		for s in self.segments:
			s.buildlists()
			for r in s.residues: self.residues.append(r)
		ResidueGroup.buildlists(self)
	def __repr__(self):
		return '< '+self.__class__.__name__+' with '\
			+`len(self.segments)`+' segments, '\
			+`len(self.residues)`+' residues, and '\
			+`len(self.atoms)`+' atoms >'

def _sround(x,n):
	raw = str(round(x,n))
	if string.find(raw,'.') == -1 :
		raw = raw + '.'
	while len(raw) - string.find(raw,'.') <= n :
		raw = raw + '0'
	return raw

def _Ftoi(s):
	return string.atoi(string.strip(s))

def _Ftof(s):
	return string.atof(string.strip(s))

class Molecule(SegmentGroup):
	"""Complete interface for pdb/psf molecule files.

Data: pdbfile, psffile, pdbremarks, psfremarks

Methods:
   m = Molecule([pdb],[psf]) - read molecule from file(s)
   m.buildrefs() - assigns molecule for segments
   m.writepdb([file]) - write pdb to file (pdbfile by default)
   m.display([file]) - write to file (tempfile by default) and display

See also: Segment, pdbdisplayfunction
"""
	def __init__(self,pdb=None,psf=None):
		SegmentGroup.__init__(self)
		self.pdbfile = pdb
		self.psffile = psf
		self.pdbremarks = []
		self.psfremarks = []
		pdb = self.pdbfile
		psf = self.psffile
		if not ( pdb or psf ):
			raise "No data files specified."
		if pdb:
			pdbf = open(self.pdbfile,'r')
			pdbrec = pdbf.readline()
			while len(pdbrec) and pdbrec[0:6] == 'REMARK':
				self.pdbremarks.append(string.strip(pdbrec))
				print self.pdbremarks[-1]
				pdbrec = pdbf.readline()
		if psf:
			psff = open(self.psffile,'r')
			psfline = psff.readline()
			psfrec = string.split(psfline)
			while len(psfline) and not (len(psfrec) > 1 and psfrec[1] == '!NTITLE'):
				psfline = psff.readline()
				psfrec = string.split(psfline)
			nrecs = string.atoi(psfrec[0])
			for i in range(0,nrecs):
				psfrec = psff.readline()
				self.psfremarks.append(string.strip(psfrec))
				print self.psfremarks[-1]
			psfline = psff.readline()
			psfrec = string.split(psfline)
			while len(psfline) and not (len(psfrec) > 1 and psfrec[1] == '!NATOM'):
				psfline = psff.readline()
				psfrec = string.split(psfline)
			nrecs = string.atoi(psfrec[0])
		moretogo = 0
		if pdb:
			if len(pdbrec) and pdbrec[0:6] in ('ATOM  ','HETATM'): moretogo = 1
		if psf:
			psfrec = string.split(psff.readline())
			if nrecs > len(self.atoms): moretogo = 1
		curseg = None
		curres = None
		numread = 0
		while moretogo:
			moretogo = 0
			if psf:
				if (not curseg) or psfrec[1] != curseg.name:
					curseg = Segment()
					self.segments.append(curseg)
					curseg.name = psfrec[1]
				if (not curres) or string.atoi(psfrec[2]) != curres.id:
					curres = Residue()
					curseg.residues.append(curres)
					curres.id = string.atoi(psfrec[2])
					curres.name = psfrec[3]
					curres.type = curres.name
			else:
				if (not curseg) or string.strip(pdbrec[67:]) != curseg.name:
					curseg = Segment()
					self.segments.append(curseg)
					curseg.name = string.strip(pdbrec[67:])
				if (not curres) or _Ftoi(pdbrec[22:26]) != curres.id:
					curres = Residue()
					curseg.residues.append(curres)
					curres.id = _Ftoi(pdbrec[22:26])
					curres.name = string.strip(pdbrec[17:21])
					curres.type = curres.name
			curatom = Atom()
			curres.atoms.append(curatom)
			numread = numread + 1
			if pdb:
				curatom.name = string.strip(pdbrec[12:16])
				curatom.type = curatom.name
				curatom.id = _Ftoi(pdbrec[6:11])
				curatom.x = _Ftof(pdbrec[30:38])
				curatom.y = _Ftof(pdbrec[38:46])
				curatom.z = _Ftof(pdbrec[46:54])
				curatom.q = _Ftof(pdbrec[54:60])
				curatom.b = _Ftof(pdbrec[60:66])
				pdbrec = pdbf.readline()
				if len(pdbrec) and pdbrec[0:6] in ('ATOM  ','HETATM'): moretogo = 1
			if psf:
				curatom.name = psfrec[4]
				curatom.type = psfrec[5]
				curatom.id = string.atoi(psfrec[0])
				curatom.mass = string.atof(psfrec[7])
				curatom.charge = string.atof(psfrec[6])
				psfrec = string.split(psff.readline())
				if nrecs > numread: moretogo = 1
		if pdb: pdbf.close()
		if psf: psff.close()
		self.buildlists()
		self.buildrefs()
	def buildrefs(self):
		for s in self.segments:
			s.molecule = self
			s.buildrefs()
	def writepdb(self,pdbfile=None):
		if not pdbfile:
			pdbfile = self.pdbfile
		if not pdbfile:
			raise "No pdb file specified."
		f = open(pdbfile,'w')
		for r in self.pdbremarks:
			if r[0:6] == 'REMARK' :
				f.write(r+'\n')
			else:
				f.write('REMARK '+r+'\n')
		for a in self.atoms:
			f.write('ATOM  ')
			f.write(string.rjust(str(a.id),5)+' ')
			if len(a.name) > 3:
				f.write(string.ljust(a.name,4)+' ')
			else:
				f.write(' '+string.ljust(a.name,3)+' ')
			f.write(string.ljust(a.residue.name,4))
			f.write(' '+string.rjust(str(a.residue.id),4)+'    ')
			f.write(string.rjust(_sround(a.x,3),8))
			f.write(string.rjust(_sround(a.y,3),8))
			f.write(string.rjust(_sround(a.z,3),8))
			f.write(string.rjust(_sround(a.q,2),6))
			f.write(string.rjust(_sround(a.b,2),6))
			f.write(string.rjust(a.residue.segment.name,10))
			f.write('\n')
		f.write('END\n')
		f.close()
	def display(self,file=None):
		if file is None:
			self.writepdb(pdbdisplayfunction())
		else:
			self.writepdb(file)
			pdbdisplayfunction(file)
	def __repr__(self):
		return '< Molecule with '\
			+`len(self.segments)`+' segments, '\
			+`len(self.residues)`+' residues, and '\
			+`len(self.atoms)`+' atoms >'

#
# Trans class hierarcy:
#                                        Trans
#

class Trans:
	"""Transformation matrix generator.

Data: matrix

Methods:
   t = Trans([shift],[center],[axis],[angle],[units])
      NOTE: (x,y,z) or (x,y,z,1) treated as Vector, (x,y,z,0) as Coord
      shift=Vector: translate by this vector (applied last)
      shift=Coord: translate this coordinate to the origin (applied last)
      center=Coord: rotate about this coordinate
      axis=Vector: rotate around line along this direction from center
      axis=Coord: rotate around line from center to this coordinate
      angle: amount to rotate in units
   t(atom) - modify coordinates of an atom
   t(group) - modify coordinates of a group of atoms
   t(trans2) - left-multiply another transformation

See also: HomoCoord, 'angles'
"""
	def __init__(self,shift=(0.,0.,0.),center=(0.,0.,0.),axis=(0.,0.,1.),angle=0.,units=angledefault):
		angle = angleconvert(angle,units,'rad')
		if len(axis) == 4 and axis[3] == 1:
			axis = (axis[0]-center[0],axis[1]-center[1],axis[2]-center[2])
		mc = [[1.,0.,0.,-center[0]],[0.,1.,0.,-center[1]],[0.,0.,1.,-center[2]],[0.,0.,0.,1.]]
		mci = [[1.,0.,0.,center[0]],[0.,1.,0.,center[1]],[0.,0.,1.,center[2]],[0.,0.,0.,1.]]
		if ( axis[0] or axis[1] ):
			theta = 0.5*math.pi - math.atan2(axis[1],axis[0])
		else:
			theta = 0.
		mx = [[math.cos(theta),-math.sin(theta),0.,0.],[math.sin(theta),math.cos(theta),0.,0.],[0.,0.,1.,0.],[0.,0.,0.,1.]]
		mxi = [[math.cos(theta),math.sin(theta),0.,0.],[-math.sin(theta),math.cos(theta),0.,0.],[0.,0.,1.,0.],[0.,0.,0.,1.]]
		axis = (mx[0][0]*axis[0]+mx[0][1]*axis[1],mx[1][0]*axis[0]+mx[1][1]*axis[1],axis[2])
		if ( axis[1] or axis[2] ):
			theta = 0.5*math.pi - math.atan2(axis[2],axis[1])
		else:
			theta = 0.
		my = [[1.,0.,0.,0.],[0.,math.cos(theta),-math.sin(theta),0.],[0.,math.sin(theta),math.cos(theta),0.],[0.,0.,0.,1.]]
		myi = [[1.,0.,0.,0.],[0.,math.cos(theta),math.sin(theta),0.],[0.,-math.sin(theta),math.cos(theta),0.],[0.,0.,0.,1.]]
		mz = [[math.cos(angle),-math.sin(angle),0.,0.],[math.sin(angle),math.cos(angle),0.,0.],[0.,0.,1.,0.],[0.,0.,0.,1.]]
		m0 = [[1.,0.,0.,0.],[0.,1.,0.,0.],[0.,0.,1.,0.],[0.,0.,0.,1.]]
		m1 = [[0.,0.,0.,0.],[0.,0.,0.,0.],[0.,0.,0.,0.],[0.,0.,0.,0.]]
		for i in range(0,4):
			for j in range(0,4):
				for k in range(0,4):
					m1[i][j] = m1[i][j] + mc[i][k]*m0[k][j]
		m0 = m1
		m1 = [[0.,0.,0.,0.],[0.,0.,0.,0.],[0.,0.,0.,0.],[0.,0.,0.,0.]]
		for i in range(0,4):
			for j in range(0,4):
				for k in range(0,4):
					m1[i][j] = m1[i][j] + mx[i][k]*m0[k][j]
		m0 = m1
		m1 = [[0.,0.,0.,0.],[0.,0.,0.,0.],[0.,0.,0.,0.],[0.,0.,0.,0.]]
		for i in range(0,4):
			for j in range(0,4):
				for k in range(0,4):
					m1[i][j] = m1[i][j] + my[i][k]*m0[k][j]
		m0 = m1
		m1 = [[0.,0.,0.,0.],[0.,0.,0.,0.],[0.,0.,0.,0.],[0.,0.,0.,0.]]
		for i in range(0,4):
			for j in range(0,4):
				for k in range(0,4):
					m1[i][j] = m1[i][j] + mz[i][k]*m0[k][j]
		m0 = m1
		m1 = [[0.,0.,0.,0.],[0.,0.,0.,0.],[0.,0.,0.,0.],[0.,0.,0.,0.]]
		for i in range(0,4):
			for j in range(0,4):
				for k in range(0,4):
					m1[i][j] = m1[i][j] + myi[i][k]*m0[k][j]
		m0 = m1
		m1 = [[0.,0.,0.,0.],[0.,0.,0.,0.],[0.,0.,0.,0.],[0.,0.,0.,0.]]
		for i in range(0,4):
			for j in range(0,4):
				for k in range(0,4):
					m1[i][j] = m1[i][j] + mxi[i][k]*m0[k][j]
		m0 = m1
		m1 = [[0.,0.,0.,0.],[0.,0.,0.,0.],[0.,0.,0.,0.],[0.,0.,0.,0.]]
		for i in range(0,4):
			for j in range(0,4):
				for k in range(0,4):
					m1[i][j] = m1[i][j] + mci[i][k]*m0[k][j]
		m0 = m1
		if len(shift) == 4 and shift[3] == 1:
			shift = (-shift[0],-shift[1],-shift[2])
		ms = [[1.,0.,0.,shift[0]],[0.,1.,0.,shift[1]],[0.,0.,1.,shift[2]],[0.,0.,0.,1.]]
		m1 = [[0.,0.,0.,0.],[0.,0.,0.,0.],[0.,0.,0.,0.],[0.,0.,0.,0.]]
		for i in range(0,4):
			for j in range(0,4):
				for k in range(0,4):
					m1[i][j] = m1[i][j] + ms[i][k]*m0[k][j]
		self.matrix = tuple(map(tuple,m1))
	def __call__(self,coord):
		if ( hasattr(coord,'x') and hasattr(coord,'y') and hasattr(coord,'z') ):
			xnew = ( self.matrix[0][0] * coord.x + self.matrix[0][1] * coord.y +
				self.matrix[0][2] * coord.z + self.matrix[0][3] )
			ynew = ( self.matrix[1][0] * coord.x + self.matrix[1][1] * coord.y +
				self.matrix[1][2] * coord.z + self.matrix[1][3] )
			znew = ( self.matrix[2][0] * coord.x + self.matrix[2][1] * coord.y +
				self.matrix[2][2] * coord.z + self.matrix[2][3] )
			coord.x = xnew
			coord.y = ynew
			coord.z = znew
		elif hasattr(coord,'atoms'):
			for a in coord.atoms:
				self(a)
		elif hasattr(coord,'residues'):
			for a in coord.residues:
				self(a)
		elif hasattr(coord,'matrix'):
			m2 = coord.matrix
			m1 = self.matrix
			m = [[0.,0.,0.,0.],[0.,0.,0.,0.],[0.,0.,0.,0.],[0.,0.,0.,0.]]
			for i in range(0,4):
				for j in range(0,4):
					for k in range(0,4):
						m[i][j] = m[i][j] + m1[i][k]*m2[k][j]
			coord.matrix = tuple(map(tuple,m))
	def __repr__(self):
		str = "< Trans ("
		for i in self.matrix:
			str = str + "("
			for j in i:
				str = str + repr(j) + ","
			str = str[:-1] + "),"
		str = str[:-1] + ")"
		return str + " >"

#
# DCD class hierarcy:
#                                        DCD
#                                         |
#                                      DCDWrite
#

class DCD:
	"""Reads from a DCD file.

Data: dcdfile, file, posNSET, NSET, ISTART, NSAVC, NAMNF, DELTA, remarks, NTITLE, N, FREEINDEXES, fixed, pos1, pos2, rlen, numframes, numatoms

Methods:
   d = DCD(dcdfile) - open dcdfile and read header
   len(d) - return number of frames
   d[10] - return a frame as a list of (x,y,z)
   s = d.asel() - return a selection of dummy atoms for all atoms in d
   sf = d.aselfree(mol) - return a selection from mol of the free atoms in d
   sf2 = d.aselfree() - return a selection of dummy atoms for free atoms in d
   sf.getmolframe(d[10]) - load only free atoms from d into mol

See also: AtomGroup, ASel
"""
	def __init__(self,dcdfile):
		self.dcdfile = dcdfile
		self.file = open(dcdfile,'rb')
		# Abbreviations
		up = struct.unpack
		cs = struct.calcsize
		f = self.file
		# Read header
		dat = up('i4c',f.read(cs('i4c')))
		if dat[0] != 84 or dat[1:5] != ('C','O','R','D') :
			raise "DCD format error 1"
		self.posNSET = f.tell()
		self.NSET = up('i',f.read(cs('i')))[0]
		self.ISTART = up('i',f.read(cs('i')))[0]
		self.NSAVC = up('i',f.read(cs('i')))[0]
		f.read(cs('4i'))
		f.read(cs('i')) # Why?
		self.NAMNF = up('i',f.read(cs('i')))[0]
		self.DELTA = up('d',f.read(cs('d')))[0]
		f.read(cs('i')) # Why?
		f.read(cs('8i'))
		dat = up('i',f.read(cs('i')))[0]
		if dat != 84 :
			raise "DCD format error 2"
		size = up('i',f.read(cs('i')))[0]
		if (size-4)%80 != 0 :
			raise "DCD format error 3"
		self.remarks = []
		self.NTITLE = up('i',f.read(cs('i')))[0]
		for i in range(0,self.NTITLE):
			dat = up('80c',f.read(cs('80c')))
			self.remarks.append(
				string.strip(string.joinfields(dat,'')))
			print self.remarks[-1]
		if up('i',f.read(cs('i')))[0] != size :
			raise "DCD format error 4"
		if up('i',f.read(cs('i')))[0] != 4 :
			raise "DCD format error 5"
		self.N = up('i',f.read(cs('i')))[0]
		if up('i',f.read(cs('i')))[0] != 4 :
			raise "DCD format error 6"
		if self.NAMNF:
			size = up('i',f.read(cs('i')))[0]
			if size != (self.N-self.NAMNF)*cs('i') :
				raise "DCD format error 7"
			self.FREEINDEXES = up(repr(self.N-self.NAMNF)+'i',
				f.read(cs(repr(self.N-self.NAMNF)+'i')))
			if up('i',f.read(cs('i')))[0] != size :
				raise "DCD format error 8"
		else:
			self.FREEINDEXES = ()
		self.fixed = self.NAMNF
		self.pos1 = f.tell()
		self.pos2 = self.pos1 + cs(repr(3*self.N)+'f6i')
		self.rlen = cs(repr(3*(self.N-self.NAMNF))+'f6i')
		if self.fixed :
			size = up('i',f.read(cs('i')))[0]
			if size != cs(repr(self.N)+'f') :
				raise "DCD format error 9"
			self.xbuff = up(repr(self.N)+'f',f.read(cs(repr(self.N)+'f')))
			size = up('i',f.read(cs('i')))[0]
			if size != cs(repr(self.N)+'f') :
				raise "DCD format error 10"
			size = up('i',f.read(cs('i')))[0]
			if size != cs(repr(self.N)+'f') :
				raise "DCD format error 11"
			self.ybuff = up(repr(self.N)+'f',f.read(cs(repr(self.N)+'f')))
			size = up('i',f.read(cs('i')))[0]
			if size != cs(repr(self.N)+'f') :
				raise "DCD format error 12"
			size = up('i',f.read(cs('i')))[0]
			if size != cs(repr(self.N)+'f') :
				raise "DCD format error 13"
			self.zbuff = up(repr(self.N)+'f',f.read(cs(repr(self.N)+'f')))
			size = up('i',f.read(cs('i')))[0]
			if size != cs(repr(self.N)+'f') :
				raise "DCD format error 14"
		f.seek(0,2)
		self.numframes = (f.tell()-self.pos2)/self.rlen + 1
		self.numatoms = self.N
	def __getitem__(self,fn):
		# Abbreviations
		up = struct.unpack
		cs = struct.calcsize
		f = self.file
		# Find the right point in the file
		if fn < -1*self.numframes or fn >= self.numframes :
			raise IndexError
		elif fn == 0 and self.fixed :
			return map(None,self.xbuff,self.ybuff,self.zbuff)
		elif fn < 0 :
			return self.__getitem__(self.numframes + fn)
		else :
			f.seek(self.pos2 + (fn-1)*self.rlen)
		# Read data
		size = up('i',f.read(cs('i')))[0]
		if self.fixed == 0 :
			if size != cs(repr(self.N)+'f') :
				raise "DCD format error 9"
			x = up(repr(self.N)+'f',f.read(cs(repr(self.N)+'f')))
			size = up('i',f.read(cs('i')))[0]
			if size != cs(repr(self.N)+'f') :
				raise "DCD format error 10"
			size = up('i',f.read(cs('i')))[0]
			if size != cs(repr(self.N)+'f') :
				raise "DCD format error 11"
			y = up(repr(self.N)+'f',f.read(cs(repr(self.N)+'f')))
			size = up('i',f.read(cs('i')))[0]
			if size != cs(repr(self.N)+'f') :
				raise "DCD format error 12"
			size = up('i',f.read(cs('i')))[0]
			if size != cs(repr(self.N)+'f') :
				raise "DCD format error 13"
			z = up(repr(self.N)+'f',f.read(cs(repr(self.N)+'f')))
			size = up('i',f.read(cs('i')))[0]
			if size != cs(repr(self.N)+'f') :
				raise "DCD format error 14"
			frame = map(None,x,y,z)
		else:
			free = len(self.FREEINDEXES)
			fm = repr(free)+'f'
			sz = cs(fm)
			if size != sz :
				raise "DCD format error 9"
			xfree = up(fm,f.read(sz))
			size = up('i',f.read(cs('i')))[0]
			if size != sz :
				raise "DCD format error 10"
			size = up('i',f.read(cs('i')))[0]
			if size != sz :
				raise "DCD format error 11"
			yfree = up(fm,f.read(sz))
			size = up('i',f.read(cs('i')))[0]
			if size != sz :
				raise "DCD format error 12"
			size = up('i',f.read(cs('i')))[0]
			if size != sz :
				raise "DCD format error 13"
			zfree = up(fm,f.read(sz))
			size = up('i',f.read(cs('i')))[0]
			if size != sz :
				raise "DCD format error 14"
			frame = map(None,self.xbuff,self.ybuff,self.zbuff)
			ii = 0
			for i in self.FREEINDEXES:
				frame[i-1] = (xfree[ii],yfree[ii],zfree[ii])
				ii = ii + 1
		return frame
	def asel(self):
		fakemol = AtomGroup()
		for i in range(0,self.numatoms):
   			a = Atom()
			a.id = i + 1
			fakemol.atoms.append(a)
		return fakemol
	def aselfree(self,mol=None):
		if ( mol is None ):
			if ( self.fixed ):
				fakemol = AtomGroup()
				for id in self.FREEINDEXES:
   					a = Atom()
					a.id = id
					fakemol.atoms.append(a)
				return fakemol
			else:
				return self.asel()
		else:
			if ( self.fixed ):
				return ASel(mol,lambda a,l=self.FREEINDEXES: a.id in l)
			else:
				return ASel(mol,lambda a,l=range(1,self.numatoms+1): a.id in l)
	def __len__(self):
		return self.numframes
	def __del__(self):
		self.file.close()
	def __repr__(self):
		return "< DCD " + self.dcdfile + " with " + repr(self.numframes) + " frames of " + repr(self.numatoms) + " atoms (" + repr(self.fixed) + " fixed) >"

class DCDWrite(DCD):
	"""Writes a DCD file.  Can only append.

Data: allatoms, freeatoms.

Methods:
   d = DCD(dcdfile,atoms,[free],[ISTART],[NSAVC],[DELTA]) - open dcdfile
      dcdfile: must not exist
      atoms=AtomGroup: all atoms to save to the file
      free=AtomGroup: if there are fixed atoms, these are the free ones
      ISTART: first timestep
      NSAVC: saving frequency
      DELTA: timestep
   d.append() - append the current coordinates of atoms (free) to the file

Note: DO NOT modify atoms or free while DCDWrite is being used.

See also: AtomGroup
"""
	def __init__(self,dcdfile,atoms,free=None,**header):
		self.allatoms = atoms
		if free is None: self.freeatoms = self.allatoms
		else: self.freeatoms = free
		self.dcdfile = dcdfile
		try: open(dcdfile,'rb').close()
		except: self.file = open(dcdfile,'w+b')
		else: raise 'file '+dcdfile+' exists and this class will not overwrite it'
		# Abbreviations
		p = struct.pack
		cs = struct.calcsize
		f = self.file
		# Write header
		f.write(p('i4c',84,'C','O','R','D'))
		self.posNSET = f.tell()
		self.NSET = 0
		self.numframes = 0
		f.write(p('i',self.NSET))
		try:    self.ISTART = header['ISTART']
		except: self.ISTART = 0
		f.write(p('i',self.ISTART))
		try:    self.NSAVC = header['NSAVC']
		except: self.NSAVC = 0
		f.write(p('i',self.NSAVC))
		f.write(p('4i',0,0,0,0))
		f.write(p('i',0)) # Why?
		self.NAMNF = len(self.allatoms.atoms) - len(self.freeatoms.atoms)
		f.write(p('i',self.NAMNF))
		self.fixed = self.NAMNF
		try:    self.DELTA = header['DELTA']
		except: self.DELTA = 0.
		f.write(p('d',self.DELTA))
		f.write(p('i',0)) # Why?
		f.write(p('8i',0,0,0,0,0,0,0,0))
		f.write(p('i',84))
		rawremarks = [ \
			'REMARKS FILENAME='+self.dcdfile+' CREATED BY MDTools for Python',
			'REMARKS DATE: '+time.ctime(time.time())+' CREATED BY USER: '+os.environ['USER']]
		self.remarks = []
		for r in rawremarks: self.remarks.append(string.ljust(r,80)[0:80])
		size = cs('i')+80*len(self.remarks)
		f.write(p('i',size))
		self.NTITLE = len(self.remarks)
		f.write(p('i',self.NTITLE))
		for r in self.remarks:
			f.write(apply(p,tuple(['80c']+map(None,r))))
			print string.strip(r)
		f.write(p('i',size))
		f.write(p('i',cs('i')))
		self.N = len(self.allatoms.atoms)
		self.numatoms = self.N
		f.write(p('i',self.N))
		f.write(p('i',cs('i')))
		if self.NAMNF:
			size = (self.N-self.NAMNF)*cs('i')
			f.write(p('i',size))
			#fi = []
			#for i in range(0,len(self.allatoms.atoms)):
			#	if self.allatoms.atoms[i] in self.freeatoms.atoms: fi.append(i+1)
			fi = map(lambda a,l=self.allatoms.atoms: l.index(a)+1,self.freeatoms.atoms)
			self.FREEINDEXES = tuple(fi)
			if len(self.FREEINDEXES) != self.N-self.NAMNF:
				raise "some free atoms were not in atoms list"
			fi[:0] = [repr(self.N-self.NAMNF)+'i']
			f.write(apply(p,tuple(fi)))
			f.write(p('i',size))
		else:
			self.FREEINDEXES = ()
		self.pos1 = f.tell()
		self.pos2 = self.pos1 + cs(repr(3*self.N)+'f6i')
		self.rlen = cs(repr(3*(self.N-self.NAMNF))+'f6i')
		f.flush()
	def append(self):
		p = struct.pack
		cs = struct.calcsize
		f = self.file
		fn = self.NSET
		self.NSET = self.NSET + 1
		self.numframes = self.NSET
		f.seek(self.posNSET)
		f.write(p('i',self.NSET))
		f.seek(0,2)
		if fn:
			fr = []
			self.freeatoms.putframe(fr)
			fr[:0] = [None]
			(x,y,z) = tuple(apply(map,tuple(fr)))
		else:
			fr = []
			self.allatoms.putframe(fr)
			fr[:0] = [None]
			(x,y,z) = tuple(apply(map,tuple(fr)))
			if self.fixed :
				(self.xbuff,self.ybuff,self.zbuff) = (x,y,z)
		fm = repr(len(x))+'f'
		size = cs(fm)
		f.write(p('i',size))
		f.write(apply(p,(fm,)+x))
		f.write(p('i',size))
		f.write(p('i',size))
		f.write(apply(p,(fm,)+y))
		f.write(p('i',size))
		f.write(p('i',size))
		f.write(apply(p,(fm,)+z))
		f.write(p('i',size))
		f.flush()
	def __repr__(self):
		return "< DCDWrite " + self.dcdfile + " with " + repr(self.numframes) + " frames of " + repr(self.numatoms) + " atoms (" + repr(self.fixed) + " fixed) >"

#
# Data class hierarchy
#                                       Data
#                                         |
#                                     NAMDOutput
#

class Data:
	"""General structure for sequence of data points.

Data: fields, names, data

Methods:
   d = Data(fields,[data])
      fields: list or tuple of field names as in ('t','x','y','z')
   d.append(rec) - append a record such as (1,1.2,3.2,5.1)
   d[8:13] - return a list of records
   len(d) - return number of records
   addfield(name,args,func) - add a field based on other fields
      name: name of new field, as in 'x+y'
      args: tuple of arguments for func, as in ('x','y'), just one, as in 'x'
      func: function to create new field, as in lambda x,y:x+y
   addindex([offset],[name]) - add an index field
   filter(args,func) - return new Data of records that pass filter func
   average(args,[func]) - mean of function (default is x)
   deviation(args,[func]) - STD of function (default is x)
   Warning: deviation() divides by N - 1 to estimate the STD from a sample
   plot([args],[file]) - save args to a file (tempfile by default) and plot
   list([args],[file],[titles]) - print to screen (file) w/ or w/o titles

See also: xyplotfunction
"""
	def __init__(self,fields,data=[]):
		self.fields = {}
		self.names = tuple(fields)
		self.data = []
		for f in fields:
			if type(f) is types.IntType:
				raise "integer field names not allowed"
			if f in self.fields.keys():
				raise "duplicate field name "+`f`
			self.fields[f] = len(self.fields)
		for d in data:
			self.data.append(tuple(d))
	def __getitem__(self,key):
		if type(key) is types.IntType:
			return self.data[key]
		if type(key) is types.TupleType:
			fs = map(lambda k,d=self.fields: d[k],key)
			tfunc = lambda rec,f=fs: \
				tuple(map(lambda e,r=rec:r[e],f))
			return map(tfunc,self.data)
		else:
			return map(lambda r,f=self.fields[key]: r[f],self.data)
	def __getslice__(self,i,j):
		return self.data[i:j]
	def __len__(self):
		return len(self.data)
	def __repr__(self):
		return '< Data with '+`len(self.data)`+' frames of '+`self.names`+' data >'
	def append(self,rec):
		if ( len(rec) != len(self.fields) ):
			raise 'appending wrong length record'
		self.data.append(tuple(rec))
	def addfield(self,name,args,func):
		if type(args) is not types.TupleType: args = (args,)
		fs = map(lambda k,d=self.fields: d[k],args)
		tfunc = lambda rec,f=fs: \
			tuple(map(lambda e,r=rec:r[e],f))
		data = []
		for d in self.data:
			dl = map(None,d)
			dl.append(apply(func,tfunc(d)))
			data.append(tuple(dl))
		self.fields[name] = len(self.fields)
		nl = map(None,self.names)
		nl.append(name)
		self.names = tuple(nl)
		self.data = data
	def addindex(self,offset=0,name='index'):
		if name in self.names:
			raise 'field name '+`name`+' already in use'
		for i in range(0,len(self.data)):
			dl = map(None,self.data[i])
			dl[:0] = [i+offset]
			self.data[i] = tuple(dl)
		nl = map(None,self.names)
		nl[:0] = [name]
		self.names = tuple(nl)
		for n in self.fields.keys():
			self.fields[n] = self.fields[n] + 1
		self.fields[name] = 0
	def filter(self,args,func):
		if type(args) is not types.TupleType: args = (args,)
		fs = map(lambda k,d=self.fields: d[k],args)
		tfunc = lambda rec,f=fs: \
			tuple(map(lambda e,r=rec:r[e],f))
		ffunc = lambda r,t=tfunc,f=func: apply(f,t(r))
		return Data(self.names,filter(ffunc,self.data))
	def average(self,args,func=lambda x:x,zero=0.):
		if type(args) is not types.TupleType: args = (args,)
		fs = map(lambda k,d=self.fields: d[k],args)
		tfunc = lambda rec,f=fs: \
			tuple(map(lambda e,r=rec:r[e],f))
		ffunc = lambda s,r,t=tfunc,f=func: s+apply(f,t(r))
		return reduce(ffunc,self.data,zero)/len(self.data)
	def deviation(self,args,func=lambda x:x,zero=0.):
		if type(args) is not types.TupleType: args = (args,)
		fs = map(lambda k,d=self.fields: d[k],args)
		tfunc = lambda rec,f=fs: \
			tuple(map(lambda e,r=rec:r[e],f))
		ffunc = lambda s,r,t=tfunc,f=func: s+apply(f,t(r))
		avg = reduce(ffunc,self.data,zero)/len(self.data)
		ffunc = lambda s,r,t=tfunc,f=func,a=avg: s+pow(apply(f,t(r))-a,2)
		return math.sqrt(reduce(ffunc,self.data,zero)/(len(self.data)-1))
	def plot(self,args=None,file=None):
		if args is None: args = self.names
		if type(args) is not types.TupleType: args = (args,)
		if file is None: wfile = xyplotfunction()
		else: wfile = file
		f = open(wfile,'w')
		for e in args:
			f.write(e+' ')
		f.write('\n')
		for r in self.data:
			if None not in r:
				for e in args:
					f.write(`r[self.fields[e]]`+' ')
				f.write('\n')
		f.close()
		if file is not None: xyplotfunction(file)
	def list(self,args=None,file=None,titles=1):
		if args is None: args = self.names
		if type(args) is not types.TupleType: args = (args,)
		if file is None: f = sys.stdout
		else: f = open(file,'w')
		if titles:
			for e in args:
				f.write(string.center(e,18))
		f.write('\n')
		for r in self.data:
			for e in args:
				f.write(string.center(`r[self.fields[e]]`,18))
			f.write('\n')
		if file is not None: f.close()

def _NAMD_infochop(s):
	p = string.find(s,'>')
	if p == -1:
		return ''
	else:
		return s[p+1:]

class NAMDOutput(Data):
	"""Reads output files of the molecular dynamics program NAMD.

Data: timestep, namdfields

Methods:
   d = NAMDOutput(namdfile,[fields=('TS','TEMP')])
   d.append(namdfile) - append another output file
   d.addtime() - add a 'TIME' field based on timestep
   d.plot([args],[file]) - same as Data but eliminates 'TS' if 'TIME' present

See also: http://www.ks.uiuc.edu/Research/namd/
"""
	def __init__(self,namdfile,fields=('TS','TEMP')):
		Data.__init__(self,fields)
		self.namdfile = namdfile
		self.namdfields = {}
		self.timestep = 0
		f = open(self.namdfile,'r')
		raw = f.readline()
		rec = string.split(_NAMD_infochop(raw))
		while len(raw) and not self.namdfields :
			if len(rec) and rec[0] == 'ETITLE:' :
				self.namdfields = {}
				for fn in self.names:
					self.namdfields[fn] = rec.index(fn)
			elif len(rec) and rec[0] == 'TIMESTEP' :
				self.timestep = string.atof(rec[1]) / 1000.0
			raw = f.readline()
			rec = string.split(_NAMD_infochop(raw))
		fieldnums = []
		for fn in self.names:
			fieldnums.append(self.namdfields[fn])
		while len(raw) :
			if len(rec) and rec[0] == 'ENERGY:' :
				dr = []
				for i in fieldnums:
					dr.append(string.atof(rec[i]))
				self.data.append(tuple(dr))
			raw = f.readline()
			rec = string.split(_NAMD_infochop(raw))
		f.close()
	def append(self,namdfile):
		f = open(self.namdfile,'r')
		fieldnums = []
		for fn in self.names:
			fieldnums.append(self.namdfields[fn])
		raw = f.readline()
		rec = string.split(_NAMD_infochop(raw))
		while len(raw) :
			if len(rec) and rec[0] == 'ENERGY:' :
				dr = []
				for i in fieldnums:
					dr.append(string.atof(rec[i]))
				self.data.append(tuple(dr))
			raw = f.readline()
			rec = string.split(_NAMD_infochop(raw))
		f.close()
	def addtime(self):
		self.addfield('TIME','TS',lambda ts,d=self.timestep:ts*d)
	def plot(self,args=None,file=None):
		if args is None and 'TIME' in self.names:
			l = map(None,(self.names))
			l.remove('TIME')
			try: l.remove('TS')
			except: pass
			l[0:0] = ['TIME']
			Data.plot(self,tuple(l),file)
		else:
			Data.plot(self,args,file)
