/***************************************************************************
 *cr
 *cr            (C) Copyright 2007-2009 The Board of Trustees of the
 *cr                        University of Illinois
 *cr                         All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: OpenCLUtils.C,v $
 *      $Author: johns $        $Locker:  $             $State: Exp $
 *      $Revision: 1.6 $      $Date: 2010/03/23 18:31:18 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *   OpenCL utility functions for use in VMD
 *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if defined(__APPLE__)
#include <OpenCL/cl.h>
#else
#include <CL/cl.h>
#endif

#if defined(VMDOPENCL)
#include "Inform.h"
#endif

int vmd_cl_print_platform_info(void) {
  cl_int clerr;
  cl_uint numplatforms;
  cl_platform_id *platformlist;
  clerr=clGetPlatformIDs(0, NULL, &numplatforms);
  platformlist = (cl_platform_id *) malloc(sizeof(cl_platform_id)*numplatforms);
  clerr=clGetPlatformIDs(numplatforms, platformlist, NULL);

  cl_uint i;
  for (i=0; i<numplatforms; i++) {
    char platname[80];
    clerr=clGetPlatformInfo(platformlist[i], CL_PLATFORM_NAME, 
                            sizeof(platname), (void *) platname, NULL);

    char platprofile[80];
    clerr=clGetPlatformInfo(platformlist[i], CL_PLATFORM_PROFILE, 
                            sizeof(platprofile), (void *) platprofile, NULL);

#if 0
    char platvendor[80];
    clerr=clGetPlatformInfo(platformlist[i], CL_PLATFORM_VENDOR, 
                            sizeof(platvendor), (void *) platvendor, NULL);
#endif

    cl_uint numdevs;
    clerr=clGetDeviceIDs(platformlist[i], CL_DEVICE_TYPE_ALL,
                         0, NULL, &numdevs);    

    char platforminfo[4096];
#if !defined(VMDOPENCL)
    printf("OpenCL Platform[%d]: %s, %s  Devices: %u\n",
           i, platname, platprofile, numdevs);
#else
    sprintf(platforminfo, "OpenCL Platform[%d]: %s, %s  Devices: %u\n",
           i, platname, platprofile, numdevs);
    msgInfo << platforminfo << sendmsg;
#endif
  }

  free(platformlist);

  return 0;
}


cl_platform_id vmd_cl_get_platform_index(int i) {
  cl_int clerr;
  cl_uint numplatforms;
  cl_platform_id *platformlist;
  cl_platform_id plat;    
  clerr=clGetPlatformIDs(0, NULL, &numplatforms);
  if (i >= (int) numplatforms)
    return NULL;

  platformlist = (cl_platform_id *) malloc(sizeof(cl_platform_id)*numplatforms);
  clerr=clGetPlatformIDs(numplatforms, platformlist, NULL);
  if (clerr != CL_SUCCESS) {
    free(platformlist);
    return NULL;
  }

  plat=platformlist[i];
  free(platformlist);

  return plat;
}  


int vmd_cl_context_num_devices(cl_context clctx) {
  size_t parmsz;
  cl_int clerr = clGetContextInfo(clctx, CL_CONTEXT_DEVICES, 0, NULL, &parmsz);
  if (clerr != CL_SUCCESS)
    return 0;
  
  return (int) (parmsz / sizeof(size_t));
}


cl_command_queue vmd_cl_create_command_queue(cl_context clctx, int dev) {
  size_t parmsz;
  cl_int clerr = clGetContextInfo(clctx, CL_CONTEXT_DEVICES, 0, NULL, &parmsz);
  if (clerr != CL_SUCCESS)
    return NULL;

  cl_device_id* cldevs = (cl_device_id *) malloc(parmsz);
  clerr = clGetContextInfo(clctx, CL_CONTEXT_DEVICES, parmsz, cldevs, NULL);
  if (clerr != CL_SUCCESS)
    return NULL;

  cl_command_queue clcmdq = clCreateCommandQueue(clctx, cldevs[dev], 0, &clerr);
  free(cldevs);
  if (clerr != CL_SUCCESS)
    return NULL;

  return clcmdq;
}


cl_kernel vmd_cl_compile_kernel(cl_context clctx, const char *kernname,
                                 const char *srctext, const char *flags, 
                                 cl_int *clerr, int verbose) {
  char buildlog[8192];
  cl_program clpgm = NULL;
  cl_kernel clkern = NULL;

  clpgm = clCreateProgramWithSource(clctx, 1, &srctext, NULL, clerr);
  if (clerr != CL_SUCCESS) {
    if (verbose)
      printf("Failed to compile OpenCL kernel: '%s'\n", kernname);
    return NULL;
  }
  *clerr = clBuildProgram(clpgm, 0, NULL, flags, NULL, NULL);

#if 1
  if (verbose) {
    memset(buildlog, 0, sizeof(buildlog));

    size_t parmsz;
    *clerr |= clGetContextInfo(clctx, CL_CONTEXT_DEVICES, 0, NULL, &parmsz);

    cl_device_id* cldevs = (cl_device_id *) malloc(parmsz);
    *clerr |= clGetContextInfo(clctx, CL_CONTEXT_DEVICES, parmsz, cldevs, NULL);

    size_t len=0;
    *clerr = clGetProgramBuildInfo(clpgm, cldevs[0], CL_PROGRAM_BUILD_LOG, sizeof(buildlog), buildlog, &len);
    if (len > 1) {
      printf("OpenCL kernel compilation log:\n");
      printf("  '%s'\n", buildlog);
    }
  }   
#endif

  clkern = clCreateKernel(clpgm, "clenergy", clerr);
  if (clerr != CL_SUCCESS) {
    if (verbose)
      printf("Failed to create OpenCL kernel: '%s'\n", kernname);
    return NULL;
  }

  return clkern;
} 


