set pi 3.14159265358979323846

# Shift the atom coordinates to the origin
proc calc_Rj {args} {
  # Number of atoms
  set num_atoms [expr [llength $args] / 3]
  # Center of geometry
  set cog_x 0.0
  set cog_y 0.0
  set cog_z 0.0
  for {set i 0} {$i < $num_atoms} {incr i} {
    set i_atom [expr $i * 3]
    set cog_x [expr $cog_x + [lindex $args [expr $i_atom + 0]]]
    set cog_y [expr $cog_y + [lindex $args [expr $i_atom + 1]]]
    set cog_z [expr $cog_z + [lindex $args [expr $i_atom + 2]]]
  }
  set cog_x [expr $cog_x / $num_atoms]
  set cog_y [expr $cog_y / $num_atoms]
  set cog_z [expr $cog_z / $num_atoms]
  # R_j = r_j - R_cog
  set result [list]
  for {set i 0} {$i < $num_atoms} {incr i} {
    set i_atom [expr $i * 3]
    set R_jx [expr [lindex $args [expr $i_atom + 0]] - $cog_x]
    set R_jy [expr [lindex $args [expr $i_atom + 1]] - $cog_y]
    set R_jz [expr [lindex $args [expr $i_atom + 2]] - $cog_z]
    lappend result [list $R_jx $R_jy $R_jz]
  }
  return $result
}

# Calculate z_j, args are R_j
proc calc_zj {args} {
  # Calculate R' and R''
  set N [llength $args]
  set Rpx 0.0
  set Rpy 0.0
  set Rpz 0.0
  set Rppx 0.0
  set Rppy 0.0
  set Rppz 0.0
  global pi
  # Calculate R' and R''
  for {set i 0} {$i < $N} {incr i} {
    set factor [expr (2.0 * $pi * $i) / $N]
    set sin_f [expr sin($factor)]
    set cos_f [expr cos($factor)]
    set R_j [lindex $args $i]
    set R_jx [lindex $R_j 0]
    set R_jy [lindex $R_j 1]
    set R_jz [lindex $R_j 2]
    set Rpx [expr $Rpx + $sin_f * $R_jx]
    set Rpy [expr $Rpy + $sin_f * $R_jy]
    set Rpz [expr $Rpz + $sin_f * $R_jz]
    set Rppx [expr $Rppx + $cos_f * $R_jx]
    set Rppy [expr $Rppy + $cos_f * $R_jy]
    set Rppz [expr $Rppz + $cos_f * $R_jz]
  }
  # Calculate the normal vector (step 1): R' cross R''
  set cross_x [expr $Rpy * $Rppz - $Rpz * $Rppy]
  set cross_y [expr $Rpz * $Rppx - $Rpx * $Rppz]
  set cross_z [expr $Rpx * $Rppy - $Rpy * $Rppx]
  # Calculate the normal vector (step 2): |R' cross R''|
  set n_norm_factor [expr sqrt($cross_x * $cross_x + $cross_y * $cross_y + $cross_z * $cross_z)]
  set n_x [expr $cross_x / $n_norm_factor]
  set n_y [expr $cross_y / $n_norm_factor]
  set n_z [expr $cross_z / $n_norm_factor]
  set z [list]
  set n [list]
  # Project R_j onto the normal vector
  for {set i 0} {$i < $N} {incr i} {
    set R_j [lindex $args $i]
    set R_jx [lindex $R_j 0]
    set R_jy [lindex $R_j 1]
    set R_jz [lindex $R_j 2]
    set z_j [expr $R_jx * $n_x + $R_jy * $n_y + $R_jz * $n_z]
    lappend z $z_j
  }
  return $z
}

# Calculate A, B and C from z_j
proc calc_ABC {args} {
  set N [llength $args]
  set A 0.0
  set B 0.0
  set C 0.0
  global pi
  for {set i 0} {$i < $N} {incr i} {
    set factor [expr 2.0 * $pi / $N * 2 * $i]
    set sin_f [expr sin($factor)]
    set cos_f [expr cos($factor)]
    set A [expr $A + [lindex $args $i] * $sin_f]
    set B [expr $B + [lindex $args $i] * $cos_f]
    set C [expr $C + [lindex $args $i] * ((-2.0) * ($i % 2) + 1.0)]
  }
  return [list $A $B $C]
}

proc calc_cptheta {args} {
  # args is a list of atom coordinates of length 3*N
  set R_j [calc_Rj {*}[lindex $args 0]]
  set z_j [calc_zj {*}$R_j]
  lassign [calc_ABC {*}$z_j] A B C
  global pi
  set cptheta [expr 180.0 / $pi * \
    acos($C / sqrt(2.0 * $A * $A + 2.0 * $B * $B + $C * $C))]
  return $cptheta
}

proc calc_cpphi {args} {
  # args is a list of atom coordinates of length 3*N
  set R_j [calc_Rj {*}[lindex $args 0]]
  set z_j [calc_zj {*}$R_j]
  lassign [calc_ABC {*}$z_j] A B C
  global pi
  set cpphi [expr 180.0 / $pi * (atan2(-$A, $B))]
  if {$cpphi < 0} {
    set $cpphi [expr $cpphi + 360.0]
  }
  return $cpphi
}

proc calc_cpQ {args} {
  # args is a list of atom coordinates of length 3*N
  set R_j [calc_Rj {*}[lindex $args 0]]
  set z_j [calc_zj {*}$R_j]
  lassign [calc_ABC {*}$z_j] A B C
  set q [expr sqrt((2.0 * $A * $A + 2.0 * $B * $B + $C * $C) / 6.0)]
  return $q
}

#############
# Gradients #
#############

proc cross_ab {a1 a2 a3 b1 b2 b3} {
  set cross_x [expr $a2 * $b3 - $a3 * $b2]
  set cross_y [expr $a3 * $b1 - $a1 * $b3]
  set cross_z [expr $a1 * $b2 - $a2 * $b1]
  return [list $cross_x $cross_y $cross_z]
}

proc dot_ab {a1 a2 a3 b1 b2 b3} {
  return [expr $a1 * $b1 + $a2 * $b2 + $a3 * $b3]
}

proc calc_ABC_gradient {args} {
  # Number of atoms
  set num_atoms [expr [llength $args] / 3]
  # Center of geometry
  set cog_x 0.0
  set cog_y 0.0
  set cog_z 0.0
  for {set i 0} {$i < $num_atoms} {incr i} {
    set i_atom [expr $i * 3]
    set cog_x [expr $cog_x + [lindex $args [expr $i_atom + 0]]]
    set cog_y [expr $cog_y + [lindex $args [expr $i_atom + 1]]]
    set cog_z [expr $cog_z + [lindex $args [expr $i_atom + 2]]]
  }
#   puts "num_atoms = $num_atoms"
  set cog_x [expr $cog_x / $num_atoms]
  set cog_y [expr $cog_y / $num_atoms]
  set cog_z [expr $cog_z / $num_atoms]
  # R_j = r_j - R_cog
  set result [list]
  for {set i 0} {$i < $num_atoms} {incr i} {
    set i_atom [expr $i * 3]
    set R_jx [expr [lindex $args [expr $i_atom + 0]] - $cog_x]
    set R_jy [expr [lindex $args [expr $i_atom + 1]] - $cog_y]
    set R_jz [expr [lindex $args [expr $i_atom + 2]] - $cog_z]
    lappend result [list $R_jx $R_jy $R_jz]
  }
  # Calculate R' and R''
  set N $num_atoms
  set Rpx 0.0
  set Rpy 0.0
  set Rpz 0.0
  set Rppx 0.0
  set Rppy 0.0
  set Rppz 0.0
  global pi
  for {set i 0} {$i < $N} {incr i} {
    set factor [expr (2.0 * $pi * $i) / $N]
    set sin_f [expr sin($factor)]
    set cos_f [expr cos($factor)]
    set R_j [lindex $result $i]
    set R_jx [lindex $R_j 0]
    set R_jy [lindex $R_j 1]
    set R_jz [lindex $R_j 2]
    set Rpx [expr $Rpx + $sin_f * $R_jx]
    set Rpy [expr $Rpy + $sin_f * $R_jy]
    set Rpz [expr $Rpz + $sin_f * $R_jz]
    set Rppx [expr $Rppx + $cos_f * $R_jx]
    set Rppy [expr $Rppy + $cos_f * $R_jy]
    set Rppz [expr $Rppz + $cos_f * $R_jz]
  }
  # Calculate the normal vector (step 1): R' cross R''
  set cross_x [expr $Rpy * $Rppz - $Rpz * $Rppy]
  set cross_y [expr $Rpz * $Rppx - $Rpx * $Rppz]
  set cross_z [expr $Rpx * $Rppy - $Rpy * $Rppx]
  # Calculate the normal vector (step 2): |R' cross R''|
  set denominator [expr sqrt($cross_x * $cross_x + $cross_y * $cross_y + $cross_z * $cross_z)]
  # Derivative of R_j \cdot R' \times R'' with respect to r_j
  set dA_dr {}
  set dB_dr {}
  set dC_dr {}
  set tmp2 [expr 1.0 - 1.0 / $N]
  set tmp3 [expr -1.0 / $N]
  set one_denom [expr 1.0 / $denominator]
  set one_denom_sq [expr $one_denom * $one_denom]
  for {set j 0} {$j < $N} {incr j} {
    set dA_drjx 0
    set dA_drjy 0
    set dA_drjz 0
    set dB_drjx 0
    set dB_drjy 0
    set dB_drjz 0
    set dC_drjx 0
    set dC_drjy 0
    set dC_drjz 0
    # ∂R'/∂rj
    set dRp_drj_f 0
    set dRpp_drj_f 0
    for {set k 0} {$k < $N} {incr k} {
      set factor_j [expr (2.0 * $pi * $k) / $N]
      set sin_f [expr sin($factor_j)]
      set cos_f [expr cos($factor_j)]
      if {$j == $k} {
        set dRp_drj_f [expr $dRp_drj_f + $sin_f * $tmp2]
        set dRpp_drj_f [expr $dRpp_drj_f + $cos_f * $tmp2]
      } else {
        set dRp_drj_f [expr $dRp_drj_f + $sin_f * $tmp3]
        set dRpp_drj_f [expr $dRpp_drj_f + $cos_f * $tmp3]
      }
    }
    # ∂R'/∂rj × R''
    set dRp_drjx_times_Rpp [cross_ab $dRp_drj_f 0 0 $Rppx $Rppy $Rppz]
    set dRp_drjy_times_Rpp [cross_ab 0 $dRp_drj_f 0 $Rppx $Rppy $Rppz]
    set dRp_drjz_times_Rpp [cross_ab 0 0 $dRp_drj_f $Rppx $Rppy $Rppz]
    # R' × ∂R''/∂rj
    set Rp_times_dRpp_drjx [cross_ab $Rpx $Rpy $Rpz $dRpp_drj_f 0 0]
    set Rp_times_dRpp_drjy [cross_ab $Rpx $Rpy $Rpz 0 $dRpp_drj_f 0]
    set Rp_times_dRpp_drjz [cross_ab $Rpx $Rpy $Rpz 0 0 $dRpp_drj_f]
    # Derivative of the norm
    set dnorm_dx [expr $one_denom * ([dot_ab $cross_x $cross_y $cross_z {*}$dRp_drjx_times_Rpp] + [dot_ab $cross_x $cross_y $cross_z {*}$Rp_times_dRpp_drjx])]
    set dnorm_dy [expr $one_denom * ([dot_ab $cross_x $cross_y $cross_z {*}$dRp_drjy_times_Rpp] + [dot_ab $cross_x $cross_y $cross_z {*}$Rp_times_dRpp_drjy])]
    set dnorm_dz [expr $one_denom * ([dot_ab $cross_x $cross_y $cross_z {*}$dRp_drjz_times_Rpp] + [dot_ab $cross_x $cross_y $cross_z {*}$Rp_times_dRpp_drjz])]
    for {set k 0} {$k < $N} {incr k} {
      set R_k [lindex $result $k]
      set dtriple_dx 0
      set dtriple_dy 0
      set dtriple_dz 0
      if {$j == $k} {
        set dtriple_dx [expr $tmp2 * $cross_x + [dot_ab {*}$R_k {*}$dRp_drjx_times_Rpp] + [dot_ab {*}$R_k {*}$Rp_times_dRpp_drjx]]
        set dtriple_dy [expr $tmp2 * $cross_y + [dot_ab {*}$R_k {*}$dRp_drjy_times_Rpp] + [dot_ab {*}$R_k {*}$Rp_times_dRpp_drjy]]
        set dtriple_dz [expr $tmp2 * $cross_z + [dot_ab {*}$R_k {*}$dRp_drjz_times_Rpp] + [dot_ab {*}$R_k {*}$Rp_times_dRpp_drjz]]
      } else {
        set dtriple_dx [expr $tmp3 * $cross_x + [dot_ab {*}$R_k {*}$dRp_drjx_times_Rpp] + [dot_ab {*}$R_k {*}$Rp_times_dRpp_drjx]]
        set dtriple_dy [expr $tmp3 * $cross_y + [dot_ab {*}$R_k {*}$dRp_drjy_times_Rpp] + [dot_ab {*}$R_k {*}$Rp_times_dRpp_drjy]]
        set dtriple_dz [expr $tmp3 * $cross_z + [dot_ab {*}$R_k {*}$dRp_drjz_times_Rpp] + [dot_ab {*}$R_k {*}$Rp_times_dRpp_drjz]]
      }
      # Derivative of z_j wrt r_j
      set triple [dot_ab {*}$R_k $cross_x $cross_y $cross_z]
      set dzk_drjx [expr $one_denom_sq * ($dtriple_dx * $denominator - $dnorm_dx * $triple)]
      set dzk_drjy [expr $one_denom_sq * ($dtriple_dy * $denominator - $dnorm_dy * $triple)]
      set dzk_drjz [expr $one_denom_sq * ($dtriple_dz * $denominator - $dnorm_dz * $triple)]
      set sin_f2 [expr sin(2.0 * $pi * $k * 2.0 / $N)]
      set cos_f2 [expr cos(2.0 * $pi * $k * 2.0 / $N)]
      set fC [expr (-2.0) * ($k % 2) + 1.0]
      set dA_drjx [expr $dA_drjx + $sin_f2 * $dzk_drjx]
      set dA_drjy [expr $dA_drjy + $sin_f2 * $dzk_drjy]
      set dA_drjz [expr $dA_drjz + $sin_f2 * $dzk_drjz]
      set dB_drjx [expr $dB_drjx + $cos_f2 * $dzk_drjx]
      set dB_drjy [expr $dB_drjy + $cos_f2 * $dzk_drjy]
      set dB_drjz [expr $dB_drjz + $cos_f2 * $dzk_drjz]
      set dC_drjx [expr $dC_drjx + $fC * $dzk_drjx]
      set dC_drjy [expr $dC_drjy + $fC * $dzk_drjy]
      set dC_drjz [expr $dC_drjz + $fC * $dzk_drjz]
    }
    lappend dA_dr $dA_drjx $dA_drjy $dA_drjz
    lappend dB_dr $dB_drjx $dB_drjy $dB_drjz
    lappend dC_dr $dC_drjx $dC_drjy $dC_drjz
  }
  return [list $dA_dr $dB_dr $dC_dr]
}

proc calc_cpQ_gradient {args} {
  # args is a list of atom coordinates of length 3*N
  set R_j [calc_Rj {*}[lindex $args 0]]
  set z_j [calc_zj {*}$R_j]
  lassign [calc_ABC {*}$z_j] A B C
  lassign [calc_ABC_gradient {*}[lindex $args 0]] dA dB dC
  set q [expr sqrt((2.0 * $A * $A + 2.0 * $B * $B + $C * $C) / 6.0)]
  set N [llength [lindex $args 0]]
  set grad [list]
  for {set i 0} {$i < $N} {incr i} {
    lappend grad [expr (2.0 * $A * [lindex $dA $i] + 2.0 * $B * [lindex $dB $i] + $C * [lindex $dC $i]) / (6.0 * $q)]
  }
  return [list $grad]
}

proc calc_cpphi_gradient {args} {
  global pi
  # args is a list of atom coordinates of length 3*N
  set R_j [calc_Rj {*}[lindex $args 0]]
  set z_j [calc_zj {*}$R_j]
  lassign [calc_ABC {*}$z_j] A B C
  lassign [calc_ABC_gradient {*}[lindex $args 0]] dA dB dC
  set N [llength [lindex $args 0]]
  set grad [list]
  set factor [expr 180.0 / $pi / ($A * $A + $B * $B)]
  for {set i 0} {$i < $N} {incr i} {
    lappend grad [expr $factor * (-([lindex $dA $i] * $B) + ([lindex $dB $i] * $A))]
  }
  return [list $grad]
}

proc calc_cptheta_gradient {args} {
  global pi
  set N [llength [lindex $args 0]]
  # args is a list of atom coordinates of length 3*N
  set R_j [calc_Rj {*}[lindex $args 0]]
  set z_j [calc_zj {*}$R_j]
  lassign [calc_ABC {*}$z_j] A B C
  lassign [calc_ABC_gradient {*}[lindex $args 0]] dA dB dC
  set tmp1 [expr 2.0 * ($A * $A + $B * $B) + $C * $C]
  set factor [expr -180.0 / $pi / sqrt(1.0 - ($C * $C / $tmp1)) * (1.0 / $tmp1)]
  set tmp2 [expr sqrt($tmp1)]
  set tmp3 [expr 1.0 / $tmp2]
  set grad [list]
  for {set i 0} {$i < $N} {incr i} {
    lappend grad [expr $factor * ($tmp2 * [lindex $dC $i] - $C * $tmp3 * (2.0 * $A * [lindex $dA $i] + 2.0 * $B * [lindex $dB $i] + $C * [lindex $dC $i]))]
  }
  return [list $grad]
}

#############
# Test code #
#############

# proc ABC_numerical_grad {args {delta 0.0001}} {
#   set grad_A {}
#   set grad_B {}
#   set grad_C {}
#   set N [llength $args]
#   for {set i 0} {$i < $N} {incr i} {
#     set prev $args
#     set next $args
#     lset prev $i [expr [lindex $args $i] - $delta]
#     lset next $i [expr [lindex $args $i] + $delta]
#     set R_j_prev [calc_Rj {*}$prev]
#     set z_j_prev [calc_zj {*}$R_j_prev]
#     set R_j_next [calc_Rj {*}$next]
#     set z_j_next [calc_zj {*}$R_j_next]
#     lassign [calc_ABC {*}$z_j_prev] A_prev B_prev C_prev
#     lassign [calc_ABC {*}$z_j_next] A_next B_next C_next
#     lappend grad_A [expr ($A_next - $A_prev) / (2.0 * $delta)]
#     lappend grad_B [expr ($B_next - $B_prev) / (2.0 * $delta)]
#     lappend grad_C [expr ($C_next - $C_prev) / (2.0 * $delta)]
#   }
#   return [list $grad_A $grad_B $grad_C]
# }
#
proc cpQphitheta_numerical_grad {args {delta 0.0001}} {
  set grad_A {}
  set grad_B {}
  set grad_C {}
  set N [llength $args]
  for {set i 0} {$i < $N} {incr i} {
    set prev $args
    set next $args
    lset prev $i [expr [lindex $args $i] - $delta]
    lset next $i [expr [lindex $args $i] + $delta]
    set Q_prev [calc_cpQ $prev]
    set Q_next [calc_cpQ $next]
    set phi_prev [calc_cpphi $prev]
    set phi_next [calc_cpphi $next]
    set theta_prev [calc_cptheta $prev]
    set theta_next [calc_cptheta $next]
    lappend grad_A [expr ($Q_next - $Q_prev) / (2.0 * $delta)]
    lappend grad_B [expr ($phi_next - $phi_prev) / (2.0 * $delta)]
    lappend grad_C [expr ($theta_next - $theta_prev) / (2.0 * $delta)]
  }
  return [list $grad_A $grad_B $grad_C]
}
#
proc diff_rmsd {la lb} {
  set sum 0.0
  foreach x $la y $lb {
    set sum [expr $sum + ($x - $y) * ($x - $y)]
  }
  set N [llength $la]
  set E [expr sqrt($sum)]
  return $E
}
#
# proc check_ABC_numerical_grad {args} {
#   set ABC_ag [calc_ABC_gradient {*}$args]
#   set delta 0.1
#   for {set i 0} {$i < 4} {incr i} {
#     set ABC_ng [ABC_numerical_grad $args $delta]
#     set error_A [diff_rmsd [lindex $ABC_ag 0] [lindex $ABC_ng 0]]
#     set error_B [diff_rmsd [lindex $ABC_ag 1] [lindex $ABC_ng 1]]
#     set error_C [diff_rmsd [lindex $ABC_ag 2] [lindex $ABC_ng 2]]
#     puts [format "delta = %12.5e, error_A = %12.5e, error_B = %12.5e, error_C = %12.5e" $delta $error_A $error_B $error_C]
#     set delta [expr $delta / 10.0]
#   }
# }
#
# proc test_ABC_grad {} {
#   set coords {-12.531   7.936 -13.956 -11.395   8.534 -13.420 -11.302   8.113 -11.928 -10.941   6.618 -11.997 -12.010   5.902 -12.765 -12.375   6.477 -14.077}
# #   set ABC_ng [ABC_numerical_grad $coords 1e-3]
# #   puts "$ABC_ng\n"
#   set ABC_ag [calc_ABC_gradient {*}$coords]
#   puts "$ABC_ag\n"
# #   check_ABC_numerical_grad {*}$coords
# }
#
proc test {} {
  set coords {-12.531   7.936 -13.956 -11.395   8.534 -13.420 -11.302   8.113 -11.928 -10.941   6.618 -11.997 -12.010   5.902 -12.765 -12.375   6.477 -14.077}
  set gQ [calc_cpQ_gradient $coords]
  lassign [cpQphitheta_numerical_grad $coords 0.001] gQ_numerical gphi_numerical gtheta_numerical
  puts "$gQ"
  puts "$gQ_numerical"
  set gphi [calc_cpphi_gradient $coords]
  puts "\n$gphi"
  puts "$gphi_numerical"
  set gtheta [calc_cptheta_gradient $coords]
  puts "\n$gtheta"
  puts "$gtheta_numerical"
}
