#!/usr/bin/env tclsh

# $Id: ams_genSharedData.tcl,v 1.5 2015/07/08 19:50:40 mayne Exp $

# Written by: Christopher G. Mayne
# last updated: 07/08/2015
# version # 20150708a

# TCL script to generate AMS shared data files from initialized system replicas
# Will write out *.status, *.sys, *.iter, *.rcf

# Script can be run from terminal or VMD tkcon

# Usage proc called when an input error is detected
proc usage {} {
	puts "\n"
	puts "-----------------------------------------------------------"
	puts "Generate AMS Input from Initialization *.zdat Files:"
	puts "Script written by Christopher G. Mayne"
	puts "-----------------------------------------------------------"
	puts "Required Input:"
	puts "\t-prefix <text>            | e.g., -prefix ams.example1 "
	puts "\t-zMax <number>            | e.g., -zMax 4.5"
	puts "Optional Input:"
	puts "\t-datadir <path/to/zdat>   | e.g., -datadir ./amsdata "
	puts "-----------------------------------------------------------"
	puts "Generated Output:"
	puts "\t<prefix>.iter, <prefix>.status, <prefix>.sys, <prefix>.rcf"
	puts "-----------------------------------------------------------"
	puts "\n"
	flush stdout
}

# script argument proccessing
if { $argc < 4 || $argc > 6  || [expr {$argc % 2}] != 0 } { eval usage; return }
# set defaults for optional arguments
set datadir "."
# parse flags
foreach {flag val} $argv { set [string range $flag 1 end] $val }

# very minimal sanity check
# test datadir
if { ![file isdirectory $datadir] } { 
	puts "ERROR: -datadir ($datadir) not found"; flush stdout
	eval usage
	return
} elseif { ![file writable $datadir] } {
	puts "ERROR: -datadir ($datadir) is not writable"; flush stdout
	eval usage
	return
}
########## LAURA: remember to check if zMax was given!

# begin normal run
puts -nonewline "generating AMS shared data files..."; flush stdout

# build a list of zdat files
set zdatList [lsort -increasing -dictionary [glob -directory $datadir *.zdat]]

# open output files
set sysFile    [open [file join $datadir ${prefix}.sys   ] w]
#set statusFile [open [file join $datadir ${prefix}.status] w]
# status file is made by writezdat program
set rcfFile    [open [file join $datadir ${prefix}.rcf   ] w]
# gather some data from zdat files
set sysCount [llength $zdatList]
for {set i 0} {$i < $sysCount} {incr i} {
	# pick out the current zdat file
	set zdat [lindex $zdatList $i]
	set zdatRoot [file rootname [file tail $zdat]]

	# parse system ID from the zdat filename
	# correct naming scheme should be path/to/prefix.000N.zdat, where N is the (zero-padded) target
    set sysID [scan [lindex [split $zdatRoot "."] end] %d]
	
	# read through zdat to find max measured z (last line)
	set infile [open $zdat r]
	set maxF -1; set maxZ -inf
	while {![eof $infile]} {
		set inline [string trim [gets $infile]]
		if { [llength $inline] != 0 } {
			set maxF [lindex $inline 0]
			set maxZ [lindex $inline 1]
		}
	}
	close $infile

	#puts "sysID: $sysID -- maxF: $maxF \t maxZ: $maxZ"

	# write relevant data to files
	puts $sysFile "$sysID  $maxZ  \{\{$zdatRoot $maxF\}\}"; flush $sysFile
#	if { $maxZ < $zMax } {
#		puts $statusFile "$sysID  s  $maxZ"; flush $statusFile
#	} else {
#		puts $statusFile "$sysID  b  $maxZ"; flush $statusFile
#	}
	puts $rcfFile "$zdatRoot  1"; flush $rcfFile
}

# close output files
close $sysFile
#close $statusFile
close $rcfFile

# write out the *.iter file
# Laura
set rep [scan [lindex [split [file rootname [file tail [lindex $zdatList end]]] "."] end] %d]
set iter [expr $rep + 1 ]
set killFile  [open [file join $datadir ${prefix}.killed ] w]
puts $killFile $iter
close $killFile
# end Laura
set padding [string length [lindex [split [lindex $zdatList 0] "."] end-1]]

set iterFile [open [file join $datadir ${prefix}.iter] w]
puts $iterFile [format "%0${padding}i" $iter]
close $iterFile

# wrap up
puts "done"; flush stdout
# end script
