#include "scheduler.h"
#include "dbinterface.h"
#include "BasicStrategy.h"
#include <stdlib.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <unistd.h>
#include <pwd.h>
#include <time.h>
#include <fcntl.h>

#include "sockRoutines.h"

char *CONV_HOST;
#define MACHINE_PERMANENT_PORT 22

#define RSH_GROUP 10002

#define XTERM "/usr/bin/X11/xterm"
#define GDB "/usr/bin/gdb"

#define EXECUTE_POLL_TIME 1
#define REMOVE_POLL_TIME 2

#define ACTIVELIST_POLL_TIME 300
#define ACTIVELIST_INVOKE_MULT 6

DbInterface dbi;
Scheduler *theScheduler;
int event_occured = 0;               // Event Variable.

int shutdown_flag  = 0;

// The CONV_RSH environment variable for the system.
char SERVER_RSH[SMALL_BUF_SIZE];
// Path for conv-host for CHARM jobs.

void default_options(){
    printf("scheduler <port> <#nodes> [<nodelist file>] [<dbname>]\n");
}

double getCurrentTime(){
    struct timeval tv;
    gettimeofday(&tv, NULL);
    
    double time = tv.tv_usec;
    time = tv.tv_sec + time/1000000;  //returns time in secs.

    return time;
}    
/*
void my_system(char *cmd, char* arg1, char* arg2, char* arg3, int id){

    setuid(0);
    if(fork() == 0){
        setuid(0);
        if(setgid(RSH_GROUP) != 0)
            printf("Setgid failed\n");
        
        setuid(id);

        printf("ID's uid=%d euid=%d gid=%d egid=%d\n", getuid(), geteuid(), getgid(), getegid());

        char *argv[6];
        argv[0] = cmd;
        argv[1] = arg1;
        argv[2] = arg2;
        argv[3] = arg3;
        argv[4] = NULL;
            
        printf("Executing %s %s %s %s\n", cmd, arg1, arg2, arg3);
    
        if(execv(cmd, argv) != 0){
            printf("Error executing command for Job #%s\n", arg3);
	}
        exit(0);
    }
    
    int status = 0;
    wait(&status);
}
*/


// reg_unreg == 0 means register, 1 means unregister
int register_with_central_manager(int reg_unreg, int my_port, 
				  char *dest, int port){

    int sockfd, numbytes;
    char buf[SMALL_BUF_SIZE];
    struct hostent *he;
    struct sockaddr_in their_addr; /* connector's address information */
    int status, argcount;

    if ((he=gethostbyname(dest)) == NULL) {  /* get the host info */
        perror("CMM gethostbyname");
        return 0;
    }

    if ((sockfd = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
        perror("CMM socket");
        return 0;
    }
    
    their_addr.sin_family = AF_INET;         /* host byte order */
    their_addr.sin_port = htons(port);       /* short, network byte order */
    their_addr.sin_addr = *((struct in_addr *)he->h_addr);
    bzero(&(their_addr.sin_zero), 8);        /* zero the rest of the struct */

    // SIGPIPE is received if the socket stream connection is broken.
    signal(SIGPIPE, SIG_IGN);

    if (connect(sockfd, (struct sockaddr *)&their_addr,
		sizeof(struct sockaddr)) == -1) {
        perror("CMM connect");
        return 0;
    }

    // Prepare and send the reg/unreg command to the scheduler.
    char my_name[SMALL_BUF_SIZE+1];
    gethostname(my_name, SMALL_BUF_SIZE);
    char str[SMALL_BUF_SIZE+1];
    snprintf(str, SMALL_BUF_SIZE, "%s %s %d\n",
            (reg_unreg==0)? "CMRegister": "CMUnRegister",
            my_name, my_port);
    
    printf("Sending string [%s] to %s:%d\n", str, dest, port);
    write(sockfd, str, strlen(str));
    snprintf(str, SMALL_BUF_SIZE, "q\n");
    write(sockfd, str, strlen(str));

    close(sockfd);

    // Go back to default handling of broken streams
    signal(SIGPIPE,SIG_DFL);

    return 1; // success
}

int abortFn(int code, const char *msg) { //  throw(CcsException){

    printf("In abort function\n");

    CcsException ex(code, msg);

    throw ex;

    return -1;
}

int main(int argc, char ** argv){

    //    seteuid(getuid());
    setuid(0);
    setgid(RSH_GROUP);
    skt_set_abort(abortFn);

    setlinebuf(stdout);
    char pid_command[SMALL_BUF_SIZE];
    int pid = getpid();
    sprintf(pid_command, "/bin/echo %d > pid\n", pid); 
    system(pid_command);

    if(argc < 3){
	default_options();
	exit(1);
    }
    
    //if(argc > 8) 
    //register_with_central_manager(0, atoi(argv[1]), argv[9], 1999);
    //else register_with_central_manager(0, atoi(argv[1]), "localhost", 1999);
    

    // get CONV_RSH.
    if(getenv("CONV_RSH") != NULL)
	snprintf(SERVER_RSH, SMALL_BUF_SIZE, "%s", getenv("CONV_RSH"));
    else
	strncpy(SERVER_RSH, "rsh", SMALL_BUF_SIZE);

    // conv-host in current dir.
    CONV_HOST = new char[SMALL_BUF_SIZE];
    snprintf(CONV_HOST, SMALL_BUF_SIZE, "%s/charmrun", getenv("PWD"));
    printf("CONV_HOST = %s\n", CONV_HOST);
    printf("Starting scheduler.  Use Control-C to exit.\n");

    int ppn = 1;
    if(argc > 8)
        ppn =  atoi(argv[8]);

    dbi.set_nodes(atoi(argv[2]), ppn);
    if(argc > 7)
	dbi.connect(argv[5], argv[6], argv[7], argv[4]);
    else
	dbi.connect("localhost", "root", "test");

    Scheduler sch(atoi(argv[1]), atoi(argv[2]), ppn);
    theScheduler = &sch;

    if(argc >= 4) {
	sch.load_nodelist(argv[3]);
    } else {
        printf("Loading nodelist file: scheduler_nodelist.\n");
	sch.load_nodelist("scheduler_nodelist");
    }

    sch.start_scheduler();

    return 0;
}

Scheduler::Scheduler(int port, int nodes, int ppn){

    int sockfd;

    utilization = 0.0;
    qsize = 0.0;

    perr_flag = perr_count = 0;

    this->port = port;
    this->num_nodes = nodes;
    this->ppn = ppn;

#ifdef USE_UNIX
    struct sockaddr_un my_addr;
    if ((sockfd = socket(AF_UNIX, SOCK_STREAM, 0)) == -1) {
        perror("socket");
        exit(1);
    }

    my_addr.sun_family = AF_UNIX;
    snprintf(my_addr.sun_path, UNIX_PATH_MAX, "/tmp/scheduler.%d", port);
    if (bind(sockfd, (struct sockaddr *)&my_addr,
	     strlen(my_addr.sun_path) + sizeof(short)) == -1) {
        perror("bind");
        close(sockfd);
        exit(1);
    }

#else 
    struct sockaddr_in my_addr;
    if ((sockfd = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
        perror("socket");
        exit(1);
    }

    //Make port reusable immediately after the server crashes.
    int yes = 1;
    if (setsockopt (sockfd, SOL_SOCKET, SO_REUSEADDR, (char*)&yes, 
                    sizeof (yes)) == -1) {
	perror ("setsockopt");
	exit(1);
    }

    my_addr.sin_family = AF_INET;
    my_addr.sin_port = htons(port);
    my_addr.sin_addr.s_addr = INADDR_ANY;
    bzero(&(my_addr.sin_zero), 8);
    if (bind(sockfd, (struct sockaddr *)&my_addr,
	     sizeof(struct sockaddr)) == -1) {
        perror("bind");
        close(sockfd);
        exit(1);
    }
#endif
    
    this->sch_socfd = sockfd;
    // INITIALIZATIONS FROM THE DATABASE.

    free_node_vector = new char[num_nodes];
    num_free_nodes = num_nodes;

    dead_node_vector = new char[num_nodes];
    num_dead_nodes = 0;

    for(int i = 0; i < num_nodes; i++) {
	free_node_vector[i] = 1;
        dead_node_vector[i] = 0;
    }

    runq = dbi.fetch_running_jobs();

    Job *jptr = runq;
    while(jptr != NULL){

	jptr->re_init();
	for(int i = 0; i < num_nodes; i++)
	    if(jptr->bit_map[i]){
		free_node_vector[i] = 0;
		num_free_nodes --;
	    }
	if(jptr->type == MCHARM)
	    jptr->connect();
	jptr = jptr->next;
    }	
    
    max_port = dbi.get_max_port() + 1;
    if(max_port < port + 1)
	max_port = port + 1;

    strategy = new BasicStrategy(num_nodes);
    strategy->free_node_vector = free_node_vector;
    strategy->num_free_nodes = num_free_nodes;

    shared_mutex = new pthread_mutex_t;
    
    //    *shared_mutex =  PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP;

    pthread_mutex_init(shared_mutex, NULL);
    remove_job();
    
    //System performance vairables.

    startTime = getCurrentTime();
    updateUtil();
}

// Background thread that queues jobs in the database.
void* schedule_thread_handler(void *arg){

    printf("Starting schedule thread with id %d\n", getpid());

    theScheduler->wait_for_request();
    return NULL;
}

// Execute thread
void* execute_thread_handler(void *arg){
    
    printf("Starting execute thread with id %d\n", getpid());

    while(1){	
	theScheduler->execute_job();
	while(!event_occured)
	    sleep(EXECUTE_POLL_TIME);
    }
    return NULL;
}

// Background thread that waits for jobs to finish.
void* remove_thread_handler(void *){
    
    printf("Starting remove thread with id %d\n", getpid());

    while(1){
	theScheduler->remove_job();
	
	sleep(REMOVE_POLL_TIME);
    }
    return NULL;
}

void* activelist_manager_handler(void *){
   
    printf("Starting active list manager with id %d\n", getpid());

    while(1){
	theScheduler->activelist_manager();
	
	sleep(ACTIVELIST_POLL_TIME);
    }
    return NULL;
}

void Scheduler::start_scheduler(){
    
    pthread_t * cur_thread = new pthread_t;
    // Create a thread to accept requests from remote clients.
    //schedule_thread_handler(NULL);
    pthread_create(cur_thread, NULL, schedule_thread_handler, NULL);

    cur_thread = new pthread_t;
    // Create a thread to execute jobs stored in the database.
    pthread_create(cur_thread, NULL, execute_thread_handler, NULL);
    
    // Wait for jobs to exit.
    cur_thread = new pthread_t;
    pthread_create(cur_thread, NULL, remove_thread_handler, NULL);
    //remove_thread_handler(NULL);
    
    cur_thread = new pthread_t;
    // Create a thread to manage the nodelist.
    pthread_create(cur_thread, NULL, activelist_manager_handler, NULL);
    
    pthread_join(*cur_thread, NULL);

    printf("SHUTTING SYSTEM DOWN\n");
}

//Writes on to a socket.
void send_message(FILE *client_fp, char *message){
    //    FILE *client_fp = fdopen(fd, "w");

    if(client_fp == NULL)
	return;

    fprintf(client_fp, "%s", message);
    fprintf(client_fp, "\n");
    fflush(client_fp);
    //    printf("After Send Message\n");
}

void Scheduler::wait_for_request(){
    int client_fd;
    socklen_t sin_size;
    struct sockaddr_in their_addr;

    if (listen(sch_socfd, MAX_CONN) == -1) {
	perror("listen");
	exit(1);
    }

    while(1){ //Accept loop.
	
        printf("================\nAwaiting faucet request:\n");

	sin_size = sizeof(struct sockaddr_in);
	if ((client_fd = accept(sch_socfd, (struct sockaddr *)
				&their_addr, &sin_size))== -1) {
            perror("accept");
	    continue;
	}
	printf("Received faucet request:\n");

        pthread_mutex_lock(shared_mutex);
        //printf("After acquiring lock\n");

        try {
            handle_request(client_fd);
        }
        catch(...) {
            printf("Error parsing request \n");            
            close(client_fd);
        }

        printf("wait unlocking\n");
        pthread_mutex_unlock(shared_mutex);
    }
}

void Scheduler::handle_request(int fd){
    
    printf("In handle Request \n");
    FILE * client_fpr, *client_fpw;
    
    client_fpr = fdopen(fd, "r");
    client_fpw = fdopen(fd, "w");

    if((client_fpr == NULL) || (client_fpw == NULL)){
        printf("Unable to handle request \n");

        close(fd);
        return;
    }

    char input[LARGE_BUF_SIZE];
    char command_type[SMALL_BUF_SIZE];
    char username[SMALL_BUF_SIZE];

    fgets(input, LARGE_BUF_SIZE, client_fpr);
    
    char cmd[LARGE_BUF_SIZE];
    
    //    printf("Before scanf\n");
    input[LARGE_BUF_SIZE - 1] = 0; 

    if(sscanf(input, "%[^:]:%[^:]:%[^\n]", command_type, username, cmd) != 3){
	send_message(client_fpw, "Unable to Parse Command");
        fclose(client_fpr);
        fclose(client_fpw);
	close(fd);
	return;
    }

    printf("values are %s, %s, %s\n", command_type, username, cmd);

    int fd1 = dup(fd);

    if(strncasecmp(command_type, "command", 8) == 0)
	handle_command(cmd, username, fd1);
    else if(strncasecmp(command_type, "job", 4) == 0)
	handle_job(cmd, username, fd1);
    else if(strncasecmp(command_type, "remote_job", 4) == 0)
	handle_remote_job(cmd, username, fd1);
    else if(strncasecmp(command_type, "query", 8) == 0)
	handle_query(cmd, username, fd1);
    else{
        send_message(client_fpw, "Unknown Command Type");
    }

    if(shutdown_flag == 1){

        pthread_mutex_unlock(shared_mutex);
        
        close(sch_socfd);
        dbi.close();

        while(shutdown_flag != 4)
            sleep(1);
        
        send_message(client_fpw, "Shut Down Complete\n");
        close(fd);
        fclose(client_fpr);
        fclose(client_fpw);

        pthread_exit(0);
    }

    fclose(client_fpr);
    fclose(client_fpw);
    close(fd);
    printf("After Handle request\n");
    return;
}

void Scheduler::handle_command(char *cmd, char *username, int fd){
    printf("In Handle Command\n");
    FILE * client_fp;
    int dbid = 0;

    client_fp = fdopen(fd, "w");

    if(client_fp == NULL){
        printf("Unable to handle request\n");
        close(fd);
        return;
    }
    
    if (strncasecmp(cmd, "1 GETSTATUS",11) == 0) {
        dump_stats(client_fp);
    }
    else if(strncasecmp(cmd, "2 KILL ", 7) == 0){
	printf("In Kill %s\n", cmd);
	
	if(!((sscanf(cmd, "2 KILL %d", &dbid) == 1) ||
	     (sscanf(cmd, "2 kill %d", &dbid) == 1))){
	    send_message(client_fp, "Unable to Parse Command");
	    fclose(client_fp);
	    close(fd);
	    return;
	}

	//Acquire lock and also kill queued jobs.
	Job *waitq = dbi.fetch_queued_jobs();
	
	int killed = 0;
	Job *jptr = waitq;
	while(jptr != NULL){
	    if((jptr->dbid == dbid) && 
               (strcmp(username, jptr->userName) == 0)){
		dbi.update_status(dbid, FINISHED); //Add new status later.
                close(jptr->client_soc_fd);
		killed = 1;
		break;
	    }
            jptr = jptr->next;
	}
	
	printf("Killing Job %d for %s\n", dbid, username);

	if(!killed){
	    jptr = runq;
	    while(jptr != NULL) {
		printf("job %d %s\n", jptr->dbid, jptr->userName);
		if((jptr->dbid == dbid) && 
                   (strcmp(username, jptr->userName) == 0)){
		    jptr->Kill();
		    killed = 1;
		    break;
		}
		jptr = jptr->next;
	    }
	}
	
	if(killed)
	    send_message(client_fp, "Job Successfully Killed");
	else
	    send_message(client_fp, "Job Kill Error");

	dbi.free_list(waitq);
    }	
    else if(strncasecmp(cmd, "1 LISTJOBS", 10) == 0){
	list_jobs(client_fp);
    }
    else if(strncasecmp(cmd, "2 LISTJOBS", 10) == 0){
        dbid = 0;
        if((sscanf(cmd, "2 LISTJOBS %d", &dbid) != 1) &&
           (sscanf(cmd, "2 listjobs %d", &dbid) != 1)){
            
	    send_message(client_fp, "Unable to Parse Command");
	    fclose(client_fp);
	    close(fd);
	    return;
	}
        
	list_jobs(client_fp, dbid, username);
    }
    else if(strncasecmp(cmd, "1 SHUTDOWN", 10) == 0){

        if(strcmp( username, "root") == 0){
            shutdown_flag = 1;
            event_occured = 1;
            send_message(client_fp, 
                         "System Shutting Down, can take up to 30 secs");
        }
        else send_message(client_fp, 
                          "You need root previlages for a shutdown, ignored");
    }
    else send_message(client_fp, "Invalid Command");

    fclose(client_fp);
    close(fd);
    return;
}

void Scheduler::handle_job(char *cmd, char *username, int fd){

    printf("In handle job\n");

    FILE * client_fp = fdopen(fd, "w");

    if(client_fp == NULL){
        printf("Unable to handle request\n");
        close(fd);
        return;
    }    

    Job j(num_nodes, ppn);            
    j.port = max_port++;
    strncpy(j.userName, username, SMALL_BUF_SIZE);

    int parse_error = j.parse_query(cmd);
    if(parse_error == -1){
	send_message(client_fp, "Error Parsing Query");
        fclose(client_fp);
	close(fd);
	return;
    }
    
    if(j.working_directory[0] != '/'){
        send_message(client_fp, "Invalid Working Directory");
        fclose(client_fp);
        close(fd);
	return;
    }   
    
    int user_id = getuid();
    setuid(0);

    struct stat statbuf;
    stat(j.working_directory, &statbuf);
    if(!S_ISDIR(statbuf.st_mode)){
        send_message(client_fp, "Invalid Working Directory");
        fclose(client_fp);
	close(fd);
	return;
    }   

    char absolute_path[SMALL_BUF_SIZE];
    if(j.name[0] != '/')
        snprintf(absolute_path, SMALL_BUF_SIZE, "%s/%s", j.working_directory, j.name);
    else snprintf(absolute_path, SMALL_BUF_SIZE, "%s", j.name);

    stat(absolute_path, &statbuf);

    if(!S_ISREG(statbuf.st_mode) && !S_ISLNK(statbuf.st_mode)){
        send_message(client_fp, "Executable does not exist on the parallel system");
        fclose(client_fp);
	close(fd);
	return;
    }   
    seteuid(user_id);

    //printf("before acquiring the lock\n");
    
    if((strcmp(j.Stdout, "socket") == 0) || (strcmp(j.Stderr, "socket") == 0) ||
       (strcmp(j.Stdin, "socket") == 0)){
        printf("Duping file descriptor %s\n", j.Stdout);
        j.client_soc_fd = dup(fd);
    }

    int jobdbid = dbi.insert_job(j);
    printf("Job Accepted, job id is %d\n", jobdbid);

    if(jobdbid != -1){
	char message[SMALL_BUF_SIZE];
        //        sprintf(message, "Successfully Queuing Job,\nJob id is %d", 
        //jobdbid);
        sprintf(message, "%d", jobdbid);
	send_message(client_fp, message);
    }
    else
	send_message(client_fp, "Error Submitting Job");

    event_occured = 1;

    fclose(client_fp);
    close(fd);
    
    j.~Job();
}

void Scheduler::handle_remote_job(char *cmd, char *username, int fd){

    Job j(num_nodes, ppn);            
    j.client_soc_fd = fd;
    j.port = max_port++;
    strncpy(j.userName, "skumar2", SMALL_BUF_SIZE);

    FILE *client_fp = fdopen(fd, "w");

    if(client_fp == NULL){
        printf("Unable to handle request\n");
        close(fd);
        return;
    }

    int parse_error = j.parse_query(cmd);
    if(parse_error == -1){
	send_message(client_fp, "Error Parsing Query");
	close(fd);
	return;
    }
    
    int jobdbid = dbi.insert_job(j);
    printf("Job Accepted, job id is %d\n", jobdbid);

    if(jobdbid != -1){
	char message[SMALL_BUF_SIZE];
        //        sprintf(message, "Successfully Queuing Job,\nJob id is %d", 
        //     jobdbid);
        sprintf(message, "%d", jobdbid);
	send_message(client_fp, message);
    }
    else
	send_message(client_fp, "Error Submitting Job");

    event_occured = 1;

    /*
    if((j.Stdout != NULL && strcmp(j.Stdout, "socket")) &&
       (j.Stderr != NULL && strcmp(j.Stderr, "socket")) &&
       (j.Stdin  != NULL && strcmp(j.Stdin, "socket")))
	close(fd);
    */

    fclose(client_fp);
    close(fd);
    j.~Job();
}

void Scheduler::handle_query(char *cmd, char *username, int fd){

    Job j(num_nodes, ppn);            
    j.client_soc_fd = fd;
    j.port = max_port++;
    int parse_error = j.parse_query(cmd);

    FILE *client_fp = fdopen(fd, "w");

    if(client_fp == NULL){
        printf("Unable to handle request\n");
        close(fd);
        return;
    }

    if(parse_error == -1){
	send_message(client_fp, "Error Parsing Query");
	close(fd);
	return;
    }
    
    Job *waitq = dbi.fetch_queued_jobs();
    
    if (strategy->is_available(&j, waitq, runq) != -1)
	send_message(client_fp, "yes");
    else
	send_message(client_fp, "no");
    
    fclose(client_fp);
    close(fd);
    j.~Job();
    dbi.free_list(waitq);
    
    return;
}

void *start_func(void *arg){

    theScheduler->start_job((Job *)arg);
    return NULL;
}

void Scheduler::execute_job(){

    Job *waitq;
    printf("in execute job \n");
    //    printf("before locking\n");
    pthread_mutex_lock(shared_mutex);

    if(shutdown_flag == 1){

        shutdown_flag ++;

        printf("Execute Thread shutdown\n");

        pthread_mutex_unlock(shared_mutex);
        pthread_exit(0);
    }

    //    printf("before fetch\n");
    waitq = dbi.fetch_queued_jobs();

    //if(waitq == NULL){
    //printf("waitq = NULL\n");
    //}

    //    printf("after fetch\n");

    updateUtil();
    updateQSize(waitq);

    strategy->num_free_nodes = num_free_nodes;
    strategy->allocate_nodes(waitq, runq);
    num_free_nodes = strategy->num_free_nodes;
    
    //    printf("after allocate processors \n");

    Job *jptr = runq;
    while(jptr != NULL){
	if(jptr->type == MCHARM)
	    jptr->set_bitmap();

        /*
        //HACK REMOVE LATER

        //        printf("Waiting for %d in %d\n", jptr->pid, getpid());
        
        int status;
        waitpid(jptr->pid, &status, WNOHANG | WUNTRACED );
        */
	jptr = jptr->next;
    }
    
    //    printf("after setting bitmaps\n");

    jptr = waitq;
    Job *prev = NULL;

    while(jptr != NULL){
	if(jptr->bitmap_changed){
            
	    pthread_t * cur_thread = new pthread_t;

	    // Create the job starting thread.
            
	    int err = pthread_create(cur_thread, NULL, start_func, 
                                     (void *) jptr);
            if(err != 0){
                printf("Pthread Create Failed\n");
                pthread_mutex_unlock(shared_mutex);
                return;
                //                exit(0);
            }
            pthread_detach(*cur_thread);
            
	    //start_job(jptr);
	 
	    // Take jptr out of waitq
	    if(prev)
		prev->next = jptr->next;
	    else
		waitq = jptr->next;
	
	    // Assign runq	    
	    jptr->next = runq;
	    runq = jptr;
	    
	    // Assign jptr
	    if(prev)
		jptr = prev->next;
	    else
		jptr = waitq;
	}
	else{
	    prev = jptr;
	    jptr = jptr->next;
	}
    }

    // Now we inform the migratable jobs that their bitmap might have
    // been changed.
    /*
    jptr = waitq;
    printf("ids ");
    while(jptr != NULL){
        printf("%d, ", jptr->dbid); 
        jptr = jptr->next;
    }
    printf("\n");
    */
    event_occured = 0;
    dbi.free_list(waitq);
    
    pthread_mutex_unlock(shared_mutex);
    //    printf("after unlocking: execute job\n");

}

// The handler function for the job startup wait thread.
void Scheduler::Connect(Job *new_job, int jobpid){    

    int connect_failed = 0;

    pthread_mutex_lock(shared_mutex);

    if(!Job::ping(jobpid)){
        pthread_mutex_unlock(shared_mutex);
        return;
    }

    new_job->ip = 0;

    Job temp_job(num_nodes, ppn);
    temp_job.dbid = new_job->dbid;
    temp_job.ip = new_job->ip;
    temp_job.port = new_job->port;
    temp_job.status = new_job->status;
    pthread_mutex_unlock(shared_mutex);

    //Connecting to the job, sleep while the job starts.
    if(temp_job.connect() != 0){
	connect_failed = 1;
    }
    
    //CHECK (Problem when job dies between Connect and Set bitmap)!!!
    //FIXED if job is alive then its entry in runq exists or new_job exixts.
    pthread_mutex_lock(shared_mutex);
    new_job->status = temp_job.status;
    if(!connect_failed){

        if(Job::ping(jobpid)) {  
            new_job->svr = temp_job.svr;
            new_job->set_bitmap();
        }
    }
    pthread_mutex_unlock(shared_mutex);
    
    return;
}

int Scheduler::start_job(Job *j){
    int childpid;

    //    chdir("temp3");
    
    //printf("start job acquiring lock %d\n", j->dbid);
    pthread_mutex_lock(shared_mutex);   //change later!        
    //    printf("After acquiring lock: start_job\n");
    
    j->init_arg();

    int user_id = getuid();
    //become superuser.
    setuid(0);

    char message[LARGE_BUF_SIZE];
    time_t time_val = time(NULL);
    char timebuf[LARGE_BUF_SIZE];
    char *time_str = ctime_r(&time_val, timebuf);
    sprintf(message, "Starting Job #%d with %d nodes at %s\n", j->dbid, 
            j->num_allocated_nodes, time_str);

    j->send_email(message, NOTIFY_START);

    if(!(childpid = fork())){

        close(theScheduler->sch_socfd);

        setpgrp();
        chdir(j->working_directory);

	// Set CONV_RSH environment variable.
	char buf[SMALL_BUF_SIZE];
	snprintf(buf, SMALL_BUF_SIZE, "CONV_RSH=%s", SERVER_RSH);

	putenv(buf);

        if(j->flags & DEBUG_ON){
            snprintf(buf, SMALL_BUF_SIZE, "DISPLAY=%s", j->display);            
            putenv(buf);
            printf("Setting DISPLAY %s\n", j->display);
        }

        printf("%s\n", getenv("CONV_RSH"));
	
        char temp_buf[LARGE_BUF_SIZE];
        struct passwd *cur_passwd = new struct passwd;

#ifdef __GNUC__
        getpwnam_r(j->userName, cur_passwd, temp_buf, LARGE_BUF_SIZE, &cur_passwd);
#else
        getpwnam_r(j->userName, cur_passwd, temp_buf, LARGE_BUF_SIZE);
#endif
        if(cur_passwd == NULL){
            printf("Getpwnam Error, Child Exiting\n");
            exit(0);
        }

	int jobuserid = cur_passwd->pw_uid;
        delete cur_passwd;
	
        // This is the child of the fork, i.e. the new job.
        // Connect Stdin, Stdout, Stderr to the client that requested
        // the job.

        close(0);
	close(1);
	close(2);

        setuid(0);
	setgid(RSH_GROUP);
	//become user
	setuid(jobuserid);

	// Pipe the Stdin Stdout and Stderr to the required destinations. 
	int cur_fd = 0;

	if(strncmp(j->Stdin, "socket", SMALL_BUF_SIZE) == 0)
	    dup(j->client_soc_fd);
	else if(strncmp(j->Stdin, "null", SMALL_BUF_SIZE) == 0)
	    cur_fd = open("/dev/null", O_RDONLY);
	else if(j->Stdin != NULL)
	    cur_fd = open(j->Stdin, O_RDONLY);
	
	if(cur_fd == -1)
	    dup(j->client_soc_fd);

	cur_fd = 0;
	if(strncmp(j->Stdout, "socket", SMALL_BUF_SIZE) == 0)
	    dup(j->client_soc_fd);
	else if(strncmp(j->Stdout, "null", SMALL_BUF_SIZE) == 0)
	    cur_fd = open("/dev/null", O_WRONLY);
	else if(j->Stdout != NULL) {
	    cur_fd = open(j->Stdout, O_CREAT | O_WRONLY |  O_EXCL);
	    if(cur_fd == -1)
		cur_fd = open(j->Stdout, O_APPEND | O_RDWR);
            
            if(cur_fd >= 0)
                fchmod(cur_fd, S_IREAD|S_IWRITE);
	}
	
	if(cur_fd == -1)
	    dup(j->client_soc_fd);

	cur_fd = 0;
	if(strncmp(j->Stderr, "socket", SMALL_BUF_SIZE) == 0)
	    dup(j->client_soc_fd);
	else if(strncmp(j->Stderr, "null", SMALL_BUF_SIZE) == 0)
	    cur_fd = open("/dev/null", O_WRONLY);
	else if(j->Stderr != NULL) {
	    cur_fd = open(j->Stderr, O_CREAT | O_WRONLY |  O_EXCL);
	    if(cur_fd == -1)
		cur_fd = open(j->Stderr, O_APPEND | O_RDWR);
            if(cur_fd >= 0)
                fchmod(cur_fd, S_IREAD|S_IWRITE);
	}

	if(cur_fd == -1)
	    dup(j->client_soc_fd);
	
	close(j->client_soc_fd);
	
	long arg = 0;
	arg = arg | !FD_CLOEXEC;
	fcntl(0, F_SETFD, arg);
	
	//back to user.
	
        //	time_t time_val = time(NULL);
        printf(message);

	if(j->type != UNI)
	    create_nodes_file(j);
	else
	    set_unijob_host(j);

        //printf("init_arg %s, %s, %s\n", j->argv[0], j->argv[1], j->argv[2]);

	char script_name[SMALL_BUF_SIZE];
	write_script(j); //Creates script and adds it to the arguments.
	
	fflush(stdout);

	char *starting_program;

	if((j->type == MCHARM) || (j->type == NCHARM))
	    starting_program = CONV_HOST;
	else if(j->type == MPI)
	    starting_program = MPIRUN;
	else if(j->type == UNI)
	    starting_program = RSH;

        //printf("ID's uid=%d euid=%d gid=%d egid=%d\n", getuid(), geteuid(), getgid(), getegid());

	if(execv(starting_program, j->argv) == -1){
	    printf("Error Starting Job #%d\n", j->dbid);
	    fflush(stdout);
	    exit(0);
	}
    }

    // This is the parent of the fork, i.e. the scheduler.    

    if(childpid == -1){
        printf("FORK failed\n");
        pthread_mutex_unlock(shared_mutex);
        pthread_exit(0);
    }

    //    seteuid(user_id);
    j->pid = childpid;
    close(j->client_soc_fd);

    //    printf("Before Started\n");
    j->started();
    int jobtype = j->type;

    //    printf("start_job before unlocking\n");
    pthread_mutex_unlock(shared_mutex);
    
    if(jobtype == MCHARM)
        Connect(j, childpid);

    printf("After start job %d\n", j->dbid);
    //#ifdef _SOLARIS
    // For Solaris, inefficient otherwise, currently also used to get 
    //exit code of the child process
    
    int status = 0;
    int dbid = j->dbid;
    wait(&status);

    int exit_code = WEXITSTATUS(status);
    printf("Job %d exitcode is %d\n", dbid, exit_code);
    pthread_mutex_lock(shared_mutex);
    dbi.update_exitCode(dbid, exit_code);
    pthread_mutex_unlock(shared_mutex);
    //#endif
    //    pthread_exit(0);
    return 0;
}

void Scheduler::remove_job(){

    //    printf("remove_job: before acquiring lock\n");
    pthread_mutex_lock(shared_mutex);
    //printf("remove_job: after acquiring lock\n");

    if(shutdown_flag == 2){

        printf("Remove Thread shutdown\n");

        shutdown_flag ++;

        pthread_mutex_unlock(shared_mutex);
        pthread_exit(0);
    }
    
    updateUtil();
    
    Job *jptr = runq;    
    Job *prev = NULL;
    int kill_count = 0;
    while(jptr != NULL){

        if((jptr->type == MCHARM) && (jptr->status < FAILED_CONNECT)){
            prev = jptr;
            jptr = jptr->next;
            continue;
        }
        else if(jptr->status == QUEUED){
            prev = jptr;
            jptr = jptr->next;
            continue;
        }

        //        printf("Looking for running jobs %d\n", jptr->pid);

	if(!Job::ping(jptr->pid)){
	    fprintf(stdout, "Removing Job %d\n", jptr->dbid);

	    num_free_nodes += jptr->delete_all(free_node_vector);

            if(jptr->kill_node_jobs) {
                killNodeJobs(jptr);
		kill_count ++;
	    }
            
	    printf("after kill node jobs\n");

	    event_occured = 1;
	    
            //            printf("before delete\n");
	    if(prev){
		prev->next = jptr->next;
		delete jptr;
		jptr = prev->next;
	    }
	    else{
		runq = jptr->next;
		delete jptr;
		jptr = runq;
	    }		
            //            printf("after delete\n");
	}
	else{
	    prev = jptr;
	    jptr = jptr->next;
	}
    }

    jptr = runq;
    double curTime = getCurrentTime();
    while(jptr != NULL){
	
	if((jptr->status != RUNNING) || (jptr->pid <= 0)){
	    jptr = jptr->next;
            continue;
        }
	
	if(jptr->allocated_time + jptr->start_time + 10 < curTime) {   //10 second grace. change later if necessary.
	    printf("Killing Job #%d as it has crossed time limit of %d\n", jptr->dbid, jptr->allocated_time); 
	    jptr->Kill();
	}
	
	jptr = jptr->next;
    }

    //printf("Remove Job Unlocking\n");
    pthread_mutex_unlock(shared_mutex);
    
    while(kill_count > 0) {
        int status = 0;
        wait(&status);
        kill_count --;
    }
    
    //    printf("Remove job releasing lock\n");
    //    printf("After remove job\n");
}

void Scheduler::load_nodelist(char * node_listfile){
    FILE *fp;
    char buf[SMALL_BUF_SIZE];
    int nodes = 0;

    strncpy(nodesfile, node_listfile, SMALL_BUF_SIZE);
    nodesfile[SMALL_BUF_SIZE - 1] = '\0';

    freelist = NULL;
    activelist = NULL;
    fp = fopen(node_listfile, "r");

    if(fp == NULL){
        printf("Unable to open Node List\n");
        exit(1);
    }   

    fscanf(fp, "%s", buf);
    Node *nptr = NULL;
    while(1){
	if(strncmp(buf, "host", SMALL_BUF_SIZE) == 0){
	    if(fscanf(fp, "%s", buf) == 1){
		Node *cur_node = new Node;
		
		strncpy(cur_node->name, buf, SMALL_BUF_SIZE);
		cur_node->name[SMALL_BUF_SIZE - 1] = '\0';

		cur_node->next = NULL;
		
		if(nodes == 0)
		    activelist = cur_node;		
		
		if(nodes == num_nodes){
		    freelist = cur_node;
		    nptr = NULL;
		}

		if(nptr)
		    nptr->next = cur_node;		
		nptr = cur_node;
		nodes ++;
	    }
	    else break;
	}
	if(fscanf(fp, "%s", buf) != 1)
	    break;
    }

    if(nodes == 0){
	printf("incorrect scheduler_nodelist file\n");
	exit(1);
    }

    if(nodes < num_nodes){
	printf("#Processors is less than num_nodes\n");
	exit(1);
    }
    fclose(fp);
}

// When starting to execute a new non-migratable job, we have to
// create a nodelist file telling it which CPU's to use.
void Scheduler::create_nodes_file(Job *j){

    char nodesfilepath[2 * SMALL_BUF_SIZE];
    if(j->nodes_file[0] != '/')
	sprintf(nodesfilepath, "%s/%s", j->working_directory,
		 j->nodes_file);
    else 
	sprintf(nodesfilepath, "%s", j->nodes_file);

    FILE * fp = fopen(nodesfilepath, "w+");
    if(fp == NULL){
	printf("Error Writing Nodelist File: %s\n", nodesfilepath);
	return;
    }
    
    if((j->type == NCHARM) || (j->type == MCHARM))
	fprintf(fp, "group main\n");
    
    int i;
    Node *nptr;
    for(i=0, nptr = activelist; i < num_nodes && nptr != NULL; i++, nptr=nptr->next){
	if(j->type == MCHARM) {
            for(int pcount = 0; pcount < nptr->ppn; pcount++)
                fprintf(fp, "host %s\n", nptr->name);
        }
	else if(j->bit_map[i]) {
            for(int pcount = 0; pcount < j->ppn; pcount++)
                if(j->type == MPI)
                    fprintf(fp, "%s\n", nptr->name);
                else if(j->type == NCHARM)
                    fprintf(fp, "host %s\n", nptr->name);
        }
    }

    fclose(fp);
    return;
}

void Scheduler::set_unijob_host(Job *j){
    int i;
    Node *nptr;
    for(i=0, nptr = activelist; i < num_nodes&& nptr!=NULL; i++, nptr=nptr->next) {
	if(j->bit_map[i])
	    j->argv[1] = nptr->name;
    }
    return;
}

Node *Scheduler::get_live_node(){
    Node *nptr = freelist;
    Node *prev = NULL;
    
    while(nptr != NULL)
	if(is_alive(nptr->name)){
	    if(prev)
		prev->next = nptr->next;
	    else
		freelist = freelist->next;
	    return nptr;
	}
	else{
	    prev = nptr;
	    nptr = nptr->next;
	}
    return NULL;
}

void Scheduler::activelist_manager(){

    static int invoke_count = 0;
    
    invoke_count ++;

    // LOCK
    pthread_mutex_lock(shared_mutex);
    //    printf("Activelist acquiring lock\n");

    if(shutdown_flag == 3){

        printf("Nodelist Manager Thread shutdown\n");
        
        shutdown_flag ++;
        
        //        printf("Activelist releasing lock\n");
        pthread_mutex_unlock(shared_mutex);
        pthread_exit(0);
    }
    
    if(invoke_count % ACTIVELIST_INVOKE_MULT != 1){
        //        printf("Activelist releasing lock\n");
        pthread_mutex_unlock(shared_mutex);
        return;
    }

    Job *waitq = dbi.fetch_queued_jobs();
    updateUtil();
    updateQSize(waitq);
    dbi.free_list(waitq);

#ifdef SHOW_UTILIZATION
    fprintf(stdout, "Utilization = %5.3lf, mean queue size = %5.3lf\n", 
	   utilization, qsize);
#endif

    // There exists a mcharm job running on all processors.
    Job *jptr = runq;
    while(jptr != NULL)
	if(jptr->type == MCHARM){
            //            printf("Activelist releasing lock\n");
            pthread_mutex_unlock(shared_mutex);
	    return;
	}
	else jptr = jptr->next;
    
    //    printf("Activelist releasing lock\n");
    pthread_mutex_unlock(shared_mutex);
    
    int node_changed = 0;

    Node *nptr = activelist;
    Node *prev = NULL;
    int pos = 0;
    int up_nodes = num_nodes;

    while(nptr != NULL){
	if(!is_alive(nptr->name)){
            
            if(dead_node_vector[pos])
                continue;
            
            pthread_mutex_lock(shared_mutex);
            if(job_running(pos))
                killJobAtNode(pos);
            
            //            printf("Activelist acquiring lock\n");
	    up_nodes --;
	    Node *node = get_live_node();	    
	    
	    if(node != NULL){

                up_nodes ++;

		if(prev){
		    prev->next = node;
		    node->next = nptr->next;
		}
		else{
		    activelist = node;
		    node->next = nptr->next;
		}

		node = nptr;
		nptr = nptr->next;

		// add down node to free list 
		node->next = freelist;
		freelist = node;
		node_changed = 1;
	    }
            else {
                free_node_vector[pos] = 0;
                num_free_nodes --;
                dead_node_vector[pos] = 1;
                num_dead_nodes ++;
            }
            //            printf("Activelist releasing lock\n");
            pthread_mutex_unlock(shared_mutex);
	}        
	else{
            if(dead_node_vector[pos]) {
                free_node_vector[pos] = 1;
                num_free_nodes ++;
                dead_node_vector[pos] = 0;
                num_dead_nodes --;                
            }

            prev = nptr;
            nptr = nptr->next;            
	}
	pos ++;
    }

    if(up_nodes == num_nodes)
        perr_flag = 1;

    if(node_changed)
	update_nodesfile();

    //    printf("finished active list manager\n");
}

int Scheduler::is_alive(char *node){
    
    int sockfd;

    struct hostent *he;
    struct sockaddr_in their_addr; /* connector's address information */

    if ((he=gethostbyname(node)) == NULL) {  /* get the host info */
	char buf[SMALL_BUF_SIZE];
	snprintf(buf, SMALL_BUF_SIZE, "Node %s :gethostbyname", node);

        if(perr_flag) {
            perror(buf);
            perr_count ++;
        }

        return 0;
    }

    if ((sockfd = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
	char buf[SMALL_BUF_SIZE];
	snprintf(buf, SMALL_BUF_SIZE, "Node %s :socket", node);

        if(perr_flag) {
            perror(buf);
            perr_count ++;
        }
        
        return 0;
    }
    
    their_addr.sin_family = AF_INET;         /* host byte order */
    /* short, network byte order */
    their_addr.sin_port = htons(MACHINE_PERMANENT_PORT);   
    
    their_addr.sin_addr = *((struct in_addr *)he->h_addr);
    bzero(&(their_addr.sin_zero), 8);        /* zero the rest of the struct */

    // SIGPIPE is received if the socket stream connection is broken.
    signal(SIGPIPE, SIG_IGN);

    if (connect(sockfd, (struct sockaddr *)&their_addr,
		sizeof(struct sockaddr)) == -1) {
	char buf[SMALL_BUF_SIZE];
	snprintf(buf, SMALL_BUF_SIZE, "Node %s :connect", node);
        
        if(perr_flag) {
            perror(buf);
            perr_count ++;
        }
        
        return 0;
    }

    close(sockfd);

    if(perr_count >= num_nodes) {
        perr_flag = 0;
        perr_count = 0;
    }

    return 1;
}

int Scheduler::job_running(int pos){
    Job *jptr = runq;

    while(jptr != NULL)
	if(jptr->bit_map[pos])
	    return 1;
	else
	    jptr = jptr->next;

    return 0;
}

// Move to the database later.
void Scheduler::update_nodesfile(){
    FILE * fp = fopen(nodesfile, "w+");
    
    if(fp == NULL){
        printf("Unable to update nodelist\n");
        return;
    }
    
    Node *nptr = activelist;

    while(nptr != NULL){
	fprintf(fp, "host %s\n", nptr->name);
	nptr = nptr->next;
    }

    nptr = freelist;

    while(nptr != NULL){
	fprintf(fp, "host %s\n", nptr->name);
	nptr = nptr->next;
    }
    fclose(fp);
}

void Scheduler::dump_stats(FILE *fp){

    int njobs = 0;
    Job *jptr = runq;
    
    while(jptr != NULL){
	njobs ++;
	jptr = jptr->next;
    }
    
    fprintf(fp, "%d %d\n", num_nodes, njobs);
    
    jptr = runq;
    while(jptr != NULL){
        fprintf(fp, "%s %d\n", jptr->name, jptr->num_allocated_nodes);
        jptr = jptr->next;
    }
}

void Scheduler::list_jobs(FILE *fp){
    int njobs = 0;
    Job *jptr = runq;
    
    while(jptr != NULL){
	njobs ++;
	jptr = jptr->next;
    }
    
    //    fprintf(fp, "%d %d\n", num_nodes, njobs);

    Job *waitq = dbi.fetch_queued_jobs();
    jptr = waitq;
    fprintf(fp, "%-10s%-10s%-14s%-10s%-10s%-10s%-10s\n", "UserId", "JobId", 
            "Nodes", "Min Nodes", "Max Nodes", "Status", "Name");

    while(jptr != NULL){	
	fprintf(fp, "%-10s%-10d%-14d%-10d%-10d%-10s%-10s\n", jptr->userName, 
                jptr->dbid, jptr->num_allocated_nodes, jptr->min_nodes, 
                jptr->max_nodes, "QUEUED", jptr->name);
        
        jptr = jptr->next;
    }
    
    dbi.free_list(waitq);

    jptr = runq;
    while(jptr != NULL){
	char *job_status;
	if(jptr->status == QUEUED)
	    job_status = "QUEUED";
	else if(jptr->status == RUNNING || jptr->status == CONNECTED)
	    job_status = "RUNNING";
	else if(jptr->status == FINISHED)
	    job_status = "FINISHED";

        fprintf(fp, "%-10s%-10d%-14d%-10d%-10d%-10s%-10s\n", jptr->userName, 
                jptr->dbid, jptr->num_allocated_nodes, jptr->min_nodes, 
                jptr->max_nodes, job_status, jptr->name);
        
        jptr = jptr->next;
    }
    fprintf(fp,"CURRENTLY %d NODES FREE\n", num_free_nodes); 
}

void Scheduler::list_jobs(FILE *fp, int dbid, char *username){
    int njobs = 0;
    Job *jptr = runq;
    
    while(jptr != NULL){
	njobs ++;
	jptr = jptr->next;
    }
    
    Job *waitq = dbi.fetch_queued_jobs();
    jptr = waitq;
    fprintf(fp, "%-10s%-10s%-14s%-10s%-10s%-10s%-10s\n", "UserId", "JobId", 
            "Nodes", "Min Nodes", "Max Nodes", "Status", "Name");

    while(jptr != NULL){	
        if((jptr->dbid == dbid) && (strcmp(jptr->userName, username) == 0)){
            fprintf(fp, "%-10s%-10d%-14d%-10d%-10d%-10s%-10s\n", 
                    jptr->userName, jptr->dbid, jptr->num_allocated_nodes, 
                    jptr->min_nodes, jptr->max_nodes, "QUEUED", jptr->name);
            
            dbi.free_list(waitq);
            return;
        }
        
        jptr = jptr->next;
    }
    
    jptr = runq;
    while(jptr != NULL){
	char *job_status;
	
        if(jptr->status == QUEUED)
	    job_status = "QUEUED";
	else if(jptr->status == RUNNING || jptr->status == CONNECTED)
	    job_status = "RUNNING";
	else if(jptr->status == FINISHED)
	    job_status = "FINISHED";
        
        if((jptr->dbid == dbid) && (strcmp(jptr->userName, username) == 0)){
            fprintf(fp, "%-10s%-10d%-14d%-10d%-10d%-10s%-10s\n", 
                    jptr->userName, jptr->dbid, jptr->num_allocated_nodes, 
                    jptr->min_nodes, jptr->max_nodes, job_status, jptr->name);
            
            return;
        }
        
        jptr = jptr->next;
    }

    fprintf(fp, "Job Not Found\n");
}

void Scheduler::updateUtil(){
    static double previousEventTime = 0.0;      // Relative.

    double time = getCurrentTime() - startTime;  //Relative Time since start.

    if(time < 0.1)
        return;

    utilization = utilization * previousEventTime 
        + ((time - previousEventTime)*((num_nodes - num_free_nodes) * 100))/num_nodes;

    utilization = utilization / time;

    //    printf("Utilization = %5.3lf\n", utilization);
    previousEventTime = time;
}

void Scheduler::updateQSize(Job *waitq){
    static double previousEventTime = 0.0;      // Relative.

    double time = getCurrentTime() - startTime;  //Relative Time since start.

    if(time < 0.1)
        return;
    
    cur_qsize = 0;
    Job *jptr = waitq;
    while(jptr != NULL){
        cur_qsize ++;
        jptr = jptr->next;
    }

    jptr = runq;
    while(jptr != NULL){
        cur_qsize ++;
        jptr = jptr->next;
    }
    
    if(cur_qsize > 1024 * num_nodes){
        printf("TOO LONG a queue\n");
        //        TERMINATE(stdout);
        //my_end(1);
        exit(0);
    }   
    
    qsize = qsize * previousEventTime + 
        + (time - previousEventTime) * cur_qsize;
    qsize = qsize / time;

    previousEventTime = time;
}

void Scheduler::write_script(Job *j){

    char script_path[2 * SMALL_BUF_SIZE];
    
    if(j->script_name[0] != '/')
	sprintf(script_path, "%s/%s", j->working_directory,
		 j->script_name);
    else 
	sprintf(script_path, "%s", j->script_name);

    FILE *fp = fopen(script_path, "w+");

    if(fp == NULL){
	printf("Error Writing Script File: %s\n", script_path);
	return;
    }
    
    char temp_buf[LARGE_BUF_SIZE];
    struct passwd *cur_passwd = new struct passwd;

#ifdef __GNUC__
    getpwnam_r(j->userName, cur_passwd, temp_buf, LARGE_BUF_SIZE, &cur_passwd);
#else
    getpwnam_r(j->userName, cur_passwd, temp_buf, LARGE_BUF_SIZE);
#endif
    
    if(cur_passwd == NULL){
        printf("Error in getpwd : %s\n", script_path);
	return;
    }   

    char *shell = cur_passwd->pw_shell;
    
    fprintf(fp, "#!%s\n", shell);
    char name[2 * SMALL_BUF_SIZE];
    if(j->name[0] == '/')
	strcpy(name, j->name);
    else
	sprintf(name, "%s/%s", j->working_directory, j->name);
    
    if(j->type == UNI)
        fprintf(fp, "cd %s;\n", j->working_directory);
    
    if(j->kill_node_jobs){
        if(j->flags & DEBUG_ON)
            fprintf(fp, "%s -e %s %s $* %s &\n", XTERM, GDB, name, j->argbuf);
        else
            fprintf(fp, "%s $* %s &\n", name, j->argbuf);
        fprintf(fp, "/bin/touch /tmp/pid.%d\n", j->dbid);
        fprintf(fp, "echo $! >> /tmp/pid.%d\n", j->dbid);
    }
    else {
        if(j->flags & DEBUG_ON)
            fprintf(fp, "%s -e %s %s $* %s\n", XTERM, GDB, name, j->argbuf);
        else
            fprintf(fp, "%s $* %s \n", name, j->argbuf);
    }
    
    //    fprintf(fp, "wait $!\n");
    
    //    fprintf(fp, "/usr/bin/sudo -u root /usr/bin/sudo -u %s %s %s $*\n", j->userName, name, j->argbuf);
    chmod(j->script_name, S_IREAD|S_IWRITE|S_IXUSR);
    delete cur_passwd;
    fclose(fp);
}

void Scheduler::killNodeJobs(Job *j){

    printf("In Kill Node Jobs \n");

    Node *nptr = activelist;
    char command[SMALL_BUF_SIZE], arg[SMALL_BUF_SIZE]; 
    int count = 0;

    char temp_buf[LARGE_BUF_SIZE];
    struct passwd *cur_passwd = new struct passwd;

#ifdef __GNUC__
    getpwnam_r(j->userName, cur_passwd, temp_buf, LARGE_BUF_SIZE, &cur_passwd);
#else
    getpwnam_r(j->userName, cur_passwd, temp_buf, LARGE_BUF_SIZE);
#endif

    if(cur_passwd == NULL){
        printf("getpwnam error\n");
        return;
    }

    int jobuserid = cur_passwd->pw_uid;
    delete cur_passwd;

    printf("user id = %d\n", jobuserid);
    setuid(0);
    int childpid = 0;
    if((childpid = fork()) == 0) {

        close(theScheduler->sch_socfd);
        
        setuid(0);
        if(setgid(RSH_GROUP) != 0)
            printf("Setgid failed\n");
        
        setuid(jobuserid);
        
        //setpgid(0, 1);
        
	count = 0;
	nptr = activelist;
        sprintf(arg, "%d", j->dbid);
	printf("Killing Processes\n");
        while(nptr != NULL && count < num_nodes){
            if(j->bit_map[count]){
                sprintf(command, "%s %s %s/killScript %d\n", RSH, nptr->name, getenv("PWD"), j->dbid); 
                //printf(command);
                system(command);
                
                //sprintf(command, "%s/killScript", getenv("PWD")); 
                //my_system(RSH, nptr->name, command, arg, jobuserid);
		//printf("After system\n");
            }
            nptr = nptr->next;
            count ++;
        }
        
        nptr = activelist;
        count = 0;
	printf("Removing files for job %d\n", j->dbid);
        while(nptr != NULL && count < num_nodes){
            if(j->bit_map[count]){
                sprintf(command, "%s %s %s/cleanupScript %d\n", RSH, nptr->name, getenv("PWD"), j->dbid);                 
                //printf(command);
                system(command);
                //sprintf(command, "%s/cleanupScript", getenv("PWD")); 
                //my_system(RSH, nptr->name, command, arg, jobuserid);
            }
            nptr = nptr->next;
            count ++;
        }
        exit(0);
    }
}

//kills the job running on that node.
void Scheduler::killJobAtNode(int node){
    Job *jptr = runq;
    
    while(jptr != NULL){
        if(jptr->bit_map[node] == 1) {
            jptr->Kill();
            return;
        }
        jptr = jptr->next;
    }
}

CcsException::CcsException(int code, const char *msg){
    this->code = code;
    strcpy(errorMessage, msg);
}

void CcsException::getMessage(char *buf){
    strcpy(buf, errorMessage);
}


