/*
Provides the user interface--responds to mouse clicks
and keyboard events.

Orion Sky Lawlor, olawlor@acm.org, 8/29/2002
*/
#include <vector>
#include "glutmain.h"
#include "glutil.h"
#include "ckviewpoint.h"
#include "ckviewable.h"
#include "sixty.h"
#include "ckhashtable.h"

bool addFrame=false;
class glView {
	CkView *v; //The source image
	GLtexture tex; //The texture
public:
	glView(CkView *v_) 
		:v(v_), tex(v->getImage().getData(),
			v->getImage().getWidth(),v->getImage().getHeight(),
			v->getImage().getColors())
	{
		v->ref();
		//Now that we've copied the view into GL, 
		// flush the old in-memory copy:
		// v->flushImage();
	}
	~glView() {v->unref();}
	
	void draw(const CkViewpoint &vp,double alpha=1.0) {
		glColor4f(1.0,1.0,1.0,alpha);
		CkVector3d bl=v->getCorner(0);
		CkVector3d br=v->getCorner(1);
		CkVector3d tl=v->getCorner(2);
		CkVector3d tr=v->getCorner(3);
		GLtextureQuad(tex,tl,tr,bl,br,GLtexture_linear);
		if (addFrame) {
			GLline(tl,br);
			GLline(tr,bl);
		}
	}
	
	//Return true if we should be regenerated:
/*	bool outOfDate(const CkViewpoint &vp) {
		const double rmsTol=4.0;
		return v->rmsError(vp)>rmsTol;
	} */
};

//A local copy of a cached remote viewable
class Viewable {
	CkViewableID id;
	glView *view; //the last view recieved
public:
	Viewable(const CkViewableID &id_) :id(id_) 
	{
		view=NULL;
	}
	~Viewable(void) {
		delete view;
	}
	
	void addView(CkView *v) {
		//FIXME: should keep both old and new view around
		if (view!=NULL) delete view;
		view=new glView(v);
	}
	
	void draw(const CkViewpoint &vp) {
		//FIXME: should choose the best view among all we have
		if (view)
			view->draw(vp,1.0);
	}
};

//Hashtable for keeping track of all the viewables we've received info. about:
class  ViewableTable : public viewDest {
	CkHashtableT<CkViewableID,Viewable *> table;
public:
	ViewableTable() {}
	
	//Add this incoming view to our table:
	virtual void viewResponse(CkView *view) {
		const CkViewableID &id=view->getViewable();
		printf("view response: %d %d %d %d\n",id.id[0],id.id[1],id.id[2],id.id[3]);
		Viewable *dest=table.get(id);
		if (dest==NULL) {
			dest=new Viewable(id);
			table.put(id)=dest;
		}
		dest->addView(view);
		GLrepaint(30);
	}
	
	//Draw all the viewables we know about:
	void draw(const CkViewpoint &vp) {
		CkHashtableIterator *it=table.iterator();
		void *obj;
		while (NULL!=(obj=it->next())) {
			Viewable *v=*(Viewable **)obj;
			v->draw(vp);
		}
		delete it;
	}
	
};

//Control the GUI:
class uiController : public guiController {
	int wid,ht; //last known display width and height
	
	viewSource *vsource; //Place we request new views from
	
	bool enableUpdates;
	
	ViewableTable table;
	bool viewRequested;
	
	CkVector3d rotcen; //Center of rotation/scaling
	CkAxes3d axes; //Current drawing axes
	double eyeDist; //Distance from center to eye
	CkVector3d getEye(void) {
		return axes.getZ()*eyeDist+rotcen;
	}
	CkViewpoint vp; //Current viewpoint
	
	void state(int priority, const char *name) {
		if (priority>2)
			printf("%s\n",name);
	}
	
	void axesChanged(void) {
		// Update our viewpoint:
		state(1,"axesChanged");
		CkVector3d E = getEye();
		vp=CkViewpoint(E,rotcen,axes.getY());
		vp.discretize(wid,ht,60.0);
		// Update OpenGL matrix:
		glMatrixMode(GL_PROJECTION);
		double mat[16];
		vp.makeOpenGL(mat,0.1*eyeDist,10.0*eyeDist);
		glLoadMatrixd(mat);
		
		if (!viewRequested) {
			viewRequested=true;
			if (enableUpdates) {
				state(2,"viewRequest");
				vsource->viewRequest(&table,vp);
			}
			GLrepaint(0);
		}
	}
public:
	uiController(viewSource *src_) :vsource(src_) {
		eyeDist=4.0;
		rotcen=CkVector3d(0.5,0.5,0.5);
		viewRequested=false;
		enableUpdates=true;
		// axesChanged(); //< Can't call axesChanged until we know wid, ht
	}
	
	virtual void draw(void) { //Render
		state(2,"redraw");
		
		if (0) { // Tiny test sphere:
			glTranslated(0,0,1.0); glutWireSphere(0.1,5,5);
		}
		
		if (0) { // Tiny test quad:
			const static unsigned char texData[4*2*2]={
				0xff,0x00,0x00,0xff, 0x80,0x80,0x80,0xff,
				0x00,0x00,0xff,0xff, 0xc0,0xc0,0xc0,0xff};
			GLtexture tex(texData,2,2);
			GLtextureQuad(tex,CkVector3d(0,0,0), CkVector3d(1,0,0), 
				CkVector3d(0,1,0), CkVector3d(1,1,0), 
				GLtexture_linear);
		}
		table.draw(vp);
		
		
		viewRequested=false;
	}
	
	virtual void idle(void) { //Went idle
		state(0,"idle");
		vsource->viewPoll();
	}
	virtual void done(void) { //Exited
		exit(0);
	}
	
	/*The screen is now w x h pixels*/
	virtual void reshape(int wid_,int ht_) {
		state(2,"reshape");
		wid=wid_; ht=ht_;
		axesChanged();
	}
	
	virtual void mouseHover(int x,int y) {
		
	}
	int lastX,lastY;
	virtual void mouseDown(int button,int x,int y) {
		lastX=x; lastY=y;
	}
	virtual void mouseDrag(int x,int y) {
		state(1,"mouseDrag");
		double scale=0.002;
		double dx=scale*(x-lastX);
		double dy=scale*(y-lastY);
		axes.nudge(dx,dy);
		axesChanged();
		lastX=x; lastY=y;
	}
	virtual void mouseUp(int button,int x,int y) {
		
	}
	
	virtual void handleEvent(uiEvent_t uiCode) { //Hit a key or menu
		state(3,"handleEvent");
		switch(uiCode) {
		case 'u': enableUpdates=!enableUpdates; axesChanged(); break;
		case 'f': addFrame=!addFrame; GLrepaint(0); break;
		case 'e': eyeDist*=0.9; axesChanged(); break;
		case 'c': eyeDist*=1.1; axesChanged(); break;
		}
	}
};


guiController *createController(viewSource *src_) {
	return new uiController(src_);
}

