/**
 * AMPI Test program
 *  Orion Sky Lawlor, olawlor@acm.org, 2003/4/4
 */
#include <stdio.h>
#include <stdlib.h>
#include "mpi.h"
#include "charm.h" /* For CkAbort */

int getRank(void) {
	int rank; 
	MPI_Comm_rank(MPI_COMM_WORLD,&rank);
	return rank;
}

void testFailed(const char *where) {
	fprintf(stderr,"[%d] MPI TEST FAILED> %s\n",getRank(),where);
	CkAbort("MPI TEST FAILED");
	MPI_Finalize();
	exit(1);
}

void testEqual(int goodValue,int curValue,const char *where) {
	if (goodValue!=curValue) {
		fprintf(stderr,"[%d] %s> expected %d, got %d!\n",
			getRank(),where,goodValue,curValue);
		testFailed(where);
	}
}

void testRange(int loBound,int upBound,int curValue,const char *where) {
	if (curValue<loBound) {
		fprintf(stderr,"[%d] %s> expected at least %d, got %d!\n",
			getRank(),where,loBound,curValue);
		testFailed(where);
	}
	if (curValue>upBound) {
		fprintf(stderr,"[%d] %s> expected at most %d, got %d!\n",
			getRank(),where,upBound,curValue);
		testFailed(where);
	}
}

/// Change this verbosity level to get more printouts:
int verboseLevel=1;
void beginTest(int testLevel, const char *testName) {
	if (testLevel<10) // Block between important tests:
		MPI_Barrier(MPI_COMM_WORLD);
	if (testLevel<=verboseLevel) {
		int rank=getRank();
		if (rank==0 || verboseLevel>=10)
			printf("[%d] Testing: %s\n",rank,testName);
	}
}

#define TEST_MPI(routine,args) \
	beginTest(12,"     calling routine " #routine); \
	testEqual(MPI_SUCCESS,routine args, #routine " return code")


/**
 * Test out a bunch of MPI routines for this communicator.
 */
class MPI_Tester {
public:
	int rank,size; //Our rank in, and true size of the communicator
	
	MPI_Tester(MPI_Comm comm,int trueSize=-1);
	void test(void);

private: 
	MPI_Comm comm; //Communicator to test

//Little utility routines:
	/// Broadcast this value from this master:
	int bcast(int value,int master) {
		int mValue=-1; if (rank==master) mValue=value;
		TEST_MPI(MPI_Bcast,(&mValue,1,MPI_INT,master,comm));
		return mValue;
	}
	/// Reduce this value to this master:
	int reduce(int value,int master,int op) {
		int destValue=-1;
		TEST_MPI(MPI_Reduce,(&value,&destValue,1,MPI_INT,op,master,comm));
		return destValue;
	}
	/// Receive this integer from this source:
	int recv(int source,int tag) {
		int recvVal=-1;
		MPI_Status sts;
		TEST_MPI(MPI_Recv,(&recvVal,1,MPI_INT,source,tag,comm,&sts));
		testEqual(tag,sts.MPI_TAG,"Recv status tag");
		if (source!=MPI_ANY_SOURCE)
			testEqual(source,sts.MPI_SOURCE,"Recv status source");
		testEqual(comm,sts.MPI_COMM,"Recv status comm");
		/* not in standard: testEqual(1,sts.MPI_LENGTH,"Recv status length");*/
		return recvVal;
	}
};

MPI_Tester::MPI_Tester(MPI_Comm comm_,int trueSize)
	:comm(comm_)
{ /// Communicator size:
	beginTest(5,"Comm_size/rank");
	TEST_MPI(MPI_Comm_size,(comm,&size));
	if (trueSize!=-1) testEqual(trueSize,size,"MPI_Comm_size value");
	TEST_MPI(MPI_Comm_rank,(comm,&rank));
	testRange(0,size,rank,"MPI_Comm_rank value");
}

void MPI_Tester::test(void)
{ 
/// Send and receive:
	beginTest(3,"Send/recv");
	// Send to the next guy in a ring:
	int next=(rank+1)%size;
	int prev=(rank-1+size)%size;
	int tag=12387, recvVal=-1;
	
	// Forward around ring:
	TEST_MPI(MPI_Send,(&rank,1,MPI_INT,next,tag,comm));
	testEqual(prev,recv(prev,tag),"Received rank (using prev as source)");
	
	// Simultanious forward and backward:
	TEST_MPI(MPI_Send,(&rank,1,MPI_INT,next,tag,comm));
	TEST_MPI(MPI_Send,(&rank,1,MPI_INT,prev,tag,comm));
	testEqual(next,recv(next,tag),"Received rank (specifying next as source)");
	testEqual(prev,recv(prev,tag),"Received rank (specifying prev as source)");

/// Collective operations:
	beginTest(4,"Barrier");
	TEST_MPI(MPI_Barrier,(comm));
	
	beginTest(3,"Broadcast");
	const int bcastValue=123;
	testEqual(bcastValue,bcast(bcastValue,0),"Broadcast integer from 0");
	testEqual(bcastValue,bcast(bcastValue,size/2),"Broadcast integer from size/2");
	testEqual(size,bcast(size,0),"Broadcast comm size");
	
	beginTest(3,"Reduction");
	int sumSize=reduce(1,0,MPI_SUM);
	if (rank==0) testEqual(size,sumSize,"Reduce sum integer 1");
	int prodSize=reduce(1,0,MPI_PROD);
	if (rank==0) testEqual(1,prodSize,"Reduce product integer 1");
}

int main(int argc,char **argv)
{
	MPI_Init(&argc,&argv);
	MPI_Comm comm=MPI_COMM_WORLD;
	MPI_Tester *master=new MPI_Tester(comm);
	
for (int loop=0;loop<3;loop++) {
	beginTest(1,"Testing...");
	if (1) 
	{//Try out MPI_COMM_WORLD:
		beginTest(2,"Communicator = MPI_COMM_WORLD");
		master->test();
	}
	
	if (1) 
	{//Try out MPI_COMM_SELF:
		beginTest(2,"Communicator = MPI_COMM_SELF");
		(new MPI_Tester(MPI_COMM_SELF))->test();
	}
	
	if (1) 
	{ //Split the world in two and retest:
		beginTest(2,"Communicator Split");
		MPI_Comm split;
		int nPieces=2+loop;
		int myColor=(master->rank*nPieces)/master->size;
		TEST_MPI(MPI_Comm_split,(comm,myColor,0,&split));
		MPI_Tester *tester=new MPI_Tester(split);
		tester->test();
	}
}
	
	if (getRank()==0) CkPrintf("All tests passed\n");
	MPI_Finalize();
	return 0;
}
