## Virtual DNA v3.0
## TcBishop Dec 2004
## Tulane University 
##  http://dna.cbr.tulane.edu
## Please e-mail improvements or comments to
##  bishop@tulane.edu / 504-988-6203
## VERSION HISTORY
## v1: initial version
## v2: the lost version
## v3: J.S. speedups and TcB coarse grained 1 cube/bp
## OPEN ISSUES:  
##  a) do the axes really agree w/ 3DNA
##  b) a gui
##  c) really fast 
#####################################
###  NOTES:
###  this script reconstructs a 3D curve using
###   the DNA interbasepair helical parameters 
###  Gamma = (shift,slide,rise)
###  Omega = (roll,tilt,twist)
###  Default is 10 nucleosomes 146bp each. 
###  and a 30bp length of straight linker
######################################
### USAGE:
### requires VMD 1.8.2 or later available at www.ks.uiuc.edu
###  0) save this file as vdna.vmd
###  1) start vmd 
###  2) enter the following command in VMD's console window
###      source vdna.vmd
### other things to try then repeat 1 and 2 above
###  a) modify the linker or core nucleosome
###     structure by changing the values of 
###      of nLinkBp  or nCoreBp defined below
###      and  source vdna.vmd again.
###  b) ADVANCED: modify the definition
###     of the helical parameters themselves
###      in the procedures GAMMA or OMEGA
######################################


## Tell Tcl that we're a package and any dependencies we may have
package provide vdna 1.0


namespace eval ::VDNA:: {
  ##################################
  ### user defined inputs
  ### nNuc number of nucleosomes
  ### nLinkBp length of linker in basepairs
  ### nCoreBp length of nucleosome core in basepairs
  ### nNucBp number of basepairs in a nucleosome
  ### nBp total number of bp to model
  ### 3DNA = 1 creates output for use with 3DNA
  ###    additional utilities are needed
  ##################################
  variable nNuc  28.0
  variable nLinkBp 38.0
  variable nCoreBp 146.0
  variable nNucBp [expr $nCoreBp + $nLinkBp ]
  variable nBp [expr $nNucBp * $nNuc ]
  variable Phase  0.0
  variable 3DNA 0

  # these values get used over and over in Omega
  set Pi 3.14159265358979
  variable nucTwist    [expr {14.0 * 2 * $Pi / $nCoreBp}]
  variable nucKappa    [expr {1.76 * 2 * $Pi / $nCoreBp}]
  variable linkerTwist [expr {3.59 * 2 * $Pi / $nLinkBp}]
  variable gammaKappa  [expr {-20.6 * 1.75   / $nCoreBp}]

  ##################################
  ### END OF SIMPLE USER DEFINED INPUTS
  ### see the procedures
  ### GAMMA and OMEGA to redefine
  ###  the helical parameters
  ##################################

  ##################################
  #### internal state variables
  ##################################
  variable graphmol
  variable nCoreBp
  variable nLinkBp
  variable Phase

  ##################################
  ### cross-section of the rod
  ##################################
  ### ideal (Wdth)^2 + (Dpth)^2 = (20)^2
  ###  and Dpth = 1.7*Wdth
  ### following is set to match atomic models
  ###  values are in Angstrom
  variable Wdth 5.0
  variable Dpth 17.2
  ##################################
}


##########################################################
#	PROCEDURE DEFINITIONS
#  1) OMEGA  
#  2) GAMMA
#  3) DRAW_CUBE
##########################################################

#######################################
### returns (roll,tilt,twist)
### isNuc =1 returns values for the nucleosome core
### isNuc =0 returns values for the linker DNA
proc ::VDNA::Omega4 { s isNuc } {

  if { $isNuc } {
    variable nucKappa
    variable nucTwist
    variable Phase
    set Om3 $nucTwist
    set angle [expr {$s*$Om3 + $Phase}]
    set Om2 [expr  { $nucKappa * cos($angle)}]
    set Om1 [expr  { $nucKappa * sin($angle)}]	
    return [list $Om1 $Om2 $Om3]
  }
  # not nuc
  variable linkerTwist
  return [list 0 0 [expr {$linkerTwist}]]
}

#######################################
#### GAMMA (s, isNuc)
### returns (shift,slide,rise)
### isNuc =1 returns values for the nucleosome core
### isNuc =0 returns values for the linker DNA
proc ::VDNA::Gamma4 { s isNuc} {
    	
  if { $isNuc } { 
    variable Phase
    variable nucTwist
    variable gammaKappa
    set Om3 $nucTwist
    set G3  3.4
    set G2 [expr    $gammaKappa * cos($s*$Om3 + $Phase) ]
    set G1 [expr    $gammaKappa * sin($s*$Om3 + $Phase) ]

  } else {
    set G3  3.4
    set G2  0.0
    set G1  0.0
  }
  return [list $G1 $G2 $G3]
}
## end of GAMMA
#######################################

#######################################
### DRAW_CUBE from END1 to END2
#######################################
### End1 and End2 are lists of 3-vectors 
### each End contains the coordinates for 4 points in space
proc ::VDNA::draw_cube { End1 End2 } {
  variable graphmol

#  if {[llength $End1] != 4 } { return "ERROR: draw_cube inproper data format End1"}
#  if {[llength $End2] != 4 } { return "ERROR: draw_cube inproper data format End2"}

  for {set k 0 } { $k <= 3 } { incr k } { 
    set Pt(1,$k) [lindex $End1 $k]
    set Pt(2,$k) [lindex $End2 $k]
  }

  for { set k 0 } { $k <=3 } { incr k } {
    set kp  [expr {($k+1)%4}]
    set kpp [expr {($k+2)%4}]
    set km  [expr {($k-1)%4}]

    # normals for first set of triangles
    set n11  [vecnorm [vecsub $Pt(1,$km)  $Pt(1,$k)]]
    set n12  [vecnorm [vecsub $Pt(2,$kpp) $Pt(2,$kp)]]
    set n13  [vecnorm [vecsub $Pt(1,$kpp) $Pt(1,$kp)]]

    # normals for second set of triangles
    set n21  [vecnorm [vecsub $Pt(1,$km)  $Pt(1,$k)]]
    set n22  [vecnorm [vecsub $Pt(2,$km)  $Pt(2,$k)]]
    set n23  [vecnorm [vecsub $Pt(2,$kpp) $Pt(2,$kp)]]

    # use trinorms so that graphics are smooth
    graphics $graphmol color [expr {$k%2}]
    graphics $graphmol trinorm $Pt(1,$k) $Pt(2,$kp) $Pt(1,$kp) $n11 $n12 $n13
    graphics $graphmol trinorm $Pt(1,$k) $Pt(2,$k)  $Pt(2,$kp) $n21 $n22 $n23
  }
} 
### end of draw_cube


#########################
### INITIALIZE MAIN  LOOP
#########################

proc ::VDNA::vdna {} {
  variable Wdth
  variable Dpth
  variable nNuc
  variable nLinkBp
  variable nCoreBp
  variable nNucBp
  variable nBp
  variable Phase
  variable 3DNA
  variable graphmol

  ### initialize the graphics
  mol new
  set graphmol [molinfo top]
  set Pi 3.14159265358979

  ### WARNGING if DELTA S is too large
  ### the numerical integration does not converge
  ### but graphics will still be produced
  ###  deltaS of 0.05 is sufficient to capture the 
  ###  the curvature of a nucleosome but with some error
  ###  mobius strip with the same curvature and twist
  ###   as your model of interest is a nice check on convergence
  ###   
  set deltaS 0.1
  set s    0.0      
  set sp   $deltaS   

  ##################################
  ##### orientation of first basepair
  ##################################
  set R(0.0) { 0.0 0.0 0.0 }   
  set d(1,0.0) { 1.0 0.0 0.0 }  
  set d(2,0.0) { 0.0 1.0 0.0 }  
  set d(3,0.0) { 0.0 0.0 1.0 }  
  ##################################
  ### DEFINE FIRST END at S = 0.0
  ##################################
  set sign(0)  1
  set sign(1)  1
  set sign(2) -1
  set sign(3) -1
  set End1 ""
  for { set j 0 } { $j <= 3} { incr j } {
    set scale [expr $sign($j)/2.0 * $Wdth]
    set vec1  [vecscale $scale $d(1,$s) ]
    set scale [expr $sign([expr ($j-1)%4 ])/2.0 * $Dpth]
    set vec2  [vecscale $scale $d(2,$s)]
    set Pt($j) [vecadd $vec1 $vec2 ]
    set Pt($j) [vecadd $R($s) $Pt($j) ]
    set End1 [ lappend End1 $Pt($j)]
  }
  ### end of End1 loop

  puts "Completed initialization"
  
  ### open files for helical parameter output

  if {$3DNA } {
    set hplist { "Ti.dat" "Ro.dat" "Tw.dat" "Sh.dat" "Sl.dat" "Ri.dat" }
    foreach hp  $hplist { 
      set fp($hp)  [open $hp w ]
    }
    puts "opened all files "
  }


  #########################
  ###  MAIN  LOOP
  #########################
  ### s is the fiducial arch length
  ### sp is value of s plus deltaS
  ###  0.0 < s < nBp
  ################################
  set s      0.0
  set avgcnt 0
  set avgR { 0.0 0.0 0.0 }
  set isNucLink 0
  set isNuc     1

  if {$3DNA  } {
    set Om3  [ Omega4 $s 1 ] 
    set Ga3  [ Gamma4 $s 1 ] 
     puts " writing 3DNA data $s"
    foreach i { 0 1 2  } {
      puts $fp([lindex $hplist $i])  [expr 180.0/$Pi * [lindex $Om  $i  ]] 
      puts $fp([lindex $hplist [expr $i + 3 ]])  [lindex $Ga  $i ] 
    }
  }

  display update off
  while {$s <= $nBp} { 
    set sp [expr $s + $deltaS]  

    ### determine sbp as a measure relative
    ### to beginning nucleosome core or linker length
    set nuc [expr floor($s/$nNucBp) ]
    set sbp [expr $s - $nuc *$nNucBp ]
    set isNuc [expr $sbp <= $nCoreBp ]
    set isNucLink [expr $sbp == $nCoreBp ]
    if { $isNuc == 0  } { 
      set sbp [expr {($s - $nuc*$nNucBp - $nCoreBp )} ]
    } 

    set Om  [ Omega4 $sbp $isNuc ] 
    set Ga  [ Gamma4 $sbp $isNuc ]

    if {($3DNA ) && ([expr {int($s)}] == $s)} { 
      foreach i { 0 1 2  } {
        puts $fp([lindex $hplist $i])  [expr {180.0/$Pi * [lindex $Om  $i ]}]
        puts $fp([lindex $hplist [expr {$i +3} ]]) [lindex $Ga  $i ] 
      }
    }

    # puts $rvec1 
    set G1 [expr {$deltaS * [lindex $Ga 0]} ]
    set G2 [expr {$deltaS * [lindex $Ga 1]} ]
    set G3 [expr {$deltaS * [lindex $Ga 2]} ]

    # transformation of centerline to next increment 
    set vecx [vecscale $G1 $d(1,$s)]
    set vecy [vecscale $G2 $d(2,$s)]
    set vecz [vecscale $G3 $d(3,$s)]
    set rvec2 [vecadd $vecx $vecy $vecz]
    set R($sp) [vecadd $R($s) $rvec2]

    # puts $R($s)
    # transformation directors to next increment	
    set vecx [vecscale [lindex $Om 0] $d(1,$s) ]
    set vecy [vecscale [lindex $Om 1] $d(2,$s) ]
    set vecz [vecscale [lindex $Om 2] $d(3,$s) ]
    set Om   [vecadd $vecx $vecy $vecz ]
    for { set k 1 } { $k <= 3 } { incr k } {
      set dvec1($k) [veccross $Om $d($k,$s) ]
      set dvec1($k) [vecscale $dvec1($k) $deltaS]
      set dvec2($k) $d($k,$s)
      set d($k,$sp) [vecadd $dvec1($k) $dvec2($k)]
      set d($k,$sp) [vecnorm $d($k,$sp) ]
    }

    #  end of tranformation of centerline and directors 
    #   puts "  s= $s "
    #   puts " R  =  { $R($s)   }"
    #   puts " d1 =  { $d(1,$s) } "
    #   puts " d2 =  { $d(2,$s) } "
    #   puts " d3 =  { $d(3,$s) } "

    ### COARSE GRAIN TO one CUBE per BP for version 3	
    if {([expr {int($s)}] == $s)} { 
      set End2 ""
      for { set j 0 } { $j <= 3} { incr j } {
        set scale [expr {$sign($j)/2.0 * $Wdth}]
        set vec1  [vecscale $scale $d(1,$sp) ]
        set scale [expr {$sign([expr {($j-1)%4}])/2.0 * $Dpth}]
        set vec2  [vecscale $scale $d(2,$sp)]
        set Pt($j) [vecadd $vec1 $vec2 ]
        set Pt($j) [vecadd $R($sp) $Pt($j) ]
        set End2 [ lappend End2 $Pt($j)]
      }
  
      ###  draw a cube from End1 (current bp) to End2 (next bp)
      #	puts "END1 $End1  length [llength $End1]"
      #	puts "END2 $End2  length [llength $End2]"
      draw_cube $End1 $End2

      ## set up for next iteration
      set End1 $End2
    }
	
    ### keep track of the nucleosomes and draw a  for each
    if { $isNucLink } {
      puts "nucleosome counter: bp $s nucnumber: $nuc "
      set avgR [vecscale [expr {1.0/$avgcnt} ] $avgR  ]
      graphics $graphmol color [expr {int($nuc)%7}]
      graphics $graphmol sphere $avgR radius 40 resolution 20
      set avgR { 0.0 0.0 0.0 }
      set avgcnt 0
    } elseif { $isNuc } {
      set avgR [vecadd $avgR $R($s) ]
      incr avgcnt
    }

    ### setup for next iteration
    set s $sp
  } 
  ### end of main loop	

  display projection orthographic
  display update on
  display resetview

  ### close the 3DNA files
  if {$3DNA }  {
    foreach hp  $hplist { 
      close $fp($hp)
    }
  }
}

time {VDNA::vdna}
#exit

