/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: WKFThreads.h,v $
 *      $Author: johns $        $Locker:  $             $State: Exp $
 *      $Revision: 1.11 $       $Date: 2016/10/26 05:03:26 $
 *
 ***************************************************************************
 * WKFThreads.h - code for spawning threads on various platforms.
 *                Code donated by John Stone, john.stone@gmail.com
 *                This code was originally written for the
 *                Tachyon Parallel/Multiprocessor Ray Tracer.
 *                Improvements have been donated by Mr. Stone on an
 *                ongoing basis.  This code is provided under the
 *                three clause BSD Open Source License.
 *
 * NOTE: The latest version of this source file can be re-generated by
 *       running the sequence of 'sed' commands shown at the top of the
 *       "threads.c" file within the Tachyon source distribution.
 *
 ***************************************************************************/
/* Tachyon copyright reproduced below */
/*
 * Copyright (c) 1994-2016 John E. Stone
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef WKF_THREADS_INC
#define WKF_THREADS_INC 1

#ifdef __cplusplus
extern "C" {
#endif

/* define which thread calls to use */
#if defined(USEPOSIXTHREADS) && defined(USEUITHREADS)
#error You may only define USEPOSIXTHREADS or USEUITHREADS, but not both
#endif

/* POSIX Threads */
#if defined(_AIX) || defined(__APPLE__) || defined(_CRAY) || defined(__hpux) || defined(__irix) || defined(__linux) || defined(__osf__) ||  defined(__PARAGON__)
#if !defined(USEUITHREADS) && !defined(USEPOSIXTHREADS)
#define USEPOSIXTHREADS
#endif
#endif

/* Unix International Threads */
#if defined(SunOS)
#if !defined(USEPOSIXTHREADS) && !defined(USEUITHREADS)
#define USEUITHREADS
#endif
#endif


/*
 * CPU capability flags
 */
#define CPU_UNKNOWN     0x0001

/* Intel x86 CPU features we may need at runtime */
#define CPU_SSE2        0x0100
#define CPU_AVX         0x0200
#define CPU_AVX2        0x0400
#define CPU_FMA         0x0800
#define CPU_AVX512F     0x1000
#define CPU_AVX512CD    0x1000
#define CPU_AVX512ER    0x2000
#define CPU_AVX512PF    0x4000
#define CPU_KNL         (CPU_AVX512F | CPU_AVX512CD | \
                         CPU_AVX512ER | CPU_AVX512PF)

typedef struct {
  unsigned int flags;
} wkf_cpu_caps_t;


#ifdef WKFTHREADS
#ifdef USEPOSIXTHREADS
#include <pthread.h>

typedef pthread_t        wkf_thread_t;
typedef pthread_mutex_t   wkf_mutex_t;
typedef pthread_cond_t     wkf_cond_t;

typedef struct rwlock_struct {
  pthread_mutex_t lock;          /**< read/write monitor lock */
  int rwlock;                    /**< if >0 = #rdrs, if <0 = wrtr, 0=none */
  pthread_cond_t  rdrs_ok;       /**< start waiting readers */
  unsigned int waiting_writers;  /**< # of waiting writers  */
  pthread_cond_t  wrtr_ok;       /**< start waiting writers */ 
} wkf_rwlock_t;

#endif

#ifdef USEUITHREADS
#include <thread.h>

typedef thread_t  wkf_thread_t;
typedef mutex_t   wkf_mutex_t;
typedef cond_t    wkf_cond_t;
typedef rwlock_t  wkf_rwlock_t;
#endif


#ifdef _MSC_VER
#include <windows.h>
typedef HANDLE wkf_thread_t;
typedef CRITICAL_SECTION wkf_mutex_t;

#if 0 && (NTDDI_VERSION >= NTDDI_WS08 || _WIN32_WINNT > 0x0600)
/* Use native condition variables only with Windows Server 2008 and newer... */
#define WKFUSEWIN2008CONDVARS 1
typedef CONDITION_VARIABLE wkf_cond_t;
#else
/* Every version of Windows prior to Vista/WS2008 must emulate */
/* variables using manually resettable events or other schemes */

/* For higher performance, use interlocked memory operations   */
/* rather than locking/unlocking mutexes when manipulating     */
/* internal state.                                             */
#if 1
#define WKFUSEINTERLOCKEDATOMICOPS 1
#endif
#define WKF_COND_SIGNAL    0
#define WKF_COND_BROADCAST 1
typedef struct {
  LONG waiters;                  /**< MUST be 32-bit aligned for correct   */
                                 /**< operation with InterlockedXXX() APIs */
  CRITICAL_SECTION waiters_lock; /**< lock itself                          */
  HANDLE events[2];              /**< Signal and broadcast event HANDLEs.  */
} wkf_cond_t;
#endif

typedef struct rwlock_struct {
  wkf_mutex_t lock;               /**< read/write monitor lock */
  int rwlock;                    /**< if >0 = #rdrs, if <0 = wrtr, 0=none */
  wkf_cond_t  rdrs_ok;            /**< start waiting readers */
  unsigned int waiting_writers;  /**< # of waiting writers  */
  wkf_cond_t  wrtr_ok;            /**< start waiting writers */ 
} wkf_rwlock_t;

#endif
#endif /* _MSC_VER */


#ifndef WKFTHREADS
typedef int wkf_thread_t;
typedef int wkf_mutex_t;
typedef int wkf_cond_t;
typedef int wkf_rwlock_t;
#endif

#if defined(USENETBSDATOMICS) 
#include <sys/atomic.h>
#elif defined(USESOLARISATOMICS)
#include <atomic.h>
#endif

typedef struct atomic_int_struct {
  int padding1[8];        /**< Pad to avoid false sharing, cache aliasing */
  wkf_mutex_t lock;        /**< Mutex lock for the structure */
#if defined(USENETBSDATOMICS)
  unsigned int val;       /**< Integer value to be atomically manipulated */
#elif defined(USESOLARISATOMICS)
  unsigned int val;       /**< Integer value to be atomically manipulated */
#elif defined(USEWIN32ATOMICS)
  LONG val;               /**< Integer value to be atomically manipulated */
#else
  int val;                /**< Integer value to be atomically manipulated */
#endif
  int padding2[8];        /**< Pad to avoid false sharing, cache aliasing */
} wkf_atomic_int_t;


typedef struct barrier_struct {
  int padding1[8];        /**< Pad to avoid false sharing, cache aliasing */
  wkf_mutex_t lock;        /**< Mutex lock for the structure */
  int n_clients;          /**< Number of threads to wait for at barrier */
  int n_waiting;          /**< Number of currently waiting threads */
  int phase;              /**< Flag to separate waiters from fast workers */
  int sum;                /**< Sum of arguments passed to barrier_wait */
  int result;             /**< Answer to be returned by barrier_wait */
  wkf_cond_t wait_cv;      /**< Clients wait on condition variable to proceed */
  int padding2[8];        /**< Pad to avoid false sharing, cache aliasing */
} wkf_barrier_t;

typedef struct wkf_run_barrier_struct {
  int padding1[8];        /**< Pad to avoid false sharing, cache aliasing */
  wkf_mutex_t lock;        /**< Mutex lock for the structure */
  int n_clients;          /**< Number of threads to wait for at barrier */
  int n_waiting;          /**< Number of currently waiting threads */
  int phase;              /**< Flag to separate waiters from fast workers */
  void * (*fctn)(void *); /**< Fctn ptr to call, or NULL if done */
  void * parms;           /**< parms for fctn pointer */
  void * (*rslt)(void *); /**< Fctn ptr to return to barrier wait callers */
  void * rsltparms;       /**< parms to return to barrier wait callers */
  wkf_cond_t wait_cv;      /**< Clients wait on condition variable to proceed */
  int padding2[8];        /**< Pad to avoid false sharing, cache aliasing */
} wkf_run_barrier_t;


/*
 * Routines for querying processor counts, and managing CPU affinity
 */
/** number of physical processors available */
int wkf_thread_numphysprocessors(void);

/** number of processors available, subject to user override */
int wkf_thread_numprocessors(void);

/** CPU optional instruction set capability flags */
int wkf_cpu_capability_flags(wkf_cpu_caps_t *cpucaps);

/** query CPU affinity of the calling process (if allowed by host system) */
int * wkf_cpu_affinitylist(int *cpuaffinitycount);

/** set the CPU affinity of the current thread (if allowed by host system) */
int wkf_thread_set_self_cpuaffinity(int cpu);

/** set the concurrency level and scheduling scope for threads */
int wkf_thread_setconcurrency(int);


/*
 * Thread management
 */
/** create a new child thread */
int wkf_thread_create(wkf_thread_t *, void * fctn(void *), void *);

/** join (wait for completion of, and merge with) a thread */
int wkf_thread_join(wkf_thread_t, void **);


/*
 * Mutex management
 */
/** initialize a mutex */
int wkf_mutex_init(wkf_mutex_t *);

/** lock a mutex */
int wkf_mutex_lock(wkf_mutex_t *);

/** try to lock a mutex */
int wkf_mutex_trylock(wkf_mutex_t *);

/** lock a mutex by spinning only */
int wkf_mutex_spin_lock(wkf_mutex_t *);

/** unlock a mutex */
int wkf_mutex_unlock(wkf_mutex_t *);

/** destroy a mutex */
int wkf_mutex_destroy(wkf_mutex_t *);


/*
 * Condition variable management
 */
/** initialize a condition variable */
int wkf_cond_init(wkf_cond_t *);

/** destroy a condition variable */
int wkf_cond_destroy(wkf_cond_t *);

/** wait on a condition variable */
int wkf_cond_wait(wkf_cond_t *, wkf_mutex_t *);

/** signal a condition variable, waking at least one thread */
int wkf_cond_signal(wkf_cond_t *);

/** signal a condition variable, waking all threads */
int wkf_cond_broadcast(wkf_cond_t *);


/*
 * Atomic operations on integers
 */
/** initialize an atomic int variable */
int wkf_atomic_int_init(wkf_atomic_int_t * atomp, int val);

/** destroy an atomic int variable */
int wkf_atomic_int_destroy(wkf_atomic_int_t * atomp);

/** set an atomic int variable */
int wkf_atomic_int_set(wkf_atomic_int_t * atomp, int val);

/** get an atomic int variable */
int wkf_atomic_int_get(wkf_atomic_int_t * atomp);

/** fetch an atomic int and add inc to it, returning original value */
int wkf_atomic_int_fetch_and_add(wkf_atomic_int_t * atomp, int inc);

/** fetch an atomic int  and add inc to it, returning new value */
int wkf_atomic_int_add_and_fetch(wkf_atomic_int_t * atomp, int inc);


/*
 * Reader/writer lock management
 */
/** initialize a reader/writer lock */
int wkf_rwlock_init(wkf_rwlock_t *);

/** set reader lock */
int wkf_rwlock_readlock(wkf_rwlock_t *);

/** set writer lock */
int wkf_rwlock_writelock(wkf_rwlock_t *);

/** unlock reader/writer lock */
int wkf_rwlock_unlock(wkf_rwlock_t *);


/*
 * counting barrier
 */
/** initialize counting barrier primitive */
wkf_barrier_t * wkf_thread_barrier_init(int n_clients);

/** destroy counting barrier primitive */
void wkf_thread_barrier_destroy(wkf_barrier_t *barrier);

/** synchronize on counting barrier primitive */
int wkf_thread_barrier(wkf_barrier_t *barrier, int increment);


/*
 * This is a symmetric barrier routine designed to be used
 * in implementing a sleepable thread pool.
 */
/** initialize thread pool barrier */
int wkf_thread_run_barrier_init(wkf_run_barrier_t *barrier, int n_clients);

/** destroy thread pool barrier */
void wkf_thread_run_barrier_destroy(wkf_run_barrier_t *barrier);

/** sleeping barrier synchronization for thread pool */
void * (*wkf_thread_run_barrier(wkf_run_barrier_t *barrier,
                                void * fctn(void*),
                                void * parms,
                                void **rsltparms))(void *);

/** non-blocking poll to see if peers are already at the barrier */
int wkf_thread_run_barrier_poll(wkf_run_barrier_t *barrier);


/**
 * Task tile struct for stack, iterator, and scheduler routines;
 * 'start' is inclusive, 'end' is exclusive.  This yields a
 * half-open interval that corresponds to a typical 'for' loop.
 */
typedef struct wkf_tasktile_struct {
  int start;         /**< starting task ID (inclusive) */
  int end;           /**< ending task ID (exclusive) */
} wkf_tasktile_t;


/*
 * tile stack
 */
#define WKF_TILESTACK_EMPTY -1

/**
 * stack of work tiles, for error handling
 */
typedef struct {
  wkf_mutex_t mtx;    /**< Mutex lock for the structure */
  int growthrate;    /**< stack growth chunk size */
  int size;          /**< current allocated stack size */
  int top;           /**< index of top stack element */
  wkf_tasktile_t *s;  /**< stack of task tiles */
} wkf_tilestack_t;

/** initialize task tile stack (to empty) */
int wkf_tilestack_init(wkf_tilestack_t *s, int size);

/** destroy task tile stack */
void wkf_tilestack_destroy(wkf_tilestack_t *);

/** shrink memory buffers associated with task tile stack if possible */
int wkf_tilestack_compact(wkf_tilestack_t *);

/** push a task tile onto the stack */
int wkf_tilestack_push(wkf_tilestack_t *, const wkf_tasktile_t *);

/** pop a task tile off of the stack */
int wkf_tilestack_pop(wkf_tilestack_t *, wkf_tasktile_t *);

/** pop all of the task tiles off of the stack */
int wkf_tilestack_popall(wkf_tilestack_t *);

/** query if the task tile stack is empty or not */
int wkf_tilestack_empty(wkf_tilestack_t *);


/**
 * Shared iterators intended for trivial CPU/GPU load balancing with no
 * exception handling capability (all work units must complete with
 * no errors, or else the whole thing is canceled).
 */
#define WKF_SCHED_DONE     -1   /**< no work left to process        */
#define WKF_SCHED_CONTINUE  0   /**< some work remains in the queue */

/** iterator used for dynamic load balancing */
typedef struct wkf_shared_iterator_struct {
  wkf_mutex_t mtx;      /**< mutex lock */
  int start;           /**< starting value (inclusive) */
  int end;             /**< ending value (exlusive) */
  int current;         /**< current value */
  int fatalerror;      /**< cancel processing immediately for all threads */
} wkf_shared_iterator_t;

/** initialize a shared iterator */
int wkf_shared_iterator_init(wkf_shared_iterator_t *it);

/** destroy a shared iterator */
int wkf_shared_iterator_destroy(wkf_shared_iterator_t *it);

/** Set shared iterator state to half-open interval defined by tile */
int wkf_shared_iterator_set(wkf_shared_iterator_t *it, wkf_tasktile_t *tile);

/**
 * iterate the shared iterator with a requested tile size,
 * returns the tile received, and a return code of -1 if no
 * iterations left or a fatal error has occured during processing,
 * canceling all worker threads.
 */
int wkf_shared_iterator_next_tile(wkf_shared_iterator_t *it, int reqsize,
                                 wkf_tasktile_t *tile);

/** worker thread calls this to indicate a fatal error */
int wkf_shared_iterator_setfatalerror(wkf_shared_iterator_t *it);

/** master thread calls this to query for fatal errors */
int wkf_shared_iterator_getfatalerror(wkf_shared_iterator_t *it);


/*
 * Thread pool.
 */
/** shortcut macro to tell the create routine we only want CPU cores */
#define WKF_THREADPOOL_DEVLIST_CPUSONLY NULL

/** symbolic constant macro to test if we have a GPU or not */
#define WKF_THREADPOOL_DEVID_CPU -1

/** thread-specific handle data for workers */
typedef struct wkf_threadpool_workerdata_struct {
  int padding1[8];                        /**< avoid false sharing */
  wkf_shared_iterator_t *iter;             /**< dynamic work scheduler */
  wkf_tilestack_t *errorstack;             /**< stack of tiles that failed */
  int threadid;                           /**< worker thread's id */
  int threadcount;                        /**< total number of worker threads */
  int devid;                              /**< worker CPU/GPU device ID */
  float devspeed;                         /**< speed scaling for this device */
  void *parms;                            /**< fctn parms for this worker */
  void *thrpool;                          /**< void ptr to thread pool struct */
  int padding2[8];                        /**< avoid false sharing */
} wkf_threadpool_workerdata_t;

typedef struct wkf_threadpool_struct {
  int workercount;                        /**< number of worker threads */
  int *devlist;                           /**< per-worker CPU/GPU device IDs */
  wkf_shared_iterator_t iter;              /**< dynamic work scheduler */
  wkf_tilestack_t errorstack;              /**< stack of tiles that failed */
  wkf_thread_t *threads;                   /**< worker threads */
  wkf_threadpool_workerdata_t *workerdata; /**< per-worker data */
  wkf_run_barrier_t runbar;                /**< master/worker run barrier */
} wkf_threadpool_t;

/** create a thread pool with a specified number of worker threads */
wkf_threadpool_t * wkf_threadpool_create(int workercount, int *devlist);

/** launch threads onto a new function, with associated parms */
int wkf_threadpool_launch(wkf_threadpool_t *thrpool,
                         void *fctn(void *), void *parms, int blocking);

/** wait for all worker threads to complete their work */
int wkf_threadpool_wait(wkf_threadpool_t *thrpool);

/** join all worker threads and free resources */
int wkf_threadpool_destroy(wkf_threadpool_t *thrpool);

/** query number of worker threads in the pool */
int wkf_threadpool_get_workercount(wkf_threadpool_t *thrpool);

/** worker thread can call this to get its ID and number of peers */
int wkf_threadpool_worker_getid(void *voiddata, int *threadid, int *threadcount);

/** worker thread can call this to get its CPU/GPU device ID */
int wkf_threadpool_worker_getdevid(void *voiddata, int *devid);

/**
 * Worker thread calls this to set relative speed of this device
 * as determined by the SM/core count and clock rate
 * Note: this should only be called once, during the worker's
 * device initialization process
 */
int wkf_threadpool_worker_setdevspeed(void *voiddata, float speed);

/**
 * Worker thread calls this to get relative speed of this device
 * as determined by the SM/core count and clock rate
 */
int wkf_threadpool_worker_getdevspeed(void *voiddata, float *speed);

/**
 * worker thread calls this to scale max tile size by worker speed
 * as determined by the SM/core count and clock rate
 */
int wkf_threadpool_worker_devscaletile(void *voiddata, int *tilesize);

/** worker thread can call this to get its client data pointer */
int wkf_threadpool_worker_getdata(void *voiddata, void **clientdata);

/** Set dynamic scheduler state to half-open interval defined by tile */
int wkf_threadpool_sched_dynamic(wkf_threadpool_t *thrpool, wkf_tasktile_t *tile);

/**
 * worker thread calls this to get its next work unit
 * iterate the shared iterator, returns -1 if no iterations left
 */
int wkf_threadpool_next_tile(void *thrpool, int reqsize, wkf_tasktile_t *tile);

/**
 * worker thread calls this when it fails computing a tile after
 * it has already taken it from the scheduler
 */
int wkf_threadpool_tile_failed(void *thrpool, wkf_tasktile_t *tile);

/** worker thread calls this to indicate that an unrecoverable error occured */
int wkf_threadpool_setfatalerror(void *thrparms);

/** master thread calls this to query for fatal errors */
int wkf_threadpool_getfatalerror(void *thrparms);


/**
 * Routines to generate a pool of threads which then grind through
 * a dynamically load balanced work queue implemented as a shared iterator.
 * No exception handling is possible, just a simple all-or-nothing attept.
 * Useful for simple calculations that take very little time.
 * An array of threads is generated, launched, and joined all with one call.
 */
typedef struct wkf_threadlaunch_struct {
  int padding1[8];              /**< avoid false sharing, cache aliasing */
  wkf_shared_iterator_t *iter;   /**< dynamic scheduler iterator */
  int threadid;                 /**< ID of worker thread */
  int threadcount;              /**< number of workers */
  void * clientdata;            /**< worker parameters */
  int padding2[8];              /**< avoid false sharing, cache aliasing */
} wkf_threadlaunch_t;

/** launch up to numprocs threads using shared iterator as a load balancer */
int wkf_threadlaunch(int numprocs, void *clientdata, void * fctn(void *),
                    wkf_tasktile_t *tile);

/** worker thread can call this to get its ID and number of peers */
int wkf_threadlaunch_getid(void *thrparms, int *threadid, int *threadcount);

/** worker thread can call this to get its client data pointer */
int wkf_threadlaunch_getdata(void *thrparms, void **clientdata);

/**
 * worker thread calls this to get its next work unit
 * iterate the shared iterator, returns -1 if no iterations left
 */
int wkf_threadlaunch_next_tile(void *voidparms, int reqsize,
                              wkf_tasktile_t *tile);

/** worker thread calls this to indicate that an unrecoverable error occured */
int wkf_threadlaunch_setfatalerror(void *thrparms);


#ifdef __cplusplus
}
#endif

#endif
