/* Dave Watson				      email: watson@maths.uwa.edu.au
   Department of Mathematics				Tel: (61 9) 380 3359
   The University of Western Australia			FAX: (61 9) 380 1028
   Nedlands, WA 6009  Australia.		    PERTH, PEerless on eaRTH
 */
/* nnsort() finds the Delaunay triangulation of the two- or three-component vectors
 * in 'data_list' and returns a list of simplex vertices in 'vertices' with
 * the corresponding circumcentre and squared radius in the rows of 'circentres'.
 * nnsort() also can be used to find the ordered convex hull of the two- or three-
 * component vectors in 'data_list' and returns a list of (d-1)-facet vertices
 * in 'vertices' (dummy filename for 'circentres' must be used).
 * nsort() was written by Dave Watson and uses the algorithm described in -
 *	Watson, D.F., 1981, Computing the n-dimensional Delaunay tessellation
 *	with application to Voronoi polytopes: The Computer J., 24(2), p.
 *	167-172.
 *
 * additional information about this algorithm can be found in -
 *    CONTOURING: A guide to the analysis and display of spatial data,
 *    by David F. Watson, Pergamon Press, 1992, ISBN 0 08 040286 0
 */
/* Reformatted, changed to a procedural interface, and made into legal
 *  C++/ANSI C code by Jon Leech (leech@cs.unc.edu) 2/6/96.
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <malloc.h>
#include "nnsort.h"

#define SQ(x)	((x) * (x))
#define BIGNUM	1E37
#define EPSILON 0.00001
#define TSIZE	75	/* temporary storage size factor */
#define RANGE	10.0	/* factor (>=1) for radius of control points */
#define AND	&&
#define OR	||
#define EQ	==
#define NE	!=

static int *IntVect(int ncols)
{
    int *vectptr;
    if ((vectptr = (int *) malloc(ncols * sizeof(int))) EQ NULL) {
	fprintf(stderr, "\nnsort: Unable to allocate storage for ivector");
	return NULL;
    }
    return vectptr;
}

static void FreeVecti(int *vectptr)
{
    free(vectptr);
}

static int **IntMatrix(int nrows, int ncols)
{
    int i0;
    int **matptr;
    if (nrows < 2)
	nrows = 2;
    if (ncols < 2)
	ncols = 2;
    if ((matptr = (int **) malloc(nrows * sizeof(int *)))EQ NULL) {
	fprintf(stderr, "\nnsort: Unable to allocate storage for **imatrix");
	return NULL;
    }
    if ((matptr[0] = (int *) malloc(nrows * ncols * sizeof(int))) EQ NULL) {
	fprintf(stderr, "\nnsort: Unable to allocate storage for imatrix[]");
	free(matptr);
	return NULL;
    }
    for (i0 = 1; i0 < nrows; i0++)
	matptr[i0] = matptr[0] + i0 * ncols;
    return matptr;
}

static void FreeMatrixi(int **matptr)
{
    free(matptr[0]);
    free(matptr);
}

static double **DoubleMatrix(int nrows, int ncols)
{
    int i0;
    double **matptr;
    if (nrows < 2)
	nrows = 2;
    if (ncols < 2)
	ncols = 2;
    if ((matptr = (double **) malloc(nrows * sizeof(double *)))EQ NULL) {
	fprintf(stderr, "\nnsort: Unable to allocate storage for **dmatrix");
	return NULL;
    }
    if ((matptr[0] = (double *) malloc(nrows * ncols * sizeof(double))) EQ NULL) {
	fprintf(stderr, "\nnsort: Unable to allocate storage for dmatrix[]");
	free(matptr);
	return NULL;
    }
    for (i0 = 1; i0 < nrows; i0++)
	matptr[i0] = matptr[0] + i0 * ncols;
    return matptr;
}

static void FreeMatrixd(double **matptr)
{
    free(matptr[0]);
    free(matptr);
}

static void *nnalloc(int size) {
    void *ptr = malloc(size);

    if (ptr EQ NULL) {
	fprintf(stderr, "\nnnsort: Unable to allocate %d bytes in nnalloc()\n",
	    size);
	return NULL;
    }

    return ptr;
}

/* Returns number of facets or simplices created */
static int nnsort(
    double *pt,     /* list of 2/3 element vertices */
    int nrs,	    /* # of vertices */
    int dim,	    /* dimension of vertices, 2 or 3 */
    int chl,	    /* returns convex hull in 'hull' if nonzero, simplices otherwise */
    int **hull,     /* list of 2/3 element facet vertex indices */
    int **simplex,  /* list of 3/4 element simplex vertex indices */
    double **center,/* list of 2/3 element circumcentres */
    double **radius)/* list of radii squared */
{
    double xx, bgs, **mxy, **wrk, **pts, **ccr;
    int i0, i1, i2, i3, i4, i5, i6, i7, i8, i9, i11, ii[3], dm, dim1,
      nts, tsz, *id, **tmp, **a3s;
    int nobj;

    /* only 2D and 3D supported */
    if (dim < 2 OR dim > 3)
	return -1;
    mxy = DoubleMatrix(2, dim);
    for (i0 = 0; i0 < dim; i0++)
	mxy[0][i0] = -(mxy[1][i0] = BIGNUM);
    dim1 = dim + 1;
    wrk = DoubleMatrix(dim, dim1);
    for (i0 = 0; i0 < dim; i0++)
	for (i1 = 0; i1 < dim1; i1++)
	    wrk[i0][i1] = -RANGE;
    for (i0 = 0; i0 < dim; i0++)
	wrk[i0][i0] = RANGE * (3 * dim - 1);

    pts = DoubleMatrix(nrs + dim1, dim);
    for (i0 = 0; i0 < nrs; i0++) {
	pts[i0][0] = *pt++;
	pts[i0][1] = *pt++;
	if (dim > 2)
	    pts[i0][2] = *pt++;
	for (i1 = 0; i1 < dim; i1++) {
	    if (mxy[0][i1] < pts[i0][i1])
		mxy[0][i1] = pts[i0][i1];
	    if (mxy[1][i1] > pts[i0][i1])
		mxy[1][i1] = pts[i0][i1];
	}
    }

    for (bgs = 0, i0 = 0; i0 < dim; i0++) {
	mxy[0][i0] -= mxy[1][i0];
	if (bgs < mxy[0][i0])
	    bgs = mxy[0][i0];
    }
    bgs *= EPSILON;
    for (i0 = 0; i0 < nrs; i0++)
	for (i1 = 0; i1 < dim; i1++)
	    pts[i0][i1] += bgs * (0.5 - (double) rand() / 0x7fffffff);
    for (i0 = 0; i0 < dim1; i0++)
	for (i1 = 0; i1 < dim; i1++)
	    pts[nrs + i0][i1] = mxy[1][i1] + wrk[i1][i0] * mxy[0][i1];
    for (i1 = 1, i0 = 2; i0 < dim1; i0++)
	i1 *= i0;
    tsz = TSIZE * i1;
    tmp = IntMatrix(tsz + 1, dim);
    i1 *= (nrs + i1);
    id = IntVect(i1);
    for (i0 = 0; i0 < i1; i0++)
	id[i0] = i0;
    a3s = IntMatrix(i1, dim1);
    ccr = DoubleMatrix(i1, dim1);
    for (a3s[0][0] = nrs, i0 = 1; i0 < dim1; i0++)
	a3s[0][i0] = a3s[0][i0 - 1] + 1;
    for (ccr[0][dim] = BIGNUM, i0 = 0; i0 < dim; i0++)
	ccr[0][i0] = 0;
    nts = i4 = 1;
    dm = dim - 1;
    for (i0 = 0; i0 < nrs; i0++) {
	i1 = i7 = -1;
	i9 = 0;
	for (i11 = 0; i11 < nts; i11++) {
	    i1++;
	    while (a3s[i1][0] < 0)
		i1++;
	    xx = ccr[i1][dim];
	    for (i2 = 0; i2 < dim; i2++) {
		xx -= SQ(pts[i0][i2] - ccr[i1][i2]);
		if (xx < 0)
		    goto Corner3;
	    }
	    i9--;
	    i4--;
	    id[i4] = i1;
	    for (i2 = 0; i2 < dim1; i2++) {
		ii[0] = 0;
		if (ii[0] EQ i2)
		    ii[0]++;
		for (i3 = 1; i3 < dim; i3++) {
		    ii[i3] = ii[i3 - 1] + 1;
		    if (ii[i3] EQ i2)
			ii[i3]++;
		}
		if (i7 > dm) {
		    i8 = i7;
		    for (i3 = 0; i3 <= i8; i3++) {
			for (i5 = 0; i5 < dim; i5++)
			    if (a3s[i1][ii[i5]] NE tmp[i3][i5])
				goto Corner1;
			for (i6 = 0; i6 < dim; i6++)
			    tmp[i3][i6] = tmp[i8][i6];
			i7--;
			goto Corner2;
Corner1:;
		    }
		}
		if (++i7 > tsz) {
		    fprintf(stderr, "\nnsort: Temporary storage exceeded - increase TSIZE");
		    return -1;
		}
		for (i3 = 0; i3 < dim; i3++)
		    tmp[i7][i3] = a3s[i1][ii[i3]];
Corner2:;
	    }
	    a3s[i1][0] = -1;
Corner3:;
	}
	for (i1 = 0; i1 <= i7; i1++) {
	    if (!(chl AND tmp[i1][0] < nrs)) {
		for (i2 = 0; i2 < dim; i2++) {
		    for (wrk[i2][dim] = 0, i3 = 0; i3 < dim; i3++) {
			wrk[i2][i3] = pts[tmp[i1][i2]][i3] - pts[i0][i3];
			wrk[i2][dim] += wrk[i2][i3] * (pts[tmp[i1][i2]][i3] + pts[i0][i3]) / 2;
		    }
		}
		if (dim < 3) {
		    xx = wrk[0][0] * wrk[1][1] - wrk[1][0] * wrk[0][1];
		    ccr[id[i4]][0] = (wrk[0][2] * wrk[1][1] - wrk[1][2] * wrk[0][1]) / xx;
		    ccr[id[i4]][1] = (wrk[0][0] * wrk[1][2] - wrk[1][0] * wrk[0][2]) / xx;
		} else {
		    xx = (wrk[0][0] * (wrk[1][1] * wrk[2][2] - wrk[2][1] * wrk[1][2]))
			- (wrk[0][1] * (wrk[1][0] * wrk[2][2] - wrk[2][0] * wrk[1][2]))
			+ (wrk[0][2] * (wrk[1][0] * wrk[2][1] - wrk[2][0] * wrk[1][1]));
		    ccr[id[i4]][0] = ((wrk[0][3] * (wrk[1][1] * wrk[2][2] - wrk[2][1] * wrk[1][2]))
				      - (wrk[0][1] * (wrk[1][3] * wrk[2][2] - wrk[2][3] * wrk[1][2]))
				      + (wrk[0][2] * (wrk[1][3] * wrk[2][1] - wrk[2][3] * wrk[1][1]))) / xx;
		    ccr[id[i4]][1] = ((wrk[0][0] * (wrk[1][3] * wrk[2][2] - wrk[2][3] * wrk[1][2]))
				      - (wrk[0][3] * (wrk[1][0] * wrk[2][2] - wrk[2][0] * wrk[1][2]))
				      + (wrk[0][2] * (wrk[1][0] * wrk[2][3] - wrk[2][0] * wrk[1][3]))) / xx;
		    ccr[id[i4]][2] = ((wrk[0][0] * (wrk[1][1] * wrk[2][3] - wrk[2][1] * wrk[1][3]))
				      - (wrk[0][1] * (wrk[1][0] * wrk[2][3] - wrk[2][0] * wrk[1][3]))
				      + (wrk[0][3] * (wrk[1][0] * wrk[2][1] - wrk[2][0] * wrk[1][1]))) / xx;
		}
		for (ccr[id[i4]][dim] = 0, i2 = 0; i2 < dim; i2++) {
		    ccr[id[i4]][dim] += SQ(pts[i0][i2] - ccr[id[i4]][i2]);
		    a3s[id[i4]][i2] = tmp[i1][i2];
		}
		a3s[id[i4]][dim] = i0;
		i4++;
		i9++;
	    }
	}
	nts += i9;
    }

    if (chl) {
	/* Allocate & construct convex hull.
	 * At most 'nts' facets will be returned.
	 */
	int size = nts * sizeof(int) * ((dim < 3) ? 2 : 3);
	int *p = *hull = (int *)nnalloc(size);

	if (!hull)
	    return -1;

	i0 = -1;
	for (i11 = 0, nobj = 0; i11 < nts; i11++) {
	    i0++;
	    while (a3s[i0][0] < 0)
		i0++;
	    if (a3s[i0][1] < nrs) {
		/* Add a new facet */
		*p++ = a3s[i0][1];
		*p++ = a3s[i0][2];
		if (dim > 2)
		    *p++ = a3s[i0][3];
		nobj++;
	    }
	}
    } else {
	/* Allocate & construct Delaunay triangulation.
	 * At most 'nts' points will be returned.
	 */
	int ssize = nts * sizeof(int) * ((dim < 3) ? 3 : 4);
	int rsize = nts * sizeof(double);
	int csize = nts * sizeof(double) * ((dim < 3) ? 2 : 3);
	int *sptr = *simplex = (int *)nnalloc(ssize);
	double *rptr = *radius = (double *)nnalloc(rsize);
	double *cptr = *center = (double *)nnalloc(csize);

	if (!sptr || !rptr || !cptr)
	    return -1;

	i0 = -1;
	for (i11 = 0, nobj = 0; i11 < nts; i11++) {
	    i0++;
	    while (a3s[i0][0] < 0)
		i0++;
	    if (a3s[i0][0] < nrs) {
		for (i1 = 0; i1 < dim; i1++)
		    for (i2 = 0; i2 < dim; i2++)
			wrk[i1][i2] = pts[a3s[i0][i1]][i2] - pts[a3s[i0][dim]][i2];
		if (dim < 3) {
		    xx = wrk[0][0] * wrk[1][1] - wrk[0][1] * wrk[1][0];
		    if (fabs(xx) > EPSILON) {
			if (xx < 0) {
			    *sptr++ = a3s[i0][0];
			    *sptr++ = a3s[i0][2];
			    *sptr++ = a3s[i0][1];
			} else {
			    *sptr++ = a3s[i0][0];
			    *sptr++ = a3s[i0][1];
			    *sptr++ = a3s[i0][2];
			}

			*cptr++ = ccr[i0][0];
			*cptr++ = ccr[i0][1];

			*rptr++ = ccr[i0][2];
			nobj++;
		    }
		} else {
		    xx = ((wrk[0][0] * (wrk[1][1] * wrk[2][2] - wrk[2][1] * wrk[1][2]))
			  - (wrk[0][1] * (wrk[1][0] * wrk[2][2] - wrk[2][0] * wrk[1][2]))
			  + (wrk[0][2] * (wrk[1][0] * wrk[2][1] - wrk[2][0] * wrk[1][1])));
		    if (fabs(xx) > EPSILON) {
			if (xx < 0) {
			    *sptr++ = a3s[i0][0];
			    *sptr++ = a3s[i0][1];
			    *sptr++ = a3s[i0][3];
			    *sptr++ = a3s[i0][2];
			} else {
			    *sptr++ = a3s[i0][0];
			    *sptr++ = a3s[i0][1];
			    *sptr++ = a3s[i0][2];
			    *sptr++ = a3s[i0][3];
			}
			*cptr++ = ccr[i0][0];
			*cptr++ = ccr[i0][1];
			*cptr++ = ccr[i0][2];

			*rptr++ = ccr[i0][3];
			nobj++;
		    }
		}
	    }
	}
    }
    FreeMatrixd(pts);
    FreeMatrixi(a3s);
    FreeMatrixd(ccr);
    FreeMatrixi(tmp);
    FreeVecti(id);

    return nobj;
}

#ifndef TRUE
#define TRUE 1
#endif

int nnhull3(
    double (*pt)[3],
    int npt,
    int (**facets)[3])
{
    return nnsort((double *)pt, npt, 3, TRUE, (int **)facets, NULL, NULL, NULL);
}

int nnsimplex3(
    double (*pt)[3],
    int npt,
    int (**simplex)[4],
    double (**center)[3],
    double **radius)
{
    return nnsort((double *)pt, npt, 3, TRUE, NULL,
	    (int **)simplex, (double **)center, radius);
}

int nnhull2(
    double (*pt)[2],
    int npt,
    int (**facets)[2])
{
    return nnsort((double *)pt, npt, 2, TRUE, (int **)facets, NULL, NULL, NULL);
}

int nnsimplex2(
    double (*pt)[2],
    int npt,
    int (**simplex)[3],
    double (**center)[2],
    double **radius)
{
    return nnsort((double *)pt, npt, 2, TRUE, NULL,
	    (int **)simplex, (double **)center, radius);
}
