/***************************************************************************
 *cr
 *cr            (C) Copyright 1995-2007 The Board of Trustees of the
 *cr                        University of Illinois
 *cr                         All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: vmdconsole.c,v $
 *      $Author: johns $        $Locker:  $             $State: Exp $
 *      $Revision: 1.3 $       $Date: 2007/01/12 20:08:38 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * vmd console redirector
 * (c) 2006 Axel Kohlmeyer <akohlmey@cmm.chem.upenn.edu>
 * 
 ***************************************************************************/

#if defined(VMDTKCON)

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>

#include "vmdconsole.h"

/* XXX: FIXME, this code is not at all thread safe but needs to be.  */

/* structure for linked list of pending messages. */
struct vmdcon_msg 
{
    char *txt;
    struct vmdcon_msg *next;
};
static struct vmdcon_msg *vmdcon_pending=NULL;
static struct vmdcon_msg *vmdcon_lastmsg=NULL;
static const size_t vmdcon_bufsz=4092;

/* path to console text widget */
static char *vmdcon_wpath=NULL;

/* current insertion mark */
static char *vmdcon_mark=NULL;

/* opaque pointer to the current tcl interpreter */
static void *vmdcon_interp=NULL;

/* select text output */
static int vmdcon_stdout=0;

/* turn on text mode processing */
void vmdcon_use_text(void) 
{
    vmdcon_stdout=1;
}

/* register a tk/tcl widget to be the console window.
 * we get the widget path directly from tcl, 
 * so we have to create a copy (and free it later).
 * we also need a pointer to the tcl interpreter.
 */
int vmdcon_register(const char *w_path, const char *mark, void *interp)
{
    vmdcon_interp=interp;
    if(w_path == NULL) {
        if (vmdcon_wpath != NULL) {
            free(vmdcon_wpath);
            free(vmdcon_mark);
        }
        vmdcon_wpath=NULL;
        vmdcon_mark=NULL;
    } else {
        int len;
        
        len=strlen(w_path);
        vmdcon_wpath=(char*)malloc(len+1);
        strcpy(vmdcon_wpath, w_path);
        len=strlen(mark);
        vmdcon_mark=(char*)malloc(len+1);
        strcpy(vmdcon_mark, mark);

        /* flush console log queue */
        vmdcon_purge();
    }
    return 0;
}

/* append text to console log queue.
 * we have to make copies as we might get handed 
 * a tcl object or a pointer to some larger buffer. */
int vmdcon_append(const char *txt, int len) 
{
    struct vmdcon_msg *msg;
    char *buf;

    /* len=0, i.e. don't print. */
    if (len == 0 ) return 0;

    /* create copy of message. it gets free'd after it has been 'printed'. */
    if (len < 0) {
        len=strlen(txt);
    }
    buf=(char *)calloc(len+1,1);
    strncpy(buf,txt,len);

    /* append to message queue. */
    msg=(struct vmdcon_msg *)malloc(sizeof(struct vmdcon_msg));
    msg->txt=buf;
    msg->next=NULL;
        
    if (vmdcon_pending == NULL) {
        vmdcon_pending=msg;
        vmdcon_lastmsg=msg;
    } else {
        vmdcon_lastmsg->next=msg;
        vmdcon_lastmsg=msg;
    }

    return 0;
}

/* flush current message queue to a registered 
 * console widget, if such a thing exists.
 * since vmdcon_append() allocates the storage,
 * for everything, we have to free the msg structs
 * and the strings. */
int vmdcon_purge(void) 
{
    struct vmdcon_msg *msg;

    /* purge message queue only if we have a working(?) console window */
    if ((vmdcon_wpath != NULL) || (vmdcon_stdout)) {
        while (vmdcon_pending != NULL) {
            msg=vmdcon_pending;
            if (vmdcon_stdout) {
                fputs(msg->txt,stdout);
            } else {
                tcl_vmdcon_insert(vmdcon_interp, vmdcon_wpath, vmdcon_mark, msg->txt);
            }
            free(msg->txt);
            vmdcon_pending=msg->next;
            free(msg);
        }

        if (vmdcon_stdout) 
            fflush(stdout);

    }
    return 0;
}

/* emulate printf. unfortunately, we cannot rely on 
 * snprintf being available, so we have to write to
 * a very large buffer and then free it. :-( */
int vmdprintf(const char *fmt, ...) 
{
    va_list ap;
    char *buf;
    int len;
    
    /* expand formated output into a single string */
    buf = (char *)malloc(vmdcon_bufsz);
    va_start(ap, fmt);
    len = vsprintf(buf, fmt, ap);

    /* check result */
    if (len >= vmdcon_bufsz) {
        fprintf(stderr,"WARNING! buffer overflow in vmdprintf. %d vs %d.\n",
                len, vmdcon_bufsz);
        free(buf);
        errno=ERANGE;
        return -1;
    }

    vmdcon_append(buf, len);
    vmdcon_purge();

    free(buf);
    return 0;    
}

#endif
