/***************************************************************************
 *cr
 *cr            (C) Copyright 1995-2007 The Board of Trustees of the
 *cr                        University of Illinois
 *cr                         All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: py_numeric.C,v $
 *      $Author: johns $        $Locker:  $             $State: Exp $
 *      $Revision: 1.13 $       $Date: 2007/01/12 20:08:37 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *  Python interface to the Python Numeric module.
 ***************************************************************************/

#include "py_commands.h"
#if defined(__APPLE__)
// use the Apple-provided Python framework
#include "Python/Numeric/arrayobject.h"
#else
#include "Numeric/arrayobject.h"
#endif

#include "AtomSel.h"
#include "VMDApp.h"
#include "MoleculeList.h"

// make python happier
extern "C" {

// Return a direct reference to a timestep.  The atom coordinates are
// NOT copied; thus any modification done to this array will affect
// the coordinates, and if the object is used after the timestep has been
// deleted, expect to crash and burn.  The idea here is absolute efficiency;
// for less efficiency but more safety, use the atomselect get and set methods.
static PyObject *timestep(PyObject *self, PyObject *args) {
  int molid = 0;
  int frame = 0;
  if (!PyArg_ParseTuple(args, (char *)"ii", &molid, &frame))
    return NULL;
  VMDApp *app = get_vmdapp();
  DrawMolecule *mol = app->moleculeList->mol_from_id(molid);
  if (!mol) {
    PyErr_SetString(PyExc_ValueError, "invalid molid");
    return NULL;
  }
  Timestep *ts = mol->get_frame(frame);
  if (!ts) {
    PyErr_SetString(PyExc_ValueError, "invalid frame");
    return NULL;
  }
  int dims[1]; dims[0] = mol->nAtoms*3;
  PyArrayObject *result = (PyArrayObject *)PyArray_FromDimsAndData(
      1, dims, PyArray_FLOAT, (char *)ts->pos);
  return PyArray_Return(result);
}
 
// Return an array of ints representing flags for on/off atoms in an atom
// selection.
// atomselect(molid, frame, selection) -> array
static PyObject *atomselect(PyObject *self, PyObject *args) {
  int molid = 0;
  int frame = 0;
  char *sel = 0;
  if (!PyArg_ParseTuple(args, (char *)"iis", &molid, &frame, &sel))
    return NULL;
  VMDApp *app = get_vmdapp();
  DrawMolecule *mol = app->moleculeList->mol_from_id(molid);
  if (!mol) {
    PyErr_SetString(PyExc_ValueError, "invalid molid");
    return NULL;
  }

  AtomSel *atomSel = new AtomSel(app->atomSelParser, mol->id());
  atomSel->which_frame = frame;
  if (atomSel->change(sel, mol) == AtomSel::NO_PARSE) {
    PyErr_SetString(PyExc_ValueError, "cannot parse atom selection text");
    delete atomSel;
    return NULL;
  }
  int dims[1]; dims[0] = mol->nAtoms;
  PyArrayObject *result = (PyArrayObject *)PyArray_FromDims(
      1, dims, PyArray_INT);
  memcpy(result->data, atomSel->on, mol->nAtoms*sizeof(int));
  delete atomSel;
  return PyArray_Return(result);
}


// Simple test function to return a NumPy array
static PyObject *foo(PyObject *self, PyObject *args) {
  int dims[1]; dims[0] = 10;
  PyArrayObject *arr = (PyArrayObject *)PyArray_FromDims(
      1,                // number of dimensions
      dims,             // size of each dimension
      PyArray_FLOAT     // Data type
      );
  float *data = (float *)arr->data;  // arr->data is of type char *.
  for (int i=0; i<10; i++) data[i] = (float)i;
  return (PyObject *)arr;
  //return PyArray_Return(arr);    // casts to scalar if array has zero dims.
                                   // also checks for errors
}

// end of extern "C" block
}

static PyMethodDef Methods[] = {
  {(char *)"timestep", timestep, METH_VARARGS, (char *)"Returns zero-copy timestep reference"},
  {(char *)"atomselect", atomselect, METH_VARARGS, (char *)"Create atom selection flags"},
  {(char *)"foo", foo,METH_VARARGS, (char *)"Returns array of 10 floats"},
  {NULL, NULL}
};

void initvmdnumpy() {
  PyObject *mod=NULL, *obj=NULL;

  // check that version 22 was found.  It would be nice if Numeric would
  // define a version number in their header file so that we could compare
  // against the version we compiled against rather than hardcode it
  // ourselves, but they don't.  This method to determine the version
  // number is taken from Numeric.py.
#define VMD_NUMPY_VERSION "22.0"
  if (!(mod = PyImport_ImportModule((char *)"numeric_version")) ||
      !(obj = PyObject_GetAttrString(mod, (char *)"version")) ||
       (strcmp(PyString_AsString(obj), VMD_NUMPY_VERSION) < 0)) {
    Py_XDECREF(obj);  // obj may be NULL
    fprintf(stderr, "Could not find Numeric Python version >= %s\n", VMD_NUMPY_VERSION);
    return;
  }
  Py_XDECREF(obj);
  obj=NULL;

  // The _numpy import code is taken from the import_array() macro.  I need to
  // code it myself because I can't initialize the vmdnumpy module unless 
  // the _numpy module is also loaded; otherwise VMD will crash on the first
  // call to a NumPy function.
  if (!(mod = PyImport_ImportModule((char *) "_numpy")) ||
      !(obj = PyObject_GetAttrString(mod, (char *) "_ARRAY_API")) ||
      !(PyCObject_Check(obj))) {
    Py_XDECREF(obj);
    fprintf(stderr, "Numeric Python module _numpy not found; vmdnumpy module not available.\n");
    return;
  }
  PyArray_API = (void **)PyCObject_AsVoidPtr(obj);
  Py_InitModule((char *)"vmdnumpy", Methods);
  Py_XDECREF(obj);
}

