/***************************************************************************
 *cr
 *cr            (C) Copyright 1995-2007 The Board of Trustees of the
 *cr                        University of Illinois
 *cr                         All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: py_molecule.C,v $
 *      $Author: johns $        $Locker:  $             $State: Exp $
 *      $Revision: 1.57 $       $Date: 2007/01/12 20:08:37 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *  Python molecule manipulation interface
 ***************************************************************************/

#include <ctype.h>
#include <stdlib.h>

#include "config.h"
#include "utilities.h"
#include "VMDApp.h"
#include "JString.h"
#include "Molecule.h"
#include "MoleculeList.h"

#include "py_commands.h"

// num() : number of loaded molecules
static PyObject *mol_num(PyObject *self, PyObject *args) {
  if (!PyArg_ParseTuple(args, (char *)":molecule.num"))
    return NULL;

  return PyInt_FromLong(get_vmdapp()->num_molecules());
}

// listall() : return list of all valid molid's
static PyObject *mol_listall(PyObject *self, PyObject *args) {
  if (!PyArg_ParseTuple(args, (char *)":molecule.listall"))
    return NULL;

  VMDApp *app = get_vmdapp();
  int num = app->num_molecules();
  PyObject *newlist = PyList_New(num);
  for (int i=0; i<num; i++)
    PyList_SET_ITEM(newlist, i, PyInt_FromLong(app->molecule_id(i)));

  return newlist;
}

// exists(molid): return true if molid is valid.
static PyObject *mol_exists(PyObject *self, PyObject *args) {
  int molid;
  if (!PyArg_ParseTuple(args, (char *)"i:molecule.exists", &molid))
    return NULL;
  VMDApp *app = get_vmdapp();
  return PyInt_FromLong(app->molecule_valid_id(molid));
}

// name(molid)
static PyObject *mol_name(PyObject *self, PyObject *args) {
  int molid;

  if (!PyArg_ParseTuple(args, (char *)"i:molecule.name", &molid))
    return NULL;

  VMDApp *app = get_vmdapp();
  const char *name = app->molecule_name(molid);
  if (!name) {
    PyErr_SetString(PyExc_ValueError, (char *)"Invalid molecule id");
    return NULL;
  }
  return PyString_FromString((char *)name);
}

// numatoms(molid)
static PyObject *mol_numatoms(PyObject *self, PyObject *args) {
  int molid;

  if (!PyArg_ParseTuple(args, (char *)"i:molecule.numatoms", &molid))
    return NULL;

  VMDApp *app = get_vmdapp();
  if (!app->molecule_valid_id(molid)) {
    PyErr_SetString(PyExc_ValueError, (char *)"Invalid molecule id");
    return NULL;
  }
  return PyInt_FromLong(app->molecule_numatoms(molid));
}

// new(name) -- create new molecule.  Returns id.
static PyObject *mol_new(PyObject *self, PyObject *args) {
  char *name;
  if (!PyArg_ParseTuple(args, (char *)"s:molecule.new", &name))
    return NULL;
  VMDApp *app = get_vmdapp();
  int molid = app->molecule_new(name);
  if (molid < 0) {
    PyErr_SetString(PyExc_ValueError, (char *)"Unable to create molecule.");
    return NULL;
  }
  return PyInt_FromLong(molid);
}


static PyObject *mol_load(PyObject *self, PyObject *args, PyObject *keywds) {
 
  char *structure = NULL, *coor = NULL, *sfname = NULL, *cfname = NULL;
 
  static char *kwlist[] = {
    (char *)"structure", (char *)"sfname", (char *)"coor", (char *)"cfname", 
    NULL
  };

  if (!PyArg_ParseTupleAndKeywords(args, keywds, (char *)"ss|ss:molecule.load", kwlist, 
        &structure, &sfname, &coor, &cfname))
    return NULL;

  // must specify structure and a structure file
  if (!structure) {
    PyErr_SetString(PyExc_ValueError, (char *)"No structure type specified");
    return NULL;
  }
  if (!sfname) {
    PyErr_SetString(PyExc_ValueError, (char *)"No structure file specified");
    return NULL;
  }

  // if a coordinates file type was specified, a coordinate file must be given
  // as well, and vice versa.
  if (coor && !cfname) {
    PyErr_SetString(PyExc_ValueError, (char *)"No coordinate file specified");
    return NULL;
  }
  if (cfname && !coor) {
    PyErr_SetString(PyExc_ValueError, (char *)"No coordinate type specified");
    return NULL;
  }

  // Get the VMDApp instance from the VMDApp module
  VMDApp *app = get_vmdapp();

  // Special-case graphics molecules to load as "blank" molecules
  if (!strcmp(structure, "graphics")) {
    return PyInt_FromLong(app->molecule_new(sfname));
  }
  FileSpec spec;
  int molid = app->molecule_load(-1, sfname, structure, &spec);
  if (molid < 0) {
    PyErr_SetString(PyExc_ValueError, (char *)"Unable to load structure file");
    return NULL;
  }
  if (cfname) {
    app->molecule_load(molid, cfname, coor, &spec);
  } 
  return PyInt_FromLong(molid); 
}

static PyObject *mol_cancel(PyObject *self, PyObject *args) {
  // get one integer argument
  int molid;
  if (!PyArg_ParseTuple(args, (char *)"i:molecule.cancel", &molid))
    return NULL;

  VMDApp *app = get_vmdapp();
  if (!app->molecule_valid_id(molid)) {
    PyErr_SetString(PyExc_ValueError, (char *)"Invalid molecule id");
    return NULL;
  } 
  app->molecule_cancel_io(molid);

  Py_INCREF(Py_None);
  return Py_None;
}

// delete(molid)
static PyObject *mol_delete(PyObject *self, PyObject *args) {
  int molid;
  if (!PyArg_ParseTuple(args, (char *)"i:molecule.delete", &molid))
    return NULL;

  VMDApp *app = get_vmdapp();
  if (!app->molecule_valid_id(molid)) {
    PyErr_SetString(PyExc_ValueError, (char *)"Invalid molecule id");
    return NULL;
  } 
  app->molecule_delete(molid);
  
  Py_INCREF(Py_None);
  return Py_None;
}

// get_top(molid)
static PyObject *get_top(PyObject *self, PyObject *args) {
  if (!PyArg_ParseTuple(args, (char *)":molecule.get_top"))
    return NULL;

  return PyInt_FromLong(get_vmdapp()->molecule_top());
}

// set_top(molid)
static PyObject *set_top(PyObject *self, PyObject *args) {
  int molid;
  if (!PyArg_ParseTuple(args, (char *)"i:molecule.set_top", &molid))
    return NULL;

  VMDApp *app = get_vmdapp();
  if (!app->molecule_valid_id(molid)) {
    PyErr_SetString(PyExc_ValueError, (char *)"Invalid molecule id");
    return NULL;
  } 
  app->molecule_make_top(molid);
  
  Py_INCREF(Py_None);
  return Py_None;
}


static PyObject *readorwrite(PyObject *self, PyObject *args, PyObject *keywds,
                            int do_read) {

  int molid = -1;
  char *filename = NULL, *type = NULL;
  int beg=0, end=-1, stride=1, waitfor = 1;
  PyObject *volsets = NULL, *selobj = NULL;
  int *on = NULL;

  static char *kwlist[] = {
    (char *)"molid", (char *)"type", (char *)"filename", (char *)"beg", 
    (char *)"end", (char *)"skip", (char *)"waitfor", (char *)"volsets", 
    (char *)"selection", NULL
  };

  if (!PyArg_ParseTupleAndKeywords(args, keywds, (char *)"iss|iiiiO!O:read/write", kwlist,
    &molid, &type, &filename, &beg, &end, &stride, &waitfor, &PyList_Type, 
    &volsets, &selobj))
    return NULL;

  VMDApp *app = get_vmdapp();
 
  int numframes = 0;
  if (do_read) { 
    FileSpec spec;
    spec.first = beg;
    spec.last = end;
    spec.stride = stride;
    spec.waitfor = waitfor;
    if (volsets) {
      spec.nvolsets = PyList_Size(volsets);
      spec.setids = new int[spec.nvolsets];
      for (int i=0; i<spec.nvolsets; i++) 
        spec.setids[i] = PyInt_AsLong(PyList_GET_ITEM(volsets, i));
      if (PyErr_Occurred()) return NULL;
    } else {
      // Have a default of {0} for setids so that it isn't always necessary
      // to specify the set id's.  This should be ignored if the file type 
      // can't read volumetric datasets
      spec.nvolsets = 1;
      spec.setids = new int[1];
      spec.setids[0] = 0;
    }
    return PyInt_FromLong(app->molecule_load(molid, filename, type, &spec));
  }
  if (!app->molecule_valid_id(molid)) {
    PyErr_SetString(PyExc_ValueError, (char *)"Invalid molecule id");
    return NULL;
  }
  int natoms = app->molecule_numatoms(molid);
  if (selobj && selobj != Py_None) {
    if (!PyTuple_Check(selobj)) {
      PyErr_SetString(PyExc_ValueError, (char *)"selection argument must be tuple");
      return NULL;
    }
    on = new int[natoms];
    memset(on, 0, natoms*sizeof(int));
    for (int i=0; i<PyTuple_Size(selobj); i++) {
      int ind = PyInt_AsLong(PyTuple_GET_ITEM(selobj,i));
      if (ind < 0 || ind >= natoms) {
        fprintf(stderr, 
                "molecule.animate: Skipping invalid id (%d) in selection\n",
                ind);
      } else {
        on[ind] = 1;
      }
    }
  }
  FileSpec spec;
  spec.first = beg;
  spec.last = end;
  spec.stride = stride;
  spec.waitfor = waitfor;
  spec.selection = on;
  numframes = app->molecule_savetrajectory(molid, filename, type, &spec);
  delete [] on;
  if (numframes < 0) {
    PyErr_SetString(PyExc_ValueError, (char *)"Unable to save file");
    return NULL;
  }
  return PyInt_FromLong(numframes);
}

// read(molid, filename, type, beg=-1, end=-1, skip=-1)
static PyObject *mol_read(PyObject *self, PyObject *args, PyObject *keywds) {
  return readorwrite(self, args, keywds, 1);
}

// write(molid, filename, type, beg=-1, end=-1, skip=-1)
static PyObject *mol_write(PyObject *self, PyObject *args, PyObject *keywds) {
  return readorwrite(self, args, keywds, 0);
}

// numframes(molid)
static PyObject *numframes(PyObject *self, PyObject *args) {
  int molid;

  if (!PyArg_ParseTuple(args, (char *)"i:molecule.numframes", &molid))
    return NULL;

  VMDApp *app = get_vmdapp();
  if (!app->molecule_valid_id(molid)) {
    PyErr_SetString(PyExc_ValueError, (char *)"Invalid molecule id");
    return NULL;
  }
  return PyInt_FromLong(app->molecule_numframes(molid));
}

// get_frame(molid)
static PyObject *get_frame(PyObject *self, PyObject *args) {
  int molid;

  if (!PyArg_ParseTuple(args, (char *)"i:molecule.get_frame", &molid))
    return NULL;

  VMDApp *app = get_vmdapp();
  if (!app->molecule_valid_id(molid)) {
    PyErr_SetString(PyExc_ValueError, (char *)"Invalid molecule id");
    return NULL;
  }
  return PyInt_FromLong(app->molecule_frame(molid));
}

// set_frame(molid, frame)
static PyObject *set_frame(PyObject *self, PyObject *args) {
  int molid, frame;

  if (!PyArg_ParseTuple(args, (char *)"ii:molecule.set_frame", &molid, &frame))
    return NULL;

  VMDApp *app = get_vmdapp();
  Molecule *mol = app->moleculeList->mol_from_id(molid);
  if (!mol) {
    PyErr_SetString(PyExc_ValueError, (char *)"Invalid molecule id");
    return NULL;
  }
  mol->anim()->override_current_frame(frame);
  mol->change_ts();
  Py_INCREF(Py_None);
  return Py_None;
}

// delframe(molid, beg=0, end=-1, skip=0)
static PyObject *delframe(PyObject *self, PyObject *args, PyObject *keywds) {

  int molid = 0, beg=0, end=-1, skip=0;

  static char *kwlist[] = {
    (char *)"molid", (char *)"beg", (char *)"end", (char *)"skip", NULL
  };

  if (!PyArg_ParseTupleAndKeywords(args, keywds, (char *)"i|iii:molecule.delframe", kwlist,
    &molid, &beg, &end, &skip))
    return NULL;

  VMDApp *app = get_vmdapp();
  if (!app->molecule_valid_id(molid)) {
    PyErr_SetString(PyExc_ValueError, (char *)"Invalid molid");
    return NULL;
  }

  if (!app->molecule_deleteframes(molid, beg, end, skip)) {
    PyErr_SetString(PyExc_ValueError, (char *)"Unable to delete frames");
    return NULL;
  }
 
  Py_INCREF(Py_None);
  return Py_None;
}

// dupframe(molid, frame)
static PyObject *dupframe(PyObject *self, PyObject *args) {

  int molid, frame;
  if (!PyArg_ParseTuple(args, (char *)"ii:molecule.dupframe", &molid, &frame))
    return NULL;

  VMDApp *app = get_vmdapp();
  if (!app->molecule_valid_id(molid)) {
    PyErr_SetString(PyExc_ValueError, (char *)"Invalid molid");
    return NULL;
  }

  if (!app->molecule_dupframe(molid, frame)) {
    PyErr_SetString(PyExc_ValueError, (char *)"Unable to duplicate frame");
    return NULL;
  }
  Py_INCREF(Py_None);
  return Py_None;
}

// ssrecalc(molid)
static PyObject *mol_ssrecalc(PyObject *self, PyObject *args) {
  int molid;
  if (!PyArg_ParseTuple(args, (char *)"i:molecule.ssrecalc", &molid))
    return NULL;

  VMDApp *app = get_vmdapp();
  if (!app->molecule_valid_id(molid)) {
    PyErr_SetString(PyExc_ValueError, (char *)"Invalid molecule id");
    return NULL;
  } 
  return PyInt_FromLong(app->molecule_ssrecalc(molid));
}

// rename(molid, newname)
static PyObject *mol_rename(PyObject *self, PyObject *args) {
  int molid;
  char *newname;
  if (!PyArg_ParseTuple(args, (char *)"is:molecule.rename", &molid, &newname))
    return NULL;

  VMDApp *app = get_vmdapp();
  if (!app->molecule_valid_id(molid)) {
    PyErr_SetString(PyExc_ValueError, (char *)"Invalid molecule id");
    return NULL;
  } 
  if (!app->molecule_rename(molid, newname)) {
    PyErr_SetString(PyExc_ValueError, (char *)"Unable to rename molecule.");
    return NULL;
  }
  Py_INCREF(Py_None);
  return Py_None;
}
   
// add_volumetric(molid, name, origin, xaxis, yaxis, zaxis, xsize, ysize, 
//                zsize, data)
// Keyword arguments are availableas listed here.  Data is a list.  
// name is a string name for the data
static PyObject *mol_add_volumetric(PyObject *self, PyObject *args, PyObject *keywds) {

  int molid = -1;
  int xsize = -1, ysize = -1, zsize = -1;
  int size;
  char *name;
  PyObject *data = NULL, *origin = NULL, *xaxis = NULL, *yaxis = NULL, 
           *zaxis = NULL;
  float forigin[3], fxaxis[3], fyaxis[3], fzaxis[3];

  static char *kwlist[] = {
    (char *)"molid", (char *)"name", (char *)"origin",  (char *)"xaxis",
    (char *)"yaxis",  (char *)"zaxis",  (char *)"xsize", (char *)"ysize", 
    (char *)"zsize", (char *)"data", NULL
  };

  if (!PyArg_ParseTupleAndKeywords(args, keywds, 
       (char *)"isOOOOiiiO:molecule.add_volumetric", kwlist,
       &molid, &name, &origin, &xaxis, &yaxis, &zaxis, &xsize, 
       &ysize, &zsize, &data))
    return NULL;

  VMDApp *app = get_vmdapp();
  if (!app->molecule_valid_id(molid)) {
    PyErr_SetString(PyExc_ValueError, (char *)"Invalid molecule id");
    return NULL;
  }
  if (xsize < 1 || ysize < 1 || zsize < 1) {
    PyErr_SetString(PyExc_ValueError, (char *)"Invalid size parameters");
    return NULL;
  }
  size = xsize * ysize * zsize;
  if (!PyList_Check(data)) {
    PyErr_SetString(PyExc_ValueError, (char *)"data must be a list");
    return NULL;
  }
  if (PyList_Size(data) != size) {
    PyErr_SetString(PyExc_ValueError, (char *)"size of list does not match specified sizes");
    return NULL;
  }
  // Check that any optional arguments are valid and set defaults.
  if (origin) {
    if (!py_array_from_obj(origin, forigin)) return NULL;
  } else {
    forigin[0] = forigin[1] = forigin[2] = 0;
  }
  if (xaxis) {
    if (!py_array_from_obj(xaxis, fxaxis)) return NULL;
  } else {
    fxaxis[0] = 1.0; fxaxis[1] = fxaxis[2] = 0;
  }
  if (yaxis) {
    if (!py_array_from_obj(yaxis, fyaxis)) return NULL;
  } else {
    fyaxis[1] = 1.0; fyaxis[0] = fyaxis[2] = 0;
  }
  if (zaxis) {
    if (!py_array_from_obj(zaxis, fzaxis)) return NULL;
  } else {
    fzaxis[2] = 1.0; fzaxis[0] = fzaxis[1] = 0;
  }

  // allocate float array here; pass it to molecule
  float *fdata = new float[size];
  for (int i=0; i<size; i++) {
    PyObject *elem = PyList_GET_ITEM(data, i);
    float tmp = PyFloat_AsDouble(elem);
    if (PyErr_Occurred()) {
      delete [] fdata;
      return NULL;
    }
    fdata[i] = tmp;
  }
  if (!app->molecule_add_volumetric(molid, name, forigin, fxaxis, fyaxis,
       fzaxis, xsize, ysize, zsize, fdata)) {
    PyErr_SetString(PyExc_ValueError, (char *)"Unable to add volumetric data");
    delete [] fdata;
    return NULL;
  }
  Py_INCREF(Py_None);
  return Py_None;
}
   
// get_filenames(molid): return list of files loaded in the molecule
// get_filetypes(molid): returns list of corresponding file types.
// get_databases(molid): returns list of databases of origin 
// get_accessions(molid): returns list of database accession codes
// get_remarks(molid): returns list of per-file remarks/comments
static PyObject *get_filenames(PyObject *self, PyObject *args) {
  int molid;
  if (!PyArg_ParseTuple(args, (char *)"i:molecule.get_filenames", &molid))
    return NULL;

  VMDApp *app = get_vmdapp();
  Molecule *mol = app->moleculeList->mol_from_id(molid);
  if (!mol) {
    PyErr_SetString(PyExc_ValueError, (char *)"Invalid molecule id");
    return NULL;
  }
  int num = mol->num_files();
  PyObject *result = PyList_New(num);
  for (int i=0; i<mol->num_files(); i++) {
    PyList_SET_ITEM(result, i, PyString_FromString(mol->get_file(i)));
  }
  return result;
}
static PyObject *get_filetypes(PyObject *self, PyObject *args) {
  int molid;
  if (!PyArg_ParseTuple(args, (char *)"i:molecule.get_filetypes", &molid))
    return NULL;

  VMDApp *app = get_vmdapp();
  Molecule *mol = app->moleculeList->mol_from_id(molid);
  if (!mol) {
    PyErr_SetString(PyExc_ValueError, (char *)"Invalid molecule id");
    return NULL;
  }
  int num = mol->num_files();
  PyObject *result = PyList_New(num);
  for (int i=0; i<mol->num_files(); i++) {
    PyList_SET_ITEM(result, i, PyString_FromString(mol->get_type(i)));
  }
  return result;
}
static PyObject *get_databases(PyObject *self, PyObject *args) {
  int molid;
  if (!PyArg_ParseTuple(args, (char *)"i:molecule.get_databases", &molid))
    return NULL;

  VMDApp *app = get_vmdapp();
  Molecule *mol = app->moleculeList->mol_from_id(molid);
  if (!mol) {
    PyErr_SetString(PyExc_ValueError, (char *)"Invalid molecule id");
    return NULL;
  }
  int num = mol->num_files();
  PyObject *result = PyList_New(num);
  for (int i=0; i<mol->num_files(); i++) {
    PyList_SET_ITEM(result, i, PyString_FromString(mol->get_database(i)));
  }
  return result;
}
static PyObject *get_accessions(PyObject *self, PyObject *args) {
  int molid;
  if (!PyArg_ParseTuple(args, (char *)"i:molecule.get_accessions", &molid))
    return NULL;

  VMDApp *app = get_vmdapp();
  Molecule *mol = app->moleculeList->mol_from_id(molid);
  if (!mol) {
    PyErr_SetString(PyExc_ValueError, (char *)"Invalid molecule id");
    return NULL;
  }
  int num = mol->num_files();
  PyObject *result = PyList_New(num);
  for (int i=0; i<mol->num_files(); i++) {
    PyList_SET_ITEM(result, i, PyString_FromString(mol->get_accession(i)));
  }
  return result;
}
static PyObject *get_remarks(PyObject *self, PyObject *args) {
  int molid;
  if (!PyArg_ParseTuple(args, (char *)"i:molecule.get_remarks", &molid))
    return NULL;

  VMDApp *app = get_vmdapp();
  Molecule *mol = app->moleculeList->mol_from_id(molid);
  if (!mol) {
    PyErr_SetString(PyExc_ValueError, (char *)"Invalid molecule id");
    return NULL;
  }
  int num = mol->num_files();
  PyObject *result = PyList_New(num);
  for (int i=0; i<mol->num_files(); i++) {
    PyList_SET_ITEM(result, i, PyString_FromString(mol->get_remarks(i)));
  }
  return result;
}

// get_periodic(molid, frame=-1): return dict: a, b, c, alpha, beta, gamma.
static PyObject *get_periodic(PyObject *self, PyObject *args, PyObject *kwds) {
  int molid;
  int frame=-1;
  static char *kwlist[] = { (char *)"molid", (char *)"frame", NULL };
  if (!PyArg_ParseTupleAndKeywords(args, kwds, (char *)"i|i:molecule.get_periodic", kwlist, &molid, &frame))
    return NULL;

  VMDApp *app = get_vmdapp();
  Molecule *mol = app->moleculeList->mol_from_id(molid);
  if (!mol) {
    PyErr_SetString(PyExc_ValueError, (char *)"Invalid molecule id");
    return NULL;
  }
  Timestep *ts = NULL;
  if (frame == -1) {
    ts = mol->current();
  } else if (frame == -2) {
    ts = mol->get_last_frame();
  } else {
    ts = mol->get_frame(frame);
  }
  if (!ts) {
    PyErr_SetString(PyExc_ValueError, (char *)"Invalid frame");
    return NULL;
  }
  PyObject *dict = PyDict_New();
  PyDict_SetItemString(dict, (char *)"a", PyFloat_FromDouble(ts->a_length));
  PyDict_SetItemString(dict, (char *)"b", PyFloat_FromDouble(ts->b_length));
  PyDict_SetItemString(dict, (char *)"c", PyFloat_FromDouble(ts->c_length));
  PyDict_SetItemString(dict, (char *)"alpha", PyFloat_FromDouble(ts->alpha));
  PyDict_SetItemString(dict, (char *)"beta", PyFloat_FromDouble(ts->beta));
  PyDict_SetItemString(dict, (char *)"gamma", PyFloat_FromDouble(ts->gamma));
  return dict;
}

// set_periodic(molid, frame=-1, a, b, c, alpha, beta, gamma)
static PyObject *set_periodic(PyObject *self, PyObject *args, PyObject *kwds) {
  int molid;
  int frame=-1;
  float a=-1, b=-1, c=-1, alpha=-1, beta=-1, gamma=-1;
  static char *kwlist[] = { (char *)"molid", (char *)"frame", (char *)"a",
    (char *)"b", (char *)"c", (char *)"alpha", (char *)"beta", (char *)"gamma",
    NULL
  };
  if (!PyArg_ParseTupleAndKeywords(args, kwds, 
        (char *)"i|iffffff:molecule.set_periodic", kwlist, 
        &molid, &frame, &a, &b, &c, &alpha, &beta, &gamma))
    return NULL;
  // XXX big cut 'n paste
  VMDApp *app = get_vmdapp();
  Molecule *mol = app->moleculeList->mol_from_id(molid);
  if (!mol) {
    PyErr_SetString(PyExc_ValueError, (char *)"Invalid molecule id");
    return NULL;
  }
  Timestep *ts = NULL;
  if (frame == -1) {
    ts = mol->current();
  } else if (frame == -2) {
    ts = mol->get_last_frame();
  } else {
    ts = mol->get_frame(frame);
  }
  if (!ts) {
    PyErr_SetString(PyExc_ValueError, (char *)"Invalid frame");
    return NULL;
  }
  if (a >= 0) ts->a_length = a;
  if (b >= 0) ts->b_length = b;
  if (c >= 0) ts->c_length = c;
  if (alpha > 0) ts->alpha = alpha;
  if (beta > 0) ts->beta = beta;
  if (gamma > 0) ts->gamma = gamma;

  Py_INCREF(Py_None);
  return Py_None;
}

static PyMethodDef MolMethods[] = {
  {(char *)"num", (vmdPyMethod)mol_num, METH_VARARGS, (char *)"Number of loaded molecules."},
  {(char *)"listall", (vmdPyMethod)mol_listall, METH_VARARGS},
  {(char *)"new", (vmdPyMethod)mol_new, METH_VARARGS},
  {(char *)"load", (PyCFunction)mol_load, METH_VARARGS | METH_KEYWORDS},
  {(char *)"cancel", (vmdPyMethod)mol_cancel, METH_VARARGS},
  {(char *)"delete", (vmdPyMethod)mol_delete, METH_VARARGS},
  {(char *)"read", (PyCFunction)mol_read, METH_VARARGS | METH_KEYWORDS},
  {(char *)"write", (PyCFunction)mol_write, METH_VARARGS | METH_KEYWORDS},
  {(char *)"delframe", (PyCFunction)delframe, METH_VARARGS | METH_KEYWORDS},
  {(char *)"dupframe", (vmdPyMethod)dupframe, METH_VARARGS},
  {(char *)"numframes", (vmdPyMethod)numframes, METH_VARARGS},
  {(char *)"get_frame", (vmdPyMethod)get_frame, METH_VARARGS},
  {(char *)"set_frame", (vmdPyMethod)set_frame, METH_VARARGS},
  {(char *)"numatoms", (vmdPyMethod)mol_numatoms, METH_VARARGS},
  {(char *)"exists", (vmdPyMethod)mol_exists, METH_VARARGS},
  {(char *)"name", (vmdPyMethod)mol_name, METH_VARARGS},
  {(char *)"ssrecalc", (vmdPyMethod)mol_ssrecalc, METH_VARARGS},
  {(char *)"rename", (vmdPyMethod)mol_rename, METH_VARARGS},
  {(char *)"get_top", (vmdPyMethod)get_top, METH_VARARGS},
  {(char *)"set_top", (vmdPyMethod)set_top, METH_VARARGS},
  {(char *)"add_volumetric", (PyCFunction)mol_add_volumetric, 
                              METH_VARARGS | METH_KEYWORDS},
  {(char *)"get_filenames", (vmdPyMethod)get_filenames, METH_VARARGS},
  {(char *)"get_filetypes", (vmdPyMethod)get_filetypes, METH_VARARGS},
  {(char *)"get_databases", (vmdPyMethod)get_databases, METH_VARARGS},
  {(char *)"get_accessions", (vmdPyMethod)get_accessions, METH_VARARGS},
  {(char *)"get_remarks", (vmdPyMethod)get_remarks, METH_VARARGS},
  {(char *)"get_periodic", (PyCFunction)get_periodic, METH_VARARGS | METH_KEYWORDS},
  {(char *)"set_periodic", (PyCFunction)set_periodic, METH_VARARGS | METH_KEYWORDS},
  {NULL, NULL}
};

void initmolecule() {
  (void) Py_InitModule((char *)"molecule", MolMethods);
}

