/***************************************************************************
 *cr
 *cr            (C) Copyright 1995-2007 The Board of Trustees of the
 *cr                        University of Illinois
 *cr                         All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: VolMapCreate.h,v $
 *      $Author: jordi $        $Locker:  $             $State: Exp $
 *      $Revision: 1.24 $       $Date: 2007/03/14 16:48:08 $
 *
 **************************************************************************/


class VMDApp;
class VolMap;
class AtomSel;

/// Virtual class for dealing with the computation of VolMaps, based on
/// atomic selections and atomic data and coordinates. It provides utilities
/// for computing such maps. Various algorithms and specific calculations 
/// are done by child classes. The end result is a VolMap object, which can 
/// be passed to VMD (N/A yet) or written to a file.

class VolMapCreate {
public:
  typedef enum {COMBINE_AVG, COMBINE_MIN, COMBINE_MAX, COMBINE_STDEV, COMBINE_PMF} CombineType;
  
protected:
  VMDApp *app;
  AtomSel *sel;
  float delta;            // resolution (same along x, y and z)
  int computed_frames;    // frame counter
  int checkpoint_freq;    // write checkpoint file every xxx steps
  char *checkpoint_name;  // checkpoint file name
  bool user_minmax;       // true = user specified a minmax box, false = compute default minmax
  float min_coord[3], max_coord[3]; // used to pass user defaults, avoid using for computations!

protected:
  virtual int compute_frame(int frame, float *voldata) = 0;
  int compute_init(float padding);
  
  /// called before computing individual frames
  virtual int compute_init() {return compute_init(0.);}
  
  /// called after having computed every frame
  virtual int compute_end() {return 0;}
 
  /// calculates minmax (into preallocated float[3] arrays) using volmapcreate's selection
  int calculate_minmax (float *min_coord, float *max_coord);
  
  /// calculates max_radius (into a provided float) using volmapcreate's selection
  int calculate_max_radius (float &radius);
  
  /// for checkpointing
  void combo_begin(CombineType method, void **customptr, void *params);
  void combo_addframe(CombineType method, float *voldata, void *customptr, float *framedata);
  void combo_export(CombineType method, float *voldata, void *customptr);
  void combo_end(CombineType method, void *customptr);
  
public:
  VolMap *volmap;
  
  VolMapCreate(VMDApp *app, AtomSel *sel, float resolution);
  virtual ~VolMapCreate();
  
  void set_minmax (float minx, float miny, float minz, float maxx, float maxy, float maxz);

  void set_checkpoint (int checkpointfreq, char *checkpointname);
  
  int compute_all(bool allframes, CombineType method, void *params);
};


class VolMapCreateMask: public VolMapCreate {
protected:
  int compute_init();
  int compute_frame(int frame, float *voldata);
private:
  float atomradius;
  bool antialias; //unused
public:
  VolMapCreateMask(VMDApp *app, AtomSel *sel, float res, float the_atomradius) : VolMapCreate(app, sel, res) {
    atomradius = the_atomradius;
    antialias = false;
  }
};


class VolMapCreateDensity : public VolMapCreate {
protected:
  float *weight;
  int compute_init();
  int compute_frame(int frame, float *voldata);
  int numpoints;      // OBSOLETE for now
  float radius_scale; // mult. factor for atomic radii
  
public:
  VolMapCreateDensity(VMDApp *app, AtomSel *sel, float res, float *the_weight) : VolMapCreate(app, sel, res) {
    weight = the_weight;
    set_radius_scale(1.f);
    set_numpoints(1000);
  }
  
  //sets number of random points to use for each atom's gaussian distr.
  void set_radius_scale (float value) {
    radius_scale = value;
  }
  
  //sets number of random points to use for each atom's gaussian distr.
  void set_numpoints (int numpoints_set) {
    numpoints = numpoints_set;
  }
};


class VolMapCreateOccupancy : public VolMapCreate {
private:
  bool use_points;
protected:
  int compute_init();
  int compute_frame(int frame, float *voldata);  
public:
  VolMapCreateOccupancy(VMDApp *app, AtomSel *sel, float res) : VolMapCreate(app, sel, res) {
    use_point_particles(false);
  }

  void use_point_particles(bool value) {
    use_points = value;
  }
};


class VolMapCreateDistance : public VolMapCreate {
protected:
  float max_dist;
  int compute_init();
  int compute_frame(int frame, float *voldata);  
public:
  VolMapCreateDistance(VMDApp *app, AtomSel *sel, float res, float the_max_dist) : VolMapCreate(app, sel, res) {
    max_dist = the_max_dist;
  }
};


/// Fast ligand PMF algorithm - It finds the energy of placing a monoatomic or diatomic ligand
/// at many points in the protein 
class VolMapCreateFastEnergy : public VolMapCreate {
private:
  float max_dist;  //max dist from ligand center of mass
  float cutoff;    //max dist of interaction between any 2 atoms
  float switchdist;
  bool use_switching;

  bool compute_elec;
  
  float *conformers;
  float **conformer_voldata;
  int subres;  // resolution of the subgrid
  
public:
  float probe_bondlength;
  float probe1_epsilon, probe1_rmin;
  float probe2_epsilon, probe2_rmin, probe2_charge;
  float temperature;

protected:
  int compute_init();
  int compute_frame(int frame, float *voldata);
  int compute_end();
    
public: // shouldn't really be public...
  enum {MONO, DIHOMO, DIHETERO} ligand_type;
  int num_conformers;

  float mincutoff; // cutoff for the subgrid subsampling
                   // a correction up to cutoff is added later
  float probe1_charge;

public: //settable params
  int param_subres;

public:   
  VolMapCreateFastEnergy(VMDApp *app, AtomSel *sel, float res, float epsilon, float rmin, float the_cutoff);
};




/// Slower ligand PMF algorithm with less approximations (for double-checking
/// purposes)
class VolMapCreateSlowEnergy : public VolMapCreate {
private:
  float max_dist;  //max dist from ligand center of mass
  float cutoff;    //max dist of interaction between any 2 atoms
  float switchdist;
  bool use_switching;

  bool compute_elec;
  
  float *conformers;
  float **conformer_voldata;
public:
  float probe_bondlength;
  float probe1_epsilon, probe1_rmin;
  float probe2_epsilon, probe2_rmin, probe2_charge;
  float temperature;

protected:
  int compute_init();
  int compute_frame(int frame, float *voldata);
  int compute_end();
public:
  enum {MONO, DIHOMO, DIHETERO} ligand_type;
  int num_conformers;
  float probe1_charge;
    
  VolMapCreateSlowEnergy(VMDApp *app, AtomSel *sel, float res, float epsilon, float rmin, float the_cutoff);
};



class VolMapCreateCoulombPotential : public VolMapCreate {
protected:
  int compute_init();
  int compute_frame(int frame, float *voldata);
  
public:
  VolMapCreateCoulombPotential(VMDApp *app, AtomSel *sel, float res) : VolMapCreate(app, sel, res) {
  }
};



