/***************************************************************************
 *cr
 *cr            (C) Copyright 1995-2007 The Board of Trustees of the
 *cr                        University of Illinois
 *cr                         All Rights Reserved
 *cr
 ***************************************************************************/
/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: VolMapCreate.C,v $
 *      $Author: ltrabuco $        $Locker:  $             $State: Exp $
 *      $Revision: 1.57 $      $Date: 2007/04/02 02:12:43 $
 *
 ***************************************************************************/

/* Functions for creating useful volumetric maps based on the 3-D molecular structure */

// Todo List: 
// - Document!
// - Don't just output to a DX file... give user more control (use plugins)
// - Allow a framerange param, don't just use all available frames (and get rid
// of the VMDApp dependency). Also, VMD needs a general FrameRange object...

#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "VMDApp.h"
#include "MoleculeList.h"
#include "Molecule.h"
#include "Timestep.h"
#include "Measure.h"
#include "BondSearch.h"
#include "VolCPotential.h"
#include "VolMap.h"
#include "VolMapCreate.h"
#include "utilities.h"

#include "config.h"
#if defined(VMDTKCON)
#include "vmdconsole.h"
#define MYPRINTF vmdprintf
#else
#define MYPRINTF printf
#endif

#define MIN(X,Y) (((X)<(Y))? (X) : (Y))
#define MAX(X,Y) (((X)>(Y))? (X) : (Y))

// Conversion factor between raw units (e^2 / A) and kT/e
#define POT_CONV 560.47254

/// maximum energy (above which all energies are considered infinite, for
/// performance purposes of the Energy/PMF map type)
static const float MAX_ENERGY = 150.f; 


////////////// VolMapCreate //////////////

VolMapCreate::VolMapCreate(VMDApp *the_app, AtomSel *the_sel, float resolution) {
  volmap = NULL;
  app = the_app;
  sel = the_sel;
  delta = resolution;
  checkpoint_freq = 0;
  checkpoint_name = NULL;
  volmap = new VolMap();
  user_minmax = false;
}


VolMapCreate::~VolMapCreate() {
  if (volmap) delete volmap;
}


void VolMapCreate::set_minmax (float minx, float miny, float minz, float maxx, float maxy, float maxz) {
  user_minmax = true;
  min_coord[0] = minx;
  min_coord[1] = miny;
  min_coord[2] = minz;
  max_coord[0] = maxx;
  max_coord[1] = maxy;
  max_coord[2] = maxz;
}


void VolMapCreate::set_checkpoint (int checkpointfreq_tmp, char *checkpointname_tmp) {
  if (checkpointfreq_tmp > -1) checkpoint_freq = checkpointfreq_tmp;
  if (!checkpointname_tmp) return;
  
  if (checkpoint_name) delete[] checkpoint_name;
  checkpoint_name = new char[strlen(checkpointname_tmp)+1];
  strcpy(checkpoint_name, checkpointname_tmp);
}


/// fills in the minmax values into 2 pre-allocated double[3] arrays.
int VolMapCreate::calculate_minmax (float *my_min_coord, float *my_max_coord) {
  DrawMolecule *mol = app->moleculeList->mol_from_id(sel->molid());
  int numframes = app->molecule_numframes(sel->molid()); // XXX need a frame selection object
  
  float frame_min_coord[3], frame_max_coord[3], *coords;
  
  MYPRINTF("volmap: Computing bounding box coordinates\n");
  
  for (int frame=0; frame<numframes; frame++) {
    sel->which_frame=frame;
    sel->change(NULL,mol);
    coords = sel->coordinates(app->moleculeList);
    if (!coords) continue;

    int err = measure_minmax(sel, coords, NULL, frame_min_coord, frame_max_coord);
    if (err != MEASURE_NOERR) return err;
    
    for (int i=0; i<3; i++) {
      if (!frame || frame_min_coord[i] < my_min_coord[i]) my_min_coord[i] = frame_min_coord[i];
      if (!frame || frame_max_coord[i] > my_max_coord[i]) my_max_coord[i] = frame_max_coord[i];
    }
  }
  
  return 0;
}


/// fills in the minmax values into 2 pre-allocated double[3] arrays.
int VolMapCreate::calculate_max_radius (float &max_rad) {
  DrawMolecule *mol = app->moleculeList->mol_from_id(sel->molid());
  if (!mol) return -1;
  const float *radius = mol->extra.data("radius");
  if (!radius) return MEASURE_ERR_NORADII;
  
  max_rad = 0.f;
  for (int i=0; i<sel->num_atoms; i++) 
    if (sel->on[i] && radius[i] > max_rad) max_rad = radius[i];
  
  return 0;
}

// Combo routines are used to combbine the different frame maps together into a
// final entity

// Initialize the frame combination buffer
void VolMapCreate::combo_begin(CombineType method, void **customptr, void *params) {
  int gridsize = volmap->gridsize;
  
  *customptr = NULL;
  volmap->fill(0.f);
  volmap->weight = 0.f;
  computed_frames = 0;
  
  // these combine types need additi0onal storage
  if (method == COMBINE_STDEV) {
    float *voldata2 = new float[gridsize];
    memset(voldata2, 0, gridsize*sizeof(float));
    *customptr = (void*) voldata2;
  }
}

// Add a frame to the combination buffer
void VolMapCreate::combo_addframe(CombineType method, float *voldata, void *customptr, float *frame_voldata) {
  float *voldata2 = (float*) customptr;
  int gridsize = volmap->gridsize;
  int n;
  
  computed_frames++;
  volmap->weight = computed_frames;
     
  if (computed_frames == 1) { // FIRST FRAME
    switch (method) {
    case COMBINE_AVG:
    case COMBINE_MAX:    
    case COMBINE_MIN:    
      memcpy(voldata, frame_voldata, gridsize*sizeof(float));
      break;
    case COMBINE_PMF:
      for (n=0; n<gridsize; n++)
        voldata[n] = expf(-frame_voldata[n]);
      break;
    case COMBINE_STDEV:    
      memcpy(voldata, frame_voldata, gridsize*sizeof(float));
      for (n=0; n<gridsize; n++) voldata2[n] = frame_voldata[n]*frame_voldata[n];
      break;
    }
    
    return;
  }

  // THE FOLLOWING ONLY APPLIES TO OTHER FRAMES THAN FIRST
  switch (method) {
    case COMBINE_AVG:
      for (n=0; n<gridsize; n++) voldata[n] += frame_voldata[n];
      break;
    case COMBINE_PMF:
      for (n=0; n<gridsize; n++) voldata[n] += expf(-frame_voldata[n]);
      break;
    case COMBINE_MAX:    
      for (n=0; n<gridsize; n++) voldata[n] = MAX(voldata[n], frame_voldata[n]);
      break;
    case COMBINE_MIN:    
      for (n=0; n<gridsize; n++) voldata[n] = MIN(voldata[n], frame_voldata[n]);
      break;
    case COMBINE_STDEV:    
      for (n=0; n<gridsize; n++) voldata[n] += frame_voldata[n];
      for (n=0; n<gridsize; n++) voldata2[n] += frame_voldata[n]*frame_voldata[n];
      break;
  }
}


/// Output a copy of the combination buffer, to which a final transform
/// appropriate for the combination type has been applied to. Frames can still
/// be appended to the original combo buffer. This procedure should be used to
/// create the final map as well as each checkpoint map.
void VolMapCreate::combo_export(CombineType method, float *voldata, void *customptr) {
  float *voldata2 = (float*) customptr;
  int gridsize = volmap->gridsize;
  int n;
  
  switch (method) {
  case COMBINE_AVG:
    for (n=0; n<gridsize; n++) {
      volmap->data[n] = voldata[n]/computed_frames;
    }
    break;
  case COMBINE_PMF:
    for (n=0; n<gridsize; n++) {
      float val = -logf(voldata[n]) + logf((float) computed_frames);   // this is PMF relative to vacuum (where E=0)  
      if (val > MAX_ENERGY) val = MAX_ENERGY;  // weed out outlying data
      volmap->data[n] = val;
    }
    break;
  case COMBINE_MAX:
  case COMBINE_MIN:    
    memcpy(volmap->data, voldata, gridsize*sizeof(float));    
    break;
  case COMBINE_STDEV:    
    for (n=0; n<gridsize; n++) {
      volmap->data[n] = voldata[n]/computed_frames;
      volmap->data[n] = sqrtf(voldata2[n]/computed_frames - volmap->data[n]*volmap->data[n]); 
    }
    break;
  }
}


/// Do some cleaning up of the combination buffer
void VolMapCreate::combo_end(CombineType method, void *customptr) {
  if (method == COMBINE_STDEV) {
    float *voldata2 = (float*) customptr;
    delete[] voldata2;
  }
}



/// computes volmap for all (selected) frames and takes the average, min or max
/// depending on what was specified by "method", for most cases, 
/// params is unused (set to NULL)
/// XXX the "allframes" param should be a FrameRange object, but for now its a boolean
int VolMapCreate::compute_all (bool allframes, CombineType method, void *params) {
  int err = this->compute_init();
  if (err) return err;
  
  // Special case: if only have one frame do it here the fast way
  if (!allframes) {
    if (volmap->data) delete[] volmap->data;
    volmap->data = new float[volmap->gridsize];         // final exported voldata
  
    MYPRINTF("volmap: grid size = %dx%dx%d (%.1f MB)\n", volmap->xsize, volmap->ysize, volmap->zsize, sizeof(float)*volmap->gridsize/(1024.*1024.));
    this->compute_frame(AtomSel::TS_NOW, volmap->data); // only compute "now" frame
    
    err = this->compute_end();
    if (err) return err;
    return 0; // no error
  }
  
  
  int numframes = app->molecule_numframes(sel->molid());
  MYPRINTF("volmap: total frames = %d\n", numframes);
  MYPRINTF("volmap: grid size = %dx%dx%d (%.1f MB)\n", volmap->xsize, volmap->ysize, volmap->zsize, sizeof(float)*volmap->gridsize/(1024.*1024.));

  if (volmap->data) delete[] volmap->data;
  volmap->data = new float[volmap->gridsize];         // final exported voldata
  float *frame_voldata = new float[volmap->gridsize]; // individual frame voldata
  float *voldata = new float[volmap->gridsize];       // combo cache voldata
  
  void *customptr = NULL;
  combo_begin(method, &customptr, params);
  
  // Combine frame_voldata into voldata, one frame at a time, starting with 1st frame
  for (int frame=0; frame<numframes; frame++) { 
    // XXX to-do, only take frames from a frame selection
    MYPRINTF("volmap: frame %d/%d\n", frame, numframes);
    fflush(stdout);
    this->compute_frame(frame, frame_voldata);
    combo_addframe(method, voldata, customptr, frame_voldata);
    if (checkpoint_freq && computed_frames && !(computed_frames%checkpoint_freq)) {
      combo_export(method, voldata, customptr);
      const char *filename;
      if (checkpoint_name) filename=checkpoint_name;
      else filename = "checkpoint.dx";
      volmap->write_dx_file(filename);
    }
  }
    
  delete[] frame_voldata;
  
  // All frames have been combined, perform finishing steps here
  combo_export(method, voldata, customptr);
  combo_end (method, customptr);
  delete[] voldata;
       
  err = this->compute_end();
  if (err) return err;

  return 0; // no error
}



// compute_init() sets up the grid coordinate system and dimensions
// If the user did not specify the grid's minmax boundary, it is
// defualted to the trajectory's minmax coordinates, to which "padding"
// is subtracted in all dimensions. 
int VolMapCreate::compute_init (float padding) {
  if (!sel) return MEASURE_ERR_NOSEL;
  if (sel->num_atoms == 0) return MEASURE_ERR_NOATOMS;
  
  int err, i;
  
  if (!volmap) return -1;
  
  if (user_minmax)
    padding = 0.;  // don't want to pad user's defaults
  else {
    err = calculate_minmax(min_coord, max_coord);
    if (err) return err;
  }
  
  for (i=0; i<3; i++) {
    //adjust padding and ensure that different maps are properly aligned
    min_coord[i] = (float) floor((min_coord[i] - padding)/delta)*delta;    
    max_coord[i] = (float)  ceil((max_coord[i] + padding)/delta)*delta;
  }
  
  volmap->xsize = MAX((int)((max_coord[0] - min_coord[0])/delta) + 1, 0);
  volmap->ysize = MAX((int)((max_coord[1] - min_coord[1])/delta) + 1, 0);
  volmap->zsize = MAX((int)((max_coord[2] - min_coord[2])/delta) + 1, 0);

  MYPRINTF("volmap: grid minmax = {%f %f %f} {%f %f %f}\n", min_coord[0], min_coord[1], min_coord[2], max_coord[0], max_coord[1], max_coord[2]);
  
  volmap->xdelta[0] = delta;
  volmap->xdelta[1] = 0.f;
  volmap->xdelta[2] = 0.f;
  volmap->ydelta[0] = 0.f;
  volmap->ydelta[1] = delta;
  volmap->ydelta[2] = 0.f;
  volmap->zdelta[0] = 0.f;
  volmap->zdelta[1] = 0.f;
  volmap->zdelta[2] = delta;
  
  // define origin by shifting to middle of each cell,
  // compute_frame() needs to take this into account
  for (i=0; i<3; i++) volmap->origin[i] = min_coord[i] + \
          0.5f*(volmap->xdelta[i] + volmap->ydelta[i] + \
          volmap->zdelta[i]);
  
  volmap->update_internal();
         
  return 0; // no error
}



////////////// VolMapCreateMask //////////////
/** This creates a "mask", i.e. a map that is either 1 or 0 and can be multiplied
    to another map in order to isolate a particular region in space. Currently,
    the mask is created by painting a sphere of radius 5 around each selected atom
**/

int VolMapCreateMask::compute_init() {
  char tmpstr[255];
  sprintf(tmpstr, "mask (%s.200)", sel->cmdStr);
  volmap->set_name(tmpstr);
    
  return VolMapCreate::compute_init(atomradius+0.5f);
}


int VolMapCreateMask::compute_frame (int frame, float *voldata) {
  DrawMolecule *mol = app->moleculeList->mol_from_id(sel->molid());
  if (!mol) return -1;
  
  int i;
  int GRIDX = volmap->xsize;
  int GRIDY = volmap->ysize;
  int GRIDZ = volmap->zsize;
  int gridsize=volmap->xsize*volmap->ysize*volmap->zsize;
  
  //create volumetric density grid
  memset(voldata, 0, gridsize*sizeof(float));
  sel->which_frame=frame;
  sel->change(NULL,mol);
  
  const float *coords = sel->coordinates(app->moleculeList);
  if (!coords) return -1;
  
  float min_coords[3];
  for (i=0; i<3; i++)
    min_coords[i] = volmap->origin[i] - 0.5f*(volmap->xdelta[i] + 
                                              volmap->ydelta[i] + 
                                              volmap->zdelta[i]);
  
  // paint atomic spheres on map
  int gx, gy, gz;
  for (i=0; i < sel->num_atoms; i++) { 
    if (!sel->on[i]) continue; //atom is not selected
      gx = (int) ((coords[3*i] - min_coords[0])/delta);
      gy = (int) ((coords[3*i+1] - min_coords[1])/delta);
      gz = (int) ((coords[3*i+2] - min_coords[2])/delta);
      
      float dx, dy, dz; //to measure distances
      int steps = (int)(atomradius/delta)+1;
      for (int iz=MAX(gz-steps,0); iz<=MIN(gz+steps,GRIDZ-1); iz++)
      for (int iy=MAX(gy-steps,0); iy<=MIN(gy+steps,GRIDY-1); iy++)
      for (int ix=MAX(gx-steps,0); ix<=MIN(gx+steps,GRIDX-1); ix++) {
        int n = ix + iy*GRIDX + iz*GRIDY*GRIDX;
        dx = coords[3*i] - volmap->origin[0] - ix*delta;
        dy = coords[3*i+1] - volmap->origin[1] - iy*delta;
        dz = coords[3*i+2] - volmap->origin[2] - iz*delta;
        float dist2 = dx*dx+dy*dy+dz*dz;
        if (dist2 <= atomradius*atomradius) voldata[n] = 1.f;
      }
  }
  
  return 0;
}  





/////// VolMapCreateDensity ///////
  
int VolMapCreateDensity::compute_init () {
  char tmpstr[255];
  sprintf(tmpstr, "density (%.200s) [A^-3]", sel->cmdStr);
  volmap->set_name(tmpstr);
  
  float max_rad;
  calculate_max_radius(max_rad);
    
  return VolMapCreate::compute_init(MAX(3.f*radius_scale*max_rad,10.f));
}


int VolMapCreateDensity::compute_frame (int frame, float *voldata) {
  if (!weight) return MEASURE_ERR_NOWEIGHT;
    
  DrawMolecule *mol = app->moleculeList->mol_from_id(sel->molid());
  if (!mol) return -1;
    int i;
    
  const float *radius = mol->extra.data("radius");
  if (!radius) return MEASURE_ERR_NORADII;
  
  int GRIDX = volmap->xsize;
  int GRIDY = volmap->ysize;
  int GRIDZ = volmap->zsize;
  int gridsize=volmap->xsize*volmap->ysize*volmap->zsize;

  //create volumetric density grid
  memset(voldata, 0, gridsize*sizeof(float));
  sel->which_frame=frame;
  sel->change(NULL,mol);
  const float *coords = sel->coordinates(app->moleculeList);
  if (!coords) return -1;
  
  float min_coords[3];
  for (i=0; i<3; i++) 
    min_coords[i] = volmap->origin[i] - 0.5f*(volmap->xdelta[i] +
                                              volmap->ydelta[i] +
                                              volmap->zdelta[i]);
  
  int w_index=0;
  int gx, gy, gz;   // grid coord indices
  float dx, dy, dz; // to measure distances
  for (i=0; i < sel->num_atoms; i++) { 
    if (!sel->on[i]) continue; //atom is not selected
      gx = (int) ((coords[3*i] - min_coords[0])/delta);
      gy = (int) ((coords[3*i+1] - min_coords[1])/delta);
      gz = (int) ((coords[3*i+2] - min_coords[2])/delta);
      
      float scaled_radius = 0.5f*radius_scale*radius[i];
      float exp_factor = 1.0f/(2.0f*scaled_radius*scaled_radius);
      float norm = weight[w_index++]/(sqrtf((float) (8.0f*PI*PI*PI))*scaled_radius*scaled_radius*scaled_radius);
                  
      int steps = (int)(4.1f*scaled_radius/delta);
      for (int iz=MAX(gz-steps,0); iz<=MIN(gz+steps,GRIDZ-1); iz++)
      for (int iy=MAX(gy-steps,0); iy<=MIN(gy+steps,GRIDY-1); iy++)
      for (int ix=MAX(gx-steps,0); ix<=MIN(gx+steps,GRIDX-1); ix++) {
        int n = ix + iy*GRIDX + iz*GRIDY*GRIDX;
        dx = coords[3*i] - volmap->origin[0] - ix*delta;
        dy = coords[3*i+1] - volmap->origin[1] - iy*delta;
        dz = coords[3*i+2] - volmap->origin[2] - iz*delta;
        float dist2 = dx*dx+dy*dy+dz*dz;
        voldata[n] += norm * expf(-dist2*exp_factor);
        // Uncomment the following line for a much faster implementation
        // This is useful is all you care about is the smooth visual appearance
        // voldata[n] += exp_factor/(dist2+10.f);
      }
  }
    
  return 0;
}  

  
/////// VolMapCreateOccupancy ///////
/** This creates a map that is valued at 100 for gridpoints inside atoms, and 0 for
    gridpoints outside. If averaged over many frames, it will produce the % chance
    of that gridpoint being occupied. These maps can either be created using point
    particles or by painting spheres using the VDW radius.
**/

int VolMapCreateOccupancy::compute_init () {
  char tmpstr[255];
  sprintf(tmpstr, "occupancy (%.200s)", sel->cmdStr);
  volmap->set_name(tmpstr);
  
  float max_rad;  
  if (use_points)
    max_rad = 1.f;
  else
    calculate_max_radius(max_rad);
  
  return VolMapCreate::compute_init(max_rad);
}


int VolMapCreateOccupancy::compute_frame(int frame, float *voldata) { 
  DrawMolecule *mol = app->moleculeList->mol_from_id(sel->molid());
  if (!mol) return -1;
  
  int GRIDX = volmap->xsize;
  int GRIDY = volmap->ysize;
  int GRIDZ = volmap->zsize;
  int gridsize=volmap->xsize*volmap->ysize*volmap->zsize;
  int i;
  
  //create volumetric density grid
  memset(voldata, 0, gridsize*sizeof(float));
  sel->which_frame=frame;
  sel->change(NULL,mol);
  const float *coords = sel->coordinates(app->moleculeList);
  if (!coords) return -1;
  float min_coords[3];
  for (i=0; i<3; i++) 
    min_coords[i] = volmap->origin[i] - 0.5f*(volmap->xdelta[i] +
                                              volmap->ydelta[i] + 
                                              volmap->zdelta[i]);
  int gx, gy, gz;
  
  if (use_points) { // draw single points
    for (i=0; i < sel->num_atoms; i++) { 
      if (!sel->on[i]) continue; //atom is not selected
      gx = (int) ((coords[3*i] - min_coords[0])/delta);
      if (gx<0 || gx>=GRIDX) continue;
      gy = (int) ((coords[3*i+1] - min_coords[1])/delta);
      if (gy<0 || gy>=GRIDY) continue;
      gz = (int) ((coords[3*i+2] - min_coords[2])/delta);
      if (gz<0 || gz>=GRIDZ) continue;
      voldata[gx+GRIDX*gy+GRIDX*GRIDY*gz] = 1.f; 
    }
  }
  else { // paint atomic spheres on map
    const float *radius = mol->extra.data("radius");
    if (!radius) return MEASURE_ERR_NORADII;
  
    for (i=0; i < sel->num_atoms; i++) { 
      if (!sel->on[i]) continue; //atom is not selected
        gx = (int) ((coords[3*i] - min_coords[0])/delta);
        gy = (int) ((coords[3*i+1] - min_coords[1])/delta);
        gz = (int) ((coords[3*i+2] - min_coords[2])/delta);
      
        float dx, dy, dz; //to measure distances
        int steps = (int)(radius[i]/delta)+1;
        for (int iz=MAX(gz-steps,0); iz<=MIN(gz+steps,GRIDZ-1); iz++)
        for (int iy=MAX(gy-steps,0); iy<=MIN(gy+steps,GRIDY-1); iy++)
        for (int ix=MAX(gx-steps,0); ix<=MIN(gx+steps,GRIDX-1); ix++) {
          int n = ix + iy*GRIDX + iz*GRIDY*GRIDX;
          dx = coords[3*i] - volmap->origin[0] - ix*delta;
          dy = coords[3*i+1] - volmap->origin[1] - iy*delta;
          dz = coords[3*i+2] - volmap->origin[2] - iz*delta;
          float dist2 = dx*dx+dy*dy+dz*dz;
          if (dist2 <= radius[i]*radius[i]) voldata[n] = 1.f;
        }
    }
}
  
  
    
  return 0;
}



/////// VolMapCreateDistance ///////


int VolMapCreateDistance::compute_init () {
  char tmpstr[255];
  sprintf(tmpstr, "distance (%.200s) [A]", sel->cmdStr);
  volmap->set_name(tmpstr);
  
  float max_rad;
  calculate_max_radius(max_rad);
  
  return VolMapCreate::compute_init(max_rad+max_dist);
}
 

/// Computes, for each gridpoint, the distance to the nearest atom
/// boundary, as defined by the VMD's atomic VDW radii.
int VolMapCreateDistance::compute_frame(int frame, float *voldata) { 
  int i, n;  
  DrawMolecule *mol = app->moleculeList->mol_from_id(sel->molid());
  if (!mol) return -1;
  const float *radius = mol->extra.data("radius");
  if (!radius) return MEASURE_ERR_NORADII;
  
  int GRIDX = volmap->xsize;
  int GRIDY = volmap->ysize;
  int gridsize=volmap->xsize*volmap->ysize*volmap->zsize;

  float dx, dy, dz;
  float dist, mindist, r;
  
  float max_rad;
  calculate_max_radius(max_rad);
  
  // 1. Create a fake "molecule" containing all of the grid points
  //    this is quite memory intensive but _MUCH_ faster doing it point-by point!
  
  float *gridpos = new float[3*gridsize]; 
  int *gridon = new int[gridsize]; 
  for (n=0; n<gridsize; n++) {
    gridpos[3*n] = (n%GRIDX)*delta + volmap->origin[0]; //position of grid cell's center
    gridpos[3*n+1] = ((n/GRIDX)%GRIDY)*delta + volmap->origin[1];
    gridpos[3*n+2] = (n/(GRIDX*GRIDY))*delta + volmap->origin[2]; 
    gridon[n] = 1;
  }

  GridSearchPair *pairlist, *p;

  sel->which_frame=frame;
  sel->change(NULL,mol);
  const float *coords = sel->coordinates(app->moleculeList);
  if (!coords) return -1;
  
  // initialize all grid points to be the maximal allowed distance = cutoff
  for (n=0; n<gridsize; n++) voldata[n] = max_dist;
  
  // 2. Create a list of all bonds between the grid and the real molecule
  //    which are within the user-set cutoff distance 
  //    (the use of a cutoff is purely to speed this up tremendously)
  
  pairlist = vmd_gridsearch3(gridpos, gridsize, gridon, coords,
                             sel->num_atoms, sel->on, max_dist+max_rad, true, -1);
  for (p=pairlist; p; p=p->next) {
    n = p->ind1;
    // if a grid point is already known to be inside an atom, skip it and save some time
    if ((mindist = voldata[n]) == 0.f) continue;
    i = p->ind2;
    r = radius[i];
    dx = gridpos[3*n] - coords[3*i];
    dy = gridpos[3*n+1] - coords[3*i+1];
    dz = gridpos[3*n+2] - coords[3*i+2];
    
    // 3. At each grid point, store the _smallest_ recorded distance
    //    to a nearby atomic surface
      
    dist = sqrtf(dx*dx+dy*dy+dz*dz) - r;
    if (dist < 0) dist = 0.f;
    if (dist < mindist) voldata[n] = dist;
  }
  
  // delete pairlist
  for (p=pairlist; p;) {
    GridSearchPair *tmp = p;
    p = p->next;
    free(tmp);
  }  

  delete gridpos; 
  delete gridon; 

  return MEASURE_NOERR; 
}




/////// VolMapCreateSlowEnergy ///////

VolMapCreateSlowEnergy::VolMapCreateSlowEnergy(VMDApp *app, AtomSel *sel, float res, float epsilon, float rmin, float the_cutoff) : VolMapCreate(app, sel, res) {
    cutoff = the_cutoff;
    ligand_type = MONO;
    num_conformers = 10;

    probe1_epsilon = epsilon;
    probe1_rmin = rmin;
    probe1_charge = 0.f;
    probe_bondlength = 1.12f;  //1.23 in CHARMM
    probe2_epsilon = epsilon;
    probe2_rmin = rmin;
    probe2_charge = 0.f;
    
    temperature = 300.f;
    
    switchdist = 10.;
    use_switching = false;
    //if (cutoff >= 11.) use_switching = true;
    
    max_dist = cutoff + probe_bondlength/2.;
}

int VolMapCreateSlowEnergy::compute_init () {
  MYPRINTF("\nThis computes the potential of mean force (free energy) over\n"
         "a grid, for an implicit monoatomic or diatomic ligand\n"
         "If you use this method in your work, please cite:\n\n"
         "  J COHEN, A ARKHIPOV, R BRAUN and K SCHULTEN, \"Imaging the\n"
         "  migration pathways for O2, CO, NO, and Xe inside myoglobin\".\n"
         "  Biophysical Journal 91:1844-1857, 2006.\n\n");
  MYPRINTF("Using temperature T=%g\n", temperature);
  
  // init params
  if (ligand_type == MONO) {
    num_conformers = 0;
    probe_bondlength = 0.f;
  }
  cutoff += probe_bondlength/2.;
  max_dist = cutoff;
  
  // go  
  char tmpstr[255];
  sprintf(tmpstr, "slowligand pmf (%.200s) [kT]", sel->cmdStr);
  volmap->set_name(tmpstr);
  volmap->temperature=temperature;
  
  int retVal = VolMapCreate::compute_init(1.-cutoff);
  int gridsize = volmap->gridsize;
    
  compute_elec = (probe1_charge != 0.f); // compute electrostatics?

  
  //1. Create diatomic conformers
  if (ligand_type == MONO) {
    MYPRINTF("Using a monoatomic ligand: epsilon=%g, rmin/2=%g, charge=%g\n", probe1_epsilon, probe1_rmin, probe1_charge);
    MYPRINTF("Electrostatics: ");
    if (compute_elec)
      MYPRINTF("on\n");
    else
      MYPRINTF("off\n");
    conformers = NULL;
    conformer_voldata = NULL;
  }
  else {
    MYPRINTF("Using a diatomic ligand:\n");
    MYPRINTF("  atom1: epsilon=%g, rmin/2=%g, charge=%g\n", probe1_epsilon, probe1_rmin, probe1_charge);
    MYPRINTF("  atom2: epsilon=%g, rmin/2=%g, charge=%g\n", probe2_epsilon, probe2_rmin, probe2_charge);
    MYPRINTF("  bondlength=%g\n", probe_bondlength);
    MYPRINTF("Electrostatics: ");
    if (compute_elec)
      MYPRINTF("on\n");
    else
      MYPRINTF("off\n");
    MYPRINTF("Generating %d conformers\n", num_conformers);
    conformers = new float[6*num_conformers];
    memset(conformers, 0, 6*num_conformers*sizeof(float));
  
    // generate randomly oriented conformers
    const float halfbond = probe_bondlength/2.f;
    const float RAND_MAX_INV = 1.0f/VMD_RAND_MAX;
    int conf;
    for (conf=0; conf<num_conformers; conf++) {
      float u1 = (float) vmd_random()*RAND_MAX_INV;
      float u2 = (float) vmd_random()*RAND_MAX_INV;
      float z = (2.0f*u1 -1.0f);
      float phi = 2.0f*PI*u2;
      float R = halfbond*sqrtf(1.0f-z*z);
      conformers[6*conf  ] = R*cosf(phi);
      conformers[6*conf+1] = R*sinf(phi);
      conformers[6*conf+2] = halfbond*z;
      conformers[6*conf+3] = -conformers[6*conf  ];
      conformers[6*conf+4] = -conformers[6*conf+1];
      conformers[6*conf+5] = -conformers[6*conf+2];
    }
    
    
    int elec_factor=1;
    if (compute_elec) elec_factor=2;
    
    conformer_voldata = new float*[elec_factor*num_conformers];
    for (conf=0; conf<elec_factor*num_conformers; conf++)
      conformer_voldata[conf] = new float[gridsize];
  }
  
  return retVal;
}



/// Computes, for each gridpoint, the VdW energy to the nearest atoms
int VolMapCreateSlowEnergy::compute_frame(int frame, float *voldata) { 
  const float beta = 503.2206f*temperature; // 1/k [kcal/mol] = 1.0/(1.38066*6.022/4184) = 503.2206
  int i, n, conf;  
  DrawMolecule *mol = app->moleculeList->mol_from_id(sel->molid());
  if (!mol) return -1;
  const float *radius = mol->radius();
  if (!radius) return MEASURE_ERR_NORADII;
  const float *occupancy = mol->occupancy();
  if (!occupancy) return MEASURE_ERR_NORADII;
  const float *charge = mol->charge();
  if (!charge) return MEASURE_ERR_NORADII;
      
  int GRIDX = volmap->xsize;
  int GRIDY = volmap->ysize;
  int GRIDZ = volmap->zsize;
  int gridsize = volmap->gridsize;
     
  float min_coords[3];
  for (i=0; i<3; i++) min_coords[i] = volmap->origin[i] - \
          0.5f*(volmap->xdelta[i] + volmap->ydelta[i] + \
          volmap->zdelta[i]);

  sel->which_frame=frame;
  sel->change(NULL,mol);
  const float *coords = sel->coordinates(app->moleculeList);
  if (!coords) return -1;

  float dx, dy, dz, dist2, idist6; //to measure distances
  float vdw;
  const int steps = (int)(max_dist/delta)+1;

  const float origin_x = volmap->origin[0]; //proxies for speed increase (10%)
  const float origin_y = volmap->origin[1];
  const float origin_z = volmap->origin[2];
  
  const float cutoff2 = cutoff*cutoff;
  const float switchdist2 = switchdist*switchdist;
  const float switchfactor = 1.f/((cutoff2 - switchdist2)*(cutoff2 - switchdist2)*(cutoff2 - switchdist2));
  
  const float coulombs_constant = 331.9; // kcal A/mol/e^2

      
  if (ligand_type == MONO) {
    float elec = 0.f;   
    
    memset(voldata, 0, gridsize*sizeof(float));
    
    for (i=0; i < sel->num_atoms; i++) { 
      if (!sel->on[i]) continue; //atom is not selected
      const int gx = int ((coords[3*i] - min_coords[0])/delta);
      const int gy = int ((coords[3*i+1] - min_coords[1])/delta);
      const int gz = int ((coords[3*i+2] - min_coords[2])/delta);
     
      const float epsilon = beta * sqrtf(occupancy[i] * probe1_epsilon); //adjust per atom
      const float rmin = radius[i] + probe1_rmin; //adjust per atom
      const float coulomb = beta * coulombs_constant * probe1_charge *  charge[i];
      const float coulomb_base = coulomb/cutoff;  
        
      for (int ihz=MAX(gz-steps,0); ihz<=MIN(gz+steps,GRIDZ-1); ihz++)
      for (int ihy=MAX(gy-steps,0); ihy<=MIN(gy+steps,GRIDY-1); ihy++)
      for (int ihx=MAX(gx-steps,0); ihx<=MIN(gx+steps,GRIDX-1); ihx++) {
        n = ihx + ihy*GRIDX + ihz*GRIDY*GRIDX;
        dx = coords[3*i] - origin_x - ihx*delta;
        dy = coords[3*i+1] - origin_y - ihy*delta;
        dz = coords[3*i+2] - origin_z - ihz*delta;
        dist2 = dx*dx + dy*dy + dz*dz;
        if (dist2 <= cutoff*cutoff) { 
          idist6 = rmin*rmin/dist2;
          idist6 = idist6*idist6*idist6;
          vdw = epsilon*(idist6*idist6 - 2.*idist6);
          if (use_switching && dist2 > switchdist2)
            vdw *= switchfactor*(cutoff2 - dist2)*(cutoff2 - dist2)*(cutoff2 - 3.f*switchdist2 + 2.f*dist2);
          if (compute_elec) elec = coulomb/sqrtf(dist2)-coulomb_base; //coulomb/sqrtf(dist2);
          voldata[n] += vdw + elec; //Lennard-Jones
        }
      }
    }
  }
  else if (ligand_type == DIHOMO) { // if ligand_type != MONO
    float elec = 0.f; 
    int elec_numconf=1;
    if (compute_elec) elec_numconf=2;
    
    for (conf=0; conf<elec_numconf*num_conformers; conf++)
      memset(conformer_voldata[conf], 0, gridsize*sizeof(float));

    float *confgrid, *confgrid2; // voldata for a given diatomic conformer
        
    for (i=0; i<sel->num_atoms; i++) { 
      if (!sel->on[i]) continue; //atom is not selected
      int gx = int ((coords[3*i] - min_coords[0])/delta);
      int gy = int ((coords[3*i+1] - min_coords[1])/delta);
      int gz = int ((coords[3*i+2] - min_coords[2])/delta);
     
      float epsilon = beta * sqrtf(occupancy[i] * probe1_epsilon); //adjust per atom
      float rmin = radius[i] + probe1_rmin; //adjust per atom
      float coulomb = beta * coulombs_constant * probe1_charge * charge[i];
      float px, py, pz;
      float totalvdw;
    
      for (int ihz=MAX(gz-steps,0); ihz<=MIN(gz+steps,GRIDZ-1); ihz++)
      for (int ihy=MAX(gy-steps,0); ihy<=MIN(gy+steps,GRIDY-1); ihy++)
      for (int ihx=MAX(gx-steps,0); ihx<=MIN(gx+steps,GRIDX-1); ihx++) {
        px = coords[3*i] - origin_x - ihx*delta;
        py = coords[3*i+1] - origin_y - ihy*delta;
        pz = coords[3*i+2] - origin_z - ihz*delta;

        dist2 = px*px + py*py + pz*pz;    
        if (dist2 > cutoff2) continue;
               
        // looping over conf (conformers) after looping over i (selection atoms) results in a ~15% speed increase,
        // but requires num_conf/2 times more memory...:   
        for (conf=0; conf<num_conformers; conf++) { 
          confgrid = conformer_voldata[conf];
          confgrid2 = conformer_voldata[conf+num_conformers];
          n = ihx + ihy*GRIDX + ihz*GRIDY*GRIDX;
          
          if (compute_elec) {
            if (confgrid[n] > MAX_ENERGY && confgrid2[n] > MAX_ENERGY) continue;
          }
          else if (confgrid[n] > MAX_ENERGY) continue;
           
          elec = 0.f;
          totalvdw = 0.f;

          // first atom
          dx = px - conformers[6*conf];
          dy = py - conformers[6*conf+1];
          dz = pz - conformers[6*conf+2];
          dist2 = dx*dx + dy*dy + dz*dz;
         // if (dist2 <= cutoff2) { 
            idist6 = rmin*rmin/dist2;
            idist6 = idist6*idist6*idist6;
            vdw = epsilon*(idist6*idist6 - 2.*idist6);
            if (use_switching && dist2 > switchdist2)
              vdw *= switchfactor*(cutoff2 - dist2)*(cutoff2 - dist2)*(cutoff2 - 3.f*switchdist2 + 2.f*dist2);
            if (compute_elec) elec += coulomb/sqrtf(dist2);
            if (vdw == vdw) totalvdw += vdw; //Lennard-Jones
            else totalvdw += MAX_ENERGY;
         // }
        
          // second atom
          dx = px - conformers[6*conf+3];
          dy = py - conformers[6*conf+4];
          dz = pz - conformers[6*conf+5];
          dist2 = dx*dx + dy*dy + dz*dz;
          //if (dist2 <= cutoff2) { 
            idist6 = rmin*rmin/dist2;
            idist6 = idist6*idist6*idist6;
            vdw = epsilon*(idist6*idist6 - 2.*idist6);
            if (use_switching && dist2 > switchdist2)
              vdw *= switchfactor*(cutoff2 - dist2)*(cutoff2 - dist2)*(cutoff2 - 3.f*switchdist2 + 2.f*dist2);
            if (compute_elec) elec -= coulomb/sqrtf(dist2);
            if (vdw == vdw) totalvdw += vdw; //Lennard-Jones
            else totalvdw += MAX_ENERGY;
         // }
          
          if (compute_elec) {// average both orientations
            confgrid[n]  += totalvdw + elec;
            confgrid2[n] += totalvdw - elec;
          }
          else
            confgrid[n] += totalvdw;
        }
      }
    }
    
    float density;
    for (n=0; n<gridsize; n++) {
      density = 0.f;
      for (conf=0; conf<elec_numconf*num_conformers; conf++) {
        confgrid = conformer_voldata[conf];
        density += expf(-confgrid[n]);
      }
      voldata[n] = -logf(density/(elec_numconf*num_conformers));  // conformational pmf in kT
    }
    
  }
  else if (ligand_type == DIHETERO) { // CO, NO, etc.
    float elec = 0.f; 
    int elec_numconf=1;
    
    for (conf=0; conf<elec_numconf*num_conformers; conf++)
      memset(conformer_voldata[conf], 0, gridsize*sizeof(float));

    float *confgrid; // voldata for a given diatomic conformer
        
    for (i=0; i<sel->num_atoms; i++) { 
      if (!sel->on[i]) continue; //atom is not selected
      int gx = int ((coords[3*i] - min_coords[0])/delta);
      int gy = int ((coords[3*i+1] - min_coords[1])/delta);
      int gz = int ((coords[3*i+2] - min_coords[2])/delta);
     
      float epsilon1 = beta * sqrtf(occupancy[i] * probe1_epsilon); //adjust per atom
      float epsilon2 = beta * sqrtf(occupancy[i] * probe2_epsilon); //adjust per atom
      float rmin1 = radius[i] + probe1_rmin; //adjust per atom
      float rmin2 = radius[i] + probe2_rmin; //adjust per atom
      float coulomb = beta * coulombs_constant * probe1_charge * charge[i];
      float px, py, pz;
      float totalvdw;
    
      for (int ihz=MAX(gz-steps,0); ihz<=MIN(gz+steps,GRIDZ-1); ihz++)
      for (int ihy=MAX(gy-steps,0); ihy<=MIN(gy+steps,GRIDY-1); ihy++)
      for (int ihx=MAX(gx-steps,0); ihx<=MIN(gx+steps,GRIDX-1); ihx++) {
        px = coords[3*i] - origin_x - ihx*delta;
        py = coords[3*i+1] - origin_y - ihy*delta;
        pz = coords[3*i+2] - origin_z - ihz*delta;

        // Note: We need to compate the _center_ of the molecule to the cutoff, because of electrostatic convergence effects
        dist2 = px*px + py*py + pz*pz;    
        if (dist2 > cutoff2) continue;
               
        // looping over conf (conformers) after looping over i (selection atoms) results in a ~15% speed increase,
        // but requires num_conf/2 times more memory...:   
        for (conf=0; conf<num_conformers; conf++) { 
          confgrid = conformer_voldata[conf];
          n = ihx + ihy*GRIDX + ihz*GRIDY*GRIDX;
          
          if (compute_elec) {
            if (confgrid[n] > MAX_ENERGY) continue;
          }
          else if (confgrid[n] > MAX_ENERGY) continue;
           
          elec = 0.f;
          totalvdw = 0.f;

          // first atom
          dx = px - conformers[6*conf];
          dy = py - conformers[6*conf+1];
          dz = pz - conformers[6*conf+2];
          dist2 = dx*dx + dy*dy + dz*dz;
         // if (dist2 <= cutoff2) { 
            idist6 = rmin1*rmin1/dist2;
            idist6 = idist6*idist6*idist6;
            vdw = epsilon1*(idist6*idist6 - 2.*idist6);
            if (use_switching && dist2 > switchdist2)
              vdw *= switchfactor*(cutoff2 - dist2)*(cutoff2 - dist2)*(cutoff2 - 3.f*switchdist2 + 2.f*dist2);
            if (compute_elec) elec += coulomb/sqrtf(dist2);
            if (vdw == vdw) totalvdw += vdw; //Lennard-Jones
            else totalvdw += MAX_ENERGY;
         // }
        
          // second atom
          dx = px - conformers[6*conf+3];
          dy = py - conformers[6*conf+4];
          dz = pz - conformers[6*conf+5];
          dist2 = dx*dx + dy*dy + dz*dz;
          //if (dist2 <= cutoff2) { 
            idist6 = rmin2*rmin2/dist2;
            idist6 = idist6*idist6*idist6;
            vdw = epsilon2*(idist6*idist6 - 2.*idist6);
            if (use_switching && dist2 > switchdist2)
              vdw *= switchfactor*(cutoff2 - dist2)*(cutoff2 - dist2)*(cutoff2 - 3.f*switchdist2 + 2.f*dist2);
            if (compute_elec) elec -= coulomb/sqrtf(dist2);
            if (vdw == vdw) totalvdw += vdw; //Lennard-Jones
            else totalvdw += MAX_ENERGY;
         // }
          
          if (compute_elec) {
            confgrid[n]  += totalvdw + elec;
          }
          else
            confgrid[n] += totalvdw;
        }
      }
    }
    
       
    float density;
    for (n=0; n<gridsize; n++) {
      density = 0.f;
      for (conf=0; conf<elec_numconf*num_conformers; conf++) {
        confgrid = conformer_voldata[conf];
        density += expf(-confgrid[n]); // convert kcal/mol -> kT
      }
      voldata[n] = -logf(density/(elec_numconf*num_conformers));  // conformational pmf in kT
    }
    
  }
  
  return MEASURE_NOERR; 

}


int VolMapCreateSlowEnergy::compute_end () {
  if (conformers) delete[] conformers;
  
  int elec_factor=1;
  if (compute_elec) elec_factor=2;
    
  if (conformer_voldata) {
    for (int conf=0; conf<elec_factor*num_conformers; conf++)
      delete[] conformer_voldata[conf];
    delete[] conformer_voldata;
    conformer_voldata = NULL;
  }
    
  return 0;
}




/////// VolMapCreateFastEnergy ///////

VolMapCreateFastEnergy::VolMapCreateFastEnergy(VMDApp *app, AtomSel *sel, float res, float epsilon, float rmin, float the_cutoff) : VolMapCreate(app, sel, res) {
    cutoff = the_cutoff;
    ligand_type = MONO;
    num_conformers = 20;
    
    probe1_epsilon = epsilon;
    probe1_rmin = rmin;
    probe1_charge = 0.f;
    probe_bondlength = 1.12f; 
    probe2_epsilon = epsilon;
    probe2_rmin = rmin;
    probe2_charge = 0.f;
    temperature = 300.f;
    
    switchdist = 10.;
    use_switching = false;
    //if (cutoff >= 11.) use_switching = true;
    
    mincutoff = 4.f;
    
    max_dist = cutoff + probe_bondlength/2.;
    
    //settable params
    param_subres = -1;
}


int VolMapCreateFastEnergy::compute_init () {
  MYPRINTF("\nThis computes the potential of mean force (free energy) over\n"
         "a grid, for an implicit monoatomic or diatomic ligand\n"
         "If you use this method in your work, please cite:\n\n"
         "  J COHEN, A ARKHIPOV, R BRAUN and K SCHULTEN, \"Imaging the\n"
         "  migration pathways for O2, CO, NO, and Xe inside myoglobin\".\n"
         "  Biophysical Journal 91:1844-1857, 2006.\n\n");
  MYPRINTF("Using temperature T=%g\n", temperature);
  
  compute_elec = (probe1_charge != 0.f); // compute electrostatics?
    
  //init params
  if (ligand_type == MONO) {
    num_conformers = 0;
    probe_bondlength = 0.f;
  }
    
  if (ligand_type == MONO) 
    subres = 3;
  else {
    subres = 1;    
    float maxsuboffest = 0.5f*delta*sqrtf(3.f);
    if (mincutoff > cutoff) mincutoff = cutoff;
    cutoff += probe_bondlength/2.f;
    mincutoff += probe_bondlength/2.f + maxsuboffest;
    //if (compute_elec) mincutoff += 4.f;
  }
  
  if (param_subres > 0) subres = param_subres;

  max_dist = cutoff;
  
  //go

  char tmpstr[255];
  sprintf(tmpstr, "ligand pmf (%.200s) [kT]", sel->cmdStr);
  volmap->set_name(tmpstr);
  volmap->temperature=temperature;

  int retVal = VolMapCreate::compute_init(1.-cutoff);
  int gridsize = volmap->gridsize;

   
  //1. Create diatomic conformers
  if (ligand_type == MONO) {
    MYPRINTF("Using a monoatomic ligand: epsilon=%g, rmin/2=%g, charge=%g\n", probe1_epsilon, probe1_rmin, probe1_charge);
    MYPRINTF("Electrostatics: ");
    if (compute_elec) MYPRINTF("on\n");
    else MYPRINTF("off\n");
    MYPRINTF("Params: mincutoff/cutoff=%g/%g, subres=%d\n", mincutoff, cutoff, subres);
    conformers = NULL;
    conformer_voldata = NULL;
  }
  else {
    MYPRINTF("Using a diatomic ligand:\n");
    MYPRINTF("  atom1: epsilon=%g, rmin/2=%g, charge=%g\n", probe1_epsilon, probe1_rmin, probe1_charge);
    MYPRINTF("  atom2: epsilon=%g, rmin/2=%g, charge=%g\n", probe2_epsilon, probe2_rmin, probe2_charge);
    MYPRINTF("  bondlength=%g\n", probe_bondlength);
    MYPRINTF("Electrostatics: ");
    if (compute_elec) MYPRINTF("on\n");
    else MYPRINTF("off\n");
    MYPRINTF("Params: mincutoff/cutoff=%g/%g, subres=%d\n", mincutoff, cutoff, subres);
    MYPRINTF("Generating %d conformers\n", num_conformers);
    conformers = new float[6*num_conformers];
    memset(conformers, 0, 6*num_conformers*sizeof(float));
  
    // generate randomly oriented conformers
    const float halfbond = probe_bondlength/2.f;
    const float RAND_MAX_INV = 1.0f/VMD_RAND_MAX;
    int conf;
    for (conf=0; conf<num_conformers; conf++) {
      float u1 = (float) vmd_random()*RAND_MAX_INV;
      float u2 = (float) vmd_random()*RAND_MAX_INV;
      float z = (2.0f*u1 -1.0f);
      float phi = 2.0f*PI*u2;
      float R = halfbond*sqrtf(1.0f-z*z);
      conformers[6*conf  ] = R*cosf(phi);
      conformers[6*conf+1] = R*sinf(phi);
      conformers[6*conf+2] = halfbond*z;
      conformers[6*conf+3] = -conformers[6*conf  ];
      conformers[6*conf+4] = -conformers[6*conf+1];
      conformers[6*conf+5] = -conformers[6*conf+2];
    }
    
    int elec_factor = 1;
    if (compute_elec && ligand_type == DIHOMO) elec_factor = 2;
    
    conformer_voldata = new float*[elec_factor*num_conformers];
    for (conf=0; conf<elec_factor*num_conformers; conf++)
      conformer_voldata[conf] = new float[gridsize];
    
  }

  return retVal;
}




/// Computes, for each gridpoint, the VdW energy to the nearest atoms
int VolMapCreateFastEnergy::compute_frame(int frame, float *voldata) { 
  const float beta = 503.2206f/temperature; // 1/k [kcal/mol] = 1.0/(1.38066*6.022/4184) = 503.2206
  int i, n, conf;  
  DrawMolecule *mol = app->moleculeList->mol_from_id(sel->molid());
  if (!mol) return -1;
  const float *radius = mol->extra.data("radius");
  if (!radius) return MEASURE_ERR_NORADII;
  const float *occupancy = mol->extra.data("occupancy");
  if (!occupancy) return MEASURE_ERR_NORADII;
  const float *charge = mol->extra.data("charge");
  if (!charge) return MEASURE_ERR_NORADII;
  
  const float probe1_epsilonsqrt = beta*sqrtf(-probe1_epsilon);  // include beta
  const float probe2_epsilonsqrt = beta*sqrtf(-probe2_epsilon);  // include beta
  float *epsilonlist = new float[sel->num_atoms];
  for (i=0; i<sel->num_atoms; i++) {
    epsilonlist[i] = sqrtf(-occupancy[i]);
  }  
  
  int GRIDX = volmap->xsize;
  int GRIDY = volmap->ysize;
  int GRIDZ = volmap->zsize;
  int gridsize = volmap->gridsize;
     
  float min_coords[3];
  for (i=0; i<3; i++) min_coords[i] = volmap->origin[i] - \
          0.5f*(volmap->xdelta[i] + volmap->ydelta[i] + \
          volmap->zdelta[i]);

  sel->which_frame=frame;
  sel->change(NULL,mol);
  const float *coords = sel->coordinates(app->moleculeList);
  if (!coords) return -1;

  float dx, dy, dz, dist2, idist6; //to measure distances
  float vdw;
  const int steps = (int)(max_dist/delta)+1;
  
  const float cutoff2 = cutoff*cutoff;
  const float mincutoff2 = mincutoff*mincutoff;
  const float switchdist2 = switchdist*switchdist;
  const float switchfactor = 1.f/((cutoff2 - switchdist2)*(cutoff2 - switchdist2)*(cutoff2 - switchdist2));
  
  const float coulombs_constant = 331.9; // kcal A/mol/e^2

  memset(voldata, 0, gridsize*sizeof(float));
  
  //Build atom pairlists for interactions
  //quite memory intensive but _MUCH_ faster than normal way!!
  
  float *gridpos = new float[3*gridsize];
  int *gridon = new int[gridsize]; 
  for (n=0; n<gridsize; n++) {
    gridpos[3*n] = (n%GRIDX)*delta + volmap->origin[0]; //position of grid cell's center
    gridpos[3*n+1] = ((n/GRIDX)%GRIDY)*delta + volmap->origin[1];
    gridpos[3*n+2] = (n/(GRIDX*GRIDY))*delta + volmap->origin[2]; 
    gridon[n] = 1;
  }
  float *gridpos_ref = new float[3*gridsize]; 
  memcpy(gridpos_ref, gridpos, 3*gridsize*sizeof(float));
        
  GridSearchPair *pairlist, *p;
      
  pairlist = vmd_gridsearch3(gridpos, gridsize, gridon, coords,
                           sel->num_atoms, sel->on, mincutoff, true, -1);

  GridSearchPair **parray, *pnext, *p2;
  parray = new GridSearchPair*[gridsize];
  memset(parray, 0, gridsize*sizeof(GridSearchPair*));
  
  // Bin the pairlist for speed (100000x speedup, no kidding!!!)
  for (p=pairlist; p; p=pnext) {
    n = p->ind1;
    
    // this will copy a whole block at a time (speeds things up)
    for (p2=p; (p2->next && p2->next->ind1 == n); p2=p2->next) ;
    pnext = p2->next;
    p2->next=NULL;
    
    if (!parray[n]) parray[n] = p;
    else {
      for (p2=parray[n]; p2->next; p2=p2->next) ; 
      p2->next = p;
    }
  }
  
  // Build minigrid for additional per-voxel subsampling
  const int minigridsize = subres*subres*subres;
  float *minigridoffset = new float[3*minigridsize];
  float *minigrid = new float[minigridsize];

  float subdelta = delta/subres;
  float suboffest = -0.5f*delta+0.5f*subdelta;
  for (int mi=0; mi<subres; mi++)
  for (int mj=0; mj<subres; mj++)
  for (int mk=0; mk<subres; mk++) {
    minigridoffset[3*(mi+subres*mj+subres*subres*mk)] = suboffest + mi*subdelta;
    minigridoffset[3*(mi+subres*mj+subres*subres*mk)+1] = suboffest + mj*subdelta;
    minigridoffset[3*(mi+subres*mj+subres*subres*mk)+2] = suboffest + mk*subdelta;
  }
    
  if (ligand_type == MONO) {
    float elec=0.f;
    
    // This code calculates the subsmapling for r<mincutoff
    for (n=0; n<gridsize; n++) {
      memset(minigrid, 0, minigridsize*sizeof(float));
      for (p=parray[n]; p; p=p->next) {
        i = p->ind2;
        float epsilon = epsilonlist[i] * probe1_epsilonsqrt; //adjust per atom
        float coulomb = beta * coulombs_constant * probe1_charge * charge[i];
        float rmin = radius[i] + probe1_rmin; //adjust per atom
        
        for (int m=0; m<minigridsize; m++) {
          if (minigrid[m] > MAX_ENERGY) continue;
          dx = gridpos[3*n] + minigridoffset[3*m] - coords[3*i];
          dy = gridpos[3*n+1] + minigridoffset[3*m+1] - coords[3*i+1];
          dz = gridpos[3*n+2] + minigridoffset[3*m+2] - coords[3*i+2];
      
          dist2 = dx*dx+dy*dy+dz*dz;
          idist6 = rmin*rmin/dist2;
          idist6 = idist6*idist6*idist6;
          vdw = epsilon*(idist6*idist6 - 2.*idist6);
          if (compute_elec) elec = coulomb/sqrtf(dist2);
          minigrid[m] += vdw + elec; //Lennard-Jones
        }
      }
      
      float Z = 0.f;
      for (int m=0; m<minigridsize; m++)
        if (minigrid[m] < MAX_ENERGY)
          Z += expf(-minigrid[m]);
      
      if (Z > 0.f) voldata[n] = -logf(Z/minigridsize);
      else voldata[n] = MAX_ENERGY;

    }

    if (mincutoff < cutoff)
    for (i=0; i < sel->num_atoms; i++) { 
      if (!sel->on[i]) continue; //atom is not selected
      int gx = int ((coords[3*i] - min_coords[0])/delta);
      int gy = int ((coords[3*i+1] - min_coords[1])/delta);
      int gz = int ((coords[3*i+2] - min_coords[2])/delta);
     
      float epsilon = epsilonlist[i] * probe1_epsilonsqrt; //adjust per atom
      float coulomb = beta*coulombs_constant * probe1_charge * charge[i];
      float rmin = radius[i] + probe1_rmin; //adjust per atom
      
      for (int ihz=MAX(gz-steps,0); ihz<=MIN(gz+steps,GRIDZ-1); ihz++)
      for (int ihy=MAX(gy-steps,0); ihy<=MIN(gy+steps,GRIDY-1); ihy++)
      for (int ihx=MAX(gx-steps,0); ihx<=MIN(gx+steps,GRIDX-1); ihx++) {
        n = ihx + ihy*GRIDX + ihz*GRIDY*GRIDX;
        if (voldata[n] > MAX_ENERGY) continue; // if energy is huge, don't waste more time computing it
        dx = coords[3*i] - gridpos[3*n];
        dy = coords[3*i+1] - gridpos[3*n+1];
        dz = coords[3*i+2] - gridpos[3*n+2];
        dist2 = dx*dx + dy*dy + dz*dz;
        if (dist2 <= cutoff2 && dist2 > mincutoff2) { 
          idist6 = rmin*rmin/dist2;
          idist6 = idist6*idist6*idist6;
          vdw = epsilon*(idist6*idist6 - 2.*idist6);
          if (use_switching && dist2 > switchdist2)
            vdw *= switchfactor*(cutoff2 - dist2)*(cutoff2 - dist2)*(cutoff2 - 3.f*switchdist2 + 2.f*dist2);
          if (compute_elec) elec = coulomb/sqrtf(dist2);
          else elec = 0.f;
          voldata[n] += vdw + elec; //Lennard-Jones
        }
      }
    } 
  }
  else { // if ligand_type = DIHOMO, DIHETERO
    int elecfactor = 1;
    if (compute_elec && ligand_type == DIHOMO) elecfactor = 2;  // this means we will compute swapped charges as well
    float px, py, pz;
    float elec=0.f;
    float *confdata = new float[elecfactor*num_conformers*minigridsize];
    const float halfbond = probe_bondlength/2.f;
    float *efield = NULL;  //electric force
    if (compute_elec) {
      efield = new float[3*gridsize];
      memset(efield, 0, 3*gridsize*sizeof(float));
    }
      
    // Here, only calculate long-distance VDW interactions, as well as E-field grid...
    // XXXX Currently electrostatics results here do not match the slow energy
    // one (i.e. do not use electrostatics (yet)!!)
    if (mincutoff < cutoff)
    for (i=0; i < sel->num_atoms; i++) { 
      if (!sel->on[i]) continue; //atom is not selected
      int gx = int ((coords[3*i] - min_coords[0])/delta);
      int gy = int ((coords[3*i+1] - min_coords[1])/delta);
      int gz = int ((coords[3*i+2] - min_coords[2])/delta);
     
      float epsilon1 = epsilonlist[i] * probe1_epsilonsqrt; //adjust per atom
      float rmin1 = radius[i] + probe1_rmin; //adjust per atom
      float epsilon2 = epsilonlist[i] * probe2_epsilonsqrt; //adjust per atom
      float rmin2 = radius[i] + probe2_rmin; //adjust per atom
      float coulomb = beta * coulombs_constant * probe1_charge * charge[i];
                    
      for (int ihz=MAX(gz-steps,0); ihz<=MIN(gz+steps,GRIDZ-1); ihz++)
      for (int ihy=MAX(gy-steps,0); ihy<=MIN(gy+steps,GRIDY-1); ihy++)
      for (int ihx=MAX(gx-steps,0); ihx<=MIN(gx+steps,GRIDX-1); ihx++) {
        n = ihx + ihy*GRIDX + ihz*GRIDY*GRIDX;
        //if (voldata[n] > MAX_ENERGY) continue; // if energy is huge, don't waste more time computing it
        px = coords[3*i] - gridpos[3*n];
        py = coords[3*i+1] - gridpos[3*n+1];
        pz = coords[3*i+2] - gridpos[3*n+2];
        dist2 = px*px + py*py + pz*pz;
        if (dist2 > cutoff2 || dist2 <= mincutoff2) continue; 
  
        if (compute_elec) {
          // XXX TODO Calculate E-field for dipole calculations...
          // XXX speed this all up by inlining
          float idist = 1.f/sqrtf(dist2);
          float E = coulomb/dist2;
          
          efield[3*n]   += E*px*idist; // this is really the coulomb *Force*, not field...
          efield[3*n+1] += E*py*idist;
          efield[3*n+2] += E*pz*idist;
        }
        
        // NOTE: here, instead of using conformers, we attempt to use the energy from a 
        // tetrahedral distribution (4 atoms), divided by 2 (since 2 molecules). This should
        // provide a good average energy (err = 0.002kT for chargeless O2 in water)

        // This has not been tested for heterogeneous diatomic molecules
        
        float A = halfbond/sqrtf(3.f);
        float total_E = 0.f;

        dx = px + A;
        dy = py + A;
        dz = pz + A;        
        dist2 = dx*dx + dy*dy + dz*dz;
        idist6 = rmin1*rmin1/dist2;
        idist6 = idist6*idist6*idist6;
        vdw = epsilon1*(idist6*idist6 - 2.*idist6);
        if (use_switching && dist2 > switchdist2)
          vdw *= switchfactor*(cutoff2 - dist2)*(cutoff2 - dist2)*(cutoff2 - 3.f*switchdist2 + 2.f*dist2);
        total_E += vdw; //Lennard-Jones
      
        dx = px + A;
        dy = py - A;
        dz = pz - A;        
        dist2 = dx*dx + dy*dy + dz*dz;
        idist6 = rmin2*rmin2/dist2;
        idist6 = idist6*idist6*idist6;
        vdw = epsilon2*(idist6*idist6 - 2.*idist6);
        if (use_switching && dist2 > switchdist2)
          vdw *= switchfactor*(cutoff2 - dist2)*(cutoff2 - dist2)*(cutoff2 - 3.f*switchdist2 + 2.f*dist2);
        total_E += vdw; //Lennard-Jones
      
        dx = px - A;
        dy = py + A;
        dz = pz - A;        
        dist2 = dx*dx + dy*dy + dz*dz;
        idist6 = rmin1*rmin1/dist2;
        idist6 = idist6*idist6*idist6;
        vdw = epsilon1*(idist6*idist6 - 2.*idist6);
        if (use_switching && dist2 > switchdist2)
          vdw *= switchfactor*(cutoff2 - dist2)*(cutoff2 - dist2)*(cutoff2 - 3.f*switchdist2 + 2.f*dist2);
        total_E += vdw; //Lennard-Jones
      
        dx = px - A;
        dy = py - A;
        dz = pz + A;        
        dist2 = dx*dx + dy*dy + dz*dz;
        idist6 = rmin2*rmin2/dist2;
        idist6 = idist6*idist6*idist6;
        vdw = epsilon2*(idist6*idist6 - 2.*idist6);
        if (use_switching && dist2 > switchdist2)
          vdw *= switchfactor*(cutoff2 - dist2)*(cutoff2 - dist2)*(cutoff2 - 3.f*switchdist2 + 2.f*dist2);
        total_E += vdw; //Lennard-Jones

        voldata[n] += 0.5f*total_E;          
      }
    }
    
    
    // This code calculates the subsampling for r<mincutoff
    for (n=0; n<gridsize; n++) {
      memset(confdata, 0, elecfactor*num_conformers*minigridsize*sizeof(float));
      
      //long-range electrostatics (approximated by dipole interaction)
      if (compute_elec)
      for (conf=0; conf<num_conformers; conf++) {
        // should really use (conf1-conf2) instead of 2*conf1, U = + q1*(X2-X1)*E
        float longrange_elec = 2.f*dot_prod(conformers+6*conf, efield+3*n); // XXX CHECK SIGN!!
        for (int m=0; m<minigridsize; m++) {
          confdata[conf*minigridsize + m] = longrange_elec;
          if (elecfactor > 1) confdata[(num_conformers+conf)*minigridsize + m] = -longrange_elec;
        }
      }
            
      for (p=parray[n]; p; p=p->next) {
        i = p->ind2;
        float epsilon1 = epsilonlist[i] * probe1_epsilonsqrt; //adjust per atom
        float rmin1 = radius[i] + probe1_rmin; //adjust per atom
        float epsilon2 = epsilonlist[i] * probe2_epsilonsqrt; //adjust per atom
        float rmin2 = radius[i] + probe2_rmin; //adjust per atom
        float coulomb = beta*coulombs_constant * probe1_charge * charge[i];

        // is the following redundant?
        float px = gridpos[3*n] - coords[3*i];
        float py = gridpos[3*n+1] - coords[3*i+1];
        float pz = gridpos[3*n+2] - coords[3*i+2];
        dist2 = px*px+py*py+pz*pz;        
        if (dist2 >= mincutoff2) continue;
                          
        for (int m=0; m<minigridsize; m++) {
          float px = gridpos[3*n] + minigridoffset[3*m] - coords[3*i];
          float py = gridpos[3*n+1] + minigridoffset[3*m+1] - coords[3*i+1];
          float pz = gridpos[3*n+2] + minigridoffset[3*m+2] - coords[3*i+2];
          
          for (conf=0; conf<num_conformers; conf++) {
            if (confdata[conf*minigridsize + m] > MAX_ENERGY) continue;
            elec = 0.f;
            
            dx = px + conformers[6*conf];
            dy = py + conformers[6*conf+1];
            dz = pz + conformers[6*conf+2];
            dist2 = dx*dx+dy*dy+dz*dz;
            
            idist6 = rmin1*rmin1/dist2;
            idist6 = idist6*idist6*idist6;
            if (compute_elec) elec += coulomb/sqrtf(dist2);
            vdw = epsilon1*(idist6*idist6 - 2.*idist6);
            
            dx = px + conformers[6*conf+3];
            dy = py + conformers[6*conf+4];
            dz = pz + conformers[6*conf+5];
            dist2 = dx*dx+dy*dy+dz*dz;

            idist6 = rmin2*rmin2/dist2;
            idist6 = idist6*idist6*idist6;
            if (compute_elec) elec -= coulomb/sqrtf(dist2);
            vdw += epsilon2*(idist6*idist6 - 2.*idist6);

            if (compute_elec) { // average both orientations
              confdata[conf*minigridsize+m] += vdw + elec;
              if (elecfactor > 1) confdata[num_conformers*minigridsize + conf*minigridsize+m] += vdw - elec;
            }
            else
              confdata[conf*minigridsize+m] += vdw;
          }
        }
      }
      
      float Z = 0.f;
      for (conf=0; conf<elecfactor*num_conformers*minigridsize; conf++)
        if (confdata[conf] < MAX_ENERGY)
          Z += expf(-confdata[conf]);
      
      if (Z > 0.f) voldata[n] += -logf(Z/(elecfactor*num_conformers*minigridsize));
      else voldata[n] = MAX_ENERGY;
      
      if (voldata[n] > MAX_ENERGY) voldata[n] = MAX_ENERGY;  // clamp energies to avoid infs
    } 

        
    if (compute_elec) delete[] efield;    
    delete[] confdata;

  } // end of ligand = DIHOMO, DIHETERO
  
  
  // delete pairlists
  for (n=0; n<gridsize; n++) {
    for (p=parray[n]; p;) {
      GridSearchPair *tmp = p;
      p = p->next;
      free(tmp);
    }
  }
  
  delete[] minigridoffset;
  delete[] minigrid;
  delete[] parray;
    
  delete[] epsilonlist;
  delete[] gridpos; 
  delete[] gridon; 
  delete[] gridpos_ref;
  
  return MEASURE_NOERR; 
}


int VolMapCreateFastEnergy::compute_end () {
  if (conformers) delete[] conformers;
  
  if (conformer_voldata) {
    for (int conf=0; conf<num_conformers; conf++)
      delete[] conformer_voldata[conf];
    delete[] conformer_voldata;
    conformer_voldata = NULL;
  }
    
  return 0;
}


/////// VolMapCreateCoulombPotential ///////
  
int VolMapCreateCoulombPotential::compute_init () {
  char tmpstr[255];
  sprintf(tmpstr, "Potential (kT/e at 298.15K) (%.200s)", sel->cmdStr);
  volmap->set_name(tmpstr);
  
  float max_rad;
  calculate_max_radius(max_rad);
    
  return VolMapCreate::compute_init(10.f);
}


int VolMapCreateCoulombPotential::compute_frame(int frame, float *voldata) {
  DrawMolecule *mol = app->moleculeList->mol_from_id(sel->molid());
  if (!mol) return -1;
    int i;
    
  const float *charge = mol->extra.data("charge");
  if (!charge) return MEASURE_ERR_NORADII; // XXX fix this later

  int gridsize=volmap->xsize*volmap->ysize*volmap->zsize;

  // create volumetric density grid
  memset(voldata, 0, gridsize*sizeof(float));
  sel->which_frame=frame;
  sel->change(NULL,mol);
  const float *coords = sel->coordinates(app->moleculeList);
  if (!coords) return -1;
 
  float min_coords[3];
  for (i=0; i<3; i++) 
    min_coords[i] = volmap->origin[i] - 0.5f*(volmap->xdelta[i] +
                                              volmap->ydelta[i] +
                                              volmap->zdelta[i]);

  // copy selected atom coordinates and charges to a contiguous memory
  // buffer and translate them to the starting corner of the map.
  float *xyzq = (float *) malloc(sel->selected * 4 * sizeof(float));
  float *curatom = xyzq;
  for (i=0; i < sel->num_atoms; i++) { 
    if (sel->on[i]) {
      curatom[0] = coords[3*i  ] - min_coords[0];
      curatom[1] = coords[3*i+1] - min_coords[1];
      curatom[2] = coords[3*i+2] - min_coords[2];
      curatom[3] = charge[i] * POT_CONV;
      curatom += 4;
    }
  }

  vol_cpotential(sel->selected, xyzq, voldata, 
                 volmap->zsize, volmap->ysize, volmap->xsize, delta);

  free(xyzq);
 
  return 0;
}  


