/***************************************************************************
 *cr
 *cr            (C) Copyright 1995-2007 The Board of Trustees of the
 *cr                        University of Illinois
 *cr                         All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: VolMap.C,v $
 *      $Author: johns $       $Locker:  $             $State: Exp $
 *      $Revision: 1.11 $       $Date: 2007/01/12 20:08:35 $
 *
 **************************************************************************/

#include <errno.h>
#include <math.h>
#include <stdio.h>
#include <string.h>

#include "utilities.h"   
#include "VolMap.h"   
#include "VolumetricData.h"
#include "Inform.h"

#define MIN(X,Y) (((X)<(Y))? (X) : (Y))
#define MAX(X,Y) (((X)>(Y))? (X) : (Y))

const float vmd_NAN = sqrtf(-1.f); // a portable NAN!

/// constructor

VolMap::VolMap() {
  xsize = 0;
  ysize = 0;
  zsize = 0;
  data = NULL;
  owns_data = true;
  dataname = NULL;
  weight = 1.f;
  temperature = -1.0;
}

/// constructor that wraps a VolumetricData structure inside a VolMap cloak

VolMap::VolMap(const VolumetricData *voldata) {
  xsize = voldata->xsize;
  ysize = voldata->ysize;
  zsize = voldata->zsize;
  data = voldata->data;
  owns_data = false;
  for (int d=0; d<3; d++) {
    origin[d] = (float) voldata->origin[d];
     xaxis[d] = (float) voldata->xaxis[d];
     yaxis[d] = (float) voldata->yaxis[d]; 
     zaxis[d] = (float) voldata->zaxis[d]; 
    xdelta[d] = xaxis[d]/(xsize-1);
    ydelta[d] = yaxis[d]/(ysize-1);
    zdelta[d] = zaxis[d]/(zsize-1);
  }
  gridsize = xsize*ysize*zsize;
  dataname = NULL;
  set_name(voldata->name);
}

/// destructor
VolMap::~VolMap() {
  if (owns_data && data) delete[] data;
  if (dataname) delete[] dataname;
}


void VolMap::update_internal() {
  for (int d=0; d<3; d++) {
    xaxis[d] = xdelta[d]*(xsize-1);
    yaxis[d] = ydelta[d]*(ysize-1);
    zaxis[d] = zdelta[d]*(zsize-1);
  }
  gridsize = xsize*ysize*zsize;
}


// Copy everything, including data*
void VolMap::clone(const VolMap *volmap) {
  xsize = volmap->xsize;
  ysize = volmap->ysize;
  zsize = volmap->zsize;
  
  for (int d=0; d<3; d++) {
    origin[d] = volmap->origin[d];
    xdelta[d] = volmap->xaxis[d]/(xsize-1);
    ydelta[d] = volmap->yaxis[d]/(ysize-1);
    zdelta[d] = volmap->zaxis[d]/(zsize-1);
  }
  
  update_internal();
  
  if (volmap->data) {
    data = new float[gridsize];
    memcpy(data, volmap->data, gridsize*sizeof(float));
  }
  else data = NULL;


  dataname = NULL;
  set_name(volmap->get_name());
}


/// Makes a copy of newname and stores it internally
void VolMap::set_name(const char *newname) {
  // XXX TODO makes sure that name contains no quotes
  if (dataname) delete[] dataname;
  dataname = new char[strlen(newname)+1];
  strcpy(dataname, newname);
}

const char *VolMap::get_name() const {
  if (dataname) return dataname;
  return "(no name)";
}



/// return voxel, after safely clamping index to valid range
float VolMap::voxel_value_safe(int x, int y, int z) const {
  int xx, yy, zz; 
  xx = (x > 0) ? ((x < xsize) ? x : xsize-1) : 0;
  yy = (y > 0) ? ((y < ysize) ? y : ysize-1) : 0;
  zz = (z > 0) ? ((z < zsize) ? z : zsize-1) : 0;
  int index = zz*xsize*ysize + yy*xsize + xx;
  return data[index];
}

/// return interpolated value from 8 nearest neighbor voxels
float VolMap::voxel_value_interpolate(float xv, float yv, float zv) const {
  int x = (int) xv;
  int y = (int) yv;
  int z = (int) zv;
  float xf = xv - x;
  float yf = yv - y;
  float zf = zv - z;
  float xlerps[4];
  float ylerps[2];
  float tmp;

  tmp = voxel_value_safe(x, y, z);
  xlerps[0] = tmp + xf*(voxel_value_safe(x+1, y, z) - tmp);

  tmp = voxel_value_safe(x, y+1, z);
  xlerps[1] = tmp + xf*(voxel_value_safe(x+1, y+1, z) - tmp);

  tmp = voxel_value_safe(x, y, z+1);
  xlerps[2] = tmp + xf*(voxel_value_safe(x+1, y, z+1) - tmp);

  tmp = voxel_value_safe(x, y+1, z+1);
  xlerps[3] = tmp + xf*(voxel_value_safe(x+1, y+1, z+1) - tmp);

  ylerps[0] = xlerps[0] + yf*(xlerps[1] - xlerps[0]);
  ylerps[1] = xlerps[2] + yf*(xlerps[3] - xlerps[2]);

  return ylerps[0] + zf*(ylerps[1] - ylerps[0]);
}


int VolMap::coord_to_index(float x, float y, float z) {
  x -= origin[0];
  y -= origin[1];
  z -= origin[2];
  // XXX Needs to be fixed for non-orthog cells (subtract out projected component every step)
  int gx = int((x*xaxis[0] + y*xaxis[1] + z*xaxis[2])/norm(xaxis));
  int gy = int((x*yaxis[0] + y*yaxis[1] + z*yaxis[2])/norm(yaxis));
  int gz = int((x*zaxis[0] + y*zaxis[1] + z*zaxis[2])/norm(zaxis));
  return (gx + gy*xsize + gz*ysize*xsize);
}
  
/// return value of voxel, based on atomic coords.
/// XXX need to account for non-orthog. cells
float VolMap::voxel_value_from_coord(float xpos, float ypos, float zpos) const {
  float min_coord[3];
  for (int i=0; i<3; i++) min_coord[i] = (float) origin[i] - 0.5f*(xdelta[i] + ydelta[i] + zdelta[i]);
  xpos -= min_coord[0];
  ypos -= min_coord[1];
  zpos -= min_coord[2];
  // XXX the following is very wrong for non-orthog cells.
  int gx = (int) (xpos/xdelta[0]); 
  int gy = (int) (ypos/ydelta[1]);
  int gz = (int) (zpos/zdelta[2]);
  if (gx < 0 || gx >= xsize) return vmd_NAN;
  if (gy < 0 || gy >= ysize) return vmd_NAN;
  if (gz < 0 || gz >= zsize) return vmd_NAN;
  return data[gz*xsize*ysize + gy*xsize + gx];
}


/// return interpolated value of voxel, based on atomic coords.
/// XXX need to account for non-orthog. cells
float VolMap::voxel_value_interpolate_from_coord(float xpos, float ypos, float zpos) const {
  xpos = (xpos-(float)origin[0])/xdelta[0];
  ypos = (ypos-(float)origin[1])/ydelta[1];
  zpos = (zpos-(float)origin[2])/zdelta[2];
  int gx = (int) floor(xpos); // XXX this is wrong for non-orthog cells.
  int gy = (int) floor(ypos);
  int gz = (int) floor(zpos);
  if (gx < 0 || gx >= xsize) return vmd_NAN;
  if (gy < 0 || gy >= ysize) return vmd_NAN;
  if (gz < 0 || gz >= zsize) return vmd_NAN;
  return voxel_value_interpolate(xpos, ypos, zpos);
}



int VolMap::write_dx_file (const char *filename) {
  if (!data) return -1; // XXX is this a good random error code?
  int gridsize = xsize*ysize*zsize;
  int i;
  
  msgInfo << "volmap: writing file \"" << filename << "\"." << sendmsg;
  
  FILE *fout = fopen(filename, "w");
  if (!fout) {
    msgErr << "volmap: Cannot open file \"" << filename
           << "\" for writing." << sendmsg;
    return errno;
  };
    
  fprintf(fout, "# Data calculated by the VMD volmap function\n");
  fprintf(fout, "# VMDTAG WEIGHT %g\n", weight);
  if (temperature>=0) {
    fprintf(fout, "# VMDTAG TEMPERATURE %g Kelvin\n", temperature);
  }
  fprintf(fout, "object 1 class gridpositions counts %d %d %d\n", xsize, ysize, zsize);
  fprintf(fout, "origin %g %g %g\n", origin[0], origin[1], origin[2]);
  fprintf(fout, "delta %g %g %g\n", xdelta[0], xdelta[1], xdelta[2]);
  fprintf(fout, "delta %g %g %g\n", ydelta[0], ydelta[1], ydelta[2]);
  fprintf(fout, "delta %g %g %g\n", zdelta[0], zdelta[1], zdelta[2]);
  fprintf(fout, "object 2 class gridconnections counts %d %d %d\n", xsize, ysize, zsize);
  fprintf(fout, "object 3 class array type double rank 0 items %d data follows\n", gridsize);
  
  // This reverses the ordering from x fastest to z fastest changing variable
  float val1,val2,val3;
  int gx=0, gy=0, gz=-1;
  for (i=0; i < (gridsize/3)*3; i+=3)  {
    if (++gz >= zsize) {
      gz=0;
      if (++gy >= ysize) {gy=0; gx++;}
    }
    val1 = voxel_value(gx,gy,gz);
    if (++gz >= zsize) {
      gz=0;
      if (++gy >= ysize) {gy=0; gx++;}
    }
    val2 = voxel_value(gx,gy,gz);
    if (++gz >= zsize) {
      gz=0;
      if (++gy >= ysize) {gy=0; gx++;}
    }
    val3 = voxel_value(gx,gy,gz);    
    fprintf(fout, "%g %g %g\n", val1, val2, val3);
  }
  for (i=(gridsize/3)*3; i < gridsize; i++) {
    if (++gz >= zsize) {
      gz=0;
      if (++gy >= ysize) {gy=0; gx++;}
    }
    fprintf(fout, "%g ", voxel_value(gx,gy,gz));
  }
  fprintf(fout, "\n");
  
  // XXX todo: make sure that "dataname" contains no quotes
  fprintf(fout, "object \"%s\" class field\n", get_name());
  
  fclose(fout);
  return 0;
}


void VolMap::fill(float value) {
  if (value)
    for (int n=0; n<xsize*ysize*zsize; n++) data[n] = value;
  else
    memset(data, 0, sizeof(float)*xsize*ysize*zsize);
}



/// El cheapo convolution. Ideally should be more powerful and accept an
/// arbitrary matrix, but all I needed was a simple smooth to test out the
/// feature...
void VolMap::smooth () {
  int gx,gy,gz,n;
  float *data2 = new float[xsize*ysize*zsize];
  memcpy(data2,data,sizeof(float)*xsize*ysize*zsize);
  
  for (gz=1; gz<zsize-1; gz++) 
  for (gy=1; gy<ysize-1; gy++) 
  for (gx=1; gx<xsize-1; gx++) {
    n = gx + xsize*gy + xsize*ysize*gz;
    data[n] = (6.f*data2[n] + (data2[n-1] + data2[n+1] + data2[n-xsize] + data2[n+xsize] + data2[n+ysize*xsize] + data2[n-ysize*xsize]))/12.f;
  }

  delete[] data2;
}
