/***************************************************************************
 *cr
 *cr            (C) Copyright 1995-2007 The Board of Trustees of the
 *cr                        University of Illinois
 *cr                         All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: VMDThreads.C,v $
 *      $Author: johns $        $Locker:  $             $State: Exp $
 *      $Revision: 1.23 $       $Date: 2007/01/12 20:08:34 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * VMDThreads.C - code for spawning threads on various platforms.
 *                Code donated by John Stone, johns@megapixel.com 
 *                This code was originally written for the
 *                Tachyon Parallel/Multiprocessor Ray Tracer. 
 *                Improvements have been donated by Mr. Stone on an 
 *                ongoing basis. 
 *
 ***************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include "VMDThreads.h"

#ifdef _MSC_VER
#include <windows.h>
#include <winbase.h>
#endif

/* needed for call to sysconf() */
#if defined(__sun) || defined(ARCH_IRIX6) || defined(ARCH_IRIX6_64) || defined(ARCH_LINUX) || defined(ARCH_LINUXALHPA) || defined(ARCH_LINUXAMD64) || defined(ARCH_LINUXIA64) || defined(ARCH_LINUXPPC) || defined(_CRAY) || defined(__osf__) || defined(ARCH_AIX4) || defined(ARCH_AIX5) || defined(ARCH_AIX5_64)
#include<unistd.h>
#endif

#if defined(__APPLE__) && defined(VMDTHREADS)
#include <Carbon/Carbon.h> /* Carbon APIs for Multiprocessing */
#endif

#if defined(ARCH_HPUX11) ||  defined(ARCH_HPUX10)
#include <sys/mpctl.h>
#endif


int vmd_thread_numprocessors(void) {
  int a=1;

#ifdef VMDTHREADS
  // Allow the user to override the number of CPUs for use
  // in scalability testing, debugging, etc.
  char *forcecount = getenv("VMDFORCECPUCOUNT");
  if (forcecount != NULL) {
    if (sscanf(forcecount, "%d", &a) == 1) {
      return a; // if we got a valid count, return it
    } else {
      a=1;      // otherwise use the real available hardware CPU count
    }
  }

#if defined(__APPLE__) 
  a = MPProcessorsScheduled(); /* Number of active/running CPUs */
#endif

#ifdef _MSC_VER
  struct _SYSTEM_INFO sysinfo;
  GetSystemInfo(&sysinfo);
  a = sysinfo.dwNumberOfProcessors; /* total number of CPUs */
#endif /* _MSC_VER */

#if defined(_CRAY)
  a = sysconf(_SC_CRAY_NCPU);
#endif

#if defined(__sun) || defined(ARCH_LINUX) || defined(ARCH_LINUXALPHA) || defined(ARCH_LINUXAMD64) || defined(ARCH_LINUXIA64) || defined(ARCH_LINUXPPC) || defined(__osf__) || defined(ARCH_AIX4) || defined(ARCH_AIX5) || defined(ARCH_AIX5_64)
  a = sysconf(_SC_NPROCESSORS_ONLN); /* number of active/running CPUs */
#endif /* SunOS */

#if defined(ARCH_IRIX6) || defined(ARCH_IRIX6_64)
  a = sysconf(_SC_NPROC_ONLN); /* number of active/running CPUs */
#endif /* IRIX */

#if defined(ARCH_HPUX11) || defined(ARCH_HPUX10)
  a = mpctl(MPC_GETNUMSPUS, 0, 0); /* total number of CPUs */
#endif /* HPUX */
#endif /* VMDTHREADS */

  return a;
}


int vmd_thread_setconcurrency(int nthr) {
  int status=0;

#ifdef VMDTHREADS
#if defined(__sun) 
#ifdef USEPOSIXTHREADS 
  status = pthread_setconcurrency(nthr);
#else
  status = thr_setconcurrency(nthr);
#endif
#endif /* SunOS */

#if defined(ARCH_IRIX6) || defined(ARCH_IRIX6_64) || defined(ARCH_AIX4) || defined(ARCH_AIX5) || defined(ARCH_AIX5_64)
  status = pthread_setconcurrency(nthr);
#endif
#endif /* VMDTHREADS */

  return status;
}

// Typedef to eliminate compiler warning caused by C/C++ linkage conflict.
extern "C" {
  typedef void * (*VMDTHREAD_START_ROUTINE)(void *);
}

int vmd_thread_create(vmd_thread_t * thr, void * routine(void *), void * arg) {
  int status=0;

#ifdef VMDTHREADS 
#ifdef _MSC_VER
  int tid; /* thread id, msvc only */
 
  *thr = CreateThread(NULL, 8192,
                    (LPTHREAD_START_ROUTINE) routine, arg, 0, &tid);
 
  if (*thr == NULL) {
    status = -1;
  }
#endif /* _MSC_VER */

#ifdef USEPOSIXTHREADS 
#if defined(ARCH_AIX4) || defined(ARCH_AIX5) || defined(ARCH_AIX5_64)
  {
    pthread_attr_t attr;
    pthread_attr_init(&attr);
    pthread_attr_setscope(&attr, PTHREAD_SCOPE_SYSTEM);
    status = pthread_create(thr, &attr, routine, arg);
    pthread_attr_destroy(&attr);
  }
#else   
  status = pthread_create(thr, NULL, (VMDTHREAD_START_ROUTINE)routine, arg);
#endif 
#endif /* USEPOSIXTHREADS */

#ifdef USEUITHREADS 
  status = thr_create(NULL, 0, routine, arg, 0, thr); 
#endif /* USEUITHREADS */
#endif /* VMDTHREADS */
 
  return status;
}


int vmd_thread_join(vmd_thread_t thr, void ** stat) {
  int status=0;  

#ifdef VMDTHREADS
#ifdef _MSC_VER
  DWORD wstatus = 0;
 
  wstatus = WAIT_TIMEOUT;
 
  while (wstatus != WAIT_OBJECT_0) {
    wstatus = WaitForSingleObject(thr, INFINITE);
  }
#endif /* _MSC_VER */

#ifdef USEPOSIXTHREADS
  status = pthread_join(thr, stat);
#endif /* USEPOSIXTHREADS */

#ifdef USEUITHREADS
  status = thr_join(thr, NULL, stat);
#endif /* USEPOSIXTHREADS */
#endif /* VMDTHREADS */

  return status;
}  


int vmd_mutex_init(vmd_mutex_t * mp) {
  int status=0;

#ifdef VMDTHREADS
#ifdef USEPOSIXTHREADS
  status = pthread_mutex_init(mp, 0);
#endif /* USEPOSIXTHREADS */

#ifdef USEUITHREADS 
  status = mutex_init(mp, USYNC_THREAD, NULL);
#endif /* USEUITHREADS */
#endif /* VMDTHREADS */

  return status;
}


int vmd_mutex_lock(vmd_mutex_t * mp) {
  int status=0;

#ifdef VMDTHREADS
#ifdef USEPOSIXTHREADS
  status = pthread_mutex_lock(mp);
#endif /* USEPOSIXTHREADS */

#ifdef USEUITHREADS
  status = mutex_lock(mp);
#endif /* USEUITHREADS */
#endif /* VMDTHREADS */

  return status;
}


int vmd_mutex_unlock(vmd_mutex_t * mp) {
  int status=0;

#ifdef VMDTHREADS  
#ifdef USEPOSIXTHREADS
  status = pthread_mutex_unlock(mp);
#endif /* USEPOSIXTHREADS */

#ifdef USEUITHREADS
  status = mutex_unlock(mp);
#endif /* USEUITHREADS */
#endif /* VMDTHREADS */

  return status;
}


int vmd_mutex_destroy(vmd_mutex_t * mp) {
  int status=0;

#ifdef VMDTHREADS
#ifdef USEPOSIXTHREADS
  status = pthread_mutex_destroy(mp);
#endif /* USEPOSIXTHREADS */

#ifdef USEUITHREADS 
  status = mutex_destroy(mp);
#endif /* USEUITHREADS */
#endif /* VMDTHREADS */

  return status;
}

#if !defined(VMDTHREADS)

int vmd_thread_barrier_init(vmd_barrier_t *barrier, int n_clients) {
  return 0;
}

void vmd_thread_barrier_destroy(vmd_barrier_t *barrier) {
}

int vmd_thread_barrier(vmd_barrier_t *barrier, int increment) {
  return 0;
}

#else 

#ifdef USEPOSIXTHREADS
int vmd_thread_barrier_init(vmd_barrier_t *barrier, int n_clients) {
  if (barrier != NULL) {
    barrier->n_clients = n_clients;
    barrier->n_waiting = 0;
    barrier->phase = 0;
    barrier->sum = 0;

#if defined(VMDCAVE)
    // XXX the CAVE requires that we use a special synchronization
    //     mode so that mutexes and condition variables among multiple 
    //     processes in shared memory will work correctly.
    pthread_mutexattr_t mattr;
    pthread_condattr_t  cattr;

    printf("Setting barriers to have system scope...\n");

    pthread_mutexattr_init(&mattr);
    if (pthread_mutexattr_setpshared(&mattr, PTHREAD_PROCESS_SHARED) != 0) {
      printf("WARNING: could not set mutex to process shared scope\n");
    }

    pthread_condattr_init(&cattr);
    if (pthread_condattr_setpshared(&cattr, PTHREAD_PROCESS_SHARED) != 0) {
      printf("WARNING: could not set mutex to process shared scope\n");
    }

    pthread_mutex_init(&barrier->lock, &mattr);
    pthread_cond_init(&barrier->wait_cv, &cattr);

    pthread_condattr_destroy(&cattr);
    pthread_mutexattr_destroy(&mattr);
#else
    pthread_mutex_init(&barrier->lock, NULL);
    pthread_cond_init(&barrier->wait_cv, NULL);
#endif
  }

  return 0;
}

void vmd_thread_barrier_destroy(vmd_barrier_t *barrier) {
  pthread_mutex_destroy(&barrier->lock);
  pthread_cond_destroy(&barrier->wait_cv);
  free(barrier);
}

int vmd_thread_barrier(vmd_barrier_t *barrier, int increment) {
  int my_phase;

  pthread_mutex_lock(&barrier->lock);
  my_phase = barrier->phase;
  barrier->sum += increment;
  barrier->n_waiting++;

  if (barrier->n_waiting == barrier->n_clients) {
    barrier->result = barrier->sum;
    barrier->sum = 0;
    barrier->n_waiting = 0;
    barrier->phase = 1 - my_phase;
    pthread_cond_broadcast(&barrier->wait_cv);
  }

  while (barrier->phase == my_phase) {
    pthread_cond_wait(&barrier->wait_cv, &barrier->lock);
  }

  pthread_mutex_unlock(&barrier->lock);

  return (barrier->result);
}
#endif


#ifdef USEUITHREADS

int vmd_thread_barrier_init(vmd_barrier_t * barrier, int n_clients) {
  if (barrier != NULL) {
    barrier->n_clients = n_clients;
    barrier->n_waiting = 0;
    barrier->phase = 0;
    barrier->sum = 0;
    mutex_init(&barrier->lock, USYNC_THREAD, NULL);
    cond_init(&barrier->wait_cv, USYNC_THREAD, NULL);
  }

  return 0;
}

void vmd_thread_barrier_destroy(vmd_barrier_t *barrier) {
  mutex_destroy(&barrier->lock);
  cond_destroy(&barrier->wait_cv);
  free(barrier);
}

int vmd_thread_barrier(vmd_barrier_t *barrier, int increment) {
  int my_phase;

  mutex_lock(&barrier->lock);
  my_phase = barrier->phase;
  barrier->sum += increment;
  barrier->n_waiting++;

  if (barrier->n_waiting == barrier->n_clients) {
    barrier->result = barrier->sum;
    barrier->sum = 0;
    barrier->n_waiting = 0;
    barrier->phase = 1 - my_phase;
    cond_broadcast(&barrier->wait_cv);
  }

  while (barrier->phase == my_phase) {
    cond_wait(&barrier->wait_cv, &barrier->lock);
  }

  mutex_unlock(&barrier->lock);

  return (barrier->result);
}

#endif /* USEUITHREADS */


#endif /* VMDTHREADS */



