/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995-2007 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Timestep.C,v $
 *	$Author: johns $	$Locker:  $		$State: Exp $
 *	$Revision: 1.49 $	$Date: 2007/01/12 20:08:33 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The Timestep class, which stores coordinates, energies, etc. for a
 * single timestep.
 *
 * Note: As more data is stored for each step, it should go in here.  For
 * example, H-Bonds could be calculated each step.
 ***************************************************************************/

#include <math.h>
#include "Timestep.h"
#include "Inform.h"
#include "Atom.h"

///  constructor  
Timestep::Timestep(int n) {
  for(int i=0; i < TSENERGIES; energy[i++] = 0.0);
  num = n;
  pos = new float[3 * num];
  force = NULL;
  user = NULL;
  memset(tscov, 0, sizeof(tscov));
  scalefactor = 1.0;
  a_length = b_length = c_length = 1;
  alpha = beta = gamma = 90;
  timesteps=0;
  Initialized = FALSE;
}


/// copy constructor
Timestep::Timestep(const Timestep& ts) {
  num = ts.num;

  pos = new float[3 * num];
  memcpy(pos, ts.pos, 3 * num * sizeof(float));

  if (ts.force) {
    force = new float[3 * num];
    memcpy(force, ts.force, 3 * num * sizeof(float));
  } else {
    force = NULL;
  }

  if (ts.user) {
    user = new float[num];
    memcpy(user, ts.user, num*sizeof(float));
  } else {
    user = NULL;
  }

  memcpy(energy, ts.energy, sizeof(ts.energy));
  memcpy(tscov, ts.tscov, sizeof(tscov));
  scalefactor = ts.scalefactor;
  a_length = ts.a_length;
  b_length = ts.b_length;
  c_length = ts.c_length;
  alpha = ts.alpha;
  beta = ts.beta;
  gamma = ts.gamma;
  timesteps=ts.timesteps;
  Initialized = ts.Initialized;
}


/// destructor  
Timestep::~Timestep(void) {
  delete [] force;
  delete [] pos;
  delete [] user;
}


//////////////////////////  public routines  

/// calculate the max/min values for all the quantities, and anything else
/// required based on atom coordinates, etc.  Used for scaling
/// and translating purposes. 
//  XXX init() should be called again if/when atom coordinates are ever
//      changed for this timestep, or at least the Initialized flag should
//      be invalidated so it gets called again if necessary.
void Timestep::init(void) {
  int i;
  float covx, covy, covz;
  float minposx, minposy, minposz;
  float maxposx, maxposy, maxposz;
  const float *mpos = pos;

  // only do this if there are atoms
  if (!num)
    return;		

  covx = covy = covz = 0.0;

  // inialize min/max positions with values from the first atom
  minposx = maxposx = pos[0];
  minposy = maxposy = pos[1];
  minposz = maxposz = pos[2];

  // calc ranges for all atoms
  // XXX This calculation is very expensive and is a major cause of slower
  //     trajectory reading performance compared to standalone plugin tests.
  //     Just this code by itself makes a 1.4GB trajectory take 16.5 seconds
  //     to load rather than 12, so making this faster will help a lot.
  //     The same test with just a standalone plugin takes 3.5 seconds.
  int maxnum = num*3;
  for (i=0; i < maxnum; i+=3) {
    const float xpos = mpos[i    ];
    const float ypos = mpos[i + 1]; 
    const float zpos = mpos[i + 2]; 

    covx += xpos;
    covy += ypos;
    covz += zpos;

    if (xpos < minposx)
      minposx = xpos;
    if (xpos > maxposx)
      maxposx = xpos;

    if (ypos < minposy)
      minposy = ypos;
    if (ypos > maxposy)
      maxposy = ypos; 

    if (zpos < minposz)
      minposz = zpos;
    if (zpos > maxposz)
      maxposz = zpos;
  }

  // set the center of volume variable now
  tscov[0] = covx; 
  tscov[1] = covy; 
  tscov[2] = covz; 
  vec_scale(tscov, 1.0f / (float) num, tscov);
 
  // calculate center-of-volume and scale factor
  scalefactor = maxposx - minposx;

  // prevent getting a zero-scaled scene when loading a single atom.
  if (num == 0 || scalefactor == 0.0) {
    scalefactor = 3.0;
  }

  if ((maxposx - minposx) > scalefactor)
    scalefactor = maxposx - minposx;
  if ((maxposy - minposy) > scalefactor)
    scalefactor = maxposy - minposy;
  if ((maxposz - minposz) > scalefactor)
    scalefactor = maxposz - minposz;

  scalefactor = 1.5f / scalefactor;

  Initialized = TRUE;
}

float Timestep::scale_factor(void) {
  if (Initialized != TRUE)
    init(); // if the timestep scalefactor and tscov aren't set, do it now

  return scalefactor;
}

void Timestep::cov(float &x, float &y, float &z) {
  if (Initialized != TRUE)
    init(); // if the timestep scalefactor and tscov aren't set, do it now

  x = tscov[0];
  y = tscov[1];
  z = tscov[2]; 
}

// reset coords and related items to 0
void Timestep::zero_values(void) {
  if (num <= 0) 
    return;
    
  memset(pos,0,3*num*sizeof(float));
   
  for(int i=0; i < TSENERGIES; energy[i++] = 0.0);
  timesteps=0;
  
  scalefactor = 1.0;
  init();
}

#define DEG2RAD 3.14159625359/180.0
void Timestep::get_transforms(Matrix4 &a, Matrix4 &b, Matrix4 &c) const {
  // notes: a, b, c are side lengths of the unit cell
  // alpha = angle between b and c
  //  beta = angle between a and c
  // gamma = angle between a and b

  // convert from degrees to radians
  double cosBC = cos(alpha*DEG2RAD);
  double cosAC = cos(beta*DEG2RAD);
  double cosAB = cos(gamma*DEG2RAD);
  double sinAB = sin(gamma*DEG2RAD);

  // A will lie along the positive x axis.
  // B will lie in the x-y plane
  // The origin will be (0,0,0).
  float Ax = (float) (a_length);
  float Bx = (float) (b_length * cosAB);
  float By = (float) (b_length * sinAB);

  float Cx=0, Cy=0, Cz=0;
  // If sinAB is zero, then we can't determine C uniquely since it's defined
  // in terms of the angle between A and B.
  if (sinAB > 0) {
    Cx = (float) cosAC;
    Cy = (float) ((cosBC - cosAC * cosAB) / sinAB);
    Cz = sqrtf(1.0f - Cx*Cx - Cy*Cy);
  }
  Cx *= c_length;
  Cy *= c_length;
  Cz *= c_length;

  a.translate(Ax,  0,  0);
  b.translate(Bx, By,  0);
  c.translate(Cx, Cy, Cz);
}

void Timestep::get_transform_from_cell(const int *cell, Matrix4 &mat) const {
  // 99% cut/pasted from get_transforms()

  // notes: a, b, c are side lengths of the unit cell
  // alpha = angle between b and c
  //  beta = angle between a and c
  // gamma = angle between a and b

  // convert from degrees to radians
  double cosBC = cos(alpha*DEG2RAD);
  double cosAC = cos(beta*DEG2RAD);
  double cosAB = cos(gamma*DEG2RAD);
  double sinAB = sin(gamma*DEG2RAD);

  // A will lie along the positive x axis.
  // B will lie in the x-y plane
  // The origin will be (0,0,0).
  float Ax = (float) (a_length);
  float Bx = (float) (b_length * cosAB);
  float By = (float) (b_length * sinAB);

  float Cx=0, Cy=0, Cz=0;
  // If sinAB is zero, then we can't determine C uniquely since it's defined
  // in terms of the angle between A and B.
  if (sinAB > 0) {
    Cx = (float) cosAC;
    Cy = (float) ((cosBC - cosAC * cosAB) / sinAB);
    Cz = sqrtf(1.0f - Cx*Cx - Cy*Cy);
  }
  Cx *= c_length;
  Cy *= c_length;
  Cz *= c_length;

  mat.identity();
  mat.mat[12] = cell[0]*Ax + cell[1]*Bx + cell[2]*Cx;
  mat.mat[13] =              cell[1]*By + cell[2]*Cy;            
  mat.mat[14] =                           cell[2]*Cz;
}





