/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995-2007 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: TclVolMap.C,v $
 *	$Author: jordi $	$Locker:  $		$State: Exp $
 *	$Revision: 1.35 $	$Date: 2007/03/14 16:48:08 $
 *
 ***************************************************************************
 * DESCRIPTION:
 * These are essentially just Tcl wrappers for the volmap commands in
 * VolMapCreate.C.
 *
 ***************************************************************************/

#include <stdlib.h>
#include <tcl.h>
#include "TclCommands.h"
#include "AtomSel.h"
#include "VMDApp.h"
#include "MoleculeList.h"
#include "config.h"
#include "Measure.h"
#include "VolMap.h"
#include "VolMapCreate.h"


// XXX Todo List: 
// - Make this into an independent Tcl function 
//  (e.g.: "volmap new density <blah...>"
// - Make friends with as-of-yet non-existing VMD volumetric map I/O
// - parse and allow user to set useful params

// Function: volmap <maptype> <selection> [-weight <none|atom value|string array>] 
// Options for all maptypes:
//   -allframes
//   -res <res>
// Options for "density":
//   -weight <none|atom value|string array>
//   -scale <radius scale>


static int vmd_volmap_new_fromtype(VMDApp *app, int argc, Tcl_Obj * const objv[], Tcl_Interp *interp)
{
  // 1. Figure out which map type we are dealing with:
  
  enum {UNDEF_MAP, DENS_MAP, DIST_MAP, OCCUP_MAP, MASK_MAP, ENERGY_MAP, \
        SLOWENERGY_MAP, CPOTENTIAL_MAP} maptype=UNDEF_MAP; // which map type 

  char *maptype_string=Tcl_GetString(objv[0]); 
  if (!strcmp(maptype_string, "density")) maptype=DENS_MAP;
  else if (!strcmp(maptype_string, "distance")) maptype=DIST_MAP;
  else if (!strcmp(maptype_string, "occupancy")) maptype=OCCUP_MAP; 
  else if (!strcmp(maptype_string, "mask")) maptype=MASK_MAP; 
  else if (!strcmp(maptype_string, "ligand")) maptype=ENERGY_MAP;
  else if (!strcmp(maptype_string, "slowligand")) maptype=SLOWENERGY_MAP;
  else if (!strcmp(maptype_string, "coulombpotential")) maptype=CPOTENTIAL_MAP;
  else if (!strcmp(maptype_string, "coulomb")) maptype=CPOTENTIAL_MAP;
  
  // 2. Define and initialize the optional arguments
  bool need_sel1 = false;     // requires a specified selection
  bool accept_res = true;     // allow a user-specified resolution
  bool accept_minmax = true;  // allow user-specified grid boundaries
  bool accept_weight = false; // allow a user-specified weight for each atom
  bool accept_cutoff = false; // parse a user-specified cutoff distance
  bool accept_numpts = false; // parse a user-specified number of sampling points
  bool accept_usepoints = false; // allows use of point particles
  bool accept_radius = false; // allow radius multiplicator for density
  bool accept_probe_params = false; // parse energy probe VDW parameters
  bool accept_subres = false; // energy sub-resolution
  bool accept_temperature = false; // -T parameter

  bool export_to_file = false;
  bool use_all_frames = false;
  int export_molecule = -1;
  
  bool bad_usage = false;
  char *filebase = NULL;
  char *filename = NULL;

  bool use_point_particles = false;  // for MASK map
  double cutoff=5.; //default
  
  // File export options
  int checkpoint_freq = 500;
      
  // for ENERGY/SLOWENERGY maps:
  double probe1_epsilon=-0.12, probe1_radius=1.7, probe_charge=0.; //defaults, for "EnergyMap"
  double probe2_epsilon=-0.12, probe2_radius=1.7; //defaults, for "EnergyMap"
  double temperature=300.;
  int param_subres = 1;
  int probe_type = 0;  
  double bondlength = 1.12f;
  int num_conf = 0;  // number of ligand rotamers


  // Specify required/accepted options for each maptype as well as default values.
  switch(maptype) {
    case DENS_MAP:
      need_sel1 = accept_res = true;
      accept_weight = true;
      accept_numpts = true;
      accept_radius = true;
      break;
    case DIST_MAP:
      need_sel1 = accept_res = true;
      accept_cutoff = true;
      cutoff = 3.;
      break;
    case ENERGY_MAP:
      accept_subres = true;
      param_subres = 1;
      // no break here!!
    case SLOWENERGY_MAP:
      accept_temperature = true;
      need_sel1 = accept_res = true;
      accept_cutoff = true;
      cutoff = 10.;
      accept_probe_params = true;
      probe1_epsilon = -0.12;
      probe1_radius = 1.7;
      probe2_epsilon = -0.12;
      probe2_radius = 1.7;
      temperature = 300.;
      probe_charge = 0.;
      probe_type = 0;
      num_conf = 0;
      break;
    case OCCUP_MAP:
      need_sel1=accept_res = true;
      accept_usepoints = true;
      break;
    case MASK_MAP:
      need_sel1=accept_res = true;
      accept_cutoff = true;
      cutoff = 4.;
      break;
    case CPOTENTIAL_MAP:
      need_sel1 = accept_res = true;
      break; 
    case UNDEF_MAP:
      bad_usage = true;
      break;   
  }
  
  
  // 3. Parse the command-line here
  
  int arg_sel1=0, arg_weight=0, arg_res=0, arg_combine=0, arg_minmax=0;
  int arg_cutoff=0, arg_mol=0, arg_radius=0;
  int arg_probe1_epsilon=0, arg_probe1_radius=0;
  int arg_probe2_epsilon=0, arg_probe2_radius=0;
  int arg_probe_charge=0, arg_bond=0;
  int arg_subres=0, arg_conf=0;
  int arg_checkpoint=0, arg_temperature=0;
  
  // Parse the arguments keys and identify the value fields for later parsing 
  for (int arg=1; arg<argc && !bad_usage; arg++) {
    if (accept_weight && !strcmp(Tcl_GetStringFromObj(objv[arg], NULL), "-weight")) {
      arg_weight=arg+1;
      arg++;
      if (arg_weight>=argc) bad_usage=true;
    }
    else if (accept_res && !strcmp(Tcl_GetStringFromObj(objv[arg], NULL), "-res")) {
      arg_res=arg+1;
      arg++;
      if (arg_res>=argc) bad_usage=true;
    }
    else if (accept_minmax && !strcmp(Tcl_GetStringFromObj(objv[arg], NULL), "-minmax")) {
      arg_minmax=arg+1;
      arg++;
      if (arg_minmax>=argc) bad_usage=true;
    }
    else if (accept_cutoff && !strcmp(Tcl_GetStringFromObj(objv[arg], NULL), "-cutoff")) {
      arg_cutoff=arg+1;
      arg++;
      if (arg_cutoff >= argc) bad_usage=true;
    }
    else if (accept_radius && !strcmp(Tcl_GetStringFromObj(objv[arg], NULL), "-radscale")) {
      arg_radius=arg+1;
      arg++;
      if (arg_radius >= argc) bad_usage=true;
    }
    else if (!strcmp(Tcl_GetStringFromObj(objv[arg], NULL), "-mol")) { // add volmap to mol
      arg_mol=arg+1;
      arg++;
      if (arg_mol >= argc) bad_usage=true;
    }
    else if (!strcmp(Tcl_GetStringFromObj(objv[arg], NULL), "-combine")) {
      arg_combine=arg+1;
      arg++;
      if (arg_combine>=argc) bad_usage=true;
    }
    else if (!strcmp(Tcl_GetStringFromObj(objv[arg], NULL), "-checkpoint")) {
      arg_checkpoint=arg+1;
      arg++;
      if (arg_checkpoint >= argc) bad_usage=true;
    }
    else if (!strcmp(Tcl_GetStringFromObj(objv[arg], NULL), "-allframes")) {
      use_all_frames=true;
    }
    else if (accept_usepoints && !strcmp(Tcl_GetStringFromObj(objv[arg], NULL), "-points")) {
      use_point_particles=true;
    }
    else if (need_sel1 && !arg_sel1) {
      arg_sel1=arg;
    }
    else if (!strcmp(Tcl_GetStringFromObj(objv[arg], NULL), "-dx") || !strcmp(Tcl_GetStringFromObj(objv[arg], NULL), "-o")) {
      if (arg+1>=argc) {bad_usage=true; break;}
      filebase = Tcl_GetString(objv[arg+1]);
      if (filebase) export_to_file = true;
      arg++;
    }
    
    // ligand map type arguments
    else if (accept_probe_params && (!strcmp(Tcl_GetStringFromObj(objv[arg], NULL), "-probe") || !strcmp(Tcl_GetStringFromObj(objv[arg], NULL), "-probe1"))) {
      arg_probe1_epsilon=arg+1;
      arg_probe1_radius=arg+2;
      arg+=2;
      if (arg_probe1_epsilon>=argc) bad_usage=true;
      if (arg_probe1_radius>=argc) bad_usage=true;
      // if the user only passes one argument here, followed by another directive,
      // there is a possibility for messiness. e.g., -probe -0.2 -res 0.5
      const char *secondarg_str=Tcl_GetString(objv[arg_probe1_radius]);
      if (secondarg_str[0] == '-') bad_usage=true;
    }
    else if (accept_probe_params && !strcmp(Tcl_GetStringFromObj(objv[arg], NULL), "-probe2")) {
      arg_probe2_epsilon=arg+1;
      arg_probe2_radius=arg+2;
      arg+=2;
      if (arg_probe2_epsilon>=argc) bad_usage=true;
      if (arg_probe2_radius>=argc) bad_usage=true;
      // if the user only passes one argument here, followed by another directive,
      // there is a possibility for messiness. e.g., -probe -0.2 -res 0.5
      const char *secondarg_str=Tcl_GetString(objv[arg_probe2_radius]);
      if (secondarg_str[0] == '-') bad_usage=true;
    }
    else if (accept_probe_params && !strcmp(Tcl_GetStringFromObj(objv[arg], NULL), "-charge")) {
      arg_probe_charge=arg+1;
      arg+=1;
      if (arg_probe_charge>=argc) bad_usage=true;
    }
    else if (accept_probe_params && !strcmp(Tcl_GetStringFromObj(objv[arg], NULL), "-dihomo")) {
      probe_type=1;
    }
    else if (accept_probe_params && !strcmp(Tcl_GetStringFromObj(objv[arg], NULL), "-dihetero")) {
      probe_type=2;
    }
    else if (accept_probe_params && !strcmp(Tcl_GetStringFromObj(objv[arg], NULL), "-conf")) {
      arg_conf=arg+1;
      arg++;
      if (arg_conf>=argc) bad_usage=true;
    }
    else if (accept_probe_params && !strcmp(Tcl_GetStringFromObj(objv[arg], NULL), "-subres")) {
      arg_subres=arg+1;
      arg++;
      if (arg_subres>=argc) bad_usage=true;
    }
    else if (accept_probe_params && !strcmp(Tcl_GetStringFromObj(objv[arg], NULL), "-bond")) {
      arg_bond=arg+1;
      arg++;
      if (arg_bond>=argc) bad_usage=true;
    }    
    else if (accept_temperature && !strcmp(Tcl_GetStringFromObj(objv[arg], NULL), "-T")) {
      arg_temperature=arg+1;
      arg++;
      if (arg_temperature>=argc) bad_usage=true;
    }    
    
    else //unknown arg
      bad_usage=true; 
  }
  
  if (need_sel1 && !arg_sel1) bad_usage=true; 
    
  if (bad_usage) {
    if (maptype == UNDEF_MAP)
      Tcl_AppendResult(interp, "volmap: unknown map type.", NULL);
    else
      Tcl_WrongNumArgs(interp, 2, objv-1, (char *)"<selection> [options...]");

    return TCL_ERROR;
  }
  
  
  // 4. Assign optional parameters here
  
  AtomSel *sel1 = NULL;
  if (need_sel1) {  
    sel1 = tcl_commands_get_sel(interp, Tcl_GetStringFromObj(objv[arg_sel1],NULL));
    if (!sel1) {
      Tcl_AppendResult(interp, "volmap: no atom selection.", NULL);
      return TCL_ERROR;
    }
    if (!sel1->selected) {
      Tcl_AppendResult(interp, "volmap: no atoms selected.", NULL);
      return TCL_ERROR;
    }
  }

  // parse -mol output
  if (arg_mol) {
    if (!strcmp(Tcl_GetStringFromObj(objv[arg_mol], NULL), "top"))
      export_molecule = app->molecule_top(); 
    else 
      Tcl_GetIntFromObj(interp, objv[arg_mol], &export_molecule);

    if (!app->molecule_valid_id(export_molecule)) {
      Tcl_AppendResult(interp, "volmap: molecule specified for ouput is invalid. (-mol)", NULL);
      return TCL_ERROR;
    }
  }
  
  // parse -radscale
  double radius_factor = 1.;
  if (arg_radius) {
    Tcl_GetDoubleFromObj(interp, objv[arg_radius], &radius_factor);
    if (radius_factor < 0.) {
      Tcl_AppendResult(interp, "volmap: radscale must be positive. (-radscale)", NULL);
      return TCL_ERROR;
    }
  }
  
  // parse resolution
  double resolution=1.;
  if (arg_res)
    Tcl_GetDoubleFromObj(interp, objv[arg_res], &resolution);
  if (resolution <= 0.) {
    Tcl_AppendResult(interp, "volmap: resolution must be positive. (-res)", NULL);
    return TCL_ERROR;
  }
  
   // parse minmax
  double minmax[6];
  if (arg_minmax) {
    int num, i;
    Tcl_Obj **data, **vecmin, **vecmax;
    
    // get the list containing minmax
    if (Tcl_ListObjGetElements(interp, objv[arg_minmax++], &num, &data) != TCL_OK) {
      Tcl_SetResult(interp, (char *)"volmap: could not read parameter (-minmax)", TCL_STATIC);
      return TCL_ERROR;
    }
    if (num != 2) {
      Tcl_SetResult(interp, (char *)"volmap: minmax requires a list with two vectors (-minmax)", TCL_STATIC);
      return TCL_ERROR;
    }
    // get the list containing min
    if (Tcl_ListObjGetElements(interp, data[0], &num, &vecmin) != TCL_OK) {
      return TCL_ERROR;
    }
    if (num != 3) {
      Tcl_SetResult(interp, (char *)"volmap: the first vector does not contain 3 elements (-minmax)", TCL_STATIC);
      return TCL_ERROR;
    }
    // get the list containing max
    if (Tcl_ListObjGetElements(interp, data[1], &num, &vecmax) != TCL_OK) {
      return TCL_ERROR;
    }
    if (num != 3) {
      Tcl_SetResult(interp, (char *)"volmap: the second vector does not contain 3 elements (-minmax)", TCL_STATIC);
      return TCL_ERROR;
    }

    // read min
    for (i=0; i<3; i++)
      if (Tcl_GetDoubleFromObj(interp, vecmin[i], minmax+i) != TCL_OK)
        return TCL_ERROR;
    
    // read max
    for (i=0; i<3; i++)
      if (Tcl_GetDoubleFromObj(interp, vecmax[i], minmax+i+3) != TCL_OK)
        return TCL_ERROR;
  }
  
    
  // parse cutoff
  if (arg_cutoff)
    Tcl_GetDoubleFromObj(interp, objv[arg_cutoff], &cutoff);
  if (cutoff <= 0.) {
    Tcl_AppendResult(interp, "volmap: cutoff must be positive. (-cutoff)", NULL);
    return TCL_ERROR;
  }
  
  // parse map combination type
  VolMapCreate::CombineType combine_type=VolMapCreate::COMBINE_AVG;
  if (arg_combine) {
    char *combine_str=Tcl_GetString(objv[arg_combine]);
    if (!strcmp(combine_str, "avg") || !strcmp(combine_str, "average")) 
      combine_type=VolMapCreate::COMBINE_AVG;
    else if (!strcmp(combine_str, "max") || !strcmp(combine_str, "maximum")) 
      combine_type=VolMapCreate::COMBINE_MAX;
    else if (!strcmp(combine_str, "min") || !strcmp(combine_str, "minimum")) 
      combine_type=VolMapCreate::COMBINE_MIN;
    else if (!strcmp(combine_str, "stdev")) 
      combine_type=VolMapCreate::COMBINE_STDEV;
    else if (!strcmp(combine_str, "pmf")) 
      combine_type=VolMapCreate::COMBINE_PMF;
    else {
      Tcl_AppendResult(interp, "volmap: -combine argument must be: avg, min, \
                                max, stdev, pmf", NULL);
      return TCL_ERROR;
    }
  }
 
  // parse -weight
  int ret_val=0;
  float *weights = NULL;
  if (accept_weight) {
    weights = new float[sel1->selected];

    if (arg_weight) 
      ret_val = tcl_get_weights(interp, app, sel1, objv[arg_weight], weights);
    else
      ret_val = tcl_get_weights(interp, app, sel1, NULL, weights);
    
    if (ret_val < 0) {
      Tcl_AppendResult(interp, "volmap: ", measure_error(ret_val), NULL);
      delete [] weights;
      return TCL_ERROR;
    }
  }
  
  // parse ligand probe1 parameters
  if (arg_probe1_radius && arg_probe1_epsilon) {
    Tcl_GetDoubleFromObj(interp, objv[arg_probe1_epsilon], &probe1_epsilon);
    Tcl_GetDoubleFromObj(interp, objv[arg_probe1_radius],  &probe1_radius);
  }
  if (probe1_epsilon > 0. || probe1_radius < 0.) {
    Tcl_AppendResult(interp, "volmap: invalid -probe1 parameters", NULL);
    return TCL_ERROR;
  }
  
  // parse ligand probe2 parameters
  if (arg_probe2_radius && arg_probe2_epsilon) {
    Tcl_GetDoubleFromObj(interp, objv[arg_probe2_epsilon], &probe2_epsilon);
    Tcl_GetDoubleFromObj(interp, objv[arg_probe2_radius],  &probe2_radius);
  }
  if (probe2_epsilon > 0. || probe2_radius < 0.) {
    Tcl_AppendResult(interp, "volmap: invalid -probe2 parameters", NULL);
    return TCL_ERROR;
  }
  
  // parse ligand bond parameter
  if (arg_bond) {
    Tcl_GetDoubleFromObj(interp, objv[arg_bond], &bondlength);
  }
  if (bondlength < 0.f) {
    Tcl_AppendResult(interp, "volmap: invalid value for -bond", NULL);
    return TCL_ERROR;
  }

  // parse ligand charge parameter
  if (arg_probe_charge) {
    Tcl_GetDoubleFromObj(interp, objv[arg_probe_charge], &probe_charge);
  }
  
  // parse ligand conformers parameter
  if (arg_conf) {
    Tcl_GetIntFromObj(interp, objv[arg_conf], &num_conf);
  }
  if (num_conf < 0) {
    Tcl_AppendResult(interp, "volmap: invalid -conf parameters", NULL);
    return TCL_ERROR;
  }
  
  // parse ligand subres parameter
  if (arg_subres) {
    Tcl_GetIntFromObj(interp, objv[arg_subres], &param_subres);
  }
  if (param_subres < 1) {
    Tcl_AppendResult(interp, "volmap: invalid -subres parameter", NULL);
    return TCL_ERROR;
  }
  
  // parse ligand subres parameter
  if (arg_checkpoint) {
    Tcl_GetIntFromObj(interp, objv[arg_checkpoint], &checkpoint_freq);
  }
  if (checkpoint_freq < 0) {
    Tcl_AppendResult(interp, "volmap: invalid -checkpoint parameter", NULL);
    return TCL_ERROR;
  }

  // parse ligand temperature parameter
  if (arg_temperature) {
    Tcl_GetDoubleFromObj(interp, objv[arg_temperature], &temperature);
  }
  if (temperature < 0.) {
    Tcl_AppendResult(interp, "volmap: invalid -T parameter", NULL);
    return TCL_ERROR;
  }

  
  // 5. Create the volmap
  VolMapCreate *volcreate = NULL;   
  
  // init the map creator objects and set default filenames
  switch(maptype) {
    case DENS_MAP: 
      volcreate = (VolMapCreate*) new VolMapCreateDensity(app, sel1, (float)resolution, weights);
      if (!filebase) filebase = "density_out.dx";
      ((VolMapCreateDensity*) volcreate)->set_radius_scale((float) radius_factor);
      break;
    case DIST_MAP:
      volcreate = (VolMapCreate*) new VolMapCreateDistance(app, sel1, (float)resolution, (float)cutoff);
      if (!filebase) filebase = "distance_out.dx";
      break;
    case OCCUP_MAP:
      volcreate = (VolMapCreate*) new VolMapCreateOccupancy(app, sel1, (float)resolution);
      if (!filebase) filebase = "occupancy_out.dx";
      ((VolMapCreateOccupancy*) volcreate)->use_point_particles(use_point_particles);
      break;
    case MASK_MAP:
      volcreate = (VolMapCreate*) new VolMapCreateMask(app, sel1, (float)resolution, (float)cutoff);
      if (!filebase) filebase = "mask_out.dx";
      break;
    case SLOWENERGY_MAP:
      volcreate = (VolMapCreate*) new VolMapCreateSlowEnergy(app, sel1, (float)resolution, (float)probe1_epsilon, (float)probe1_radius, (float)cutoff);
      // the object casting is quite ugly, but at least it stops here
      if (probe_type == 1) ((VolMapCreateSlowEnergy*) volcreate)->ligand_type = VolMapCreateSlowEnergy::DIHOMO;
      else if (probe_type == 2) {
        ((VolMapCreateSlowEnergy*) volcreate)->probe2_epsilon = probe2_epsilon;
        ((VolMapCreateSlowEnergy*) volcreate)->probe2_rmin = probe2_radius;
        ((VolMapCreateSlowEnergy*) volcreate)->ligand_type = VolMapCreateSlowEnergy::DIHETERO;
      }
      if (probe_type && !num_conf) num_conf = 10;
      if (probe_type) ((VolMapCreateSlowEnergy*) volcreate)->num_conformers = num_conf;
      ((VolMapCreateSlowEnergy*) volcreate)->probe1_charge = (float)  probe_charge;
      ((VolMapCreateSlowEnergy*) volcreate)->probe2_charge = (float) -probe_charge;
      if (arg_bond) ((VolMapCreateSlowEnergy*) volcreate)->probe_bondlength = bondlength;
      if (arg_temperature) ((VolMapCreateSlowEnergy*) volcreate)->temperature = temperature;
      if (!filebase) filebase = "slowligand_out.dx";
      break; 
    case ENERGY_MAP:
      volcreate = (VolMapCreate*) new VolMapCreateFastEnergy(app, sel1, (float)resolution, (float)probe1_epsilon, (float)probe1_radius, (float)cutoff);
      // the object casting is quite ugly, but at least it stops here
      if (probe_type == 1) ((VolMapCreateFastEnergy*) volcreate)->ligand_type = VolMapCreateFastEnergy::DIHOMO;
      if (probe_type == 2) {
        ((VolMapCreateFastEnergy*) volcreate)->probe2_epsilon = probe2_epsilon;
        ((VolMapCreateFastEnergy*) volcreate)->probe2_rmin = probe2_radius;
        ((VolMapCreateFastEnergy*) volcreate)->ligand_type = VolMapCreateFastEnergy::DIHETERO;
      }
      if (probe_type && !num_conf) num_conf = 10;
      if (probe_type) ((VolMapCreateFastEnergy*) volcreate)->num_conformers = num_conf;
      ((VolMapCreateFastEnergy*) volcreate)->probe1_charge = probe_charge;
      ((VolMapCreateFastEnergy*) volcreate)->probe2_charge = -probe_charge;
      if (arg_subres) ((VolMapCreateFastEnergy*) volcreate)->param_subres = param_subres;
      if (arg_bond) ((VolMapCreateFastEnergy*) volcreate)->probe_bondlength = bondlength;
      if (arg_temperature) ((VolMapCreateFastEnergy*) volcreate)->temperature = temperature;
      if (!filebase) filebase = "ligand_out.dx";
      break; 
    case CPOTENTIAL_MAP:
      volcreate = (VolMapCreate*) new VolMapCreateCoulombPotential(app, sel1, (float)resolution);
      if (!filebase) filebase = "coulomb_out.dx";
      break;

    default:  // silence compiler warnings
      break;  
  }

  // generate and write out volmap
  if (volcreate) {
    //Pass parameters common to all volmap type
    if (arg_minmax) volcreate->set_minmax(minmax[0],minmax[1],minmax[2], minmax[3], minmax[4], minmax[5]);
    
    //Setup checkpointing
    if (checkpoint_freq) {
      char *checkpointname = new char[32+strlen(filebase)+1];
      char *tailname = strrchr(filebase, '/');
      if (!tailname) tailname = filebase;
      else tailname = tailname+1;
      char *dirname = new char[strlen(filebase)+1];
      strcpy(dirname, filebase);
      char *sep = strrchr(dirname, '/');
      if (sep) {
        *sep = '\0';
        sprintf(checkpointname, "%s/checkpoint:%s", dirname, tailname);
      }
      else {
        *dirname = '\0';
        sprintf(checkpointname, "checkpoint:%s", tailname);
      }
      delete[] dirname;
      Tcl_AppendResult(interp, "CHECKPOINTNAME = ", checkpointname, NULL);
      volcreate->set_checkpoint(checkpoint_freq, checkpointname);
      delete[] checkpointname;
    }
    
    // Create map...
    volcreate->compute_all(use_all_frames, combine_type, NULL);
    
    // Export volmap to a file:
    if (export_to_file || !arg_mol) {
      // add .dx suffix to filebase if it is missing
      filename = new char[strlen(filebase)+16];
      strcpy(filename,filebase);
      int suffixpos = strlen(filename)-3;
      if (!strcmp(filename+suffixpos,".dx")) filename[suffixpos] = '\0';
      strcat(filename, ".dx");
      volcreate->volmap->write_dx_file(filename);
      delete[] filename;
    }
    
    // Export volmap to a molecule:
    if (arg_mol) {
      VolMap *volmap = volcreate->volmap;
      int err = app->molecule_add_volumetric(export_molecule, volmap->get_name(), volmap->origin, volmap->xaxis, volmap->yaxis, volmap->zaxis, volmap->xsize, volmap->ysize, volmap->zsize, volmap->data);
      if (err != 1) {
        Tcl_AppendResult(interp, "ERROR: export of volmap into molecule was unsuccessful!", NULL);
      }
      else volmap->data=NULL; // avoid data being deleted by volmap's destructor (it is now owned by the molecule)
    }

    delete volcreate;
  }
  
  if (weights) delete [] weights;
  if (ret_val < 0) {
    Tcl_AppendResult(interp, "volmap: ", measure_error(ret_val), NULL);
    return TCL_ERROR;
  }

  return TCL_OK;
}


int obj_volmap(ClientData cd, Tcl_Interp *interp, int argc, Tcl_Obj * const objv[]) {
    
  if (argc < 2) {
    Tcl_SetResult(interp,
      "usage: volmap <command> <args...>\n"
      "\nVolmap Creation:\n"
      " volmap <maptype> <selection> [opts...]   -- create a new volmap file\n"
      " maptypes:\n"
      "   density    -- arbitrary-weight density map [atoms/A^3]\n"
      "   distance   -- distance nearest atom surface [A]\n"
      "   occupancy  -- percent atomic occupancy of gridpoints [%]\n"
      "   mask       -- binary mask by painting spheres around atoms\n"
      "   ligand     -- energy of implicit ligand, optimized [kT]\n"
      "   slowligand -- energy of implicit ligand, slow [kT]\n"
      "   coulomb    -- Coulomb electrostatic potential [kT/e]\n"
      " options common to all maptypes:\n"
      "   -o <filename>           -- output DX format file name (use .dx extension)\n"
      "   -mol <molid>            -- export volmap into the specified mol\n"
      "   -res <float>            -- resolution in A of smallest cube\n"
      "   -allframes              -- compute for all frames of the trajectory\n"
      "   -combine <arg>          -- rule for combining the different frames\n"
      "                              <arg> = avg, min, max, stdev or pmf\n"
      "   -minmax <list of 2 vectors>   -- specify boundary of output grid\n"
      " options specific to certain maptypes:\n"
      "   -points                 -- use point particles for occupancy\n"
      "   -cutoff <float>         -- distance cutoff for calculations [A]\n"
      "   -radscale <float>       -- premultiply all atomic radii by a factor\n"
      "   -weight <str/list>      -- per atom weights for calculation\n"
      " options for ligand/slowligand:\n"
      "   see documentation\n", NULL);
    return TCL_ERROR;
  }

  VMDApp *app = (VMDApp *)cd;
  char *arg1 = Tcl_GetStringFromObj(objv[1],NULL);

  // If maptype is recognized, proceed with the map creation (vs. yet-unimplemented map operations)...
  if (argc > 1 && !strupncmp(arg1, "occupancy", CMDLEN))
    return vmd_volmap_new_fromtype(app, argc-1, objv+1, interp);
  if (argc > 1 && !strupncmp(arg1, "density", CMDLEN))
    return vmd_volmap_new_fromtype(app, argc-1, objv+1, interp);
  if (argc > 1 && !strupncmp(arg1, "distance", CMDLEN))
    return vmd_volmap_new_fromtype(app, argc-1, objv+1, interp);
  if (argc > 1 && !strupncmp(arg1, "mask", CMDLEN))
    return vmd_volmap_new_fromtype(app, argc-1, objv+1, interp);
  if (argc > 1 && !strupncmp(arg1, "ligand", CMDLEN))
    return vmd_volmap_new_fromtype(app, argc-1, objv+1, interp);
  if (argc > 1 && !strupncmp(arg1, "slowligand", CMDLEN))
    return vmd_volmap_new_fromtype(app, argc-1, objv+1, interp);   
  if (argc > 1 && (!strupncmp(arg1, "coulombpotential", CMDLEN) || !strupncmp(arg1, "coulomb", CMDLEN)))
    return vmd_volmap_new_fromtype(app, argc-1, objv+1, interp);   

  Tcl_SetResult(interp, "Type 'volmap' for summary of usage\n",NULL);
  return TCL_ERROR;
}
