/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995-2007 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: TclMeasure.C,v $
 *	$Author: johns $	$Locker:  $		$State: Exp $
 *	$Revision: 1.88 $	$Date: 2007/04/02 20:13:44 $
 *
 ***************************************************************************
 * DESCRIPTION:
 * These are essentially just Tcl wrappers for the measure commands in
 * Measure.C.
 *
 ***************************************************************************/

#include <stdlib.h>
#include <tcl.h>
#include <ctype.h>
#include "TclCommands.h"
#include "AtomSel.h"
#include "Matrix4.h"
#include "SymbolTable.h"
#include "VMDApp.h"
#include "MoleculeList.h"
#include "utilities.h"
#include "config.h"
#include "Measure.h"
#include "BondSearch.h"
#include "Atom.h"
#include "Molecule.h"


// compute the center of mass
// Takes an atom selection and one of three possible weights
//  1) if weight == NULL, uses weights of 1
//  2) if num == sel.selected ; assumes there is one weight per
//           selected atom
//  3) if num == sel.num_atoms; assumes weight[i] is for atom[i]
// returns center coordinate in float com[3]
// returns 0 if valid data or an appropriate MEASURE_ERR_xxx code otherwise.

// get weights from a Tcl item.  data must have space to hold sel->selected
// items.
// If there is no item, or if it's "none", return a list of ones.
// If the item matches an atom selection keyword, and the keyword returns 
// floating point data, return that data, otherwise error.
// Otherwise, the item must be a list of floats.  Set the selected items
// from the list.   
int tcl_get_weights(Tcl_Interp *interp, VMDApp *app, AtomSel *sel, 
                       Tcl_Obj *weight_obj, float *data) {

  char *weight_string = NULL;
  if (!sel) return MEASURE_ERR_NOSEL;
  if (!app->molecule_valid_id(sel->molid())) return MEASURE_ERR_NOMOLECULE;
  if (weight_obj)
    weight_string = Tcl_GetStringFromObj(weight_obj, NULL);

  if (!weight_string || !strcmp(weight_string, "none")) {
    for (int i=0; i<sel->selected; i++) {
      data[i] = 1.0;
    }
    return 0;
  }
  // if a selection string was given, check the symbol table
  SymbolTable *atomSelParser = app->atomSelParser; 
  // weights must return floating point values, so the symbol must not 
  // be a singleword, so macro is NULL.
  atomsel_ctxt context(atomSelParser, 
                       app->moleculeList->mol_from_id(sel->molid()), 
                       sel->which_frame, NULL);

  int fctn = atomSelParser->find_attribute(weight_string);
  if (fctn >= 0) {
    // the keyword exists, so get the data
    // first, check to see that the function returns floats.
    // if it doesn't, it makes no sense to use it as a weight
    if (atomSelParser->fctns.data(fctn)->returns_a != SymbolTableElement::IS_FLOAT) {
      Tcl_AppendResult(interp, 
        "weight attribute must have floating point values", NULL);
      return MEASURE_ERR_BADWEIGHTPARM;  // can't understand weight parameter 
    }
    double *tmp_data = new double[sel->num_atoms];
    atomSelParser->fctns.data(fctn)->keyword_double(
        &context, sel->num_atoms,tmp_data, sel->on);
    int j=0;
    for (int i=0; i<sel->num_atoms; i++) {
      if (sel->on[i])
        data[j++] = (float)tmp_data[i];
    }
    delete [] tmp_data;
    return 0;
  }
  // and see if this is a Tcl list with the right number of atoms
  int list_num;
  Tcl_Obj **list_data;
  if (Tcl_ListObjGetElements(interp, weight_obj, &list_num, &list_data) 
      != TCL_OK) {
    return MEASURE_ERR_BADWEIGHTPARM;
  }
  if (list_num != sel->selected && list_num != sel->num_atoms) 
    return MEASURE_ERR_BADWEIGHTNUM;
  
  int j = 0;
  for (int i=0; i<list_num; i++) {
    double tmp_data;
    if (Tcl_GetDoubleFromObj(interp, list_data[i], &tmp_data) != TCL_OK) 
      return MEASURE_ERR_NONNUMBERPARM;
    if (list_num == sel->selected) {
      //assert(i < sel->selected);
      data[i] = (float)tmp_data;
    } else {
      if (sel->on[i]) {
        //assert(j < sel->selected);
        data[j++] = (float)tmp_data;
      }
    }
  }
  return 0;
}

// get the  atom index re-ordering list for use by measure_fit
int tcl_get_orders(Tcl_Interp *interp, int selnum, 
                       Tcl_Obj *order_obj, int *data) {
  int list_num;
  Tcl_Obj **list_data;

  if (Tcl_ListObjGetElements(interp, order_obj, &list_num, &list_data)
      != TCL_OK) {
    return MEASURE_ERR_NOSEL;
  }

  // see if this is a Tcl list with the right number of atom indices
  if (list_num != selnum) return MEASURE_ERR_NOSEL;

  for (int i=0; i<list_num; i++) {
    if (Tcl_GetIntFromObj(interp, list_data[i], &data[i]) != TCL_OK)
      return MEASURE_ERR_NONNUMBERPARM;

    // order indices are 0-based
    if (data[i] < 0 || data[i] >= selnum)
      return MEASURE_ERR_BADORDERINDEX; // order index is out of range
  }

  return 0;
}

/*
Function:  vmd_measure_center
Parameters:  <selection>               // computes with weight == 1
Parameters:  <selection> weight [none|atom value|string array] 
  computes with the weights based on the following:
      none   => weights all 1
      atom value => value from atomSelParser (eg
         mass  => use weight based on mass
         index => use weight based on atom index (0 to n-1)
      string => use string to get weights for each atom.  The string can
         have number == number of selected atoms or total number of atoms

 Examples: 
    vmd_measure_center atomselect12
    vmd_measure_center atomselect12 weight mass
    vmd_measure_center atomselect12 {12 13} [atomselect top "index 2 3"]
 If no weight is given, no weight term is used (computes center of number)
 */
static int vmd_measure_center(VMDApp *app, int argc, Tcl_Obj *const objv[], Tcl_Interp *interp)
{
  if (argc != 2 && argc != 4 ) {
    Tcl_WrongNumArgs(interp, 2, objv-1, 
      (char *)"<selection> ?weight <weightstring>?");
    return TCL_ERROR;
  }
  if (argc == 4 && strcmp(Tcl_GetStringFromObj(objv[2],NULL), "weight")) {
    Tcl_SetResult(interp, (char *) "measure center: parameter can only be 'weight'", TCL_STATIC);
    return TCL_ERROR;
  }
  
  // get the selection
  AtomSel *sel = tcl_commands_get_sel(interp, Tcl_GetStringFromObj(objv[1],NULL));
  if (!sel) {
    Tcl_SetResult(interp, (char *) "measure center: no atom selection", TCL_STATIC);
    return TCL_ERROR;
  }

  // get the weight
  float *weight = new float[sel->selected];
  {
    int ret_val;
    if (argc == 2) {          // only from atom selection, so weight is 1
      ret_val = tcl_get_weights(interp, app, sel, NULL, weight);
    } else {
      ret_val = tcl_get_weights(interp, app, sel, objv[3], weight);
    }
    if (ret_val < 0) {
      Tcl_AppendResult(interp, "measure center: ", measure_error(ret_val),
		       NULL);
      delete [] weight;
      return TCL_ERROR;
    }
  }

  // compute the center of "mass"
  {
    float com[3];
    const float *framepos = sel->coordinates(app->moleculeList);
    int ret_val = measure_center(sel, framepos, weight, com);
    delete [] weight;
    if (ret_val < 0) {
      Tcl_AppendResult(interp, "measure center: ", measure_error(ret_val),
		       NULL);
      return TCL_ERROR;
    }
    Tcl_Obj *tcl_result = Tcl_NewListObj(0, NULL);
    Tcl_ListObjAppendElement(interp, tcl_result, Tcl_NewDoubleObj(com[0]));
    Tcl_ListObjAppendElement(interp, tcl_result, Tcl_NewDoubleObj(com[1]));
    Tcl_ListObjAppendElement(interp, tcl_result, Tcl_NewDoubleObj(com[2]));
    Tcl_SetObjResult(interp, tcl_result);
  }

  return TCL_OK;
}


// measure sum of weights for selected atoms
static int vmd_measure_sumweights(VMDApp *app, int argc, Tcl_Obj *const objv[], Tcl_Interp *interp) {
  if (argc != 4) {
    Tcl_WrongNumArgs(interp, 2, objv-1, (char *)"<selection> ?weight <weightstring>?");
    return TCL_ERROR;
  }
  if (strcmp(Tcl_GetStringFromObj(objv[2],NULL), "weight")) {
    Tcl_SetResult(interp, (char *) "measure sumweight: parameter can only be 'weight'", TCL_STATIC);
    return TCL_ERROR;
  }
 
  // get the selection
  AtomSel *sel = tcl_commands_get_sel(interp, Tcl_GetStringFromObj(objv[1],NULL)
);
  if (!sel) {
    Tcl_SetResult(interp, (char *) "measure sumweight: no atom selection", TCL_STATIC);
    return TCL_ERROR;
  }

  // get the weight
  float *weight = new float[sel->selected];
  {
    int ret_val = tcl_get_weights(interp, app, sel, objv[3], weight);
    if (ret_val < 0) {
      Tcl_AppendResult(interp, "measure center: ", measure_error(ret_val),
                       NULL);
      delete [] weight;
      return TCL_ERROR;
    }
  }

  // compute the sum of the weights
  {
    float weightsum;
    int ret_val = measure_sumweights(sel, sel->selected, weight, &weightsum);
    delete [] weight;
    if (ret_val < 0) {
      Tcl_AppendResult(interp, "measure center: ", measure_error(ret_val),
                       NULL);
      return TCL_ERROR;
    }
    Tcl_Obj *tcl_result = Tcl_NewListObj(0, NULL);
    Tcl_ListObjAppendElement(interp, tcl_result, Tcl_NewDoubleObj(weightsum));
    Tcl_SetObjResult(interp, tcl_result);
  }

  return TCL_OK;
}


// Function: vmd_measure_avpos <selection> first <first> last <last> step <step>
//  Returns: the average position of the selected atoms over the selected frames
//  Example: measure avpos atomselect76 0 20 1
static int vmd_measure_avpos(VMDApp *app, int argc, Tcl_Obj * const objv[], Tcl_Interp *interp) {
  int first = 0;  // start with first frame by default
  int last = -1;  // finish with last frame by default
  int step = 1;   // use all frames by default

  if (argc < 2 || argc > 8) {
    Tcl_WrongNumArgs(interp, 2, objv-1, (char *) "?selection first <first> last <last> step <step>?");
    return TCL_ERROR;
  }
  AtomSel *sel = tcl_commands_get_sel(interp, Tcl_GetStringFromObj(objv[1],NULL)
);
  if (!sel) {
    Tcl_AppendResult(interp, "measure avpos: no atom selection", NULL);
    return TCL_ERROR;
  }

  int i;
  for (i=2; i<argc; i+=2) {
    char *argvcur = Tcl_GetStringFromObj(objv[i],NULL);
    if (!strupncmp(argvcur, "first", CMDLEN)) {
      if (Tcl_GetIntFromObj(interp, objv[i+1], &first) != TCL_OK) {
        Tcl_AppendResult(interp, "measure avpos: bad first frame value", NULL);
        return TCL_ERROR;
      }
    } else if (!strupncmp(argvcur, "last", CMDLEN)) {
      if (Tcl_GetIntFromObj(interp, objv[i+1], &last) != TCL_OK) {
        Tcl_AppendResult(interp, "measure avpos: bad last frame value", NULL);
        return TCL_ERROR;
      }
    } else if (!strupncmp(argvcur, "step", CMDLEN)) {
      if (Tcl_GetIntFromObj(interp, objv[i+1], &step) != TCL_OK) {
        Tcl_AppendResult(interp, "measure avpos: bad frame step value", NULL);
        return TCL_ERROR;
      }
    } else {
      Tcl_AppendResult(interp, "measure avpos: invalid syntax, no such keyword: ", argvcur, NULL);
      return TCL_ERROR;
    }
  }

  float *avpos = new float[3*sel->selected];
  int ret_val = measure_avpos(sel, app->moleculeList, first, last, step, avpos);
  if (ret_val < 0) {
    Tcl_AppendResult(interp, "measure avpos: ", measure_error(ret_val), NULL);
    delete [] avpos;
    return TCL_ERROR;
  }

  Tcl_Obj *tcl_result = Tcl_NewListObj(0, NULL);
  for (i=0; i<sel->selected; i++) {
    Tcl_Obj *atom = Tcl_NewListObj(0, NULL);
    Tcl_ListObjAppendElement(interp, atom, Tcl_NewDoubleObj(avpos[i*3    ]));
    Tcl_ListObjAppendElement(interp, atom, Tcl_NewDoubleObj(avpos[i*3 + 1]));
    Tcl_ListObjAppendElement(interp, atom, Tcl_NewDoubleObj(avpos[i*3 + 2]));
    Tcl_ListObjAppendElement(interp, tcl_result, atom);
  }

  Tcl_SetObjResult(interp, tcl_result);

  delete [] avpos;

  return TCL_OK;
}


// Function: vmd_measure_dipole <selection>
//  Returns: the dipole moment for the selected atoms
static int vmd_measure_dipole(VMDApp *app, int argc, Tcl_Obj * const objv[], Tcl_Interp *interp) {
  const char *opt;
  int unitsdebye=0; // default units are elementary charges/Angstrom

  if (argc < 2) {
    Tcl_WrongNumArgs(interp, 2, objv-1, (char *) "?selection?");
    return TCL_ERROR;
  }
  AtomSel *sel = tcl_commands_get_sel(interp, Tcl_GetStringFromObj(objv[1],NULL)
);
  if (!sel) {
    Tcl_AppendResult(interp, "measure dipole: no atom selection", NULL);
    return TCL_ERROR;
  }

  if (argc == 3) {
    opt = Tcl_GetStringFromObj(objv[2], NULL);
    if (!strcmp(opt, "-debye"))
      unitsdebye=1; 
    if (!strcmp(opt, "-elementary"))
      unitsdebye=0; 
  }

  float dipole[3];
  int ret_val = measure_dipole(sel, app->moleculeList, dipole, unitsdebye);
  if (ret_val < 0) {
    Tcl_AppendResult(interp, "measure dipole: ", measure_error(ret_val), NULL);
    return TCL_ERROR;
  }

  Tcl_Obj *tcl_result = Tcl_NewListObj(0, NULL);
  Tcl_ListObjAppendElement(interp, tcl_result, Tcl_NewDoubleObj(dipole[0]));
  Tcl_ListObjAppendElement(interp, tcl_result, Tcl_NewDoubleObj(dipole[1]));
  Tcl_ListObjAppendElement(interp, tcl_result, Tcl_NewDoubleObj(dipole[2]));
  Tcl_SetObjResult(interp, tcl_result);

  return TCL_OK;
}



// Function: vmd_measure_dihed {4 atoms as {<atomid> ?<molid>?}} ?molid <default molid>?
//                             ?frame [f|all|last]? | ?first <first>? ?last <last>?
//  Returns: the dihedral angle for the specified atoms
static int vmd_measure_dihed(VMDApp *app, int argc, Tcl_Obj * const objv[], Tcl_Interp *interp) {
  int first=-1, last=-1, frame=-1;

  if(argc<2) {
    Tcl_WrongNumArgs(interp, 2, objv-1, (char *) "{{<atomid1> ?<molid1>?} {<atomid2> ?<molid2>?} {<atomid3> ?<molid3>?} {<atomid4> ?<molid4>?}} ?molid <default molid>? [?frame <frame|all|last>? | ?first <first>? ?last <last>?]");
    return TCL_ERROR;
  }

  int molid[4];
  int atmid[4];
  int defmolid = -1;
  bool allframes = false;

  // Get the geometry type dihed/imprp
  char *geomname = Tcl_GetStringFromObj(objv[0],NULL);


  // Read the atom list
  int numatms;
  Tcl_Obj **data;
  if (Tcl_ListObjGetElements(interp, objv[1], &numatms, &data) != TCL_OK) {
    Tcl_AppendResult(interp, " measure ", geomname, ": bad syntax", NULL);
    Tcl_AppendResult(interp, " Usage: measure ", geomname, " {{<atomid1> ?<molid1>?} {<atomid2> ?<molid2>?} {<atomid3> ?<molid3>?} {<atomid4> ?<molid4>?}} ?molid <default molid>? [?frame <frame|all|last>? | ?first <first>? ?last <last>?]", NULL);

    return TCL_ERROR;
  }

  if (numatms!=4) {
    Tcl_AppendResult(interp, " measure dihed: must specify exactly four atoms in a list", NULL);
    return TCL_ERROR;
  }
    

  if (argc>3) {
    int i;
    for (i=2; i<argc; i+=2) {
      char *argvcur = Tcl_GetStringFromObj(objv[i],NULL);
      if (!strupncmp(argvcur, "molid", CMDLEN)) {
	if (Tcl_GetIntFromObj(interp, objv[i+1], &defmolid) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure ", geomname, ": bad molid", NULL);
	  return TCL_ERROR;
	}
      } else if (!strupncmp(argvcur, "first", CMDLEN)) {
	if (Tcl_GetIntFromObj(interp, objv[i+1], &first) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure ", geomname, ": bad first frame value", NULL);
	  return TCL_ERROR;
	}
      } else if (!strupncmp(argvcur, "last", CMDLEN)) {
	if (Tcl_GetIntFromObj(interp, objv[i+1], &last) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure ", geomname, ": bad last frame value", NULL);
	  return TCL_ERROR;
	}
     } else if (!strupncmp(argvcur, "frame", CMDLEN)) {
	if (!strupcmp(Tcl_GetStringFromObj(objv[i+1],NULL), "all")) {
	   allframes = true;
	} else if (!strupcmp(Tcl_GetStringFromObj(objv[i+1],NULL), "last")) {
	  frame=-2;
	} else if (Tcl_GetIntFromObj(interp, objv[i+1], &frame) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure ", geomname, ": bad frame value", NULL);
	  return TCL_ERROR;
	}
      } else {
	Tcl_AppendResult(interp, "measure ", geomname, ": invalid syntax, no such keyword: ", argvcur, NULL);
	return TCL_ERROR;
      }
    }
  }

  if ((allframes || frame>=0) && (first>=0 || last>=0)) {
    Tcl_AppendResult(interp, "measure ", geomname, ": Ambiguous syntax: You cannot specify a frame AND a frame range (using first or last).", NULL);
    Tcl_AppendResult(interp, "\nUsage:\nmeasure ", geomname, " {<atomid1> ?<molid1>?} {<atomid2> ?<molid2>?} {<atomid3> ?<molid3>?} {<atomid4> ?<molid4>?}} ?molid <default molid>? [?frame <frame|all|last>? | ?first <first>? ?last <last>?]", NULL);
    return TCL_ERROR;    
  }

  if (allframes) first=0;

  // If no molecule was specified use top as default
  if (defmolid<0) defmolid = app->molecule_top();

  // Assign atom IDs and molecule IDs
  int i,numelem;
  Tcl_Obj **atmmol;
  for (i=0; i<numatms; i++) {
    if (Tcl_ListObjGetElements(interp, data[i], &numelem, &atmmol) != TCL_OK) {
      return TCL_ERROR;
    }

    if (Tcl_GetIntFromObj(interp, atmmol[0], atmid+i) != TCL_OK) {
      Tcl_AppendResult(interp, " measure ", geomname, ": bad atom index", NULL);
      return TCL_ERROR;
    }
    
    if (numelem==2) {
      if (Tcl_GetIntFromObj(interp, atmmol[1], molid+i) != TCL_OK) {
	Tcl_AppendResult(interp, " measure ", geomname, ": bad molid", NULL);
	return TCL_ERROR;
      }
    } else molid[i] = defmolid;
  }
  

  // Compute the value
  ResizeArray<float> gValues(1024);
  int ret_val;
  ret_val = measure_geom(app->moleculeList, molid, atmid, &gValues, frame, first, last,
			 defmolid, MEASURE_DIHED);
  if (ret_val<0) {
    Tcl_AppendResult(interp, measure_error(ret_val), NULL);
    return TCL_ERROR;
  }

  Tcl_Obj *tcl_result = Tcl_NewListObj(0, NULL);
  int numvalues = gValues.num();
  for (int count = 0; count < numvalues; count++) {
    Tcl_ListObjAppendElement(interp, tcl_result, Tcl_NewDoubleObj(gValues[count]));
  }
  Tcl_SetObjResult(interp, tcl_result);

  return TCL_OK;
}

// Function: vmd_measure_angle {3 atoms as {<atomid> ?<molid>?}} ?molid <default molid>? 
//                             ?frame [f|all|last]? | ?first <first>? ?last <last>?
//  Returns: the bond angle for the specified atoms
static int vmd_measure_angle(VMDApp *app, int argc, Tcl_Obj * const objv[], Tcl_Interp *interp) {
  int first=-1, last=-1, frame=-1;

  if(argc<2) {
    Tcl_WrongNumArgs(interp, 2, objv-1, (char *) "{{<atomid1> ?<molid1>?} {<atomid2> ?<molid2>?} {<atomid3> ?<molid3>?}} ?molid <default molid>? [?frame <frame|all|last>? | ?first <first>? ?last <last>?]");
    return TCL_ERROR;
  }

  int molid[3];
  int atmid[3];
  int defmolid = -1;
  bool allframes = false;

  // Read the atom list
  int numatms;
  Tcl_Obj **data;
  if (Tcl_ListObjGetElements(interp, objv[1], &numatms, &data) != TCL_OK) {
    Tcl_AppendResult(interp, " measure bond: bad syntax", NULL);
    Tcl_AppendResult(interp, " Usage: measure angle {{<atomid1> ?<molid1>?} {<atomid2> ?<molid2>?} {<atomid3> ?<molid3>?}} ?molid <default molid>? [?frame <frame|all|last>? | ?first <first>? ?last <last>?]", NULL);
    return TCL_ERROR;
  }

  if (numatms!=3) {
    Tcl_AppendResult(interp, " measure angle: must specify exactly three atoms in a list", NULL);
    return TCL_ERROR;
  }
    

  if (argc>3) {
    int i;
    for (i=2; i<argc; i+=2) {
      char *argvcur = Tcl_GetStringFromObj(objv[i],NULL);
      if (!strupncmp(argvcur, "molid", CMDLEN)) {
	if (Tcl_GetIntFromObj(interp, objv[i+1], &defmolid) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure angle: bad molid", NULL);
	  return TCL_ERROR;
	}
      } else if (!strupncmp(argvcur, "first", CMDLEN)) {
	if (Tcl_GetIntFromObj(interp, objv[i+1], &first) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure angle: bad first frame value", NULL);
	  return TCL_ERROR;
	}
      } else if (!strupncmp(argvcur, "last", CMDLEN)) {
	if (Tcl_GetIntFromObj(interp, objv[i+1], &last) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure angle: bad last frame value", NULL);
	  return TCL_ERROR;
	}
      } else if (!strupncmp(argvcur, "frame", CMDLEN)) {
	if (!strupcmp(Tcl_GetStringFromObj(objv[i+1],NULL), "all")) {
	  allframes = true;
	} else if (!strupcmp(Tcl_GetStringFromObj(objv[i+1],NULL), "last")) {
	  frame=-2;
	} else if (Tcl_GetIntFromObj(interp, objv[i+1], &frame) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure angle: bad frame value", NULL);
	  return TCL_ERROR;
	}
      } else {
	Tcl_AppendResult(interp, " measure angle: invalid syntax, no such keyword: ", argvcur, NULL);
	return TCL_ERROR;
      }
    }
  }

  if ((allframes || frame>=0) && (first>=0 || last>=0)) {
    Tcl_AppendResult(interp, "measure angle: Ambiguous syntax: You cannot specify a frame AND a frame range (using first or last).", NULL);
    Tcl_AppendResult(interp, "\nUsage:\nmeasure angle {<atomid1> ?<molid1>?} {<atomid2> ?<molid2>?} {<atomid3> ?<molid3>?}} ?molid <default molid>? [?frame <frame|all|last>? | ?first <first>? ?last <last>?]", NULL);
    return TCL_ERROR;    
  }

  if (allframes) first=0;

  // If no molecule was specified use top as default
  if (defmolid<0) defmolid = app->molecule_top();

  // Assign atom IDs and molecule IDs
  int i,numelem;
  Tcl_Obj **atmmol;
  for (i=0; i<numatms; i++) {
    if (Tcl_ListObjGetElements(interp, data[i], &numelem, &atmmol) != TCL_OK) {
      return TCL_ERROR;
    }

    if (Tcl_GetIntFromObj(interp, atmmol[0], atmid+i) != TCL_OK) {
      Tcl_AppendResult(interp, " measure angle: bad atom index", NULL);
      return TCL_ERROR;
    }
    
    if (numelem==2) {
      if (Tcl_GetIntFromObj(interp, atmmol[1], molid+i) != TCL_OK) {
	Tcl_AppendResult(interp, " measure angle: bad molid", NULL);
	return TCL_ERROR;
      }
    } else molid[i] = defmolid;
  }
  

  // Compute the value
  ResizeArray<float> gValues(1024);
  int ret_val;
  ret_val = measure_geom(app->moleculeList, molid, atmid, &gValues, frame, first, last,
			 defmolid, MEASURE_ANGLE);
  if (ret_val<0) {
    printf("ERROR\n %s\n", measure_error(ret_val));
    Tcl_AppendResult(interp, measure_error(ret_val), NULL);
    return TCL_ERROR;
  }

  Tcl_Obj *tcl_result = Tcl_NewListObj(0, NULL);
  int numvalues = gValues.num();
  for (int count = 0; count < numvalues; count++) {
    Tcl_ListObjAppendElement(interp, tcl_result, Tcl_NewDoubleObj(gValues[count]));
  }
  Tcl_SetObjResult(interp, tcl_result);

  return TCL_OK;
}

// Function: vmd_measure_bond {2 atoms as {<atomid> ?<molid>?}} ?molid <molid>? ?frame [f|all|last]? | ?first <first>? ?last <last>?
//  Returns: the bond length for the specified atoms
static int vmd_measure_bond(VMDApp *app, int argc, Tcl_Obj * const objv[], Tcl_Interp *interp) {
  int first=-1, last=-1, frame=-1;

  if (argc<2) {
    Tcl_WrongNumArgs(interp, 2, objv-1,
		     (char *) "{{<atomid1> ?<molid1>?} {<atomid2> ?<molid2>?}} ?molid <default molid>? [?frame <frame|all|last>? | ?first <first>? ?last <last>?]");
    return TCL_ERROR;
  }
  
  int molid[2];
  int atmid[2];
  int defmolid = -1;
  bool allframes = false;

  // Read the atom list
  int numatms;
  Tcl_Obj **data;
  if (Tcl_ListObjGetElements(interp, objv[1], &numatms, &data) != TCL_OK) {
    Tcl_AppendResult(interp, " measure bond: bad syntax", NULL);
    Tcl_AppendResult(interp, " Usage: measure bond {{<atomid1> ?<molid1>?} {<atomid2> ?<molid2>?}} ?molid <default>? [?frame <frame|all|last>? | ?first <first>? ?last <last>?]", NULL);
    return TCL_ERROR;
  }

  if (numatms!=2) {
    Tcl_AppendResult(interp, " measure bond: must specify exactly two atoms in a list", NULL);
    return TCL_ERROR;
  }
    

  if (argc>3) {
    int i;
    for (i=2; i<argc; i+=2) {
      char *argvcur = Tcl_GetStringFromObj(objv[i],NULL);
      if (!strupncmp(argvcur, "molid", CMDLEN)) {
	if (Tcl_GetIntFromObj(interp, objv[i+1], &defmolid) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure bond: bad molid", NULL);
	  return TCL_ERROR;
	}
      } else if (!strupncmp(argvcur, "first", CMDLEN)) {
	if (Tcl_GetIntFromObj(interp, objv[i+1], &first) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure bond: bad first frame value", NULL);
	  return TCL_ERROR;
	}
      } else if (!strupncmp(argvcur, "last", CMDLEN)) {
	if (Tcl_GetIntFromObj(interp, objv[i+1], &last) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure bond: bad last frame value", NULL);
	  return TCL_ERROR;
	}
      } else if (!strupncmp(argvcur, "frame", CMDLEN)) {
	if (!strupcmp(Tcl_GetStringFromObj(objv[i+1],NULL), "all")) {
	  allframes = true;
	} else if (!strupcmp(Tcl_GetStringFromObj(objv[i+1],NULL), "last")) {
	  frame=-2;
	} else if (Tcl_GetIntFromObj(interp, objv[i+1], &frame) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure bond: bad frame value", NULL);
	  return TCL_ERROR;
	}
      } else {
	Tcl_AppendResult(interp, " measure bond: invalid syntax, no such keyword: ", argvcur, NULL);
	return TCL_ERROR;
      }
    }
  }

  if ((allframes || frame>=0) && (first>=0 || last>=0)) {
    Tcl_AppendResult(interp, "measure bond: Ambiguous syntax: You cannot specify a frame AND a frame range (using first or last).", NULL);
    Tcl_AppendResult(interp, "\nUsage:\nmeasure bond <molid1>/<atomid1> <molid2>/<atomid2> [?frame <frame|all|last>? | ?first <first>? ?last <last>?]", NULL);
    return TCL_ERROR;    
  }

  if (allframes) first=0;

  // If no molecule was specified use top as default
  if (defmolid<0) defmolid = app->molecule_top();

  // Assign atom IDs and molecule IDs
  int i,numelem;
  Tcl_Obj **atmmol;
  for (i=0; i<numatms; i++) {
    if (Tcl_ListObjGetElements(interp, data[i], &numelem, &atmmol) != TCL_OK) {
      return TCL_ERROR;
    }

    if (Tcl_GetIntFromObj(interp, atmmol[0], atmid+i) != TCL_OK) {
      Tcl_AppendResult(interp, " measure bond: bad atom index", NULL);
      return TCL_ERROR;
    }
    
    if (numelem==2) {
      if (Tcl_GetIntFromObj(interp, atmmol[1], molid+i) != TCL_OK) {
	Tcl_AppendResult(interp, " measure bond: bad molid", NULL);
	return TCL_ERROR;
      }
    } else molid[i] = defmolid;
  }
  

  // Compute the value
  ResizeArray<float> gValues(1024);
  int ret_val;
  ret_val = measure_geom(app->moleculeList, molid, atmid, &gValues, frame, first, last,
			 defmolid, MEASURE_BOND);
  if (ret_val<0) {
    printf("ERROR\n %s\n", measure_error(ret_val));
    Tcl_AppendResult(interp, measure_error(ret_val), NULL);
    return TCL_ERROR;
  }

  Tcl_Obj *tcl_result = Tcl_NewListObj(0, NULL);
  int numvalues = gValues.num();
  for (int count = 0; count < numvalues; count++) {
    Tcl_ListObjAppendElement(interp, tcl_result, Tcl_NewDoubleObj(gValues[count]));
  }
  Tcl_SetObjResult(interp, tcl_result);

  return TCL_OK;
}

// Function: vmd_measure_rmsf <selection> first <first> last <last> step <step>
//  Returns: position variance of the selected atoms over the selected frames
//  Example: measure rmsf atomselect76 0 20 1
static int vmd_measure_rmsf(VMDApp *app, int argc, Tcl_Obj * const objv[], Tcl_Interp *interp) {
  int first = 0;  // start with first frame by default
  int last = -1;  // finish with last frame by default
  int step = 1;   // use all frames by default

  if (argc < 2 || argc > 8) {
    Tcl_WrongNumArgs(interp, 2, objv-1, (char *)"?selection first <first> last <last> step <step>?");
    return TCL_ERROR;
  }
  AtomSel *sel = tcl_commands_get_sel(interp, Tcl_GetStringFromObj(objv[1],NULL)
);
  if (!sel) {
    Tcl_AppendResult(interp, "measure rmsf: no atom selection", NULL);
    return TCL_ERROR;
  }
 
  int i;
  for (i=2; i<argc; i+=2) {
    char *argvcur = Tcl_GetStringFromObj(objv[i],NULL);
    if (!strupncmp(argvcur, "first", CMDLEN)) {
      if (Tcl_GetIntFromObj(interp, objv[i+1], &first) != TCL_OK) {
        Tcl_AppendResult(interp, "measure rmsf: bad first frame value", NULL);
        return TCL_ERROR;
      }
    } else if (!strupncmp(argvcur, "last", CMDLEN)) {
      if (Tcl_GetIntFromObj(interp, objv[i+1], &last) != TCL_OK) {
        Tcl_AppendResult(interp, "measure rmsf: bad last frame value", NULL);
        return TCL_ERROR;
      }
    } else if (!strupncmp(argvcur, "step", CMDLEN)) {
      if (Tcl_GetIntFromObj(interp, objv[i+1], &step) != TCL_OK) {
        Tcl_AppendResult(interp, "measure rmsf: bad frame step value", NULL);
        return TCL_ERROR;
      }
    } else {
      Tcl_AppendResult(interp, "measure rmsf: invalid syntax, no such keyword: ", argvcur, NULL);
      return TCL_ERROR;
    }
  }

  float *rmsf = new float[sel->selected];
  int ret_val = measure_rmsf(sel, app->moleculeList, first, last, step, rmsf);
  if (ret_val < 0) {
    Tcl_AppendResult(interp, "measure rmsf: ", measure_error(ret_val), NULL);
    delete [] rmsf;
    return TCL_ERROR;
  }

  Tcl_Obj *tcl_result = Tcl_NewListObj(0, NULL);

  for (i=0; i<sel->selected; i++) {
    Tcl_ListObjAppendElement(interp, tcl_result, Tcl_NewDoubleObj(rmsf[i]));
  }

  Tcl_SetObjResult(interp, tcl_result);

  delete [] rmsf;

  return TCL_OK;
}


// measure radius of gyration for selected atoms
static int vmd_measure_rgyr(VMDApp *app, int argc, Tcl_Obj *const objv[], Tcl_Interp *interp)
{
  if (argc != 2 && argc != 4 ) {
    Tcl_WrongNumArgs(interp, 2, objv-1,
      (char *)"<selection> ?weight <weightstring>?");
    return TCL_ERROR;
  }
  if (argc == 4 && strcmp(Tcl_GetStringFromObj(objv[2],NULL), "weight")) {
    Tcl_SetResult(interp, (char *) "measure rgyr: parameter can only be 'weight'", TCL_STATIC);
    return TCL_ERROR;
  }

  // get the selection
  AtomSel *sel = tcl_commands_get_sel(interp, Tcl_GetStringFromObj(objv[1],NULL));
  if (!sel) {
    Tcl_SetResult(interp, (char *) "measure rgyr: no atom selection", TCL_STATIC);
    return TCL_ERROR;
  }

  // get the weight
  float *weight = new float[sel->selected];
  {
    int ret_val;
    if (argc == 2) {          // only from atom selection, so weight is 1
      ret_val = tcl_get_weights(interp, app, sel, NULL, weight);
    } else {
      ret_val = tcl_get_weights(interp, app, sel, objv[3], weight);
    }
    if (ret_val < 0) {
      Tcl_AppendResult(interp, "measure rgyr: ", measure_error(ret_val),
                       NULL);
      delete [] weight;
      return TCL_ERROR;
    }
  }
  float rgyr;
  int ret_val = measure_rgyr(sel, app->moleculeList, weight, &rgyr);
  delete [] weight;
  if (ret_val < 0) {
    Tcl_AppendResult(interp, "measure rgyr: ", measure_error(ret_val),
                     NULL);
    return TCL_ERROR;
  }
  Tcl_SetObjResult(interp, Tcl_NewDoubleObj(rgyr));
  return TCL_OK;
}


// Function: vmd_measure_minmax <selection>
//  Returns: the cartesian range of a selection (min/max){x,y,z}
//  Example: vmd_measure_minmax atomselect76
//     {-5 0 0} {15 10 11.2}
static int vmd_measure_minmax(VMDApp *app, int argc, Tcl_Obj * const objv[], Tcl_Interp *interp) {
  const float *radii = NULL;
  if (argc != 2 && argc != 3) {
    Tcl_WrongNumArgs(interp, 2, objv-1, (char *)"?selection?");
    return TCL_ERROR;
  }
  if (argc == 3 && strcmp(Tcl_GetStringFromObj(objv[2],NULL), "-withradii")) {
    Tcl_SetResult(interp, (char *) "measure minmax: parameter can only be '-withradii'", TCL_STATIC);
    return TCL_ERROR;
  }

  AtomSel *sel = tcl_commands_get_sel(interp, Tcl_GetStringFromObj(objv[1],NULL));
  if (!sel) {
    Tcl_AppendResult(interp, "measure minmax: no atom selection", NULL);
    return TCL_ERROR;
  }

  float min_coord[3], max_coord[3];
  const float *framepos = sel->coordinates(app->moleculeList);
  if (!framepos) return TCL_ERROR;

  // get atom radii if requested
  if (argc == 3) {
    Molecule *mol = app->moleculeList->mol_from_id(sel->molid());
    radii = mol->extra.data("radius");
  } 

  int ret_val = measure_minmax(sel, framepos, radii, min_coord, max_coord);
  if (ret_val < 0) {
    Tcl_AppendResult(interp, "measure minmax: ", measure_error(ret_val));
    return TCL_ERROR;
  }

  Tcl_Obj *list1 = Tcl_NewListObj(0, NULL);
  Tcl_Obj *list2 = Tcl_NewListObj(0, NULL);
  Tcl_Obj *tcl_result = Tcl_NewListObj(0, NULL);

  Tcl_ListObjAppendElement(interp, list1, Tcl_NewDoubleObj(min_coord[0]));
  Tcl_ListObjAppendElement(interp, list1, Tcl_NewDoubleObj(min_coord[1]));
  Tcl_ListObjAppendElement(interp, list1, Tcl_NewDoubleObj(min_coord[2]));

  Tcl_ListObjAppendElement(interp, list2, Tcl_NewDoubleObj(max_coord[0]));
  Tcl_ListObjAppendElement(interp, list2, Tcl_NewDoubleObj(max_coord[1]));
  Tcl_ListObjAppendElement(interp, list2, Tcl_NewDoubleObj(max_coord[2]));

  Tcl_ListObjAppendElement(interp, tcl_result, list1);
  Tcl_ListObjAppendElement(interp, tcl_result, list2);
  Tcl_SetObjResult(interp, tcl_result);

  return TCL_OK;
}

/* Function: vmd_measure_rmsd <selection1> <selection2> 
                       {weight [none|atom value|string array]}

   Returns the RMSD between the two selection, taking the weight (if
any) into account.  If number of elements in the weight != num_atoms
in sel1 then (num in weight = num selected in sel1 = num selected in
sel2) else (num in weight = total num in sel1 = total num in sel2).
The weights are taken from the FIRST selection, if needed

Examples:
  set sel1 [atomselect 0 all]
  set sel2 [atomselect 1 all]
  measure rmsd $sel1 $sel2
  measure rmsd $sel1 $sel2 weight mass
  set sel3 [atomselect 0 "index 3 4 5"]
  set sel4 [atomselect 1 "index 8 5 9"]    # gets turned to 5 8 9
  measure rmsd $sel3 $sel4 weight occupancy
*/
static int vmd_measure_rmsd(VMDApp *app, int argc, Tcl_Obj * const objv[], Tcl_Interp *interp)
{
  if (argc !=3 && argc != 5) {
    Tcl_WrongNumArgs(interp, 2, objv-1, 
      (char *)"<selection1> <selection2> ?weight <weightstring>?");
    return TCL_ERROR;
  }
  // get the selections
  AtomSel *sel1 = tcl_commands_get_sel(interp, Tcl_GetStringFromObj(objv[1],NULL));
  AtomSel *sel2 = tcl_commands_get_sel(interp, Tcl_GetStringFromObj(objv[2],NULL));
  if (!sel1 || !sel2) {
    Tcl_AppendResult(interp, "measure rmsd: no atom selection", NULL);
    return TCL_ERROR;
  }

  if (sel1->selected != sel2->selected) {
    Tcl_AppendResult(interp, "measure rmsd: selections must have the same number of atoms", NULL);
    return TCL_ERROR;
  }
  if (!sel1->selected) {
    Tcl_AppendResult(interp, "measure rmsd: No atoms selected", NULL);
    return TCL_ERROR;
  }
  float *weight = new float[sel1->selected];
  {
    int ret_val;
    if (argc == 3) {
      ret_val = tcl_get_weights(interp, app, sel1, NULL, weight);
    } else {
      ret_val = tcl_get_weights(interp, app, sel1, objv[4], weight);
    }
    if (ret_val < 0) {
      Tcl_AppendResult(interp, "measure rmsd: ", measure_error(ret_val),
		       NULL);
      delete [] weight;
      return TCL_ERROR;
    }
  }
  // compute the rmsd
  {
    float rmsd = 0;
    const float *x = sel1->coordinates(app->moleculeList);
    const float *y = sel2->coordinates(app->moleculeList);
    if (!x || !y) {
      delete [] weight;
      return TCL_ERROR;
    }
    int ret_val = measure_rmsd(sel1, sel2, sel1->selected, x, y, weight, &rmsd);
    delete [] weight;
    if (ret_val < 0) {
      Tcl_AppendResult(interp, "measure rmsd: ", measure_error(ret_val),
		       NULL);
      return TCL_ERROR;
    }
    Tcl_SetObjResult(interp, Tcl_NewDoubleObj(rmsd));
  }
  return TCL_OK;
}

//////////////////////////////////////////////
// measure fit $sel1 $sel2 [weight <weights>][
static int vmd_measure_fit(VMDApp *app, int argc, Tcl_Obj * const objv[], Tcl_Interp *interp)
{
  AtomSel *sel1, *sel2;
  int *order = NULL;
  float *weight = NULL;
  int rc;

  if (argc != 3 && argc != 5 
      && argc != 7) {
    Tcl_WrongNumArgs(interp, 2, objv-1, 
       (char *)"<selection1> <selection2> ?weight <weightstring>?");
    return TCL_ERROR;
  } else if (argc == 5
             && strcmp("weight", Tcl_GetStringFromObj(objv[3], NULL)) 
             && strcmp("order", Tcl_GetStringFromObj(objv[3], NULL))) {
    Tcl_WrongNumArgs(interp, 2, objv-1, 
       (char *)"<selection1> <selection2> ?weight <weightstring>?");
    return TCL_ERROR;
  } else if (argc == 7 && 
             (strcmp("weight", Tcl_GetStringFromObj(objv[3], NULL)) || 
              strcmp("order", Tcl_GetStringFromObj(objv[5], NULL)))) {
    Tcl_WrongNumArgs(interp, 2, objv-1, 
       (char *)"<selection1> <selection2> ?weight <weightstring>?");
    return TCL_ERROR;
  }

  // get the selections
  sel1 = tcl_commands_get_sel(interp, Tcl_GetStringFromObj(objv[1],NULL));
  sel2 = tcl_commands_get_sel(interp, Tcl_GetStringFromObj(objv[2],NULL));

  if (!sel1 || !sel2) {
    Tcl_AppendResult(interp, "measure fit: no atom selection", NULL);
    return TCL_ERROR;
  }

  int num = sel1->selected;
  if (!num) {
    Tcl_AppendResult(interp, "measure fit: No atoms selected", NULL);
    return TCL_ERROR;
  }
  if (num != sel2->selected) {
    Tcl_AppendResult(interp, "measure fit: selections must have the same number of atoms", NULL);
    return TCL_ERROR;
  }

  // get the weights
  weight = new float[num]; 
  if (argc > 3 && !strcmp("weight", Tcl_GetStringFromObj(objv[3], NULL))) {
    // get user weight parameter
    rc = tcl_get_weights(interp, app, sel1, objv[4], weight);
  } else {
    // default to weights of 1.0
    rc = tcl_get_weights(interp, app, sel1, NULL, weight);
  }
  if (rc < 0) {
    Tcl_AppendResult(interp, "measure fit: ", measure_error(rc), NULL);
    delete [] weight;
    return TCL_ERROR;
  }


  int orderparam = 0;
  if (argc == 5 && !strcmp("order", Tcl_GetStringFromObj(objv[3], NULL))) {
    orderparam = 4;
  } else if (argc == 7 && !strcmp("order", Tcl_GetStringFromObj(objv[5], NULL))) {
    orderparam = 6;
  }

  if (orderparam != 0) {
    // get the atom order
    order = new int[num];
    rc = tcl_get_orders(interp, num, objv[orderparam], order);
    if (rc < 0) {
      Tcl_AppendResult(interp, "measure fit: ", measure_error(rc), NULL);
      delete [] order;
      return TCL_ERROR;
    }
  }

  // compute the transformation matrix
  Matrix4 T;
  const float *x = sel1->coordinates(app->moleculeList);
  const float *y = sel2->coordinates(app->moleculeList);

  int ret_val = MEASURE_ERR_NOMOLECULE;
  if (x && y) 
    ret_val = measure_fit(sel1, sel2, x, y, weight, order, &T);

  delete [] weight;

  if (order != NULL)
    delete [] order;

  if (ret_val < 0) {
    Tcl_AppendResult(interp, "measure fit: ", measure_error(ret_val), NULL);
    return TCL_ERROR;
  }

  // and return the matrix
  tcl_append_matrix(interp, T.mat);
  return TCL_OK;
}

//////////////////////////////////
// measure inverse 4x4matrix
static int vmd_measure_inverse(int argc, Tcl_Obj * const objv[], Tcl_Interp *interp) {
  // make there there is exactly one matrix
  if (argc != 2) {
    Tcl_WrongNumArgs(interp, 2, objv-1, (char *)"<matrix>");
    return TCL_ERROR;
  }
  // Get the first matrix
  Matrix4 inv;
  if (tcl_get_matrix("measure inverse: ",interp,objv[1],inv.mat) != TCL_OK) {
    return TCL_ERROR;
  }
  if (inv.inverse()) {
    Tcl_AppendResult(interp, "Singular Matrix; inverse not computed", NULL);
    return TCL_ERROR;
  }
  tcl_append_matrix(interp, inv.mat);
  return TCL_OK;
}

// Find all atoms p in sel1 and q in sel2 within the cutoff.  
static int vmd_measure_contacts(VMDApp *app, int argc, Tcl_Obj * const objv[], Tcl_Interp *interp) {
  
  // Cutoff, and either one or two atom selections
  if (argc != 3 && argc != 4) {
    Tcl_WrongNumArgs(interp, 2, objv-1, (char *)"cutoff selection ?selection?");
    return TCL_ERROR;
  }
  AtomSel *sel1 = tcl_commands_get_sel(interp, Tcl_GetStringFromObj(objv[2],NULL));
  if (!sel1) {
    Tcl_AppendResult(interp, "measure contacts: no atom selection", NULL);
    return TCL_ERROR;
  }
  AtomSel *sel2 = NULL;
  if (argc == 4) {
    sel2 = tcl_commands_get_sel(interp, Tcl_GetStringFromObj(objv[3],NULL));
    if (!sel2) {
      Tcl_AppendResult(interp, "measure contacts: no atom selection", NULL);
      return TCL_ERROR;
    }
  }
  if (!sel2) sel2 = sel1;
  
  double cutoff;
  if (Tcl_GetDoubleFromObj(interp, objv[1], &cutoff) != TCL_OK)
    return TCL_ERROR;

  const float *pos1 = sel1->coordinates(app->moleculeList);
  const float *pos2 = sel2->coordinates(app->moleculeList);
  if (!pos1 || !pos2) {
    Tcl_AppendResult(interp, "measure contacts: error, molecule contains no coordinates", NULL);
    return TCL_ERROR;
  }
  Molecule *mol1 = app->moleculeList->mol_from_id(sel1->molid());
  Molecule *mol2 = app->moleculeList->mol_from_id(sel2->molid());

  GridSearchPair *pairlist = vmd_gridsearch3(pos1, sel1->num_atoms, sel1->on, pos2, sel2->num_atoms, sel2->on, (float) cutoff, -1, (sel1->num_atoms + sel2->num_atoms) * 27);
  GridSearchPair *p, *tmp;
  Tcl_Obj *list1 = Tcl_NewListObj(0, NULL);
  Tcl_Obj *list2 = Tcl_NewListObj(0, NULL);
  for (p=pairlist; p != NULL; p=tmp) {
    // throw out pairs that are already bonded
    MolAtom *a1 = mol1->atom(p->ind1);
    if (mol1 != mol2 || !a1->bonded(p->ind2)) {
      Tcl_ListObjAppendElement(interp, list1, Tcl_NewIntObj(p->ind1));
      Tcl_ListObjAppendElement(interp, list2, Tcl_NewIntObj(p->ind2));
    }
    tmp = p->next;
    free(p);
  }
  Tcl_Obj *result = Tcl_NewListObj(0, NULL);
  Tcl_ListObjAppendElement(interp, result, list1);
  Tcl_ListObjAppendElement(interp, result, list2);
  Tcl_SetObjResult(interp, result);
  return TCL_OK;
}


// measure g(r) for two selections, with delta, rmax, usepbc, first/last/step 
// frame parameters the code will compute the normalized histogram.
static int vmd_measure_gofr(VMDApp *app, int argc, Tcl_Obj * const objv[], Tcl_Interp *interp) {
  int i;
  // initialize optional arguments to default values
  double rmax=10.0;
  double delta=0.1;
  int usepbc=0;
  int selupdate=0;
  int first=-1, last=-1, step=1;
  int rc;

  // argument error message
  const char *argerrmsg = "sel1 sel2 [delta <value>] [rmax <value>] [usepbc <bool>] [selupdate <bool>] [first <value>] [last <value>] [step <value>]";

  // Two atom selections and optional keyword/value pairs.
  if ((argc < 3) || (argc > 17) || (argc % 2 == 0) )  {
    Tcl_WrongNumArgs(interp, 2, objv-1, (char *)argerrmsg);
    return TCL_ERROR;
  }

  // check atom selections
  AtomSel *sel1 = tcl_commands_get_sel(interp, Tcl_GetStringFromObj(objv[1], NULL));
  if (!sel1) {
    Tcl_AppendResult(interp, "measure gofr: invalid first atom selection", NULL);
    return TCL_ERROR;
  }

  AtomSel *sel2 = tcl_commands_get_sel(interp, Tcl_GetStringFromObj(objv[2], NULL));
  if (!sel2) {
    Tcl_AppendResult(interp, "measure gofr: invalid second atom selection", NULL);
    return TCL_ERROR;
  }

  // parse optional arguments
  for (i=3; i<argc; i+=2) {
    const char *opt = Tcl_GetStringFromObj(objv[i], NULL);
    if (i==(argc-1)) {
      Tcl_WrongNumArgs(interp, 2, objv-1, (char *)argerrmsg);
      return TCL_ERROR;
    }
    if (!strcmp(opt, "delta")) {
      if (Tcl_GetDoubleFromObj(interp, objv[i+1], &delta) != TCL_OK)
        return TCL_ERROR;
      if (delta <= 0.0) {
        Tcl_AppendResult(interp, "measure gofr: invalid 'delta' value", NULL);
        return TCL_ERROR;
      }
    } else if (!strcmp(opt, "rmax")) {
      if (Tcl_GetDoubleFromObj(interp, objv[i+1], &rmax) != TCL_OK)
        return TCL_ERROR;
      if (rmax <= 0.0) {
        Tcl_AppendResult(interp, "measure gofr: invalid 'rmax' value", NULL);
        return TCL_ERROR;
      }
    } else if (!strcmp(opt, "usepbc")) {
      if (Tcl_GetBooleanFromObj(interp, objv[i+1], &usepbc) != TCL_OK)
        return TCL_ERROR;
    } else if (!strcmp(opt, "selupdate")) {
      if (Tcl_GetBooleanFromObj(interp, objv[i+1], &selupdate) != TCL_OK)
        return TCL_ERROR;
    } else if (!strcmp(opt, "first")) {
      if (Tcl_GetIntFromObj(interp, objv[i+1], &first) != TCL_OK)
        return TCL_ERROR;
    } else if (!strcmp(opt, "last")) {
      if (Tcl_GetIntFromObj(interp, objv[i+1], &last) != TCL_OK)
        return TCL_ERROR;
    } else if (!strcmp(opt, "step")) {
      if (Tcl_GetIntFromObj(interp, objv[i+1], &step) != TCL_OK)
        return TCL_ERROR;
    } else { // unknown keyword.
      Tcl_AppendResult(interp, "unknown keyword '", opt, "'. usage: measure gofr ", argerrmsg,NULL);
      return TCL_ERROR;
    }
  }

  // allocate and initialize histogram arrays
  int    count_h = (int)(rmax / delta + 1.0);
  int    *hlist  = new int[count_h];
  double *gofr   = new double[count_h];
  double *numint = new double[count_h];
  double *histog = new double[count_h];
  int *framecntr = new int[3];

  // do the gofr calculation
  rc = measure_gofr(sel1, sel2, app->moleculeList,
               count_h, hlist, gofr, numint, histog,
               (float) delta,
               first, last, step, framecntr,
               usepbc, selupdate);

  // XXX: this needs a 'case' structure to provide more meaninful error messages.
  if (rc != MEASURE_NOERR) { 
    Tcl_AppendResult(interp, "measure gofr: error during g(r) calculation.", NULL);
    return TCL_ERROR;
  }

  // convert the results of the lowlevel call to tcl lists
  // and build a list from them as return value.
  Tcl_Obj *tcl_result = Tcl_NewListObj(0, NULL);
  Tcl_Obj *tcl_rlist  = Tcl_NewListObj(0, NULL);
  Tcl_Obj *tcl_gofr   = Tcl_NewListObj(0, NULL);
  Tcl_Obj *tcl_numint = Tcl_NewListObj(0, NULL);
  Tcl_Obj *tcl_histog = Tcl_NewListObj(0, NULL);
  Tcl_Obj *tcl_frames = Tcl_NewListObj(0, NULL);

  // build lists with results ready for plotting
  for (i=0; i<count_h; i++) { 
    Tcl_ListObjAppendElement(interp, tcl_rlist,  Tcl_NewDoubleObj(delta * (double)(i+1)));
    Tcl_ListObjAppendElement(interp, tcl_gofr,   Tcl_NewDoubleObj(gofr[i]));
    Tcl_ListObjAppendElement(interp, tcl_numint, Tcl_NewDoubleObj(numint[i]));
    Tcl_ListObjAppendElement(interp, tcl_histog, Tcl_NewDoubleObj(histog[i]));
  }

  // build list with number of frames: 
  // total, skipped and processed (one entry for each algrorithm).
  Tcl_ListObjAppendElement(interp, tcl_frames, Tcl_NewIntObj(framecntr[0]));
  Tcl_ListObjAppendElement(interp, tcl_frames, Tcl_NewIntObj(framecntr[1]));
  Tcl_ListObjAppendElement(interp, tcl_frames, Tcl_NewIntObj(framecntr[2]));

  // build final list-of-lists as return value
  Tcl_ListObjAppendElement(interp, tcl_result, tcl_rlist);
  Tcl_ListObjAppendElement(interp, tcl_result, tcl_gofr);
  Tcl_ListObjAppendElement(interp, tcl_result, tcl_numint);
  Tcl_ListObjAppendElement(interp, tcl_result, tcl_histog);
  Tcl_ListObjAppendElement(interp, tcl_result, tcl_frames);
  Tcl_SetObjResult(interp, tcl_result);

  delete [] gofr;
  delete [] numint;
  delete [] histog;
  delete [] hlist;
  delete [] framecntr;
  return TCL_OK;
}


static int vmd_measure_hbonds(VMDApp *app, int argc, Tcl_Obj * const objv[], Tcl_Interp *interp) {
  
  // Cutoff, angle, and either one or two atom selections
  if (argc != 4 && argc != 5) {
    Tcl_WrongNumArgs(interp, 2, objv-1, (char *)"cutoff angle selection ?selection?");
    return TCL_ERROR;
  }
  AtomSel *sel1 = tcl_commands_get_sel(interp, Tcl_GetStringFromObj(objv[3],NULL));
  if (!sel1) {
    Tcl_AppendResult(interp, "measure hbonds: invalid first atom selection", NULL);
    return TCL_ERROR;
  }

  AtomSel *sel2 = NULL;
  if (argc == 5) {
    sel2 = tcl_commands_get_sel(interp, Tcl_GetStringFromObj(objv[4],NULL));
    if (!sel2) {
      Tcl_AppendResult(interp, "measure hbonds: invalid second atom selection", NULL);
      return TCL_ERROR;
    }
  }
  if (sel2 && sel2->molid() != sel1->molid()) {
    Tcl_AppendResult(interp, "measure hbonds: error, atom selections must come from same molecule.", NULL);
    return TCL_ERROR;
  }
  double cutoff;
  if (Tcl_GetDoubleFromObj(interp, objv[1], &cutoff) != TCL_OK) 
    return TCL_ERROR;

  double maxangle;
  if (Tcl_GetDoubleFromObj(interp, objv[2], &maxangle) != TCL_OK) 
    return TCL_ERROR;
  
  const float *pos = sel1->coordinates(app->moleculeList);
  if (!pos) {
    Tcl_AppendResult(interp, "measure bondsearch: error, molecule contains no coordinates", NULL);
    return TCL_ERROR;
  }

  // XXX the actual code for measuring hbonds doesn't belong here, it should
  //     be moved into Measure.[Ch] where it really belongs.  This file
  //     only implements the Tcl interface, and should not be doing the
  //     hard core math, particularly if we want to expose the same
  //     feature via other scripting interfaces.  Also, having a single
  //     implementation avoids having different Tcl/Python bugs in the 
  //     long-term.  Too late to do anything about this now, but should be
  //     addressed for the next major version when time allows.

  // XXX This code is close, but not identical to the HBonds code in 
  //     DrawMolItem.  Is there any good reason they aren't identical?
  //     This version does a few extra tests that the other does not.

  Molecule *mol = app->moleculeList->mol_from_id(sel1->molid());

  const int *A = sel1->on;
  const int *B = sel2 ? sel2->on : sel1->on;
 
  GridSearchPair *pairlist = vmd_gridsearch2(pos, sel1->num_atoms, A, B, (float) cutoff, sel1->num_atoms * 27);
  GridSearchPair *p, *tmp;
  float donortoH[3], Htoacceptor[3];
  Tcl_Obj *donlist = Tcl_NewListObj(0, NULL);
  Tcl_Obj *hydlist = Tcl_NewListObj(0, NULL);
  Tcl_Obj *acclist = Tcl_NewListObj(0, NULL);
  for (p=pairlist; p != NULL; p=tmp) {
    MolAtom *a1 = mol->atom(p->ind1); 
    MolAtom *a2 = mol->atom(p->ind2); 
    
    // neither the donor nor acceptor may be hydrogens
    if (mol->atom(p->ind1)->atomType == ATOMHYDROGEN ||
        mol->atom(p->ind2)->atomType == ATOMHYDROGEN) {
      tmp = p->next;
      free(p);
      continue;
    } 
    if (!a1->bonded(p->ind2)) {
      int b1 = a1->bonds;
      int b2 = a2->bonds;
      const float *coor1 = pos + 3*p->ind1;
      const float *coor2 = pos + 3*p->ind2;
      int k;
      // first treat sel1 as donor
      for (k=0; k<b1; k++) {
        const int hindex = a1->bondTo[k];
        if (mol->atom(hindex)->atomType == ATOMHYDROGEN) {         
          const float *hydrogen = pos + 3*hindex;
          vec_sub(donortoH,hydrogen,coor1);
          vec_sub(Htoacceptor,coor2,hydrogen);
          if (angle(donortoH, Htoacceptor)  < maxangle ) {
            Tcl_ListObjAppendElement(interp, donlist, Tcl_NewIntObj(p->ind1));
            Tcl_ListObjAppendElement(interp, acclist, Tcl_NewIntObj(p->ind2));
            Tcl_ListObjAppendElement(interp, hydlist, Tcl_NewIntObj(hindex));
          }
        }
      }
      // if only one atom selection was given, treat sel2 as a donor as well
      if (!sel2) {
        for (k=0; k<b2; k++) {
          const int hindex = a2->bondTo[k];
          if (mol->atom(hindex)->atomType == ATOMHYDROGEN) {
            const float *hydrogen = pos + 3*hindex;
            vec_sub(donortoH,hydrogen,coor2);
            vec_sub(Htoacceptor,coor1,hydrogen);
            if (angle(donortoH, Htoacceptor)  < maxangle ) {
              Tcl_ListObjAppendElement(interp, donlist, Tcl_NewIntObj(p->ind2));
              Tcl_ListObjAppendElement(interp, acclist, Tcl_NewIntObj(p->ind1));
              Tcl_ListObjAppendElement(interp, hydlist, Tcl_NewIntObj(hindex));
            }
          }
        }
      } 
    }
    tmp = p->next;
    free(p);
  }
  Tcl_Obj *result = Tcl_NewListObj(0, NULL);
  Tcl_ListObjAppendElement(interp, result, donlist);
  Tcl_ListObjAppendElement(interp, result, acclist);
  Tcl_ListObjAppendElement(interp, result, hydlist);
  Tcl_SetObjResult(interp, result);
  return TCL_OK;
}

  
static int vmd_measure_sasa(VMDApp *app, int argc, Tcl_Obj * const objv[], Tcl_Interp *interp) {

  int i;
  // srad and one atom selection, plus additional options
  if (argc < 3) {
    Tcl_WrongNumArgs(interp, 2, objv-1, (char *)"srad selection opt? value? ...");
    return TCL_ERROR;
  }
  // parse options
  Tcl_Obj *ptsvar = NULL;
  AtomSel *restrictsel = NULL;
  int nsamples = -1;
  int *sampleptr = NULL;
  for (i=3; i<argc-1; i+=2) {
    const char *opt = Tcl_GetStringFromObj(objv[i], NULL);
    if (!strcmp(opt, "-points")) {
      ptsvar = objv[i+1];
    } else if (!strcmp(opt, "-restrict")) {
      restrictsel = tcl_commands_get_sel(interp, 
          Tcl_GetStringFromObj(objv[i+1], NULL));
      if (!restrictsel) {
        Tcl_AppendResult(interp, "measure sasa: invalid restrict atom selection", NULL);
        return TCL_ERROR;
      }
    } else if (!strcmp(opt, "-samples")) {
      if (Tcl_GetIntFromObj(interp, objv[i+1], &nsamples) != TCL_OK)
        return TCL_ERROR;
      sampleptr = &nsamples;
    } else {
      Tcl_AppendResult(interp, "measure sasa: unknown option '", opt, "'", 
          NULL);
      return TCL_ERROR;
    }
  }

  double srad;
  if (Tcl_GetDoubleFromObj(interp, objv[1], &srad) != TCL_OK) 
    return TCL_ERROR;
  AtomSel *sel1 = tcl_commands_get_sel(interp, Tcl_GetStringFromObj(objv[2],NULL));
  if (!sel1) {
    Tcl_AppendResult(interp, "measure sasa: invalid first atom selection", NULL);
    return TCL_ERROR;
  }

  const float *pos = sel1->coordinates(app->moleculeList);
  if (!pos) {
    Tcl_AppendResult(interp, "measure sasa: error, molecule contains no coordinates", NULL);
    return TCL_ERROR;
  }
  Molecule *mol = app->moleculeList->mol_from_id(sel1->molid());
  const float *radius = mol->extra.data("radius");

  ResizeArray<float> sasapts;
  float sasa = 0;
  int rc = measure_sasa(sel1, pos, radius, (float) srad, &sasa, &sasapts, 
                        restrictsel, sampleptr);
  if (rc < 0) {
    Tcl_AppendResult(interp, "measure: sasa: ", measure_error(rc), NULL);
    return TCL_ERROR;
  }
  Tcl_SetObjResult(interp, Tcl_NewDoubleObj(sasa));
  if (ptsvar) {
    // construct list from sasapts
    Tcl_Obj *listobj = Tcl_NewListObj(0, NULL);
    i=0;
    while (i<sasapts.num()) {
      Tcl_Obj *elem = Tcl_NewListObj(0, NULL);
      Tcl_ListObjAppendElement(interp, elem, Tcl_NewDoubleObj(sasapts[i++]));
      Tcl_ListObjAppendElement(interp, elem, Tcl_NewDoubleObj(sasapts[i++]));
      Tcl_ListObjAppendElement(interp, elem, Tcl_NewDoubleObj(sasapts[i++]));
      Tcl_ListObjAppendElement(interp, listobj, elem);
    }
    Tcl_ObjSetVar2(interp, ptsvar, NULL, listobj, 0);
  }
  return TCL_OK;
}

// Function: vmd_measure_bond {2 atoms as {<atomid> ?<molid>?}} ?molid <molid>? ?frame [f|all]? | ?first <first>? ?last <last>?
//  Returns: the bond length for the specified atoms
static int vmd_measure_energy(VMDApp *app, int argc, Tcl_Obj * const objv[], Tcl_Interp *interp) {

  if (argc<3) {
    Tcl_WrongNumArgs(interp, 2, objv-1,
		     (char *) "bond|angle|dihed|imprp|vdw|elec {{<atomid1> ?<molid1>?} {<atomid2> ?<molid2>?}} ?molid <default molid>? [?frame <frame|all>? | ?first <first>? ?last <last>?]");
    return TCL_ERROR;
  }

  int geomtype, reqatoms;
  char *geomname = Tcl_GetStringFromObj(objv[1],NULL);
  if        (!strncmp(geomname, "bond", 4)) {
    reqatoms = 2; geomtype = MEASURE_BOND;
  } else if (!strncmp(geomname, "angl", 4)) {
    reqatoms = 3; geomtype = MEASURE_ANGLE;
  } else if (!strncmp(geomname, "dihe", 4)) {
    reqatoms = 4; geomtype = MEASURE_DIHED;
  } else if (!strncmp(geomname, "impr", 4)) {
    reqatoms = 4; geomtype = MEASURE_IMPRP;
  } else if (!strncmp(geomname, "vdw",  3)) {
    reqatoms = 2; geomtype = MEASURE_VDW;
  } else if (!strncmp(geomname, "elec", 4)) {
    reqatoms = 2; geomtype = MEASURE_ELECT;
  } else {
    Tcl_AppendResult(interp, " measure energy: bad syntax (must specify bond|angle|dihed|imprp|vdw|elec)", NULL);
    return TCL_ERROR;
  }

  int molid[4];
  int atmid[4];
  int defmolid = -1;
  bool allframes = false;
  char errstring[200];

  // Read the atom list
  int numatms;
  Tcl_Obj **data;
  if (Tcl_ListObjGetElements(interp, objv[2], &numatms, &data) != TCL_OK) {
    Tcl_AppendResult(interp, " measure energy: bad syntax", NULL);
    Tcl_AppendResult(interp, " Usage: measure energy bond|angle|dihed|imprp|vdw|elec {{<atomid1> ?<molid1>?} {<atomid2> ?<molid2>?} ...} ?molid <default molid>? [?frame <frame|all>? | ?first <first>? ?last <last>?]", NULL);
    return TCL_ERROR;
  }

  if (numatms!=reqatoms) {
    sprintf(errstring, " measure energy %s: must specify exactly %i atoms in list", geomname, reqatoms);
    Tcl_AppendResult(interp, errstring, NULL);
    return TCL_ERROR;
  }
    
  int first=-1, last=-1, frame=-1;
  double params[6];
  memset(params, 0, 6*sizeof(double));

  if (argc>4) {
    int i;
    for (i=3; i<argc; i+=2) {
      char *argvcur = Tcl_GetStringFromObj(objv[i],NULL);
      if (!strupncmp(argvcur, "molid", CMDLEN)) {
	if (Tcl_GetIntFromObj(interp, objv[i+1], &defmolid) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure energy: bad molid", NULL);
	  return TCL_ERROR;
	}
      } else if (!strupncmp(argvcur, "k", CMDLEN)) {
	if (Tcl_GetDoubleFromObj(interp, objv[i+1], params) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure energy: bad force constant value", NULL);
	  return TCL_ERROR;
	}
      } else if (!strupncmp(argvcur, "x0", CMDLEN)) {
	if (Tcl_GetDoubleFromObj(interp, objv[i+1], params+1) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure energy: bad equilibrium value", NULL);
	  return TCL_ERROR;
	}
      } else if (!strupncmp(argvcur, "kub", CMDLEN)) {
	if (Tcl_GetDoubleFromObj(interp, objv[i+1], params+2) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure energy: bad Urey-Bradley force constant value", NULL);
	  return TCL_ERROR;
	}
      } else if (!strupncmp(argvcur, "s0", CMDLEN)) {
	if (Tcl_GetDoubleFromObj(interp, objv[i+1], params+3) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure energy: bad Urey-Bradley equilibrium distance", NULL);
	  return TCL_ERROR;
	}
      } else if (!strupncmp(argvcur, "n", CMDLEN)) {
	if (Tcl_GetDoubleFromObj(interp, objv[i+1], params+1) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure energy: bad Urey-Bradley equilibrium distance", NULL);
	  return TCL_ERROR;
	}
      } else if (!strupncmp(argvcur, "delta", CMDLEN)) {
	if (Tcl_GetDoubleFromObj(interp, objv[i+1], params+2) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure energy: bad Urey-Bradley equilibrium distance", NULL);
	  return TCL_ERROR;
	}
      } else if (!strupncmp(argvcur, "eps1", CMDLEN)) {
	if (Tcl_GetDoubleFromObj(interp, objv[i+1], params) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure energy: bad vdw well depth", NULL);
	  return TCL_ERROR;
	}
      } else if (!strupncmp(argvcur, "rmin1", CMDLEN)) {
	if (Tcl_GetDoubleFromObj(interp, objv[i+1], params+1) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure energy: bad vdw equilibrium distance", NULL);
	  return TCL_ERROR;
	}
      } else if (!strupncmp(argvcur, "eps2", CMDLEN)) {
	if (Tcl_GetDoubleFromObj(interp, objv[i+1], params+2) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure energy: bad vdw well depth", NULL);
	  return TCL_ERROR;
	}
      } else if (!strupncmp(argvcur, "rmin2", CMDLEN)) {
	if (Tcl_GetDoubleFromObj(interp, objv[i+1], params+3) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure energy: bad vdw equilibrium distance", NULL);
	  return TCL_ERROR;
	}
      } else if (!strupncmp(argvcur, "q1", CMDLEN)) {
	if (Tcl_GetDoubleFromObj(interp, objv[i+1], params) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure energy: bad charge value", NULL);
	  return TCL_ERROR;
	}
	params[2]=1.0;
      } else if (!strupncmp(argvcur, "q2", CMDLEN)) {
	if (Tcl_GetDoubleFromObj(interp, objv[i+1], params+1) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure energy: bad charge value", NULL);
	  return TCL_ERROR;
	}
	params[3]=1.0;
      } else if (!strupncmp(argvcur, "cutoff", CMDLEN)) {
	if (Tcl_GetDoubleFromObj(interp, objv[i+1], params+4) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure energy: bad electrostatic cutoff value", NULL);
	  return TCL_ERROR;
	}
      } else if (!strupncmp(argvcur, "switchdist", CMDLEN)) {
	if (Tcl_GetDoubleFromObj(interp, objv[i+1], params+5) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure energy: bad switching distance value", NULL);
	  return TCL_ERROR;
	}
      } else if (!strupncmp(argvcur, "first", CMDLEN)) {
	if (Tcl_GetIntFromObj(interp, objv[i+1], &first) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure energy: bad first frame value", NULL);
	  return TCL_ERROR;
	}
      } else if (!strupncmp(argvcur, "last", CMDLEN)) {
	if (Tcl_GetIntFromObj(interp, objv[i+1], &last) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure energy: bad last frame value", NULL);
	  return TCL_ERROR;
	}
      } else if (!strupncmp(argvcur, "frame", CMDLEN)) {
	if (!strupcmp(Tcl_GetStringFromObj(objv[i+1],NULL), "all")) {
	  allframes = true;
	} else if (Tcl_GetIntFromObj(interp, objv[i+1], &frame) != TCL_OK) {
	  Tcl_AppendResult(interp, " measure energy: bad frame value", NULL);
	  return TCL_ERROR;
	}
      } else {
	Tcl_AppendResult(interp, " measure energy: invalid syntax, no such keyword: ", argvcur, NULL);
	return TCL_ERROR;
      }
    }
  }

  if ((allframes || frame>=0) && (first>=0 || last>=0)) {
    Tcl_AppendResult(interp, "measure energy: Ambiguous syntax: You cannot specify a frame AND a frame range (using first or last).", NULL);
    Tcl_AppendResult(interp, "\nUsage:\nmeasure bond <molid1>/<atomid1> <molid2>/<atomid2> [?frame <frame|all>? | ?first <first>? ?last <last>?]", NULL);
    return TCL_ERROR;    
  }

  if (allframes) first=0;

  // If no molecule was specified use top as default
  if (defmolid<0) defmolid = app->molecule_top();

  // Assign atom IDs and molecule IDs
  int i,numelem;
  Tcl_Obj **atmmol;
  for (i=0; i<numatms; i++) {
    if (Tcl_ListObjGetElements(interp, data[i], &numelem, &atmmol) != TCL_OK) {
      return TCL_ERROR;
    }

    if (Tcl_GetIntFromObj(interp, atmmol[0], atmid+i) != TCL_OK) {
      Tcl_AppendResult(interp, " measure energy: bad atom index", NULL);
      return TCL_ERROR;
    }
    
    if (numelem==2) {
      if (Tcl_GetIntFromObj(interp, atmmol[1], molid+i) != TCL_OK) {
	Tcl_AppendResult(interp, " measure energy: bad molid", NULL);
	return TCL_ERROR;
      }
    } else molid[i] = defmolid;
  }
  

  // Compute the value
  ResizeArray<float> gValues(1024);
  int ret_val;
  ret_val = measure_energy(app->moleculeList, molid, atmid, &gValues, frame, first, last,
			 defmolid, params, geomtype);
  if (ret_val<0) {
    printf("ERROR\n %s\n", measure_error(ret_val));
    Tcl_AppendResult(interp, measure_error(ret_val), NULL);
    return TCL_ERROR;
  }

  Tcl_Obj *tcl_result = Tcl_NewListObj(0, NULL);
  int numvalues = gValues.num();
  for (int count = 0; count < numvalues; count++) {
    Tcl_ListObjAppendElement(interp, tcl_result, Tcl_NewDoubleObj(gValues[count]));
  }
  Tcl_SetObjResult(interp, tcl_result);

  return TCL_OK;
}


int obj_measure(ClientData cd, Tcl_Interp *interp, int argc,
                            Tcl_Obj * const objv[]) {

  if (argc < 2) {
    Tcl_SetResult(interp, 
      (char *) "usage: measure <command> [args...]\n"
      "\nMeasure Commands:\n"
      "  avpos    <selection> first <first> last <last> step <step> -- average atom position\n"
      "  center   <selection> [weight <weights>]   -- geometrical (or weighted) center\n"
      "  contacts <cutoff> <sel1> [<sel2>]         -- list contacts\n" 
      "  dipole   <sel> [-elementary | -debye]     -- dipole moment\n"
      "  fit      <sel1> <sel2> [weight <weights>] -- transformation matrix from 1 to 2\n"
      "  gofr     <sel1> <sel2> [delta <value>] [rmax <value>] [usepbc <bool>] [selupdate <bool>] [first <value>] [last <value>] [step <value>]\n"
      "     -- atomic pair distribution function g(r)\n"
      "  hbonds   <cutoff> <angle> <sel1> [<sel2>]\n"
      "     -- list donors, acceptors, hydrogens involved in hydrogen bonds\n"
      "  inverse  <matrix>                         -- inverse matrix\n"
      "  minmax   [<selection>] [-withradii]       -- bounding box\n"
      "  rgyr     <selection> [weight <weights>]   -- radius of gyration\n"
      "  rmsd     <sel1> <sel2> [weight <weights>] -- RMS deviation\n"
      "  rmsf     <selection> first <star> last <last> step <step> -- RMS fluctuation\n"
      "  sasa     <srad> <selection> [-points <varname>] [-restrict <selection>]\n"
      "                            [-samples <n>]  -- solvent-accessible surface area\n"
      "  sumweights <selection> weight <weights>   -- sum of selected weights\n"
      "  bond  {{<atomid1> ?<molid1>?} {<atomid2> ?<molid2>?}} ?molid <default molid>? [?frame <frame|all>? | ?first <first>? ?last <last>?]{{<atomid1> ?<molid1>?} {<atomid2> ?<molid2>?}} ?molid <default molid>? [?frame <frame|all>? | ?first <first>? ?last <last>?] -- bond length\n"
      "  angle {{<atomid1> ?<molid1>?} {<atomid2> ?<molid2>?} {<atomid3> ?<molid3>?}} ?molid <default molid>? [?frame <frame|all>? | ?first <first>? ?last <last>?] -- angle\n"
      "  dihed {{<atomid1> ?<molid1>?} {<atomid2> ?<molid2>?} {<atomid3> ?<molid3>?} {<atomid4> ?<molid4>?}} ?molid <default molid>? [?frame <frame|all>? | ?first <first>? ?last <last>?] -- dihedral angle\n"
      "  energy bond|angle|dihed|impr|vdw|elec -- compute energy",	  
      TCL_STATIC);
    return TCL_ERROR;
  }
  VMDApp *app = (VMDApp *)cd;
  char *argv1 = Tcl_GetStringFromObj(objv[1],NULL);
  if (!strupncmp(argv1, "avpos", CMDLEN))
    return vmd_measure_avpos(app, argc-1, objv+1, interp);
  else if (!strupncmp(argv1, "center", CMDLEN))
    return vmd_measure_center(app, argc-1, objv+1, interp);
  else if (!strupncmp(argv1, "contacts", CMDLEN))
    return vmd_measure_contacts(app, argc-1, objv+1, interp);
  else if (!strupncmp(argv1, "dipole", CMDLEN))
    return vmd_measure_dipole(app, argc-1, objv+1, interp);
  else if (!strupncmp(argv1, "fit", CMDLEN)) 
    return vmd_measure_fit(app, argc-1, objv+1, interp);
  else if (!strupncmp(argv1, "minmax", CMDLEN))
    return vmd_measure_minmax(app, argc-1, objv+1, interp);
  else if (!strupncmp(argv1, "gofr", CMDLEN))
    return vmd_measure_gofr(app, argc-1, objv+1, interp);
  else if (!strupncmp(argv1, "hbonds", CMDLEN))
    return vmd_measure_hbonds(app, argc-1, objv+1, interp);
  else if (!strupncmp(argv1, "inverse", CMDLEN)) 
    return vmd_measure_inverse(argc-1, objv+1, interp);
  else if (!strupncmp(argv1, "rgyr", CMDLEN))
    return vmd_measure_rgyr(app, argc-1, objv+1, interp);
  else if (!strupncmp(argv1, "rmsd", CMDLEN))
    return vmd_measure_rmsd(app, argc-1, objv+1, interp);
  else if (!strupncmp(argv1, "rmsf", CMDLEN))
    return vmd_measure_rmsf(app, argc-1, objv+1, interp);
  else if (!strupncmp(argv1, "sasa", CMDLEN))
    return vmd_measure_sasa(app, argc-1, objv+1, interp);
  else if (!strupncmp(argv1, "sumweights", CMDLEN))
    return vmd_measure_sumweights(app, argc-1, objv+1, interp);
  else if (!strupncmp(argv1, "imprp", CMDLEN))
    return vmd_measure_dihed(app, argc-1, objv+1, interp);
  else if (!strupncmp(argv1, "dihed", CMDLEN))
    return vmd_measure_dihed(app, argc-1, objv+1, interp);
  else if (!strupncmp(argv1, "angle", CMDLEN))
    return vmd_measure_angle(app, argc-1, objv+1, interp);
  else if (!strupncmp(argv1, "bond", CMDLEN))
    return vmd_measure_bond(app, argc-1, objv+1, interp);
  else if (!strupncmp(argv1, "energy", CMDLEN))
    return vmd_measure_energy(app, argc-1, objv+1, interp);

  Tcl_SetResult(interp, (char *) "Type 'measure' for summary of usage\n", TCL_VOLATILE);
  return TCL_OK;
}

