/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995-2007 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Spaceball.C,v $
 *	$Author: johns $	$Locker:  $		$State: Exp $
 *	$Revision: 1.28 $	$Date: 2007/03/15 18:09:30 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The Spaceball UI object, which maintains the current state of the 
 * spaceball.  This code uses John Stone's spaceball library when
 * VMDSPACEBALL is defined, or the Spaceware library when VMDSPACEWARE 
 * is defined.
 *
 ***************************************************************************/

#if defined(VMDSPACEWARE) || defined(VMDSPACEBALL)

#include "Spaceball.h"
#include "DisplayDevice.h"
#include "TextEvent.h"
#include "CommandQueue.h"
#include "Inform.h"
#include "PickList.h"
#include "VMDApp.h"
#include "math.h"
#include "stdlib.h" // for getenv(), abs() etc.

// constructor
Spaceball::Spaceball(VMDApp *vmdapp)
	: UIObject(vmdapp) {

#if defined(VMDSPACEBALL) && !defined(VMDSPACEWARE)
  sball=NULL; // zero it out to begin with
  if (getenv("VMDSPACEBALLPORT") != NULL) {
    msgInfo << "Opening Spaceball on port: " << 
               getenv("VMDSPACEBALLPORT") << sendmsg;
    sball = sball_open(getenv("VMDSPACEBALLPORT"));
    if (sball == NULL) 
      msgErr << "Failed to open Spaceball port, Spaceball input disabled" << 
                sendmsg; 
  }
#endif


  // set the default translation and rotation increments
  // these really need to be made user modifiable at runtime
  transInc = 1.0f / 6000.0f;
    rotInc = 1.0f /   50.0f;
  scaleInc = 1.0f / 6000.0f;

  moveMode = ROTTRANS;
  buttonDown = 0;

  reset();
}


// destructor
Spaceball::~Spaceball(void) {

#if defined(VMDSPACEBALL) && !defined(VMDSPACEWARE)
  if (sball != NULL)
    sball_close(sball);
#endif

}

/////////////////////// virtual routines for UI init/display  /////////////
   
// reset the spaceball to original settings
void Spaceball::reset(void) {
  scaling = 1.0;
}

// update the display due to a command being executed.  Return whether
// any action was taken on this command.
// Arguments are the command type, command object, and the 
// success of the command (T or F).
int Spaceball::act_on_command(int type, Command *cmd) {
  return FALSE; // we don't take any commands presently
}


// check for an event, and queue it if found.  Return TRUE if an event
// was generated.
int Spaceball::check_event(void) {
  int retval = FALSE;
  int tx, ty, tz, rx, ry, rz, buttons, buttonchanged, rmaxval, tmaxval;
  int atx, aty, atz, arx, ary, arz;
  int tmaxvec[3];
  char rmaxaxis;

#if defined(VMDSPACEBALL) && !defined(VMDSPACEWARE)
  if (sball == NULL)
    return FALSE; // no updates of course.

  if (!sball_getstatus(sball, &tx, &ty, &tz, &rx, &ry, &rz, &buttons)) 
    return FALSE;
#endif

#if defined(VMDSPACEWARE) && !defined(VMDSPACEBALL)
   if (!app->display->spaceball(&rx, &ry, &rz, &tx, &ty, &tz, &buttons)) 
     return FALSE;
#endif

  //printf("sb: %d %d %d %d %d %d %d\n", rx, ry, rz, tx, ty, tz, buttons);
  retval = TRUE; // we got an event if we got to here.

  // find which buttons changed state
  buttonchanged = buttons ^ buttonDown; 

  // if the user presses button 1, reset the view, a very very very
  // important feature to have implemented early on... ;-)
#if   defined(VMDSPACEBALL)  &&  !defined(VMDSPACEWARE)
  if (((buttonchanged & SBALL_BUTTON_1) && (buttons & SBALL_BUTTON_1)) ||
      ((buttonchanged & SBALL_BUTTON_LEFT) && (buttons & SBALL_BUTTON_LEFT))){
#elif !defined(VMDSPACEBALL) &&  defined(VMDSPACEWARE)
  if ((buttonchanged & 2) && (buttons & 2)) {
#endif

    scaling = 1.0;
    app->scene_resetview();
    msgInfo << "Spaceball reset view orientation" << sendmsg;
  }

  // Toggle between the different modes
#if   defined(VMDSPACEBALL)  &&  !defined(VMDSPACEWARE)
  if (((buttonchanged & SBALL_BUTTON_2) && (buttons & SBALL_BUTTON_2)) ||
      ((buttonchanged & SBALL_BUTTON_RIGHT) && (buttons & SBALL_BUTTON_RIGHT))) {
#elif !defined(VMDSPACEBALL) &&  defined(VMDSPACEWARE)
  if ((buttonchanged & 4) && (buttons & 4)) {
#endif

    switch (moveMode) {
      case ROTTRANS:
        moveMode = MAXAXIS;
        msgInfo << "Spaceball set to dominant axis rotation/translation mode" << sendmsg;
        break;   

      case MAXAXIS:
        moveMode = SCALING;
        msgInfo << "Spaceball set to scaling mode" << sendmsg;
        break;   

      default: 
        moveMode = ROTTRANS;
        msgInfo << "Spaceball set to rotation/translation mode" << sendmsg;
        break;
    }
  }

  // if the user presses button 3 through N, run a User command
#if defined(VMDSPACEBALL)  &&  !defined(VMDSPACEWARE)
  if ((buttonchanged & SBALL_BUTTON_3) && (buttons & SBALL_BUTTON_3)) {
    runcommand(new UserKeyEvent('3', (int) DisplayDevice::AUX));
  }
  if ((buttonchanged & SBALL_BUTTON_4) && (buttons & SBALL_BUTTON_4)) {
    runcommand(new UserKeyEvent('4', (int) DisplayDevice::AUX));
  }
  if ((buttonchanged & SBALL_BUTTON_5) && (buttons & SBALL_BUTTON_5)) {
    runcommand(new UserKeyEvent('5', (int) DisplayDevice::AUX));
  }
  if ((buttonchanged & SBALL_BUTTON_6) && (buttons & SBALL_BUTTON_6)) {
    runcommand(new UserKeyEvent('6', (int) DisplayDevice::AUX));
  }
  if ((buttonchanged & SBALL_BUTTON_7) && (buttons & SBALL_BUTTON_7)) {
    runcommand(new UserKeyEvent('7', (int) DisplayDevice::AUX));
  }
  if ((buttonchanged & SBALL_BUTTON_8) && (buttons & SBALL_BUTTON_8)) {
    runcommand(new UserKeyEvent('8', (int) DisplayDevice::AUX));
  }
#elif !defined(VMDSPACEBALL) &&  defined(VMDSPACEWARE)
  if ((buttonchanged & 8) && (buttons & 8)) {
    runcommand(new UserKeyEvent('3', (int) DisplayDevice::AUX));
  }
  if ((buttonchanged & 16) && (buttons & 16)) {
    runcommand(new UserKeyEvent('4', (int) DisplayDevice::AUX));
  }
  if ((buttonchanged & 32) && (buttons & 32)) {
    runcommand(new UserKeyEvent('5', (int) DisplayDevice::AUX));
  }
  if ((buttonchanged & 64) && (buttons & 64)) {
    runcommand(new UserKeyEvent('6', (int) DisplayDevice::AUX));
  }
  if ((buttonchanged & 128) && (buttons & 128)) {
    runcommand(new UserKeyEvent('7', (int) DisplayDevice::AUX));
  }
  if ((buttonchanged & 256) && (buttons & 256)) {
    runcommand(new UserKeyEvent('8', (int) DisplayDevice::AUX));
  }
#endif

  // get absolute values of axis forces for min/max comparison tests
  arx = abs(rx);
  ary = abs(ry);
  arz = abs(rz);
  atx = abs(tx);
  aty = abs(ty);
  atz = abs(tz);

  // Ignore null motion events since some versions of the Windows 
  // Spaceball driver emit a constant stream of null motion event
  // packets which would otherwise cause continuous redraws, pegging the 
  // CPU and GPU at maximum load.
  if ((arx+ary+arz+atx+aty+atz) > 0) {
    switch(moveMode) {
      case MAXAXIS:
        // Z-axis rotation/trans have to be negated in order to please VMD...

        // find dominant rotation axis
        if (arx > ary) {
          if (arx > arz) {
            rmaxaxis = 'x';
            rmaxval = rx; 
          } else {
            rmaxaxis = 'z';
            rmaxval = -rz; 
          }
        } else {     
          if (ary > arz) {
            rmaxaxis = 'y';
            rmaxval = ry; 
          } else {
            rmaxaxis = 'z';
            rmaxval = -rz; 
          }
        }

        // find dominant translation axis
        tmaxvec[0] = tmaxvec[1] = tmaxvec[2] = 0;
        if (atx > aty) {
          if (atx > atz) {
            tmaxval = tx;
            tmaxvec[0] = tx; 
          } else {
            tmaxval = tz;
            tmaxvec[2] = tz; 
          }
        } else {     
          if (aty > atz) {
            tmaxval = ty;
            tmaxvec[1] = ty; 
          } else {
            tmaxval = tz;
            tmaxvec[2] = tz; 
          }
       }

       // determine whether to rotate or translate
       if (abs(rmaxval) > abs(tmaxval)) {
         app->scene_rotate_by(((float) rmaxval) * rotInc, rmaxaxis);
       } else {
         app->scene_translate_by(
            ((float) tmaxvec[0]) * transInc, 
            ((float) tmaxvec[1]) * transInc, 
           -((float) tmaxvec[2]) * transInc);
       }
       break;

      case ROTTRANS:
        // Z-axis rotation/trans have to be negated in order to please VMD...
        app->scene_rotate_by(((float)  rx) * rotInc, 'x');
        app->scene_rotate_by(((float)  ry) * rotInc, 'y');
        app->scene_rotate_by(((float) -rz) * rotInc, 'z');
        app->scene_translate_by(tx * transInc, ty * transInc, -tz * transInc);
        break;

      case SCALING:
        float scf = scaling + scaleInc * (float) tz;
        if (scf < 0.0)
          scf = 0.0;
        app->scene_scale_by(scf);
        break;
    }
  }

  // update button status for next time through
  buttonDown = buttons;

  return retval;
}

#endif
