/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995-2007 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: OpenGLRenderer.C,v $
 *	$Author: johns $	$Locker:  $		$State: Exp $
 *	$Revision: 1.381 $	$Date: 2007/03/29 19:16:31 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Subclass of DisplayDevice, this object has routines used by all the
 * different display devices that use OpenGL for rendering.
 * Will render drawing commands into a window.
 * This is not the complete definition,
 * however, of a DisplayDevice; something must provide routines to open
 * windows, reshape, clear, set perspective, etc.  This object contains the
 * code to render a display command list.
 ***************************************************************************/

#include "OpenGLRenderer.h"
#include "DispCmds.h"
#include "Inform.h"
#include "utilities.h"
#include "VMDDisplayList.h"
#include "Hershey.h"

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "OpenGLStipples.h"

// enable WireGL support
#define VMDWIREGL 1

#if defined(VMDUSEOPENGLSHADER)
#define VMDUSEGLSLSPHERES 1
#endif

#if 0
#define OGLERR { GLenum err; if ((err = glGetError()) != GL_NO_ERROR) {  \
	msgErr << __FILE__ << " line " << __LINE__ << " " << \
	(const char *) gluErrorString(err) << sendmsg; }}
#else 
#define OGLERR
#endif

#define MIN_SPHERE_RES 4
#define MAX_SPHERE_RES 30

void OpenGLRenderer::setup_initial_opengl_state(void) {
  int i; 

  if (getenv("VMDSIMPLEGRAPHICS") == NULL) {
    simplegraphics = 0; // use all available OpenGL features and extensions
  } else {
    simplegraphics = 1; // limit VMD to OpenGL ~1.0 with no extensions
    msgWarn << "Simple graphics mode: OpenGL 1.0, no extensions" << sendmsg;
  }

  // Create quadric objects for spheres, cylinders, and disks
  // Default to drawing filled objects, may be changed by the detail level
  objQuadric = gluNewQuadric();
  pointsQuadric = gluNewQuadric();
  gluQuadricDrawStyle(objQuadric, (GLenum)GLU_FILL);
  gluQuadricDrawStyle(pointsQuadric, (GLenum)GLU_POINT);

  // display list caching
  ogl_cachelistbase = 2000;
  ogl_cacheenabled = 0;
  ogl_cachedebug = 0;
  if (getenv("VMDCACHEDLDEBUG") != NULL) {
    ogl_cachedebug = 1; // enable verbose output for display list caching
  }

  wiregl = 0;          // wiregl not detected yet
  immersadeskflip = 0; // Immersadesk right-eye X-axis mode off by default
  shearstereo = 0;     // use stereo based on eye rotation by default

  // initialize state caching variables so they get overwritten on 
  // first reference in the call to render()
  oglmaterialindex = -1;
  oglopacity = -1.0f;
  oglambient = -1.0f;
  ogldiffuse = -1.0f;
  oglspecular = -1.0f;
  oglshininess = -1.0f;
  ogl_useblendedtrans = 0;
  ogl_transpass = 0; 
  ogl_useglslshader = 0;
  ogl_acrobat3dcapture = 0;
  ogl_lightingenabled = 0;
  ogl_rendstateserial = 1;    // force GLSL update on 1st pass
  ogl_glslserial = 0;         // force GLSL update on 1st pass
  ogl_glsltoggle = 1;         // force GLSL update on 1st pass
  ogl_glslmaterialindex = -1; // force GLSL update on 1st pass
  ogl_glslprojectionmode = DisplayDevice::PERSPECTIVE; 
  ogl_glsltexturemode = 0;    // initialize GLSL projection to off

  // identify the rendering hardware we're using
  ext->find_renderer();

  // find all available OpenGL extensions, unless the user doesn't want us to.
  if (!simplegraphics) {
    ext->find_extensions(); ///< register available OpenGL extensions
  }


#if 0
// XXX the performance workaround aspect of this must still be tested
// on Linux to verify that the existing code path is actually a benefit.

// not tested on other platforms yet
#if defined(ARCH_MACOSX) || defined(ARCH_MACOSXX86)
  // Detect NVidia graphics cards, which have a semi-broken stereo 
  // implementation that favors rendering in stereo mode all the time, 
  // as the alternative is 20-50x performance hit on Linux.
  // XXX on MacOS X, the behavior is much more serious than just a performance
  // hit, they actually fail to draw/clear the back right color buffer 
  // when drawing to GL_BACK.
  if (ext->hasstereo && ext->oglrenderer == OpenGLExtensions::NVIDIA) {
    msgInfo << "nVidia card detected, enabling mono drawing performance workaround" << sendmsg;

    // force drawing in stereo even when VMD is set for mono mode
    ext->stereodrawforced = 1;
  }
#endif
#endif

#if defined(ARCH_LINUX) || defined(ARCH_LINUXAMD64) || defined(ARCH_LINUXIA64)
  // Detect ATI Linux driver and disable unsafe extensions
  if (ext->oglrenderer == OpenGLExtensions::ATI) {
    if (getenv("VMDDISABLEATILINUXWORKAROUND") == NULL) {
      msgInfo << "ATI Linux driver detected, limiting features to avoid driver bugs." << sendmsg;
      msgInfo << "  Set the environment variable VMDDISABLEATILINUXWORKAROUND" << sendmsg;
      msgInfo << "  to enable full functionality on a known-safe driver version." << sendmsg;

      simplegraphics = 1; 
    }
  }
#endif

#if defined(VMDWIREGL)
  // Detect WireGL and shut off unsupported rendering features if so.
  if ((getenv("VMDWIREGL") != NULL) || 
      (strstr((char *) glGetString(GL_RENDERER), "WireGL") != NULL)) {
    msgInfo << "WireGL renderer detected, disabling unsupported OpenGL features." << sendmsg;
    wiregl=1;

    // Shut off unsupported rendering features if so.
    ext->hastex2d = 0;
    ext->hastex3d = 0;
  }
#endif

  glDepthFunc(GL_LEQUAL);
  glEnable(GL_DEPTH_TEST);           // use zbuffer for hidden-surface removal
  glClearDepth(1.0);
 
#if 1
  // VMD now renormalizes all the time since non-uniform scaling 
  // operations must be applied for drawing ellipsoids and other 
  // warped geometry, and this is a final cure for people remote
  // displaying on machines with broken rescale normal extensions
  glEnable(GL_NORMALIZE);            // automatically normalize normals
#else
  // Rescale normals, assumes they are initially normalized to begin with, 
  // and that only uniform scaling is applied, and non-warping modelview
  // matrices are used. (i.e. no shear in the modelview matrix...)
  // Gets rid of a square root on every normal, but still tracks
  // uniform scaling operations.  If not available, enable full
  // normalization.
  if (simplegraphics || wiregl || ogl_acrobat3dcapture) {
    // Renormalize normals if we're in 'simplegraphics' mode.
    // WireGL doesn't work correctly with the various normal rescaling
    // features and extensions, so we have to make it use GL_NORMALIZE.
    glEnable(GL_NORMALIZE);            // automatically normalize normals
  } else {
#if defined(_MSC_VER) || defined(ARCH_IRIX6) || defined(ARCH_IRIX6_64) || defined(ARCH_MACOSX) || defined(ARCH_MACOSXX86)
    // XXX The Microsoft "GDI Generic", MacOS X, and IRIX OpenGL renderers
    // malfunction when we use GL_RESCALE_NORMAL, so we disable it here
    glEnable(GL_NORMALIZE);            // automatically normalize normals
#else
#if defined(GL_VERSION_1_2)
    ext->hasrescalenormalext = 1;
    glEnable(GL_RESCALE_NORMAL);       // automatically rescale normals
#elif defined(GL_RESCALE_NORMAL_EXT)
    if (ext->vmdQueryExtension("GL_RESCALE_NORMAL_EXT")) {
      ext->hasrescalenormalext = 1;
      glEnable(GL_RESCALE_NORMAL_EXT); // automatically rescale normals
    } else {
      glEnable(GL_NORMALIZE);          // automatically normalize normals
    }
#else
    glEnable(GL_NORMALIZE);            // automatically normalize normals
#endif
#endif
  }  
#endif

  // configure for dashed lines ... but initially show solid lines
  glLineStipple(1, 0x3333);
  glDisable(GL_LINE_STIPPLE);

  // configure the fogging characteristics ... color and position of fog
  // are set during the clear routine
  // Note about FOG/Depth Cueing:
  //    Several popular video accelerators in SGI, Sun, and equipment from
  //    other vendors has a documented bug in the functioning of the
  //    GL_LINEAR type fog.  The original bug was in SGI hardware and
  //    software, and was propagated to a large number of machines.
  //    For this reason, GL_LINEAR fog does not display the same on
  //    most machines, so I have changed VMD to use exponential fog instead.
  //    Exponential fog is somewhat slower, but at least it displays
  //    consistently on many machines.
  glFogi(GL_FOG_MODE, GL_EXP2);
  glFogf(GL_FOG_DENSITY, 0.40f);

  // configure the light model
  glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);
  glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER, GL_FALSE);

  glColorMaterial(GL_FRONT_AND_BACK, GL_DIFFUSE);
  glEnable(GL_COLOR_MATERIAL);       // have materials set by curr color
  glDisable(GL_POLYGON_SMOOTH);      // make sure not to antialias polygons

  // disable all lights by default
  for (i=0; i < DISP_LIGHTS; i++) {
    ogl_lightstate[i] = 0; // off by default
  }

  // disable all clipping planes by default
  for (i=0; i < VMD_MAX_CLIP_PLANE; i++) {
    ogl_clipmode[i] = 0; // off by default
    glDisable((GLenum) (GL_CLIP_PLANE0 + i));
  }

  // load transformation matrices on stack, initially with identity transforms
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();

  // generate sphere display lists
  glMatrixMode(GL_MODELVIEW);
  for (i=MIN_SPHERE_RES; i<=MAX_SPHERE_RES; i++) {
    GLuint solidlist = glGenLists(1);
    glNewList(solidlist, GL_COMPILE);
    gluSphere(objQuadric, 1.0, i, i);
    glEndList(); 
    solidSphereLists.append(solidlist);

    GLuint pointlist = glGenLists(1);
    glNewList(pointlist, GL_COMPILE);
    gluSphere(pointsQuadric, 1.0, i, i);
    glEndList(); 
    pointSphereLists.append(pointlist);
  }

  // create font display lists for use in displaying text
  ogl_textMat.identity();
  fontListBase = glGenLists(256);
  glListBase(fontListBase);
  for (i=0 ; i<256 ; i++) {
    glNewList(fontListBase+i, GL_COMPILE);
    //note- we use the same info twice here
    hersheyDrawLetter(hersheyFontData[i%128]);
    glEndList();
  }

  // display lists are now initialized, so they must be freed when this
  // OpenGL context is destroyed
  dpl_initialized = 1;

#if defined(GL_VERSION_1_1)
  if (!(simplegraphics || ogl_acrobat3dcapture)) {
    // enable vertex arrays.
    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_NORMAL_ARRAY);
    glEnableClientState(GL_COLOR_ARRAY);
  }
#endif


#if defined(GL_VERSION_1_1)
  if (ext->hastex2d) {
    int i, sz;
    GLint x, y;

    // test actual maximums for desired format
    glGetIntegerv(GL_MAX_TEXTURE_SIZE, &max2DtexSize);
    
    for (i=0; (sz = 1 << i) <= max2DtexSize; i++) {
      glTexImage2D(GL_PROXY_TEXTURE_2D, 0, GL_RGB8,
                   sz, sz, 0, 
                   GL_RGB, GL_UNSIGNED_BYTE, NULL);
       
      glGetTexLevelParameteriv(GL_PROXY_TEXTURE_2D, 0, GL_TEXTURE_WIDTH, &x);
      glGetTexLevelParameteriv(GL_PROXY_TEXTURE_2D, 0, GL_TEXTURE_HEIGHT, &y);
    
      if (x > 0 && y > 0) { 
        max2DtexX = x;
        max2DtexY = y;
      }
    }

    if (max2DtexX > max2DtexSize)
      max2DtexX = max2DtexSize;
        
    if (max2DtexY > max2DtexSize)
      max2DtexY = max2DtexSize;
  } 
#endif

#if defined(GL_VERSION_1_2)
  if (ext->hastex3d) {
    int i, sz;
    GLint x, y, z;

    // test actual maximums for desired format
    max3DtexSize = 0; // until successfully queried from OpenGL
    max3DtexX = 0;
    max3DtexY = 0;
    max3DtexZ = 0;
    glGetIntegerv(GL_MAX_3D_TEXTURE_SIZE, &max3DtexSize);
    
    for (i=0; (sz = 1 << i) <= max3DtexSize; i++) {
      GLTEXIMAGE3D(GL_PROXY_TEXTURE_3D, 0, GL_RGB8, 
                   sz, sz, sz, 0, 
                   GL_RGB, GL_UNSIGNED_BYTE, NULL);
 
      glGetTexLevelParameteriv(GL_PROXY_TEXTURE_3D, 0, GL_TEXTURE_WIDTH,  &x);
      glGetTexLevelParameteriv(GL_PROXY_TEXTURE_3D, 0, GL_TEXTURE_HEIGHT, &y);
      glGetTexLevelParameteriv(GL_PROXY_TEXTURE_3D, 0, GL_TEXTURE_DEPTH,  &z);
   
      if (x > 0 && y > 0 && z > 0) {
        max3DtexX = x; 
        max3DtexY = y; 
        max3DtexZ = z; 
      }  
    }

    if (max3DtexX > max3DtexSize)
      max3DtexX = max3DtexSize;
        
    if (max3DtexY > max3DtexSize)
      max3DtexY = max3DtexSize;

    if (max3DtexZ > max3DtexSize)
      max3DtexZ = max3DtexSize;

    // disable 3-D texturing on cards that return unusable max texture sizes
    if (max3DtexX < 1 || max3DtexY < 1 || max3DtexZ < 1) {
      ext->hastex3d = 0;
    }

  }  
#endif



// MacOS X has had a broken implementation of GL_SEPARATE_SPECULAR_COLOR
// for some time.
#if defined(GL_VERSION_1_2) && !defined(ARCH_MACOSX) && !defined(ARCH_MACOSXX86)
  if (((ext->oglmajor == 1) && (ext->oglminor >= 2)) || (ext->oglmajor >= 2)) {
    if (ext->hastex2d || ext->hastex3d) {
      // Make specular color calculations follow texture mapping operations
      glLightModeli(GL_LIGHT_MODEL_COLOR_CONTROL, GL_SEPARATE_SPECULAR_COLOR);
    } else {
      // Do the specular color calculations at the same time as the rest 
      glLightModeli(GL_LIGHT_MODEL_COLOR_CONTROL, GL_SINGLE_COLOR);
    }
  }
#endif

  ext->PrintExtensions(); ///< print out information on OpenGL Extensions

#if defined(VMDUSEOPENGLSHADER)
  int glslextensionsavailable=0;

  // Enable OpenGL programmable shading if it is available
  if (!(simplegraphics || ogl_acrobat3dcapture) &&
      ext->hasglshadinglangarb &&
      ext->hasglfragmentshaderarb && 
      ext->hasglvertexshaderarb   &&
      ext->hasglshaderobjectsarb) {
    glslextensionsavailable=1; // GLSL is available
  }

  if (glslextensionsavailable) {
    mainshader = new OpenGLShader(ext);

    if (mainshader) {
      char *shaderpath = NULL;

      if (getenv("VMDOGLSHADER") != NULL) {
        shaderpath = (char *) malloc(strlen(getenv("VMDOGLSHADER")) + 512);
        strcpy(shaderpath, getenv("VMDOGLSHADER"));
      } else if (getenv("VMDDIR") != NULL) {
        shaderpath = (char *) malloc(strlen(getenv("VMDDIR")) + 512);
        strcpy(shaderpath, getenv("VMDDIR"));
        strcat(shaderpath, "/shaders/vmd");
      } else {
        msgErr << "Unable to locate VMD vertex and fragment shader path, " 
               << "VMDDIR environment variable not set" << sendmsg;
        delete mainshader;
        mainshader = NULL;
      }   

      if (mainshader) {
#if defined(_MSC_VER)
        // convert '/' to '\' for Windows...
        int i, len;
        len=strlen(shaderpath);
        for (i=0; i<len; i++) {
          if (shaderpath[i] == '\\') {
            shaderpath[i] = '/';
          }
        }
#endif

        if (mainshader->LoadShader(shaderpath)) {
          mainshader->UseShader(0); // if glsl is available, turn off initially
          // OpenGL rendering state gets propagated on-demand at render time 
          // whenever ogl_renderstateserial != ogl_glslserial, thus no need to
          // enable the shader immediately at startup anymore.
        } else {
          msgWarn << "GPU driver failed to compile shader: " << sendmsg;
          msgWarn << "  " << shaderpath << sendmsg;
          delete mainshader;
          mainshader = NULL;
        }
      }
    }
    OGLERR // enable OpenGL debugging code
  }

#if defined(VMDUSEGLSLSPHERES)
  // if the main shader compiled successfully, try loading up the 
  // sphere shader also
  if (mainshader) {
    sphereshader = new OpenGLShader(ext);
    char *shaderpath = NULL;

    if (getenv("VMDOGLSPHERESHADER") != NULL) {
      shaderpath = (char *) malloc(strlen(getenv("VMDOGLSPHERESHADER")) + 512);
      strcpy(shaderpath, getenv("VMDOGLSPHERESHADER"));
    } else if (getenv("VMDDIR") != NULL) {
      shaderpath = (char *) malloc(strlen(getenv("VMDDIR")) + 512);
      strcpy(shaderpath, getenv("VMDDIR"));
      strcat(shaderpath, "/shaders/vmdsphere");
    } else {
      msgWarn << "Unable to locate VMD sphere vertex and fragment shaders, " 
              << "VMDDIR environment variable not set" << sendmsg;
      delete sphereshader;
      sphereshader = NULL;
    }   

    if (sphereshader) {
#if defined(_MSC_VER)
      // convert '/' to '\' for Windows...
      int i, len;
      len=strlen(shaderpath);
      for (i=0; i<len; i++) {
        if (shaderpath[i] == '\\') {
          shaderpath[i] = '/';
        }
      }
#endif

      if (sphereshader->LoadShader(shaderpath)) {
        sphereshader->UseShader(0); // if glsl is available, turn off initially
        // OpenGL rendering state gets propagated on-demand at render time 
        // whenever ogl_renderstateserial != ogl_glslserial, thus no need to
        // enable the shader immediately at startup anymore.
      } else {
        msgWarn << "GPU driver failed to compile shader: " << sendmsg;
        msgWarn << "  " << shaderpath << sendmsg;
        delete sphereshader;
        sphereshader = NULL;
      }
    }
    OGLERR // enable OpenGL debugging code
  }
#endif

  if (mainshader && sphereshader) {
    msgInfo << "  Full GLSL rendering mode is available." << sendmsg;
  } else if (mainshader) {
    if (glslextensionsavailable) {
      msgWarn << "This GPU/driver is buggy, or doesn't fully implement GLSL." << sendmsg;
      msgWarn << "Set environment variable VMDGLSLVERBOSE for more info." << sendmsg;
    }
    msgInfo << "  Basic GLSL rendering mode is available." << sendmsg;
  } else {
    if (glslextensionsavailable) {
      msgWarn << "This GPU/driver is buggy, or doesn't fully implement GLSL." << sendmsg;
      msgWarn << "Set environment variable VMDGLSLVERBOSE for more info." << sendmsg;
    }
    msgInfo << "  GLSL rendering mode is NOT available." << sendmsg;
  }
#endif

  // print information on OpenGL texturing hardware
  if (ext->hastex2d || ext->hastex3d) {
    msgInfo << "  Textures: ";
  
    if (ext->hastex2d) 
      msgInfo << "2-D (" << max2DtexX << "x" << max2DtexY << ")"; 

    if (ext->hastex2d && ext->hastex3d)
      msgInfo << ", ";

    if (ext->hastex3d) 
      msgInfo << "3-D (" << max3DtexX << "x" << max3DtexY << "x" << max3DtexZ << ")";

    if ((ext->hastex2d || ext->hastex3d) && ext->multitextureunits > 0)
      msgInfo << ", ";

    if (ext->multitextureunits > 0)
      msgInfo << "Multitexture (" << ext->multitextureunits << ")";

    msgInfo << sendmsg;
  }

  // print information about special stereo configuration
  if (getenv("VMDIMMERSADESKFLIP") != NULL) {
    immersadeskflip = 1;
    msgInfo << "  Enabled Immersadesk right-eye reflection stereo mode" << sendmsg;
  }

  // print information about special stereo configuration
  if (getenv("VMDSHEARSTEREO") != NULL) {
    shearstereo = 1;
    msgInfo << "  Enabled shear matrix stereo projection mode" << sendmsg;
  }

  OGLERR // enable OpenGL debugging code
}


void OpenGLRenderer::update_lists(void) {
  // point SphereList to the proper list
  ResizeArray<GLuint> *lists = (sphereMode == 
      ::SOLIDSPHERE) ? &solidSphereLists : &pointSphereLists;
  int ind = sphereRes - MIN_SPHERE_RES;
  if (ind < 0) 
    ind = 0;
  else if (ind >= lists->num())
    ind = lists->num()-1;
  SphereList = (*lists)[ind];
}

/////////////////////////  constructor and destructor  
// constructor ... initialize some variables
OpenGLRenderer::OpenGLRenderer(const char *nm) : DisplayDevice(nm) {
  // initialize data
  dataBlock = NULL;
#if defined(VMDUSEOPENGLSHADER)
  mainshader = NULL;   // disable programmable shading functionality by default
  sphereshader = NULL; // disable programmable shading functionality by default
#endif
  ext = new OpenGLExtensions;

  dpl_initialized = 0; // display lists need to be initialized still
}


// destructor
OpenGLRenderer::~OpenGLRenderer(void) {
  gluDeleteQuadric(objQuadric);    // delete the quadrics
  gluDeleteQuadric(pointsQuadric); // delete the quadrics

  delete ext;                      // delete OpenGL extensions

#if defined(VMDUSEOPENGLSHADER)
  delete mainshader;                  // delete programmable shaders
  delete sphereshader;                // delete programmable shaders
#endif
}

// prepare to free OpenGL context (should be called from subclass destructor)
void OpenGLRenderer::free_opengl_ctx() {
  int i;
  GLuint tag;

  // delete all cached display lists
  displaylistcache.markUnused();
  while ((tag = displaylistcache.deleteUnused()) != GLCACHE_FAIL) {
    glDeleteLists(tag, 1);
  }

  // delete all cached textures
  texturecache.markUnused();
  while ((tag = texturecache.deleteUnused()) != GLCACHE_FAIL) {
    glDeleteTextures(1, &tag);
  }

  if (dpl_initialized) { 
    // free sphere display lists
    for (i=MIN_SPHERE_RES; i<=MAX_SPHERE_RES; i++) {
      glDeleteLists(solidSphereLists[i], 1);
      glDeleteLists(pointSphereLists[i], 1);
    } 

    // free the display lists used for the 3-D label/text font
    glDeleteLists(fontListBase, 256);
  }
}


/////////////////////////  protected nonvirtual routines  

// change current line width
void OpenGLRenderer::set_line_width(int w) {
  if(w > 0) {
    glLineWidth((GLfloat)w);
    lineWidth = w;
  }
}

// change current line style
void OpenGLRenderer::set_line_style(int s) {
  if(s == ::DASHEDLINE) {
    lineStyle = s;
    glEnable(GL_LINE_STIPPLE);
  } else {
    lineStyle = ::SOLIDLINE;
    glDisable(GL_LINE_STIPPLE);
  }
}


// change current sphere resolution
void OpenGLRenderer::set_sphere_res(int r) {
  // avoid unnecessary display list state changes, helps avoid some serious
  // OpenGL performance problems on MacOS X.
  if (sphereRes == r)
    return; 

  if (r > 2)
    sphereRes = r;
  else
    sphereRes = 2;

  update_lists();
}

// change current sphere type
void OpenGLRenderer::set_sphere_mode(int m) {
  // avoid unnecessary display list state changes, helps avoid some serious
  // OpenGL performance problems on MacOS X.
  if (sphereMode == m)
    return; 

  sphereMode = m;
  update_lists();
}

// this routine draws a cylinder from start to end, using rod_res panels,
// of radius rod_radius
void OpenGLRenderer::cylinder(float *end, float *start, int rod_res,
                              float rod_radius, float rod_top_radius) {
  float R, RXY, phi, theta, lenaxis[3];

  // need to do some preprocessing ... find length of vector
  lenaxis[0] = end[0] - start[0];
  lenaxis[1] = end[1] - start[1];
  lenaxis[2] = end[2] - start[2];

  R = lenaxis[0]*lenaxis[0]+lenaxis[1]*lenaxis[1]+lenaxis[2]*lenaxis[2];

  if (R <= 0.0)
    return; // early exit if cylinder is of length 0;

  R = sqrtf(R); // evaluation of sqrt() _after_ early exit 

  // determine phi rotation angle, amount to rotate about y
  phi = acosf(lenaxis[2] / R);

  // determine theta rotation, amount to rotate about z
  RXY = sqrtf(lenaxis[0]*lenaxis[0]+lenaxis[1]*lenaxis[1]);
  if (RXY <= 0.0f) {
    theta = 0.0f;
  } else {
    theta = acosf(lenaxis[0] / RXY);
    if (lenaxis[1] < 0.0f)
      theta = (float) (2.0f * PI) - theta;
  }

  glPushMatrix(); // setup transform moving cylinder from Z-axis to position
  glTranslatef((GLfloat)(start[0]), (GLfloat)(start[1]), (GLfloat)(start[2]));
  if (theta != 0.0f)
    glRotatef((GLfloat) ((theta / PI) * 180.0f), 0.0f, 0.0f, 1.0f);
  if (phi != 0.0f)
    glRotatef((GLfloat) ((phi / PI) * 180.0f), 0.0f, 1.0f, 0.0f);

  // call utility routine to draw cylinders
  gluCylinder(objQuadric, (GLdouble)rod_radius, (GLdouble)rod_top_radius,
	      (GLdouble)R, (GLint)rod_res, 1);

  // if this is a cone, we also draw a disk at the bottom
  gluQuadricOrientation(objQuadric, (GLenum)GLU_INSIDE);
  gluDisk(objQuadric, (GLdouble)0, (GLdouble)rod_radius, (GLint)rod_res, 1);
  gluQuadricOrientation(objQuadric, (GLenum)GLU_OUTSIDE);

  glPopMatrix(); // restore the previous transformation matrix
}


// this routine also draws a cylinder.  However, it assumes that
// the cylinder drawing command has precomputed the data.  This
// uses more memory, but is faster
// the data are: num == number of edges
//  edges = a normal, start, and end 
static void cylinder_full(int num, float *edges, int filled) {
  int n = num;
  float *start = edges;

  if (num < 2)
     return;

  glBegin(GL_QUAD_STRIP);
    while (n-- > 0) {
      glNormal3fv(edges);
      glVertex3fv(edges+6);
      glVertex3fv(edges+3);
      edges += 9;
    }
    glNormal3fv(start);  // loop back to the beginning
    glVertex3fv(start+6);
    glVertex3fv(start+3);
  glEnd();

  // and fill in the top and bottom, if needed
  if (filled) {
    float axis[3];
    axis[0] = start[6] - start[3];
    axis[1] = start[7] - start[4];
    axis[2] = start[8] - start[5];
    vec_normalize(axis);

    if (filled & CYLINDER_LEADINGCAP) { // do the first side
      n = num;            // get one side
      edges = start + 3;
      glBegin(GL_POLYGON);
        glNormal3fv(axis);
        while (--n >= 0) {
          glVertex3fv(edges);
          edges += 9;
        }
      glEnd();
    }
    if (filled & CYLINDER_TRAILINGCAP) { // do the other side
      n = num;          // and the other
      edges = start + 6;
      glBegin(GL_POLYGON);
        glNormal3fv(axis);       // I'm going the other direction, so
        while (--n >= 0) {
          glVertex3fv(edges);
          edges += 9;
        }
      glEnd();
    }
  }
}


/////////////////////////  protected virtual routines  

// define a new light source ... return success of operation
int OpenGLRenderer::do_define_light(int n, float *color, float *position) {
  int i;
 
  for(i=0; i < 3; i++)  {
    ogl_lightcolor[n][i] = color[i];
    ogl_lightpos[n][i] = position[i];
  }
  ogl_lightpos[n][3] = 0.0; // directional lights require w=0.0 otherwise
                            // OpenGL assumes they are positional lights.
  ogl_lightcolor[n][3] = 1.0;

  // normalize the light direction vector
  vec_normalize(&ogl_lightpos[n][0]); // 4th element is left alone

  glLightfv((GLenum)(GL_LIGHT0 + n), GL_POSITION, &ogl_lightpos[n][0]);
  glLightfv((GLenum)(GL_LIGHT0 + n), GL_SPECULAR, &ogl_lightcolor[n][0]);
  
  ogl_rendstateserial++; // cause GLSL cached state to update when necessary
  _needRedraw = 1;
  return TRUE;
}

// activate a given light source ... return success of operation
int OpenGLRenderer::do_activate_light(int n, int turnon) {
  if (turnon) {
    glEnable((GLenum)(GL_LIGHT0 + n));
    ogl_lightstate[n] = 1;
  } else {
    glDisable((GLenum)(GL_LIGHT0 + n));
    ogl_lightstate[n] = 0;
  }

  ogl_rendstateserial++; // cause GLSL cached state to update when necessary
  _needRedraw = 1;
  return TRUE;
}

void OpenGLRenderer::loadmatrix(const Matrix4 &m) {
  GLfloat tmpmat[16];
  for (int i=0; i<16; i++) tmpmat[i]=(GLfloat)(m.mat[i]);
  glLoadMatrixf(tmpmat);
}

void OpenGLRenderer::multmatrix(const Matrix4 &m) {
  GLfloat tmpmat[16];
  for (int i=0; i<16; i++) tmpmat[i]=(GLfloat)(m.mat[i]);
  glMultMatrixf(tmpmat);
}

// virtual routines to return 2D screen coordinates, given 2D or 3D world
// coordinates.  These assume the proper GL window has focus, etc.
// The coordinates returned are absolute screen coords, relative to the lower
// left corner of the display monitor
void OpenGLRenderer::abs_screen_loc_3D(float *loc, long *spos) {
  GLdouble modelMatrix[16], projMatrix[16];
  GLdouble pos[3];
  int i;

  // get current matrices and viewport for project call
  for (i=0; i<16; i++) {
    modelMatrix[i] = ogl_mvmatrix[i];
    projMatrix[i] = ogl_pmatrix[i];
  }

  // call the GLU routine to project the object coord to world coords
  if(!gluProject((GLdouble)(loc[0]), (GLdouble)(loc[1]), (GLdouble)(loc[2]),
     modelMatrix, projMatrix, ogl_viewport, pos, pos + 1, pos + 2)) {
    msgErr << "Cannot determine window position of world coordinate.";
    msgErr << sendmsg;
  } else {
    spos[0] = (long)(pos[0]) + xOrig;
    spos[1] = (long)(pos[1]) + yOrig;
  }
}

void OpenGLRenderer::abs_screen_loc_2D(float *loc, long *spos) {
  float newloc[3];
  newloc[0] = loc[0];
  newloc[1] = loc[1];
  newloc[2] = 0.0f;
  abs_screen_loc_3D(newloc, spos);
}

// Given a 3D point (pos A),
// and a 2D rel screen pos point (for pos B), computes the 3D point
// which goes with the second 2D point at pos B.  Result returned in B3D.
// NOTE: currently, this algorithm only supports the simple case where the
// eye look direction is along the Z-axis.  A more sophisticated version
// requires finding the plane to which the look direction is normal, which is
// assumed here to be the Z-axis (for simplicity in coding).
void OpenGLRenderer::find_3D_from_2D(const float *A3D, const float *B2D,
				     float *B3D) {
  GLdouble modelMatrix[16], projMatrix[16], w1[3], w2[3];
  int i;
  float lsx, lsy;		// used to convert rel screen -> abs

  // get current matrices and viewport for unproject call
  for (i=0; i<16; i++) {
    modelMatrix[i] = ogl_mvmatrix[i];
    projMatrix[i] = ogl_pmatrix[i];
  }

  // get window coordinates of 2D point
  lsx = B2D[0];
  lsy = B2D[1];
  lsx = lsx * (float)xSize;
  lsy = lsy * (float)ySize;

  // call the GLU routine to unproject the window coords to world coords
  if (!gluUnProject((GLdouble)lsx, (GLdouble)lsy, 0,
      modelMatrix, projMatrix, ogl_viewport, w1, w1 + 1, w1 + 2)) {
    msgErr << "Can't determine world coords of window position 1." << sendmsg;
    return;
  }
  if (!gluUnProject((GLdouble)lsx, (GLdouble)lsy, 1.0,
      modelMatrix, projMatrix, ogl_viewport, w2, w2 + 1, w2 + 2)) {
    msgErr << "Can't determine world coords of window position2." << sendmsg;
    return;
  }

  // finally, find the point where line returned as w1..w2 intersects the 
  // given 3D point's plane (this plane is assumed to be parallel to the X-Y
  // plane, i.e., with a normal along the Z-axis.  A more general algorithm
  // would need to find the plane which is normal to the eye look direction,
  // and which contains the given 3D point.)
  
  // special case: w1z = w2z ==> just return given 3D point, since there
  //		is either no solution, or the line is in the given plane
  if(w1[2] == w2[2]) {
    memcpy(B3D, A3D, 3*sizeof(float));
  } else {
    float relchange = (float) ((A3D[2] - w1[2]) / (w2[2] - w1[2]));
    B3D[0] = (float) ((w2[0] - w1[0]) * relchange + w1[0]);
    B3D[1] = (float) ((w2[1] - w1[1]) * relchange + w1[1]);
    B3D[2] = A3D[2];
  }
}


//
// antialiasing and depth-cueing
//

// turn on antialiasing effect
void OpenGLRenderer::aa_on(void) {
  if (inStereo == OPENGL_STEREO_STENCIL) {
    msgInfo << "Antialiasing must be disabled for scanline interleaved stereo."
<< sendmsg;
    msgInfo << "You may re-enable antialiasing when stereo is turned off." << sendmsg;
    aa_off();
    return;
  }

  if (aaAvailable && !aaEnabled) {
#if defined(GL_ARB_multisample)
    if (ext->hasmultisample) {
      glEnable(GL_MULTISAMPLE_ARB);
      aaEnabled = TRUE;
      _needRedraw = 1;
      return;
    } 
#else
#if defined(GL_SGIS_multisample)
    if (ext->hasmultisample) {
      glEnable(GL_MULTISAMPLE_SGIS);
#if defined(ARCH_IRIX6) || defined(ARCH_IRIX6_64) || defined(ARCH_IRIX5)
      glSamplePatternSGIS(GL_1PASS_SGIS);
#endif
      aaEnabled = TRUE;
      _needRedraw = 1;
      return;
    } 
#endif
#endif
    // could implement accumulation buffer antialiasing here someday
    aaEnabled = TRUE;
  }
}

// turn off antialiasing effect
void OpenGLRenderer::aa_off(void) {
  if(aaAvailable && aaEnabled) {
#if defined(GL_ARB_multisample)
    if (ext->hasmultisample) {
      glDisable(GL_MULTISAMPLE_ARB);
      aaEnabled = FALSE;
      _needRedraw = 1;
      return;
    } 
#else
#if defined(GL_SGIS_multisample)
    if (ext->hasmultisample) {
      glDisable(GL_MULTISAMPLE_SGIS);
      aaEnabled = FALSE;
      _needRedraw = 1;
      return;
    } 
#endif
#endif
    // could implement accumulation buffer antialiasing here someday
    aaEnabled = FALSE;
  }
}

// turn on hardware depth-cueing
void OpenGLRenderer::cueing_on(void) {
  if(cueingAvailable && !cueingEnabled) {
    glEnable(GL_FOG);
    cueingEnabled = TRUE;
    _needRedraw = 1;
  }
}

// turn off hardware depth-cueing
void OpenGLRenderer::cueing_off(void) {
  if(cueingAvailable && cueingEnabled) {
    glDisable(GL_FOG);
    cueingEnabled = FALSE;
    _needRedraw = 1;
  }
}


void OpenGLRenderer::culling_on(void) {
  if (cullingAvailable && !cullingEnabled) {
    glFrontFace(GL_CCW);              // set CCW as fron face direction
    glPolygonMode(GL_FRONT, GL_FILL); // set front face fill mode
    glPolygonMode(GL_BACK,  GL_LINE); // set back face fill mode
    glCullFace(GL_BACK);              // set for culling back faces
    glEnable(GL_CULL_FACE);           // enable face culling
    cullingEnabled = TRUE; 
    _needRedraw = 1;
  }
}

void OpenGLRenderer::culling_off(void) {
  if (cullingAvailable && cullingEnabled) {
    glPolygonMode(GL_FRONT, GL_FILL); // set front face fill mode
    glPolygonMode(GL_BACK,  GL_FILL); // set back face fill mode
    glCullFace(GL_BACK);              // set for culling back faces
    glDisable(GL_CULL_FACE);          // disable face culling
    cullingEnabled = FALSE; 
    _needRedraw = 1;
  }
}

void OpenGLRenderer::set_background(const float *newback) {
  GLfloat r, g, b;
  r = (GLfloat)newback[0];
  g = (GLfloat)newback[1];
  b = (GLfloat)newback[2];

  // set fog color used for depth cueing
  GLfloat fogcol[4];
  fogcol[0] = r;
  fogcol[1] = g;
  fogcol[2] = b;
  fogcol[3] = 1.0;

  glFogfv(GL_FOG_COLOR, fogcol);

  // set clear color
  glClearColor((GLclampf)r,
               (GLclampf)g,
               (GLclampf)b, 1.0);
}

void OpenGLRenderer::set_backgradient(const float *topcol, 
                                      const float *botcol) {
  int i;
  for (i=0; i<3; i++) {
    ogl_backgradient[0][i] = topcol[i]; 
    ogl_backgradient[1][i] = botcol[i]; 
  }
  ogl_backgradient[0][3] = 1.0;
  ogl_backgradient[1][3] = 1.0;
}

// change to a different stereo mode
void OpenGLRenderer::set_stereo_mode(int newMode) {
  if (inStereo == newMode)
    return;   // do nothing if current mode is same as specified mode

  if (inStereo == OPENGL_STEREO_STENCIL)
    disable_stencil_stereo(); 

  if (newMode == OPENGL_STEREO_STENCIL)
    enable_stencil_stereo(); 
  inStereo = newMode;  // set new mode
  reshape();           // adjust the viewport width/height
  normal();            // adjust the viewport size/projection matrix
                       // this is reset again when left/right are called.
  clear();             // clear the screen
  update();            // redraw

  _needRedraw = 1;
}

// change to a different caching mode
void OpenGLRenderer::set_cache_mode(int newMode) {
  cacheMode = newMode; // set new mode;
  ogl_cacheenabled = newMode;
}

// change to a different rendering mode
void OpenGLRenderer::set_render_mode(int newMode) {
  if (renderMode == newMode)
    return;   // do nothing if current mode is same as specified mode

  renderMode = newMode;  // set new mode

  switch (renderMode) {
    case OPENGL_RENDER_NORMAL:
      ogl_useblendedtrans = 0;
      ogl_useglslshader = 0;
      ogl_acrobat3dcapture = 0;
      break;

    case OPENGL_RENDER_GALPHA:
      if (ext->hasglobalalphasun) {
        ogl_useblendedtrans = 1;
      } else {
        ogl_useblendedtrans = 0;
        msgWarn << "Blended alpha transparency not available." << sendmsg;
      }
      ogl_useglslshader = 0;
      ogl_acrobat3dcapture = 0;
      break;

    case OPENGL_RENDER_GLSL:
#if defined(VMDUSEOPENGLSHADER)
      // GLSL shader state variables must now be updated to match the 
      // active fixed-pipeline state before/during the next rendering pass. 
      if (mainshader) {
        ogl_useblendedtrans = 1;
        ogl_useglslshader = 1;
      } else
#endif
      {
        ogl_useblendedtrans = 0;
        ogl_useglslshader = 0;
        msgWarn << "OpenGL Programmable Shading not available." << sendmsg;
      }
      ogl_acrobat3dcapture = 0;
      break;

    case OPENGL_RENDER_ACROBAT3D:
      ogl_useblendedtrans = 0;
      ogl_useglslshader = 0;
      ogl_acrobat3dcapture = 1;
      break;
  }

  reshape();           // adjust the viewport width/height
  normal();            // adjust the viewport size/projection matrix
                       // this is reset again when left/right are called.
  clear();             // clear the screen
  update();            // redraw

  _needRedraw = 1;
}


// set up for normal (non-stereo) drawing.  Sets the viewport and perspective.
void OpenGLRenderer::normal(void) {
  glViewport(0, 0, (GLsizei)xSize, (GLsizei)ySize);
  set_persp();

  // draw the background gradient if necessary
  draw_background_gradient();
}


void OpenGLRenderer::enable_stencil_stereo(void) {
  int i;
  
  if (!ext->hasstencilbuffer) {
    set_stereo_mode(OPENGL_STEREO_OFF); 
    msgInfo << "Stencil Buffer Stereo is NOT available." << sendmsg;
    return;
  } 

  if (aaEnabled) {
    msgInfo << "Antialiasing must be disabled for scanline interleaved stereo." << sendmsg;
    msgInfo << "Antialiasing will be re-enabled when stereo is turned off." << sendmsg;
    aaPrevious = aaEnabled;
    aa_off();
  }

  glPushMatrix();
  glDisable(GL_DEPTH_TEST);

  glViewport(0, 0, (GLsizei)xSize, (GLsizei)ySize);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  glMatrixMode (GL_PROJECTION);
  glLoadIdentity();
  
  gluOrtho2D(0.0, xSize, 0.0, ySize);

  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();

  // clearing and configuring stencil drawing
  glDrawBuffer(GL_BACK);

  glEnable(GL_STENCIL_TEST);
  glClearStencil(0);
  glClear(GL_STENCIL_BUFFER_BIT);
  glStencilOp(GL_REPLACE, GL_REPLACE, GL_REPLACE); 
  glStencilFunc(GL_ALWAYS, 1, 1); 

  glColor4f(1,1,1,0); // set draw color to all 1s

  // draw the stencil pattern on every other line of the display
  // According to Appendix G. of the OpenGL 1.2 Programming Guide
  // correct 2-D line rasterization requires placing vertices at half
  // pixel offsets.  This is mentioned specifically on page 677.
  for (i=0; i<ySize; i+=2) {
    glLineWidth(1);
    glBegin(GL_LINES);
      glVertex2f(           0.0f, (GLfloat) i + 0.5f);
      glVertex2f((GLfloat) xSize, (GLfloat) i + 0.5f);
    glEnd();
  }

  glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP); // disable changes to stencil buffer

  glEnable(GL_DEPTH_TEST);
  
  glPopMatrix();
}

void OpenGLRenderer::disable_stencil_stereo(void) {
  glDisable(GL_STENCIL_TEST);
  if (aaPrevious) {
    // XXX hack to please aa_on() so it'll reenable stereo even though
    // inStereo isn't quite back to normal yet.
    int foo = inStereo;    
    inStereo = OPENGL_STEREO_OFF;
    aa_on(); // re-enable antialiasing if we're leaving stenciling mode
    inStereo = foo;
    msgInfo << "Antialiasing re-enabled." << sendmsg;
  }
}


// set up for drawing the left eye image.  Assume always the left eye is
// drawn first (so no zclear is needed before it)
void OpenGLRenderer::left(void) {
  switch (inStereo) {
    case OPENGL_STEREO_DTISIDE:
      glViewport(0, 0, (GLsizei)xSize / 2, (GLsizei)ySize);
      set_persp(LEFTEYE);
      break;

    case OPENGL_STEREO_SIDE:
      glViewport(0, 0, (GLsizei)xSize / 2, (GLsizei)ySize);
      set_persp(LEFTEYE);
      break;

    case OPENGL_STEREO_CROSSED:
      glViewport(0, 0, (GLsizei)xSize / 2, (GLsizei)ySize);
      set_persp(RIGHTEYE);      // eyes are swapped
      break;

    case OPENGL_STEREO_LEFT:
      set_persp(LEFTEYE);
      break;

    case OPENGL_STEREO_RIGHT:
      set_persp(RIGHTEYE);
      break;

    case OPENGL_STEREO_CRYSTAL:
    case OPENGL_STEREO_CRYSTAL_REVERSE:
      if (ext->hasstereo)
        glDrawBuffer(GL_BACK_LEFT); // Z-buffer must be cleared already
      else  
        // XXX do something since we don't support non-quad buffered modes
        glViewport(0, (GLint)ySize / 2, (GLsizei)xSize, (GLsizei)ySize / 2);

      if (inStereo == OPENGL_STEREO_CRYSTAL)
        set_persp(LEFTEYE);  // OPENGL_STEREO_CRYSTAL
      else
        set_persp(RIGHTEYE); // OPENGL_STEREO_CRYSTAL_REVERSE
      break;

    case OPENGL_STEREO_STENCIL:
      glStencilFunc(GL_NOTEQUAL,1,1); // draws if stencil <> 1
      set_persp(LEFTEYE);
      break;

    case OPENGL_STEREO_ANAGLYPH:
      if(ext->hasstereo) {
        glDrawBuffer(GL_BACK_LEFT); // Z-buffer must be cleared already
        glColorMask(GL_FALSE, GL_FALSE, GL_TRUE, GL_TRUE); 
      }
      else { 
        // XXX do something since we don't support non-quad buffered modes
        //     otherwise.
        glViewport(0, (GLint)ySize / 2, (GLsizei)xSize, (GLsizei)ySize / 2);
      } 
      set_persp(LEFTEYE);
      break;

    default:
      normal(); // left called even though we're non-stereo
// not tested on other platforms yet
#if defined(ARCH_MACOSX) || defined(ARCH_MACOSXX86)
      if (ext->hasstereo && ext->stereodrawforced)
        glDrawBuffer(GL_BACK_LEFT); // draw to back-left
#endif
      break;
  }
}


// set up for drawing the right eye image.  Assume always the right eye is
// drawn last (so a zclear IS needed before it)
void OpenGLRenderer::right(void) {
  switch (inStereo) {
    case OPENGL_STEREO_DTISIDE:
      glViewport((GLsizei)xSize/2, 0, (GLsizei)xSize / 2, (GLsizei)ySize);
      set_persp(RIGHTEYE);
      break;

    case OPENGL_STEREO_SIDE:
      glViewport((GLsizei)xSize/2, 0, (GLsizei)xSize / 2, (GLsizei)ySize);
      set_persp(RIGHTEYE);
      break;

    case OPENGL_STEREO_CROSSED:
      glViewport((GLsizei)xSize/2, 0, (GLsizei)xSize / 2, (GLsizei)ySize);
      set_persp(LEFTEYE);      // eyes are swapped
      break;

    case OPENGL_STEREO_LEFT:
    case OPENGL_STEREO_RIGHT:
      // no need to do anything, already done in call to left
      break;

    case OPENGL_STEREO_CRYSTAL:
    case OPENGL_STEREO_CRYSTAL_REVERSE:
      if (ext->hasstereo) {
        glDepthMask(GL_TRUE);  // make Z-buffer writable
#if defined(ARCH_MACOSX) || defined(ARCH_MACOSXX86)
        // XXX This hack is required by MacOS X because their 
        // Quadro 4500 stereo drivers are broken such that the 
        // clear on both right/left buffers doesn't actually work.
        // This explicitly forces a second clear on the back right buffer.
        glDrawBuffer(GL_BACK_RIGHT);
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
#else
        // all other platforms work fine
        glClear(GL_DEPTH_BUFFER_BIT);
#endif
        glDrawBuffer(GL_BACK_RIGHT);
      } else {
        // XXX do something since we don't support non-quad buffered modes
        glViewport(0, 0, (GLsizei)xSize, (GLsizei)ySize / 2);
      }
      if (inStereo == OPENGL_STEREO_CRYSTAL)
        set_persp(RIGHTEYE); // OPENGL_STEREO_CRYSTAL
      else
        set_persp(LEFTEYE);  // OPENGL_STEREO_CRYSTAL_REVERSE
      break;

    case OPENGL_STEREO_STENCIL:
      glDepthMask(GL_TRUE);  // make Z-buffer writable
      glClear(GL_DEPTH_BUFFER_BIT);
      glStencilFunc(GL_EQUAL,1,1); // draws if stencil <> 0
      set_persp(RIGHTEYE);
      break;

    case OPENGL_STEREO_ANAGLYPH:
      if(ext->hasstereo) {
        glDepthMask(GL_TRUE);  // make Z-buffer writable
        glClear(GL_DEPTH_BUFFER_BIT);
        glDrawBuffer(GL_BACK_RIGHT);
        glColorMask(GL_TRUE, GL_FALSE, GL_FALSE, GL_TRUE); 
      }
      else {
        // split the window horizontally until we see how to do this
        glViewport(0, 0, (GLsizei)xSize, (GLsizei)ySize / 2);
      }
      set_persp(RIGHTEYE);
      break;

    default:
      normal(); // right called even though we're non-stereo
// not tested on other platforms yet
#if defined(ARCH_MACOSX) || defined(ARCH_MACOSXX86)
      if (ext->hasstereo && ext->stereodrawforced)
        glDrawBuffer(GL_BACK_RIGHT); // draw to back-right
#endif
      break;
  }

  // draw the background gradient if necessary
  draw_background_gradient();
}


// set the current perspective, based on the eye position and where we
// are looking.
void OpenGLRenderer::set_persp(DisplayEye my_eye) {
  // define eye and look at some point.  Assumes up vector = (0,1,0)
  GLdouble ep[3];
  switch (my_eye) {
    case NOSTEREO:
      ep[0] = eyePos[0];
      ep[1] = eyePos[1];
      ep[2] = eyePos[2];
      DisplayDevice::normal();
      break;
    case LEFTEYE:
      ep[0] = eyePos[0] - eyeSepDir[0];
      ep[1] = eyePos[1] - eyeSepDir[1];
      ep[2] = eyePos[2] - eyeSepDir[2];
      DisplayDevice::left();
      break;
    case RIGHTEYE: 
      ep[0] = eyePos[0] + eyeSepDir[0];
      ep[1] = eyePos[1] + eyeSepDir[1];
      ep[2] = eyePos[2] + eyeSepDir[2];
      DisplayDevice::right();
      break;
  }

  // setup camera system and projection transformations
  if (projection() == PERSPECTIVE) {
    ogl_glslprojectionmode = DisplayDevice::PERSPECTIVE; 

    if (shearstereo) {
      // XXX almost ready for prime time, when testing is done we may
      // make shear stereo the default and eye rotation a backwards 
      // compatibility option.
      // Use the "frustum shearing" method for creating a stereo view.  
      // The frustum shearing method is preferable to eye rotation in general.

      // Calculate the eye shift (half eye separation distance)
      // XXX hack, needs to be more general
      float eyeshift = (ep[0] - eyePos[0]);

      glMatrixMode(GL_PROJECTION);
      glLoadIdentity();
      // Shifts viewing frustum horizontally in the image plane
      // according to the stereo eye separation if rendering in stereo.
      // XXX hack, the parameterization of this projection still 
      // needs work, but the fact that it incorporates eyeDist is nice.
      glFrustum((GLdouble)cpLeft  + (eyeshift * nearClip / eyeDist),
                (GLdouble)cpRight + (eyeshift * nearClip / eyeDist),
                (GLdouble)cpDown, 
                (GLdouble)cpUp,
                (GLdouble)nearClip, 
                (GLdouble)farClip);

      // Shift the eye position horizontally by half the eye separation
      // distance if rendering in stereo.
      glTranslatef(-eyeshift, 0.0, 0.0); 

      glMatrixMode(GL_MODELVIEW);
      glLoadIdentity();
      gluLookAt(eyePos[0], eyePos[1], eyePos[2],
                (GLdouble)(eyePos[0] + eyeDir[0]),
                (GLdouble)(eyePos[1] + eyeDir[1]),
                (GLdouble)(eyePos[2] + eyeDir[2]),
                (GLdouble)(upDir[0]),
                (GLdouble)(upDir[1]),
                (GLdouble)(upDir[2]));
    } else {
      // Use the "eye rotation" method for creating a stereo view.  
      // The frustum shearing method would be preferable.
      // XXX this implementation is not currently using the eyeDist
      // parameter, though it probably should.
      glMatrixMode(GL_PROJECTION);
      glLoadIdentity();
      glFrustum((GLdouble)cpLeft,  (GLdouble)cpRight,  (GLdouble)cpDown,
                (GLdouble)cpUp,   (GLdouble)nearClip, (GLdouble)farClip);

      // Reflect the X axis of the right eye for the new LCD panel immersadesks
      // XXX experimental hack that needs more work to get lighting
      // completely correct for the Axes, Title Screen, etc.
      if (immersadeskflip && my_eye == RIGHTEYE) {
        // Scale the X axis by -1 in the GL_PROJECTION matrix
        glScalef(-1, 1, 1);
      }

      glMatrixMode(GL_MODELVIEW);
      glLoadIdentity();
      gluLookAt(ep[0], ep[1], ep[2],
                (GLdouble)(eyePos[0] + eyeDir[0]),
                (GLdouble)(eyePos[1] + eyeDir[1]),
                (GLdouble)(eyePos[2] + eyeDir[2]),
                (GLdouble)(upDir[0]),
                (GLdouble)(upDir[1]),
                (GLdouble)(upDir[2]));
    }
  } else { // ORTHOGRAPHIC
    ogl_glslprojectionmode = DisplayDevice::ORTHOGRAPHIC; 
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();

    glOrtho(-0.25 * vSize * Aspect, 0.25 * vSize * Aspect,
            -0.25 * vSize,          0.25 * vSize,
            nearClip, farClip);

    // Use the "eye rotation" method for creating a stereo view.  
    // The frustum shearing method won't work with orthographic 
    // views since the eye rays are parallel, so the rotation method 
    // is ok in this case.
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    gluLookAt(ep[0], ep[1], ep[2],
              (GLdouble)(eyePos[0] + eyeDir[0]),
              (GLdouble)(eyePos[1] + eyeDir[1]),
              (GLdouble)(eyePos[2] + eyeDir[2]),
              (GLdouble)(upDir[0]),
              (GLdouble)(upDir[1]),
              (GLdouble)(upDir[2]));
  }

  // update the cached transformation matrices for use in text display, etc.
  glGetFloatv(GL_PROJECTION_MATRIX, ogl_pmatrix);
  glGetFloatv(GL_MODELVIEW_MATRIX, ogl_mvmatrix);
  glGetIntegerv(GL_VIEWPORT, ogl_viewport);
  ogl_textMat.identity();
  ogl_textMat.multmatrix(ogl_pmatrix);
  ogl_textMat.multmatrix(ogl_mvmatrix);
}


// prepare to draw a 3D image
int OpenGLRenderer::prepare3D(int do_clear) {
  if (do_clear) {
    clear();
  } else {
    glDepthMask(GL_TRUE);  // make Z-buffer writable
    glClear(GL_DEPTH_BUFFER_BIT);
  }

  // invalidate the OpenGL material index cache since a new frame is
  // being drawn and the material state for the previous index may 
  // have changed.  
  oglmaterialindex = -1;

  // start a new frame, marking all cached IDs as "unused"
  displaylistcache.markUnused();
  texturecache.markUnused();

  // draw the background gradient if necessary
  draw_background_gradient();

  return TRUE; // must return true for normal (non file-based) renderers
}


// prepare to draw opaque objects
int OpenGLRenderer::prepareOpaque(void) {
  if (ogl_useblendedtrans) {
    glDepthMask(GL_TRUE); // make Z-buffer writable
    ogl_transpass = 0;
  }

  return 1;
}

// prepare to draw transparent objects
int OpenGLRenderer::prepareTrans(void) {
  if (ogl_useblendedtrans) {
    glDepthMask(GL_FALSE); // make Z-buffer read-only while drawing trans objs
    ogl_transpass = 1;
    return 1;
  }

  return 0;
}

// clear the display
void OpenGLRenderer::clear(void) {
  // clear the whole viewport, not just one side 
  switch (inStereo) {
    case OPENGL_STEREO_DTISIDE:
    case OPENGL_STEREO_SIDE:
    case OPENGL_STEREO_CROSSED:
    case OPENGL_STEREO_CRYSTAL:
    case OPENGL_STEREO_CRYSTAL_REVERSE:
    case OPENGL_STEREO_ANAGLYPH:
      glViewport(0, 0, (GLsizei)xSize, (GLsizei)ySize);
      break;
  }

  glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);    // reset color mask 
  glDepthMask(GL_TRUE);                               // make Z-buffer writable
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT); // Clear color/depth bufs

// not tested on other platforms yet
#if defined(ARCH_MACOSX) || defined(ARCH_MACOSXX86)
  if (ext->hasstereo && ext->stereodrawforced) {
    glDrawBuffer(GL_BACK_RIGHT);
    glClear(GL_COLOR_BUFFER_BIT); // force separate clear of right buffer
    glDrawBuffer(GL_BACK);
  }
#endif
}


// draw the background gradient
void OpenGLRenderer::draw_background_gradient(void) {
  // if the background mode is set for gradient, then draw the gradient
  // note: this must be done immediately after clearing the viewport
  if (backgroundmode != 0) {
    int i;

    // disable all clipping planes by default
    for (i=0; i < VMD_MAX_CLIP_PLANE; i++) {
      ogl_clipmode[i] = 0; // off by default
      glDisable((GLenum) (GL_CLIP_PLANE0 + i));
    }

    glDisable(GL_LIGHTING);           // disable lighting
    ogl_lightingenabled=0;            // update state var
#if defined(VMDUSEOPENGLSHADER)
    if (mainshader && ogl_useglslshader) {
      mainshader->UseShader(0);       // use fixed-func pipeline
    }
#endif
    glDisable(GL_DEPTH_TEST);         // disable depth test
    glDepthMask(GL_FALSE);            // make Z-buffer read-only

    // turn off any transparent rendering state
    glDisable(GL_POLYGON_STIPPLE);    // make sure stippling is disabled
    glDisable(GL_BLEND);              // disable blending
#if defined(GL_SUN_global_alpha)
    if (ext->hasglobalalphasun) {
      GLGLOBALALPHAFACTORFSUN(1.0);   // disable global alpha factor
      glDisable(GL_GLOBAL_ALPHA_SUN); // disable global alpha blending mode
    }
#endif

    glMatrixMode(GL_MODELVIEW);       // save existing transformation state
    glPushMatrix();
    glLoadIdentity();                 // prepare for 2-D orthographic drawing

    glMatrixMode (GL_PROJECTION);     // save existing transformation state
    glPushMatrix();
    glLoadIdentity();
    gluOrtho2D(0.0, 1.0, 0.0, 1.0);   // enable 2-D orthographics drawing

    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();                 // add one more modelview

    // draw the background polygon
    glBegin(GL_QUADS);
      glColor3fv(&ogl_backgradient[1][0]);
      glVertex2f(0.0f, 0.0f);
      glColor3fv(&ogl_backgradient[0][0]);
      glVertex2f(0.0f, 1.0f);
      glColor3fv(&ogl_backgradient[0][0]);
      glVertex2f(1.0f, 1.0f);
      glColor3fv(&ogl_backgradient[1][0]);
      glVertex2f(1.0f, 0.0f);
    glEnd();

    glPopMatrix();                     // pop off top modelview

    glMatrixMode (GL_PROJECTION);
    glPopMatrix();                     // restore projection transform state

    glMatrixMode(GL_MODELVIEW);
    glPopMatrix();                     // restore modelview transform state

    glEnable(GL_DEPTH_TEST);           // restore depth testing
    glDepthMask(GL_TRUE);              // make Z-buffer writeable
    glEnable(GL_LIGHTING);             // restore lighting
    ogl_lightingenabled=1;             // update state var
#if defined(VMDUSEOPENGLSHADER)
    if (mainshader && ogl_useglslshader) {
      mainshader->UseShader(1);        // re-enable glsl mainshader
    }
#endif
  }
}


//**********************  the rendering routine  ***********************
//* This scans the given command list until the end, doing the commands
//* in the order they appear
//**********************************************************************
void OpenGLRenderer::render(const VMDDisplayList *cmdList) {
  char *cmdptr = NULL;  // ptr to current display command data
  int tok;              // what display command was encountered
  _needRedraw = 0;      // reset the flag now that we are drawing

  // early exit if any of these conditions are true. 
  if (!cmdList) 
    return;

  if (ogl_useblendedtrans) {
    if (ogl_transpass) {
      // skip rendering mostly Opaque objects on transparent pass
      if (cmdList->opacity > 0.50) 
        return;
    } else {
      // skip rendering mostly transparent objects on opaque pass
      if (cmdList->opacity <= 0.50)
        return;
    }
  } else {
    if (cmdList->opacity < 0.0625)
      return;
  }

  // if we're rendering for Acrobat3D capture, emit materials and other
  // state changes at every opportunity, caching next to nothing by 
  // invalidating materials on every object we draw
  if (ogl_acrobat3dcapture) {
    oglmaterialindex = -1;
    oglambient   = -1;;
    ogldiffuse   = -1;;
    oglspecular  = -1;;
    oglshininess = -1;;
  } 

  //
  // set the material - only changing those items that have been updated.
  //
  if (oglmaterialindex != cmdList->materialtag) {
    float matbuf[4];
    matbuf[3] = 1.0f; 
    int recalcambientlights = 0;
    int recalcdiffuselights = 0;

    oglmaterialindex = cmdList->materialtag;
    if (oglopacity != cmdList->opacity) {
      oglopacity = cmdList->opacity; // update for next time through

      if (ogl_useblendedtrans) {
        glDisable(GL_POLYGON_STIPPLE);   
        if (oglopacity > 0.999) {
          // disable alpha-blended transparency
          glDisable(GL_BLEND);
#if defined(GL_SUN_global_alpha)
          if (ext->hasglobalalphasun) {
            GLGLOBALALPHAFACTORFSUN(1.0);
            glDisable(GL_GLOBAL_ALPHA_SUN);
          }
#endif
        } else {
          glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
          glEnable(GL_BLEND);
#if defined(GL_SUN_global_alpha)
          if (ext->hasglobalalphasun) {
            glEnable(GL_GLOBAL_ALPHA_SUN);
            GLGLOBALALPHAFACTORFSUN(oglopacity);
          }
#endif
        }
      } else {
        // disable alpha-blended transparency
        glDisable(GL_BLEND);
#if defined(GL_SUN_global_alpha)
        if (ext->hasglobalalphasun) {
          GLGLOBALALPHAFACTORFSUN(1.0);
          glDisable(GL_GLOBAL_ALPHA_SUN);
        }
#endif

        // use stipple-based transparency
        if (oglopacity > 0.9375) {  
          glDisable(GL_POLYGON_STIPPLE);   
        } else {
          // here's our transparency: opacity < 0.9375  -> transparent
          if (oglopacity > 0.875) 
            glPolygonStipple(ninesixteentone);               
          else if (oglopacity > 0.75) 
            glPolygonStipple(seveneighthtone);               
          else if (oglopacity > 0.5) 
            glPolygonStipple(threequartertone);               
          else if (oglopacity > 0.25)
            glPolygonStipple(halftone);               
          else if (oglopacity > 0.125)
            glPolygonStipple(quartertone);               
          else if (oglopacity > 0.0625)
            glPolygonStipple(eighthtone);               
          else 
            return; // skip rendering the geometry if entirely transparent
    
          glEnable(GL_POLYGON_STIPPLE);                
        }
      }
    }

    if (oglambient != cmdList->ambient) { 
      oglambient = cmdList->ambient;
      recalcambientlights = 1;  // force recalculation of ambient lighting
      matbuf[0] = matbuf[1] = matbuf[2] = oglambient; 
      glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, matbuf);
    }

    if (ogldiffuse != cmdList->diffuse) { 
      ogldiffuse = cmdList->diffuse;
      recalcdiffuselights = 1;  // force recalculation of diffuse lighting
      matbuf[0] = matbuf[1] = matbuf[2] = ogldiffuse; 
      glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, matbuf);
    }

    if (oglspecular != cmdList->specular) { 
      oglspecular = cmdList->specular;
      matbuf[0] = matbuf[1] = matbuf[2] = oglspecular; 
      glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, matbuf);
    }

    if (oglshininess != cmdList->shininess) {
      oglshininess = cmdList->shininess;
  
      // clamp shininess parameter to what OpenGL can deal with
      glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, 
                  (GLfloat) (oglshininess < 128.0f) ? oglshininess : 128.0f);
    }
 
    // change lighting to match new diffuse/ambient factors
    if (recalcambientlights) { 
      for (int z=0; z<DISP_LIGHTS; z++) {
        GLfloat d[4];
        d[0] = ogl_lightcolor[z][0] * oglambient;
        d[1] = ogl_lightcolor[z][1] * oglambient;
        d[2] = ogl_lightcolor[z][2] * oglambient;
        d[3] = 1.0;
        glLightfv((GLenum)(GL_LIGHT0 + z), GL_AMBIENT, d);
      }
    }
 
    if (recalcdiffuselights) { 
      for (int z=0; z<DISP_LIGHTS; z++) {
        GLfloat d[4];
        d[0] = ogl_lightcolor[z][0] * ogldiffuse;
        d[1] = ogl_lightcolor[z][1] * ogldiffuse;
        d[2] = ogl_lightcolor[z][2] * ogldiffuse;
        d[3] = 1.0;
        glLightfv((GLenum)(GL_LIGHT0 + z), GL_DIFFUSE, d);
      }
    }
  }
  // 
  // end material processing code for fixed-function OpenGL pipeline
  //

  // XXX shouldn't be testing this every rep, but for now this works ok 
  ogl_fogmode = 0; // fogmode for shaders

  if (cueingEnabled) {
    switch (cueMode) {
      case CUE_LINEAR: 
        glFogi(GL_FOG_MODE, GL_LINEAR);
        ogl_fogmode = 1;
        break;
  
      case CUE_EXP:
        glFogi(GL_FOG_MODE, GL_EXP);
        ogl_fogmode = 2;
        break;
  
      case CUE_EXP2:
        glFogi(GL_FOG_MODE, GL_EXP2);
        ogl_fogmode = 3;
        break;

      case NUM_CUE_MODES:
        // this should never happen
        break;
    }

    glFogf(GL_FOG_DENSITY, (GLfloat) get_cue_density());
    glFogf(GL_FOG_START,   (GLfloat) get_cue_start());
    glFogf(GL_FOG_END,     (GLfloat) get_cue_end());
  }

#if defined(VMDUSEOPENGLSHADER)
  // setup programmable shader for this object
  if (mainshader) {
    if (ogl_useglslshader) {
      mainshader->UseShader(1); // if glsl is available and enabled, use it
  
      if ((ogl_glslmaterialindex != cmdList->materialtag) || ogl_glsltoggle) {
        ogl_glslmaterialindex = cmdList->materialtag;
        ogl_glsltoggle = 0;
        update_shader_uniforms(mainshader, 1);
      }
    } else {
      mainshader->UseShader(0); // if glsl is available but disabled, turn it off
    }
  }
#endif

  // save transformation matrix
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  multmatrix(cmdList->mat);

  // set up text matrices
  GLfloat textsize = 1;
  Matrix4 textMat(ogl_textMat);
  textMat.multmatrix(cmdList->mat);
  
  // XXX Display list caching begins here
  GLuint ogl_cachedid = 0;    // reset OpenGL display list ID for cached list
  int ogl_cachecreated = 0;  // reset display list creation flag
  int ogl_cacheskip;

  // Disable display list caching if GLSL is enabled or we encounter
  // a non-cacheable representation (such as an animating structure).
  ogl_cacheskip = (cmdList->cacheskip || ogl_useglslshader);

  // enable/disable clipping planes
  for (int cp=0; cp<VMD_MAX_CLIP_PLANE; cp++) {
    // don't cache 'on' state since the parameters will likely differ,
    // just setup the clip plane from the new state
    if (cmdList->clipplanes[cp].mode) {
      GLdouble cpeq[4];
      cpeq[0] = cmdList->clipplanes[cp].normal[0];
      cpeq[1] = cmdList->clipplanes[cp].normal[1];
      cpeq[2] = cmdList->clipplanes[cp].normal[2];
  
      // Convert specification to OpenGL plane equation
      cpeq[3] = 
      -(cmdList->clipplanes[cp].normal[0] * cmdList->clipplanes[cp].center[0] +
        cmdList->clipplanes[cp].normal[1] * cmdList->clipplanes[cp].center[1] +
        cmdList->clipplanes[cp].normal[2] * cmdList->clipplanes[cp].center[2]);
      glClipPlane((GLenum) (GL_CLIP_PLANE0 + cp), cpeq);
      glEnable((GLenum) (GL_CLIP_PLANE0 + cp)); 

      // XXX if the clipping plane mode is set for rendering
      // capped clipped solids, we will have to perform several
      // rendering passes using the stencil buffer and Z-buffer 
      // in order to get the desired results.
      // http://www.nigels.com/research/wscg2002.pdf 
      // http://citeseer.ist.psu.edu/stewart02lineartime.html
      // http://citeseer.ist.psu.edu/stewart98improved.html
      // http://www.sgi.com/software/opengl/advanced97/notes/node10.html
      // http://www.opengl.org/resources/tutorials/sig99/advanced99/notes/node21.html
      // http://www.ati.com/developer/sdk/rage128sdk/OpenGL/Samples/Rage128StencilCap.html
      // The most common algorithm goes something like what is described here:
      //   0) clear stencil/color/depth buffers
      //   1) disable color buffer writes
      //   2) render clipping plane polygon writing to depth buffer
      //   3) disable depth buffer writes
      //   4) set stencil op to increment when depth test passes
      //   5) draw molecule with glCullFace(GL_BACK)
      //   6) set stencil op to decrement when depth test passes
      //   7) draw molecule with glCullFace(GL_FRONT)
      //   8) clear depth buffer
      //   9) enable color buffer writes 
      //  10) set stencil function to GL_EQUAL of 1
      //  11) draw clipping plane polygon with appropriate materials
      //  12) disable stencil buffer
      //  13) enable OpenGL clipping plane
      //  14) draw molecule
    } else {
      // if its already off, no need to disable it again.
      if (ogl_clipmode[cp] != cmdList->clipplanes[cp].mode) {
        glDisable((GLenum) (GL_CLIP_PLANE0 + cp)); 
      }
    }

    // update clip mode cache
    ogl_clipmode[cp] = cmdList->clipplanes[cp].mode;
  }

  // Periodic boundary condition rendering begins here
  // Make a list of all the transformations we want to perform.
  ResizeArray<Matrix4> pbcImages;
  find_pbc_images(cmdList, pbcImages);

  // initialize text offset variables.  These values should never be set in one
  // display list and applied in another, so we make them local variables here
  // rather than OpenGLRenderer state variables.
  float textoffset_x = 0, textoffset_y = 0;

int nimages = pbcImages.num();
for (int pbcimage = 0; pbcimage < nimages; pbcimage++) {
  glPushMatrix();
  multmatrix(pbcImages[pbcimage]);

  if (ogl_cachedebug) {
    msgInfo << "Rendering scene: cache enable=" << ogl_cacheenabled 
            << ", created=" << ogl_cachecreated << ", serial=" << (int)cmdList->serial
            << ", id=" << (int)ogl_cachedid << ", skip=" << ogl_cacheskip << sendmsg;
  }

  // find previously cached display list for this object
  if (ogl_cacheenabled && !ogl_cacheskip) {
    ogl_cachedid = displaylistcache.markUsed(cmdList->serial);

    // add to the cache and regenerate if we didn't find it
    if (ogl_cachedid == GLCACHE_FAIL) {
      ogl_cachedid = glGenLists(1);      
      displaylistcache.encache(cmdList->serial, ogl_cachedid);

      // create the display list, and execute it.
      glNewList(ogl_cachedid, GL_COMPILE_AND_EXECUTE);
      ogl_cachecreated = 1; // a new display list was created 
    } 
  }

  // XXX Draw OpenGL geometry only when caching is disabled or when
  //     we have new geometry to cache
  if ((!ogl_cacheenabled) || ogl_cacheskip || (ogl_cacheenabled && ogl_cachecreated)) {

  // scan through the list, getting each command and executing it, until
  // the end of commands token is found
  VMDDisplayList::VMDLinkIter cmditer;
  cmdList->first(&cmditer);
  while((tok = cmdList->next(&cmditer, cmdptr)) != DLASTCOMMAND) {
    OGLERR // enable OpenGL debugging code

    switch (tok) {
      case DDATABLOCK:
        #ifdef VMDCAVE
        dataBlock = (float *)cmdptr;
        #else
        dataBlock = ((DispCmdDataBlock *)cmdptr)->data;
        #endif
        break;

      case DPOINT:
        // plot a point at the given position
        glBegin(GL_POINTS);
          glVertex3fv(((DispCmdPoint *)cmdptr)->pos);
        glEnd();
        break;

      case DPOINTARRAY: 
        {
          DispCmdPointArray *pa = (DispCmdPointArray *)cmdptr;
          float *centers;
          float *colors;
          pa->getpointers(centers, colors);
#if defined(GL_VERSION_1_1)
        if (!(simplegraphics || ogl_acrobat3dcapture)) {
          // Vertex array implementation 
          glDisable(GL_LIGHTING); 
          ogl_lightingenabled=0;
          glEnableClientState(GL_VERTEX_ARRAY);
          glEnableClientState(GL_COLOR_ARRAY);
          glDisableClientState(GL_NORMAL_ARRAY);
          glVertexPointer(3, GL_FLOAT, 12, (void *) centers);
          glColorPointer(3, GL_FLOAT, 12, (void *)  colors);

#if defined(GL_EXT_compiled_vertex_array) 
          if (ext->hascompiledvertexarrayext) {
            GLLOCKARRAYSEXT(0, pa->numpoints);
          }
#endif

          // set point size, enable blending and point antialiasing
          glPointSize(pa->size); 
          glEnable(GL_POINT_SMOOTH);
          glEnable(GL_BLEND);
          glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

#if defined(GL_ARB_point_parameters) 
          int dodepthscaling = 0;
  
          // enable distance based point attenuation
          if (ext->hasglpointparametersext  && (projection() == PERSPECTIVE)) {
            dodepthscaling = 1;
  
            GLfloat abc[4] = {0.0, 0.0, 1.0};
            GLPOINTPARAMETERFVARB(GL_POINT_DISTANCE_ATTENUATION_ARB, abc);
          }
#endif

          glDrawArrays(GL_POINTS, 0, pa->numpoints);

#if defined(GL_ARB_point_parameters) 
          // disable distance based point attenuation
          if (dodepthscaling) {
            GLfloat abc[4] = {1.0, 0.0, 0.0};
            GLPOINTPARAMETERFVARB(GL_POINT_DISTANCE_ATTENUATION_ARB, abc);
          }
#endif

          // disable blending and point antialiasing
          glDisable(GL_BLEND);
          glDisable(GL_POINT_SMOOTH);

#if defined(GL_EXT_compiled_vertex_array) 
          if (ext->hascompiledvertexarrayext) {
            GLUNLOCKARRAYSEXT();
          }
#endif

          glEnableClientState(GL_NORMAL_ARRAY);
          glPointSize(1.0); // reset point size to default
        } else {
#endif
          // Immediate mode implementation
          int i, ind;
          glBegin(GL_POINTS);
          ind = 0;
          for (i=0; i<pa->numpoints; i++) {
            glColor3fv(&colors[ind]);
            glVertex3fv(&centers[ind]); 
            ind += 3;
          }
          glEnd();
#if defined(GL_VERSION_1_1)
        }
#endif
        }
        break;

      case DLITPOINTARRAY: 
        {
        DispCmdLitPointArray *pa = (DispCmdLitPointArray *)cmdptr;
        float *centers;
        float *normals;
        float *colors;
        pa->getpointers(centers, normals, colors);
#if defined(GL_VERSION_1_1)
        if (!(simplegraphics || ogl_acrobat3dcapture)) {
          // Vertex array implementation 
          glEnableClientState(GL_VERTEX_ARRAY);
          glEnableClientState(GL_COLOR_ARRAY);
          glEnableClientState(GL_NORMAL_ARRAY);
          glVertexPointer(3, GL_FLOAT, 12, (void *) centers);
          glNormalPointer(GL_FLOAT, 12, (void *) normals);
          glColorPointer(3, GL_FLOAT, 12, (void *)  colors);

#if defined(GL_EXT_compiled_vertex_array) 
          if (ext->hascompiledvertexarrayext) {
            GLLOCKARRAYSEXT(0, pa->numpoints);
          }
#endif

          // set point size, enable blending and point antialiasing
          glPointSize(pa->size); 
          glEnable(GL_POINT_SMOOTH);

#if defined(GL_ARB_point_parameters) 
          int dodepthscaling = 0;
          // enable distance based point attenuation
          if (ext->hasglpointparametersext  && (projection() == PERSPECTIVE)) {
            dodepthscaling = 1;
            GLfloat abc[4] = {0.0, 0.0, 1.0};
            GLPOINTPARAMETERFVARB(GL_POINT_DISTANCE_ATTENUATION_ARB, abc);
          }
#endif

          glDrawArrays(GL_POINTS, 0, pa->numpoints);

#if defined(GL_ARB_point_parameters) 
          // disable distance based point attenuation
          if (dodepthscaling) {
            GLfloat abc[4] = {1.0, 0.0, 0.0};
            GLPOINTPARAMETERFVARB(GL_POINT_DISTANCE_ATTENUATION_ARB, abc);
          }
#endif

          // disable blending and point antialiasing
          glDisable(GL_BLEND);
          glDisable(GL_POINT_SMOOTH);

#if defined(GL_EXT_compiled_vertex_array) 
          if (ext->hascompiledvertexarrayext) {
            GLUNLOCKARRAYSEXT();
          }
#endif

          glPointSize(1.0); // reset point size to default
        } else {
#endif
          // Immediate mode implementation 
          int i, ind;
          glBegin(GL_POINTS);
          ind = 0;
          for (i=0; i<pa->numpoints; i++) {
            glColor3fv(&colors[ind]);
            glNormal3fv(&normals[ind]); 
            glVertex3fv(&centers[ind]); 
            ind += 3;
          }
          glEnd();
#if defined(GL_VERSION_1_1)
        }
#endif
        }
        break;

      case DLINE:
        // plot a line
        glBegin(GL_LINES);
          glVertex3fv(((DispCmdLine *)cmdptr)->pos1);
          glVertex3fv(((DispCmdLine *)cmdptr)->pos2);
        glEnd();
        break;

      case DLINEARRAY:
        {
          float *v = (float *)(cmdptr);
          int nlines = (int)v[0];
          v++; // move pointer forward before traversing vertex data

#if defined(GL_VERSION_1_1)
          if (!(simplegraphics || ogl_acrobat3dcapture)) {
            // Vertex array implementation
            glInterleavedArrays(GL_V3F, 0, v);

#if defined(GL_EXT_compiled_vertex_array) 
            if (ext->hascompiledvertexarrayext) {
              GLLOCKARRAYSEXT(0, 2*nlines);
            }
#endif

            glDrawArrays(GL_LINES, 0, 2*nlines); 

#if defined(GL_EXT_compiled_vertex_array) 
            if (ext->hascompiledvertexarrayext) {
              GLUNLOCKARRAYSEXT();
            }
#endif
          } else {
#endif
            // Immediate mode implementation
            glBegin(GL_LINES);
            for (int i=0; i<nlines; i++) {
              glVertex3fv(v);
              glVertex3fv(v+3);
              v += 6;
            }
            glEnd();
#if defined(GL_VERSION_1_1)
          }
#endif
        }
        break;    

      case DPOLYLINEARRAY:
        {
          float *v = (float *)(cmdptr);
          int nverts = (int)v[0];
          v++; // move pointer forward before traversing vertex data

#if defined(GL_VERSION_1_1)
          if (!(simplegraphics || ogl_acrobat3dcapture)) {
            // Vertex array implementation
            glInterleavedArrays(GL_V3F, 0, v);

#if defined(GL_EXT_compiled_vertex_array) 
            if (ext->hascompiledvertexarrayext) {
              GLLOCKARRAYSEXT(0, nverts);
            }
#endif

            glDrawArrays(GL_LINE_STRIP, 0, nverts); 

#if defined(GL_EXT_compiled_vertex_array) 
            if (ext->hascompiledvertexarrayext) {
              GLUNLOCKARRAYSEXT();
            }
#endif
          } else {
#endif
            // Immediate mode implementation
            glBegin(GL_LINE_STRIP);
            for (int i=0; i<nverts; i++) {
              glVertex3fv(v);
              v += 3;
            }
            glEnd();
#if defined(GL_VERSION_1_1)
          }
#endif
        }
        break;    

      case DSPHERE: 
        {
        float *p = (float *)cmdptr;
        glPushMatrix();
        glTranslatef(p[0], p[1], p[2]); 
        glScalef(p[3], p[3], p[3]);
        glCallList(SphereList);
        glPopMatrix();
        }
        break;

      case DSPHERE_I:
        {
        DispCmdSphereIndex *cmd = (DispCmdSphereIndex *)cmdptr;
        GLfloat * transv = ((GLfloat *) dataBlock) + cmd->pos;
        glPushMatrix();
        glTranslatef(transv[0], transv[1], transv[2]);  
        glScalef(cmd->rad, cmd->rad, cmd->rad);
        glCallList(SphereList);
        glPopMatrix();
        }
        break;

      case DSPHEREARRAY: 
        {
        DispCmdSphereArray *sa = (DispCmdSphereArray *)cmdptr;
        int i, ind;
        float * centers;
        float * radii;
        float * colors;
        sa->getpointers(centers, radii, colors);

#if defined(VMDUSEGLSLSPHERES) 
        // Render the sphere using programmable shading via ray-casting
        if (sphereshader && ogl_useglslshader) {
          // coordinates of unit bounding box
          GLfloat v0[] = {-1.0, -1.0, -1.0}; 
          GLfloat v1[] = { 1.0, -1.0, -1.0}; 
          GLfloat v2[] = {-1.0,  1.0, -1.0}; 
          GLfloat v3[] = { 1.0,  1.0, -1.0}; 
          GLfloat v4[] = {-1.0, -1.0,  1.0}; 
          GLfloat v5[] = { 1.0, -1.0,  1.0}; 
          GLfloat v6[] = {-1.0,  1.0,  1.0}; 
          GLfloat v7[] = { 1.0,  1.0,  1.0}; 
      
          mainshader->UseShader(0);   // switch to sphere shader
          sphereshader->UseShader(1); // switch to sphere shader
          update_shader_uniforms(sphereshader, 0); // force update of shader

          // Update projection parameters for OpenGL shader
          GLfloat projparms[4];
          projparms[0] = nearClip;
          projparms[1] = farClip; 
          projparms[2] = 0.5f * (farClip + nearClip);
          projparms[3] = 1.0f / (farClip - nearClip);
          GLint projloc = GLGETUNIFORMLOCATIONARB(sphereshader->ProgramObject, "vmdprojparms");
          GLUNIFORM4FVARB(projloc, 1, projparms);
          OGLERR;

          ind = 0;
          for (i=0; i<sa->numspheres; i++) {
            glPushMatrix();
            glTranslatef(centers[ind], centers[ind + 1], centers[ind + 2]); 
            glScalef(radii[i], radii[i], radii[i]);
            glColor3fv(&colors[ind]);

            // Draw the bounding box containing the sphere, gauranteeing 
            // that it will be correctly rendered regardless of the 
            // perspective projection used, viewing direction, etc.
            // If enough is known about the projection being used, this
            // could be done with simple billboard polygons, or perhaps even
            // a large OpenGL point primitive instead of a whole cube
            glBegin(GL_QUADS);
              glVertex3fv((GLfloat *) v0); /* -Z face */
              glVertex3fv((GLfloat *) v1);
              glVertex3fv((GLfloat *) v3);
              glVertex3fv((GLfloat *) v2);

              glVertex3fv((GLfloat *) v4); /* +Z face */
              glVertex3fv((GLfloat *) v5);
              glVertex3fv((GLfloat *) v7);
              glVertex3fv((GLfloat *) v6);

              glVertex3fv((GLfloat *) v0); /* -Y face */
              glVertex3fv((GLfloat *) v1);
              glVertex3fv((GLfloat *) v5);
              glVertex3fv((GLfloat *) v4);

              glVertex3fv((GLfloat *) v2); /* +Y face */
              glVertex3fv((GLfloat *) v3);
              glVertex3fv((GLfloat *) v7);
              glVertex3fv((GLfloat *) v6);

              glVertex3fv((GLfloat *) v0); /* -X face */
              glVertex3fv((GLfloat *) v2);
              glVertex3fv((GLfloat *) v6);
              glVertex3fv((GLfloat *) v4);

              glVertex3fv((GLfloat *) v1); /* +X face */
              glVertex3fv((GLfloat *) v3);
              glVertex3fv((GLfloat *) v7);
              glVertex3fv((GLfloat *) v5);
            glEnd();
            glPopMatrix();
            ind += 3; // next sphere
          }

          sphereshader->UseShader(0); // switch back to the main shader
          mainshader->UseShader(1);   // switch back to the main shader
          OGLERR;
        } else {
#endif
          // OpenGL display listed sphere rendering implementation
          set_sphere_res(sa->sphereres); // set the current sphere resolution

          // use single-sided lighting when drawing spheres for 
          // peak rendering speed.
          glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);
          ind = 0;
          for (i=0; i<sa->numspheres; i++) {
            glPushMatrix();
            glTranslatef(centers[ind], centers[ind + 1], centers[ind + 2]); 
            glScalef(radii[i], radii[i], radii[i]);
            glColor3fv(&colors[ind]);
            glCallList(SphereList);
            glPopMatrix();
            ind += 3; // next sphere
          }
          glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);
#if defined(VMDUSEGLSLSPHERES)
        }
#endif

        }
        break;

      case DTRIANGLE: 
        {
        DispCmdTriangle *cmd = (DispCmdTriangle *)cmdptr;
        glBegin(GL_TRIANGLES);
          glNormal3fv(cmd->norm1);
          glVertex3fv(cmd->pos1);
          glNormal3fv(cmd->norm2);
          glVertex3fv(cmd->pos2);
          glNormal3fv(cmd->norm3);
          glVertex3fv(cmd->pos3);
        glEnd();
        }
        break;

      case DSQUARE:
        // draw a square, given the four points
        {
        DispCmdSquare *cmd = (DispCmdSquare *)cmdptr;
        glBegin(GL_QUADS);
          glNormal3fv((GLfloat *) cmd->norml);
          glVertex3fv((GLfloat *) cmd->pos1);
          glVertex3fv((GLfloat *) cmd->pos2);
          glVertex3fv((GLfloat *) cmd->pos3);
          glVertex3fv((GLfloat *) cmd->pos4);
        glEnd();
        }
        break;

#if 0
      case DSTRIPETEX:
        if (ext->hastex3d) {
#if defined(GL_VERSION_1_2)
#define STRIPEWIDTH 32
          GLubyte stripeImage[4 * STRIPEWIDTH];
          GLuint texName = 0;
          // glGenTextures(1, &texName);
          int i;
          for (i=0; i<STRIPEWIDTH; i++) {
            stripeImage[4*i    ] = (GLubyte) ((i>4) ? 255 : 0); // R
            stripeImage[4*i + 1] = (GLubyte) ((i>4) ? 255 : 0); // G
            stripeImage[4*i + 2] = (GLubyte) ((i>4) ? 255 : 0); // B
            stripeImage[4*i + 3] = (GLubyte) 255;               // W
          }

          glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
          glBindTexture(GL_TEXTURE_1D, texName);
          glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_WRAP_S, GL_REPEAT);
          glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_WRAP_T, GL_REPEAT);
          glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_WRAP_R, GL_REPEAT);
          glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
          glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
          glTexImage1D(GL_TEXTURE_1D, 0, GL_RGBA, STRIPEWIDTH, 
                          0, GL_RGBA, GL_UNSIGNED_BYTE, stripeImage);

          // XXX should use GL_MODULATE, but requires all polygons to be
          //     drawn "white", in order for shading to make it through the
          //     texturing process.  GL_REPLACE works well for situations
          //     where we want coloring to come entirely from texture.
          glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
          GLfloat xplaneeq[4] = {0.5, 0.0, 0.0, 0.0};
          glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, GL_EYE_LINEAR);
          glTexGenfv(GL_S, GL_EYE_PLANE, xplaneeq);
          glEnable(GL_TEXTURE_GEN_S);
          glEnable(GL_TEXTURE_1D);
#endif
        }
        break;

      case DSTRIPETEXOFF:
        if (ext->hastex3d) {
#if defined(GL_VERSION_1_2)
          glDisable(GL_TEXTURE_GEN_S);
          glDisable(GL_TEXTURE_1D);
#endif
        }
        break;
#endif

      case DVOLUMETEXTURE:
        if (ext->hastex3d)
#if defined(GL_VERSION_1_2)
        {
  
#if defined(GL_GENERATE_MIPMAP_HINT)
          // set MIP map generation hint for high quality
          glHint(GL_GENERATE_MIPMAP_HINT, GL_NICEST);
#endif
          glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);

          DispCmdVolumeTexture *cmd = (DispCmdVolumeTexture *)cmdptr;
          require_volume_texture(cmd->ID, 
              cmd->xsize, cmd->ysize, cmd->zsize, 
              cmd->texmap);

          GLfloat xplaneeq[4]; 
          GLfloat yplaneeq[4]; 
          GLfloat zplaneeq[4]; 
          int i;

          glEnable(GL_TEXTURE_3D);
          glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

#if defined(VMDUSEOPENGLSHADER)
          // Update active GLSL texturing mode
          if (mainshader && ogl_useglslshader) {
            if (!ogl_lightingenabled)
              mainshader->UseShader(1); // enable shader so state updates 
            ogl_glsltexturemode = 1;
            GLint vmdtexturemode = 1;  // enable 3-D texturing->MODULATE
            GLint loc = GLGETUNIFORMLOCATIONARB(mainshader->ProgramObject, "vmdtexturemode");
            GLUNIFORM1IARB(loc, vmdtexturemode);
      
            // Set active texture map index
            loc = GLGETUNIFORMLOCATIONARB(mainshader->ProgramObject, "vmdtex0");
            GLUNIFORM1IARB(loc, 0); // using texture unit 0
            if (!ogl_lightingenabled)
              mainshader->UseShader(0); // disable shader after state updates
          }
#endif

          // automatically generate texture coordinates by translating from
          // model coordinate space to volume coordinates.  These aren't
          // going to be used by volume slices, but the performance hit
          // is expected to be insignificant.
          for (i=0; i<3; i++) {
            xplaneeq[i] = cmd->v1[i];
            yplaneeq[i] = cmd->v2[i];
            zplaneeq[i] = cmd->v3[i];
          }
          xplaneeq[3] = cmd->v0[0];
          yplaneeq[3] = cmd->v0[1];
          zplaneeq[3] = cmd->v0[2];

          glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, GL_EYE_LINEAR); 
          glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, GL_EYE_LINEAR); 
          glTexGeni(GL_R, GL_TEXTURE_GEN_MODE, GL_EYE_LINEAR); 
          glTexGenfv(GL_S, GL_EYE_PLANE, xplaneeq);
          glTexGenfv(GL_T, GL_EYE_PLANE, yplaneeq);
          glTexGenfv(GL_R, GL_EYE_PLANE, zplaneeq);
          glEnable(GL_TEXTURE_GEN_S);
          glEnable(GL_TEXTURE_GEN_T);
          glEnable(GL_TEXTURE_GEN_R);
#endif
        }
        break;

      case DVOLTEXON:
        if (ext->hastex3d) {
#if defined(GL_VERSION_1_2)
          glEnable(GL_TEXTURE_3D);     // enable volume texturing
#if defined(VMDUSEOPENGLSHADER)
          // Update active GLSL texturing mode
          if (mainshader && ogl_useglslshader) {
            if (!ogl_lightingenabled)
              mainshader->UseShader(1); // enable shader so state updates 
            ogl_glsltexturemode = 1;
            GLint vmdtexturemode = 1;  // enable 3-D texturing->MODULATE
            GLint loc = GLGETUNIFORMLOCATIONARB(mainshader->ProgramObject, "vmdtexturemode");
            GLUNIFORM1IARB(loc, vmdtexturemode);
            if (!ogl_lightingenabled)
              mainshader->UseShader(0); // disable shader after state updates
          }
#endif
          glEnable(GL_TEXTURE_GEN_S);  // enable automatic texture 
          glEnable(GL_TEXTURE_GEN_T);  //   coordinate generation
          glEnable(GL_TEXTURE_GEN_R);
#endif
        }
        break;

      case DVOLTEXOFF:
        if (ext->hastex3d) {
#if defined(GL_VERSION_1_2)
          glDisable(GL_TEXTURE_3D);     // disable volume texturing
#if defined(VMDUSEOPENGLSHADER)
          // Update active GLSL texturing mode
          if (mainshader && ogl_useglslshader) {
            if (!ogl_lightingenabled)
              mainshader->UseShader(1); // enable shader so state updates 
            ogl_glsltexturemode = 0;
            GLint vmdtexturemode = 0;  // disable 3-D texturing
            GLint loc = GLGETUNIFORMLOCATIONARB(mainshader->ProgramObject, "vmdtexturemode");
            GLUNIFORM1IARB(loc, vmdtexturemode);
            if (!ogl_lightingenabled)
              mainshader->UseShader(0); // disable shader after state updates
          }
#endif

          glDisable(GL_TEXTURE_GEN_S);  // disable automatic texture 
          glDisable(GL_TEXTURE_GEN_T);  //   coordinate generation
          glDisable(GL_TEXTURE_GEN_R);
#endif
        }
        break;


      case DVOLSLICE:
        if (ext->hastex3d) {
          DispCmdVolSlice *cmd = (DispCmdVolSlice *)cmdptr;
#if defined(GL_VERSION_1_2)

          // DVOLUMETEXTURE does most of the work for us, but we override
          // a few of the texenv settings
          glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

          // enable or disable texture interpolation and filtering
          switch (cmd->texmode) {
            case 2:
              glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
              glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
              glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
              break;

            case 1:
              glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_DONT_CARE);
              glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
              glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
              break;
     
            case 0: 
            default:
              glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_FASTEST);
              glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
              glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
              break;
          }
      
          // use the texture edge colors rather border color when wrapping
          glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_WRAP_R, GL_CLAMP_TO_EDGE);
          glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
          glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

#if defined(VMDUSEOPENGLSHADER)
          // Update active GLSL texturing mode
          if (mainshader && ogl_useglslshader) {
            ogl_glsltexturemode = 2;
            GLint vmdtexturemode = 2;  // enable 3-D texturing->REPLACE
            GLint loc = GLGETUNIFORMLOCATIONARB(mainshader->ProgramObject, "vmdtexturemode");
            GLUNIFORM1IARB(loc, vmdtexturemode);
          }
#endif
          glBegin(GL_QUADS);        
          for (int i=0; i<4; i++) {
            glNormal3fv(cmd->normal);
            glVertex3fv(cmd->v + 3*i);
          }
          glEnd();        
#endif // GL_VERSION_1_2
        } 
        break;

      case DTRIMESH: 
        {
        // draw a triangle mesh
        DispCmdTriMesh *cmd = (DispCmdTriMesh *) cmdptr;
        int ind = cmd->numfacets * 3;
        float *cnv;
        int *f;
        cmd->getpointers(cnv, f);

#if defined(GL_VERSION_1_1)
        // Vertex array implementation
        // XXX workaround XVR-4000 driver bug if SUNZULU renderer is detected
        if (!(simplegraphics || ogl_acrobat3dcapture) &&
            ext->oglrenderer != OpenGLExtensions::SUNZULU) {
          // If OpenGL 1.1, then use vertex arrays 
          glInterleavedArrays(GL_C4F_N3F_V3F, 0, cnv);

#if defined(GL_EXT_compiled_vertex_array) 
          if (ext->hascompiledvertexarrayext) {
            GLLOCKARRAYSEXT(0, cmd->numverts);
          }
#endif

          glDrawElements(GL_TRIANGLES, ind, GL_UNSIGNED_INT, f);

#if defined(GL_EXT_compiled_vertex_array) 
          if (ext->hascompiledvertexarrayext) {
            GLUNLOCKARRAYSEXT();
          }
#endif
        } else {
#endif

          // simple graphics mode, but not Acrobat3D capture mode
          if (!ogl_acrobat3dcapture) {
            int i, ind2;
            glBegin(GL_TRIANGLES);
            for (i=0; i<ind; i++) {
              ind2 = f[i] * 10;
               glColor3fv(cnv + ind2    );
              glNormal3fv(cnv + ind2 + 4);
              glVertex3fv(cnv + ind2 + 7);
            }
            glEnd();
          } else { 
            // Version 7.0.9 of Acrobat3D can't capture multicolor
            // triangles, so we revert to averaged-single-color
            // triangles until they fix this capture bug.
            int i;
            for (i=0; i<cmd->numfacets; i++) {
              int ind = i * 3;
              float tmp[3], tmp2[3];

              int v0 = f[ind    ] * 10;
              int v1 = f[ind + 1] * 10;
              int v2 = f[ind + 2] * 10;

              vec_add(tmp, cnv + v0, cnv + v1);
              vec_add(tmp2, tmp, cnv + v2);
              vec_scale(tmp, 0.3333333, tmp2);
              glBegin(GL_TRIANGLES);
              glColor3fv(tmp);
              glNormal3fv(cnv + v0 + 4);
              glVertex3fv(cnv + v0 + 7);
              glNormal3fv(cnv + v1 + 4);
              glVertex3fv(cnv + v1 + 7);
              glNormal3fv(cnv + v2 + 4);
              glVertex3fv(cnv + v2 + 7);
              glEnd();
            }
          }

#if defined(GL_VERSION_1_1)
        }
#endif
        }
        break;
        
      case DTRISTRIP: 
        {
        // draw triangle strips
        DispCmdTriStrips *cmd = (DispCmdTriStrips *) cmdptr;
        int numstrips = cmd->numstrips;
        int strip;

        float *cnv;
        int *f;
        int *vertsperstrip;

        cmd->getpointers(cnv, f, vertsperstrip);

        // use single-sided lighting when drawing possible, for
        // peak rendering speed.
        if (!cmd->doublesided) {
          glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);
        }

#if defined(GL_VERSION_1_1)
        if (!(simplegraphics || ogl_acrobat3dcapture)) {
          // If OpenGL 1.1, then use vertex arrays
          glInterleavedArrays(GL_C4F_N3F_V3F, 0, cnv);

#if defined(GL_EXT_compiled_vertex_array) 
          if (ext->hascompiledvertexarrayext) {
            GLLOCKARRAYSEXT(0, cmd->numverts);
          }
#endif

#if defined(GL_EXT_multi_draw_arrays)
          // Try the Sun/ARB MultiDrawElements() extensions first.
          if (ext->hasmultidrawext) {
            int **indices = new int *[cmd->numstrips];

            // build array of facet list pointers to allow the renderer to
            // send everything in a single command/DMA when possible
            int qv=0;
            for (int i=0; i<numstrips; i++) {
              indices[i] = (int *) ((char *)f + qv * sizeof(int));
              qv += vertsperstrip[i]; // incr vertex index, next strip
            }

            GLMULTIDRAWELEMENTSEXT(GL_TRIANGLE_STRIP, 
                                   (GLsizei *) vertsperstrip, 
                                   GL_UNSIGNED_INT, 
                                   (const GLvoid **) indices, 
                                   numstrips);

            delete [] indices;
          }
          else  // if not MDE, then fall back to other techniques
#endif
          // Use the regular OpenGL 1.1 vertex array APIs, loop over all strips
          {
            int qv=0;
            for (strip=0; strip < numstrips; strip++) {
              glDrawElements(GL_TRIANGLE_STRIP, vertsperstrip[strip],
                             GL_UNSIGNED_INT, (int *) ((char *) f + qv * sizeof(int)));
              qv += vertsperstrip[strip];
            }
          }

#if defined(GL_EXT_compiled_vertex_array) 
          if (ext->hascompiledvertexarrayext) {
            GLUNLOCKARRAYSEXT();
          }
#endif
        } else {
#endif
          // simple graphics mode, but not Acrobat3D capture mode
          if (!ogl_acrobat3dcapture) {
            // No OpenGL 1.1? ouch, then we have to do this the slow way
            int t, ind;
            int v = 0; // current vertex index, initially 0
            // loop over all of the triangle strips
            for (strip=0; strip < numstrips; strip++) {         
              glBegin(GL_TRIANGLE_STRIP);
              // render all of the triangles in this strip
              for (t = 0; t < vertsperstrip[strip]; t++) {
                ind = f[v] * 10;
                 glColor3fv(cnv + ind    );
                glNormal3fv(cnv + ind + 4);
                glVertex3fv(cnv + ind + 7);
                v++; // increment vertex index, for the next triangle
              }
              glEnd();
            }
          } else {
            // Acrobat3D capture mode works around several bugs in the
            // capture utility provided with version 7.x.  Their capture
            // feature can't catch triangle strips, so we have to render
            // each of the triangles individually.

            // render triangle strips one triangle at a time
            // triangle winding order is:
            //   v0, v1, v2, then v2, v1, v3, then v2, v3, v4, etc.
            int strip, t, v = 0;
            int stripaddr[2][3] = { {0, 1, 2}, {1, 0, 2} };

            // loop over all of the triangle strips
            for (strip=0; strip < numstrips; strip++) {
              // loop over all triangles in this triangle strip
              glBegin(GL_TRIANGLES);

              for (t = 0; t < (vertsperstrip[strip] - 2); t++) {
                // render one triangle, using lookup table to fix winding order
                int v0 = f[v + (stripaddr[t & 0x01][0])] * 10;
                int v1 = f[v + (stripaddr[t & 0x01][1])] * 10;
                int v2 = f[v + (stripaddr[t & 0x01][2])] * 10;

#if 1
                // Version 7.0.9 of Acrobat3D can't capture multicolor
                // triangles, so we revert to averaged-single-color
                // triangles until they fix this capture bug.
                float tmp[3], tmp2[3];
                vec_add(tmp, cnv + v0, cnv + v1); 
                vec_add(tmp2, tmp, cnv + v2); 
                vec_scale(tmp, 0.3333333, tmp2);
                glColor3fv(tmp);
                glNormal3fv(cnv + v0 + 4);
                glVertex3fv(cnv + v0 + 7);
                glNormal3fv(cnv + v1 + 4);
                glVertex3fv(cnv + v1 + 7);
                glNormal3fv(cnv + v2 + 4);
                glVertex3fv(cnv + v2 + 7);
#else
                 glColor3fv(cnv + v0    );
                glNormal3fv(cnv + v0 + 4);
                glVertex3fv(cnv + v0 + 7);
                 glColor3fv(cnv + v1    );
                glNormal3fv(cnv + v1 + 4);
                glVertex3fv(cnv + v1 + 7);
                 glColor3fv(cnv + v2    );
                glNormal3fv(cnv + v2 + 4);
                glVertex3fv(cnv + v2 + 7);
#endif

                v++; // move on to next vertex
              }
              glEnd();
              v+=2; // last two vertices are already used by last triangle
            }
          }

#if defined(GL_VERSION_1_1)
        }
#endif

        // return to double-sided lighting mode if we switched
        if (!cmd->doublesided) {
          glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);
        }
        }
        break;

      case DWIREMESH: 
        {
        // draw a wire mesh
        DispCmdWireMesh *cmd = (DispCmdWireMesh *) cmdptr;
        int ind = cmd->numlines * 2;
        float *cnv;
        int *l;
        cmd->getpointers(cnv, l);
#if defined(GL_VERSION_1_1)
        if (!simplegraphics) {
          glInterleavedArrays(GL_C4F_N3F_V3F, 0, cnv);

#if defined(GL_EXT_compiled_vertex_array) 
          if (ext->hascompiledvertexarrayext) {
            GLLOCKARRAYSEXT(0, cmd->numverts);
          }
#endif

          glDrawElements(GL_LINES, ind, GL_UNSIGNED_INT, l);

#if defined(GL_EXT_compiled_vertex_array) 
          if (ext->hascompiledvertexarrayext) {
            GLUNLOCKARRAYSEXT();
          }
#endif
        } else {
#endif
          int i, ind2;
          glBegin(GL_LINES);
          for (i=0; i<ind; i++) {
            ind2 = l[i] * 10;
             glColor3fv(cnv + ind2    );
            glNormal3fv(cnv + ind2 + 4);
            glVertex3fv(cnv + ind2 + 7);
          }
          glEnd();
#if defined(GL_VERSION_1_1)
        }
#endif
        }
        break;

      case DCYLINDER:
        {
        // draw a cylinder of given radius and resolution
        float *cmd = (float *)cmdptr; 
        cylinder_full((int)(cmd[7]), cmd+9, (int)(cmd[8]));
        } 
        break;

      case DCONE:
        {
        DispCmdCone *cmd = (DispCmdCone *)cmdptr;
        // draw a cone of given radius and resolution
        cylinder(cmd->pos2, cmd->pos1, cmd->res, cmd->radius, cmd->radius2);
        }
        break;

      case DTEXTSIZE:
        textsize = ((DispCmdTextSize *)cmdptr)->size;
        break;

      case DTEXTOFFSET:
        textoffset_x = ((DispCmdTextOffset *)cmdptr)->x;
        textoffset_y = ((DispCmdTextOffset *)cmdptr)->y;
        break;

      case DTEXT:
        {
        float *pos = (float *)cmdptr;        
        char *txt = (char *)(pos+3);
        float wp[4];
        float mp[4] = { 0, 0, 0, 1};

#ifdef VMDWIREGL
        // WireGL doesn't suppor the glPushAttrib() function, so these are
        // variables used to save current OpenGL state prior to 
        // clobbering it with new state, so we can return properly.
        GLfloat   tmplineWidth;
        GLboolean tmplineStipple;
        GLint     tmplineSRepeat;
        GLint     tmplineSPattern;
#endif

        mp[0] = pos[0]; mp[1] = pos[1]; mp[2] = pos[2];
        textMat.multpoint4d(mp,wp);

        glPushMatrix();
          glLoadIdentity();
          glMatrixMode(GL_PROJECTION);
          glPushMatrix();
            glLoadIdentity();
            glTranslatef((wp[0]+textoffset_x)/wp[3], 
                         (wp[1]+textoffset_y)/wp[3], 
                          wp[2]/wp[3]);

            glScalef(textsize,textsize,textsize);

#ifdef VMDWIREGL
              glGetFloatv(GL_LINE_WIDTH,          &tmplineWidth   );
            glGetIntegerv(GL_LINE_STIPPLE_REPEAT, &tmplineSRepeat );
            glGetIntegerv(GL_LINE_STIPPLE_PATTERN,&tmplineSPattern);
            tmplineStipple = glIsEnabled(GL_LINE_STIPPLE);
#else
            glPushAttrib(GL_LINE_BIT);
#endif

            // enable line antialiasing, looks much nicer, may run slower
            glEnable(GL_BLEND);
            glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
            glEnable(GL_LINE_SMOOTH);

// #define VMDMSAAFONTTOGGLE 1

            // MSAA lines with widths > 1.0 look bad at low sample counts
            // so we either toggle MSAA off/on, or we have to stick to lines
            // of 1.0 pixels in width.
#if defined(VMDMSAAFONTTOGGLE)
            // Toggle MSAA off/on on-the-fly
            if (aaEnabled) {
#if defined(GL_ARB_multisample)
              glDisable(GL_MULTISAMPLE_ARB);
#else
#if defined(GL_SGIS_multisample)
              glDisable(GL_MULTISAMPLE_SGIS);
#endif
#endif 
            }
            glLineWidth(2.0); // looks nicer when MSAA isn't on
#elif 0
            // choose best looking line width for current state
            if (aaEnabled) {
              glLineWidth(1.0); // use line width 1.0 when MSAA is on
            } else {
              glLineWidth(2.0); // looks nicer when MSAA isn't on
            }
#else 
            // original code, always width 1.0
            glLineWidth(1.0);
#endif

            glDisable(GL_LINE_STIPPLE);
            glListBase(fontListBase);
            glCallLists(strlen(txt), GL_UNSIGNED_BYTE, (GLubyte *)txt);


#if defined(VMDMSAAFONTTOGGLE)
            // Toggle MSAA off/on on-the-fly
            if (aaEnabled) {
#if defined(GL_ARB_multisample)
              glEnable(GL_MULTISAMPLE_ARB);
#else
#if defined(GL_SGIS_multisample)
              glEnable(GL_MULTISAMPLE_SGIS);
#endif
#endif 
            }
#endif



            // disable line antialiasing, return to normal mode 
            glDisable(GL_BLEND);
            glDisable(GL_LINE_SMOOTH);

#ifdef VMDWIREGL
            glLineWidth(tmplineWidth);
            glLineStipple(tmplineSRepeat, (GLushort) tmplineSPattern);
            if (tmplineStipple == GL_TRUE)
               glEnable(GL_LINE_STIPPLE);
            else
               glDisable(GL_LINE_STIPPLE);
#else
            glPopAttrib();
#endif


          glPopMatrix();
          glMatrixMode(GL_MODELVIEW);
        glPopMatrix();
        }
        break;

      case DCOLORINDEX:
        // set the current color to the given color index ... assumes the
        // color has already been defined
        glColor3fv((GLfloat *)(colorData+3*(((DispCmdColorIndex *)cmdptr)->color)));
        break;

      case DMATERIALON:
        glEnable(GL_LIGHTING);
        ogl_lightingenabled=1;
#if defined(VMDUSEOPENGLSHADER)
        if (mainshader && ogl_useglslshader) {
          mainshader->UseShader(1); // use glsl mainshader when shading is on
        }
#endif
        break;

      case DMATERIALOFF:
        glDisable(GL_LIGHTING);
        ogl_lightingenabled=0;
#if defined(VMDUSEOPENGLSHADER)
        if (mainshader && ogl_useglslshader) {
          mainshader->UseShader(0); // use fixed-func pipeline when shading is off
        }
#endif
        break;

      case DSPHERERES:
        // set the current sphere resolution
        set_sphere_res(((DispCmdSphereRes *)cmdptr)->res);
        break;

      case DSPHERETYPE:
        // set the current sphere type
        set_sphere_mode(((DispCmdSphereType *)cmdptr)->type);
        break;

      case DLINESTYLE: 
        // set the current line style
        set_line_style(((DispCmdLineType *)cmdptr)->type);
        break;

      case DLINEWIDTH: 
        // set the current line width
        set_line_width(((DispCmdLineWidth *)cmdptr)->width);
        break;

      case DPICKPOINT:
      case DPICKPOINT_I:
      case DPICKPOINT_IARRAY:
      default:
        // msgErr << "OpenGLRenderer: Unknown drawing token " << tok
        //        << " encountered ... Skipping this command." << sendmsg;
        break;

    } 
  }
 
  } // XXX code to run render loop or not

  // Tail end of display list caching code
  if (ogl_cacheenabled && (!ogl_cacheskip)) { 
    if (ogl_cachecreated) {
      glEndList();              // finish off display list we're creating
    } else {
      if (ogl_cachedebug) {
        msgInfo << "Calling cached geometry: id=" << (int)ogl_cachedid << sendmsg;
      }
      glCallList(ogl_cachedid); // call the display list we previously cached
    }
  }

  glPopMatrix();
} // end loop over periodic images

  // restore transformation matrix
  glPopMatrix();
}

void OpenGLRenderer::render_done() {
  ogl_glsltoggle = 1; // force GLSL update next time through

  GLuint tag;
  // delete all unused display lists
  while ((tag = displaylistcache.deleteUnused()) != GLCACHE_FAIL) {
    glDeleteLists(tag, 1);
  }

  // delete all unused textures
  while ((tag = texturecache.deleteUnused()) != GLCACHE_FAIL) {
    glDeleteTextures(1, &tag);
  }
}

void OpenGLRenderer::require_volume_texture(unsigned long ID,
    unsigned xsize, unsigned ysize, unsigned zsize,
    unsigned char *texmap) {

  if (!ext->hastex3d) return;
  GLuint texName;
  if ((texName = texturecache.markUsed(ID)) == 0) {
    glGenTextures(1, &texName);
    texturecache.encache(ID, texName); // cache this texture ID
    glBindTexture(GL_TEXTURE_3D, texName);

    // set texture border color to black
    GLfloat texborder[4] = {0.0, 0.0, 0.0, 1.0};
    glTexParameterfv(GL_TEXTURE_3D, GL_TEXTURE_BORDER_COLOR, texborder);

    // use the border color when wrapping at the edge
    glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_WRAP_R, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_WRAP_T, GL_CLAMP);

    // enable texture interpolation and filtering
    glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

    if (build3Dmipmaps(xsize, ysize, zsize, texmap)) {
      msgErr << "OpenGLRenderer failed to download 3-D texture map!" 
             << sendmsg; 
    }
  } else { // already cached, so just enable.
    glBindTexture(GL_TEXTURE_3D, texName);
  }
}


int OpenGLRenderer::build3Dmipmaps(int width, int height, int depth, unsigned char *tx) {
#if defined(GL_VERSION_1_2)
  if (ext->hastex3d) {
    int xsize=width;
    int ysize=height;
    int zsize=depth;
    int xstep=1;
    int ystep=1;
    int zstep=1;
    int x,y,z;

    if (tx == NULL) {
      msgErr << "Skipping MIP map generation for NULL 3-D texture map" 
             << sendmsg;
      return 1;
    } 

    // build Mipmaps if the card can't handle the full resolution texture map
    if (xsize > max3DtexX || ysize > max3DtexY || zsize > max3DtexZ) { 
      unsigned char *texmap;

      while (xsize > max3DtexX) {
        xsize >>= 1;
        xstep <<= 1;
      }
  
      while (ysize > max3DtexY) {
        ysize >>= 1;
        ystep <<= 1;
      }

      while (zsize > max3DtexZ) {
        zsize >>= 1; 
        zstep <<= 1;
      }

      if (xsize == 0 || ysize == 0 || zsize == 0)
        return 1; // error, can't subsample the image down to required res

      texmap = (unsigned char *) malloc(xsize*ysize*zsize*3);
      if (texmap == NULL) {
        msgErr << "Failed to allocate MIP map for downsampled texture" 
               << sendmsg;
        return 1; // failed allocation
      }

#if 0
      // XXX draw a checkerboard texture until the MIPmap code is finished
      msgError << "3-D texture map can't fit into accelerator memory, aborted."
               << sendmsg;

      for (z=0; z<zsize; z++) {
        for (y=0; y<ysize; y++) {
          int addr = z*xsize*ysize + y*xsize;
          for (x=0; x<xsize; x++) {
            if ((x + y + z) % 2) {
              texmap[(addr + x)*3    ] = 0;
              texmap[(addr + x)*3 + 1] = 0;
              texmap[(addr + x)*3 + 2] = 0;
            } else {
              texmap[(addr + x)*3    ] = 255;
              texmap[(addr + x)*3 + 1] = 255;
              texmap[(addr + x)*3 + 2] = 255;
            }
          }
        }
      }

#else
      msgInfo << "Downsampling 3-D texture map from " 
              << width << "x" << height << "x" << depth << " to " 
              << xsize << "x" << ysize << "x" << zsize << sendmsg;
               
      for (z=0; z<zsize; z++) {
        for (y=0; y<ysize; y++) {
          int addr = z*xsize*ysize + y*xsize;
          for (x=0; x<xsize; x++) {
            int sumR=0, sumG=0, sumB=0;
            int texelcount = 0;
            int ox, oxs, oxe;
            int oy, oys, oye;
            int oz, ozs, oze;

            oxs = x * xstep;
            oys = y * ystep;
            ozs = z * zstep;

            oxe = oxs + xstep;
            oye = oys + ystep;
            oze = ozs + zstep;
            if (oxe > width) oxe=width;
            if (oye > height) oye=height;
            if (oze > depth) oze=depth;

            for (oz=ozs; oz<oze; oz++) {
              for (oy=oys; oy<oye; oy++) {
                int oaddr = oz*width*height + oy*width;
                for (ox=oxs; ox<oxe; ox++) {
                  int oadx = (oaddr + ox)*3;
                  sumR += tx[oadx    ];
                  sumG += tx[oadx + 1];
                  sumB += tx[oadx + 2];
                  texelcount++;
                }
              }
            }

            int adx = (addr + x)*3;
            texmap[adx    ] = (unsigned char) (sumR / ((float) texelcount));
            texmap[adx + 1] = (unsigned char) (sumG / ((float) texelcount));
            texmap[adx + 2] = (unsigned char) (sumB / ((float) texelcount));
          }
        }
      }
#endif

      glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
      GLTEXIMAGE3D(GL_TEXTURE_3D, 0, GL_RGB8, xsize, ysize, zsize,
                   0, GL_RGB, GL_UNSIGNED_BYTE, texmap);

      free(texmap); // free the generated texture map for now
    } else {
      glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
      GLTEXIMAGE3D(GL_TEXTURE_3D, 0, GL_RGB8, width, height, depth,
                   0, GL_RGB, GL_UNSIGNED_BYTE, tx);
    }
  
    return 0;
  }
#endif

  return 1; // failed to render 3-D texture map
}

void OpenGLRenderer::update_shader_uniforms(void * voidshader, int forceupdate) {
#if defined(VMDUSEOPENGLSHADER)
  OpenGLShader *sh = (OpenGLShader *) voidshader; 
  GLint loc;

  // Update GLSL projection mode (used to control normal flipping code)
  GLint vmdprojectionmode = (ogl_glslprojectionmode == DisplayDevice::PERSPECTIVE) ? 1 : 0;
  loc = GLGETUNIFORMLOCATIONARB(sh->ProgramObject, "vmdprojectionmode");
  GLUNIFORM1IARB(loc, vmdprojectionmode);

  // Update active GLSL texturing mode from cached state just in case
  GLint vmdtexturemode = ogl_glsltexturemode;
  loc = GLGETUNIFORMLOCATIONARB(sh->ProgramObject, "vmdtexturemode");
  GLUNIFORM1IARB(loc, vmdtexturemode);

  // Update material parameters for OpenGL shader.
  // XXX unnecessary once references to gl_FrontMaterial.xxx work
  GLfloat matparms[4];
  matparms[0] = oglambient;
  matparms[1] = ogldiffuse;
  matparms[2] = oglspecular;
  matparms[3] = oglshininess;
  loc = GLGETUNIFORMLOCATIONARB(sh->ProgramObject, "vmdmaterial");
  GLUNIFORM4FVARB(loc, 1, matparms);

  // Set vmdopacity uniform used for alpha-blended transparency in GLSL  
  GLfloat vmdopacity[1];
  vmdopacity[0] = oglopacity;
  loc = GLGETUNIFORMLOCATIONARB(sh->ProgramObject, "vmdopacity");
  GLUNIFORM1FVARB(loc, 1, vmdopacity);

  // Set fog mode for shader using vmdfogmode uniform
  loc = GLGETUNIFORMLOCATIONARB(sh->ProgramObject, "vmdfogmode");
  GLUNIFORM1IARB(loc, ogl_fogmode);

  // Set active texture map index
  loc = GLGETUNIFORMLOCATIONARB(sh->ProgramObject, "vmdtex0");
  GLUNIFORM1IARB(loc, 0); // using texture unit 0

  // Update the main lighting state used by GLSL if it isn't the same
  // as what is currently set in the fixed-function pipeline.
  // XXX this code will not be necessary once vendors correctly implement
  //     references to gl_LightSource[n].position in GLSL shader
  if (forceupdate || (ogl_glslserial != ogl_rendstateserial)) {
    int i;

    if (!forceupdate) {
      // Once updated, no need to do it again
      ogl_glslserial = ogl_rendstateserial;
    }

    // Set light positions and pre-calculating Blinn halfway
    // vectors for use by the shaders
    for (i=0; i<DISP_LIGHTS; i++) {
      char varbuf[32];
      sprintf(varbuf, "vmdlight%d", i);
      GLint loc = GLGETUNIFORMLOCATIONARB(sh->ProgramObject, varbuf);
      GLUNIFORM3FVARB(loc, 1, &ogl_lightpos[i][0]);

      // calculate Blinn's halfway vector 
      // L = direction to light
      // V = direction to camera
      // H=normalize(L + V)
      float L[3], V[3];
      GLfloat Hvec[3];
      (transMat.top()).multpoint3d(&ogl_lightpos[i][0], L);
      vec_scale(V, -1.0, eyeDir);
      vec_normalize(V);
      Hvec[0] = L[0] + V[0];
      Hvec[1] = L[1] + V[1];
      Hvec[2] = L[2] + V[2];
      vec_normalize(Hvec);
      sprintf(varbuf, "vmdlight%dH", i);
      loc = GLGETUNIFORMLOCATIONARB(mainshader->ProgramObject, varbuf);
      GLUNIFORM3FVARB(loc, 1, Hvec);
    } 

    // Set light on/off state for shader as well, using pre-known uniforms
    // XXX this code assumes a max of 4 lights, due to the use of a 
    //     vec4 for storing the values, despite DISP_LIGHTS sizing 
    //     the array of light scales.
    loc = GLGETUNIFORMLOCATIONARB(sh->ProgramObject, "vmdlightscale");
    GLfloat vmdlightscale[DISP_LIGHTS];
    for (i=0; i<DISP_LIGHTS; i++) {
      vmdlightscale[i] = (float) ogl_lightstate[i];
    }
    GLUNIFORM4FVARB(loc, 1, vmdlightscale);
  }
#endif
}


