/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995-2007 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: OpenGLDisplayDevice.C,v $
 *	$Author: johns $	$Locker:  $		$State: Exp $
 *	$Revision: 1.147 $	$Date: 2007/02/08 22:26:44 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Subclass of DisplayDevice, this object has routines used by all the
 * different display devices that are OpenGL-specific.  Will render drawing
 * commands into a single X window.
 *
 ***************************************************************************/

#include <stdlib.h>
#include <math.h>
#include <GL/gl.h>
#include <GL/glx.h>
#include <X11/cursorfont.h>

#include "OpenGLDisplayDevice.h"
#include "Inform.h"
#include "utilities.h"
#include "config.h"   // VMD version strings etc

// static data for this object
static const char *glStereoNameStr[OPENGL_STEREO_MODES] = 
 { "Off", 
   "CrystalEyes", 
   "CrystalEyesReversed", 
   "DTI SideBySide",
   "Scanline Interleaved", 
   "Anaglyph", 
   "CrossEyes", 
   "SideBySide", 
   "Left", 
   "Right" };

static const char *glRenderNameStr[OPENGL_RENDER_MODES] = 
{ "Normal",
  "AlphaBlend",
  "GLSL",
  "Acrobat3D" };

static const char *glCacheNameStr[OPENGL_CACHE_MODES] = 
{ "Off",
  "On" };

// determine if all of the ARB multisample extension routines are available
#if defined(GL_ARB_multisample) && defined(GLX_SAMPLES_ARB) && defined(GLX_SAMPLE_BUFFERS_ARB)
#define USEARBMULTISAMPLE 1
#endif

// colors for cursors
static XColor cursorFG = { 0, 0xffff,      0,      0, 
                         DoRed | DoGreen | DoBlue, 0 };
static XColor cursorBG = { 0, 0xffff, 0xffff, 0xffff, 
                         DoRed | DoGreen | DoBlue, 0 };

////////////////////////// static helper functions.

static XVisualInfo * vmd_get_visual(glxdata *glxsrv, int *stereo, int *msamp, int *numsamples, int *hasdvr) {
  // we want double-buffered RGB with a Z buffer (possibly with stereo)
  XVisualInfo *vi;
  int ns, dsize;
  int simplegraphics = 0;
  vi = NULL;
  *numsamples = 0;
  *msamp = FALSE; 
  *stereo = FALSE;
  *hasdvr = FALSE;

  if (getenv("VMDSIMPLEGRAPHICS")) {
    simplegraphics = 1;
  }

  // loop over a big range of depth buffer sizes, starting with biggest 
  // and working our way down from there.
  for (dsize=32; dsize >= 16; dsize-=4) { 

// Try the video resizing extension if available
#if defined(GLX_VIDEO_RESIZE_SUN) && defined(USEARBMULTISAMPLE)
    if (!simplegraphics && (!vi || (vi->c_class != TrueColor))) {
      // Stereo, multisample antialising, stencil buffer
      for (ns=16; ns>1; ns--) {
        int conf[]  = {GLX_DOUBLEBUFFER, GLX_RGBA, GLX_DEPTH_SIZE, dsize, 
                       GLX_STEREO,
                       GLX_STENCIL_SIZE, 1, GLX_VIDEO_RESIZE_SUN, 1, 
                       GLX_SAMPLE_BUFFERS_ARB, 1, GLX_SAMPLES_ARB, ns, None};
        vi = glXChooseVisual(glxsrv->dpy, glxsrv->dpyScreen, conf);
    
        if (vi && (vi->c_class == TrueColor)) {
          *numsamples = ns;
          *msamp = TRUE;
          *stereo = TRUE;
          *hasdvr = TRUE;
          break; // exit loop if we got a good visual
        } 
      }
    }
#endif

// Try the OpenGL ARB multisample extension if available
#if defined(USEARBMULTISAMPLE) 
    if (!simplegraphics && (!vi || (vi->c_class != TrueColor))) {
      // Stereo, multisample antialising, stencil buffer
      for (ns=16; ns>1; ns--) {
        int conf[]  = {GLX_DOUBLEBUFFER, GLX_RGBA, GLX_DEPTH_SIZE, dsize, 
                       GLX_STEREO,
                       GLX_STENCIL_SIZE, 1, 
                       GLX_SAMPLE_BUFFERS_ARB, 1, GLX_SAMPLES_ARB, ns, None};
        vi = glXChooseVisual(glxsrv->dpy, glxsrv->dpyScreen, conf);
  
        if (vi && (vi->c_class == TrueColor)) {
          *numsamples = ns;
          *msamp = TRUE;
          *stereo = TRUE;
          *hasdvr = FALSE;
          break; // exit loop if we got a good visual
        } 
      }
    }
#endif


// Try SGI multisample extension if available
#if defined(GLX_SGIS_multisample)
    if (!simplegraphics && (!vi || (vi->c_class != TrueColor))) {
      // Stereo, SGI multisample antialising, stencil buffer
      for (ns=16; ns>1; ns--) {
        int conf[]  = {GLX_DOUBLEBUFFER, GLX_RGBA, GLX_DEPTH_SIZE, dsize, 
                       GLX_STEREO,
                       GLX_STENCIL_SIZE, 1, 
                       GLX_SAMPLE_BUFFERS_SGIS, 1, GLX_SAMPLES_SGIS, ns, None};
        vi = glXChooseVisual(glxsrv->dpy, glxsrv->dpyScreen, conf);
    
        if (vi && (vi->c_class == TrueColor)) {
          *numsamples = ns;
          *msamp = TRUE;
          *stereo = TRUE;
          *hasdvr = FALSE;
          break; // exit loop if we got a good visual
        } 
      }
    }
#endif

// Try the video resizing extension if available
#if defined(GLX_VIDEO_RESIZE_SUN) && defined(USEARBMULTISAMPLE)
    if (!simplegraphics && (!vi || (vi->c_class != TrueColor))) {
      // Non-Stereo, multisample antialising, stencil buffer, video resizing
      for (ns=16; ns>1; ns--) {
        int conf[]  = {GLX_DOUBLEBUFFER, GLX_RGBA, GLX_DEPTH_SIZE, dsize,
                       GLX_STENCIL_SIZE, 1, GLX_VIDEO_RESIZE_SUN, 1, 
                       GLX_SAMPLE_BUFFERS_ARB, 1, GLX_SAMPLES_ARB, ns, None};
        vi = glXChooseVisual(glxsrv->dpy, glxsrv->dpyScreen, conf);
  
        if (vi && (vi->c_class == TrueColor)) {
          *numsamples = ns;
          *msamp = TRUE;
          *stereo = FALSE;
          *hasdvr = TRUE;
          break; // exit loop if we got a good visual
        } 
      }
    }
#endif

    if (getenv("VMDPREFERSTEREO") != NULL) {
      // The preferred 24-bit color, quad buffered stereo mode.
      // This hack allows NVidia Quadro users to avoid the mutually-exclusive
      // antialiasing/stereo options on their cards with current drivers.
      // This forces VMD to skip looking for multisample antialiasing capable
      // X visuals and look for stereo instead.
      if (!simplegraphics && (!vi || (vi->c_class != TrueColor))) {
        int conf[] = {GLX_DOUBLEBUFFER, GLX_RGBA, GLX_DEPTH_SIZE, dsize, 
                      GLX_STEREO,
                      GLX_STENCIL_SIZE, 1, 
                      GLX_RED_SIZE, 8, GLX_GREEN_SIZE, 8, GLX_BLUE_SIZE, 8, None};
        vi = glXChooseVisual(glxsrv->dpy, glxsrv->dpyScreen, conf);
        ns = 0; // no multisample antialiasing
        *numsamples = ns;
        *msamp = FALSE; 
        *stereo = TRUE; 
        *hasdvr = FALSE;
      }
    } else {
#if defined(USEARBMULTISAMPLE) 
      // Try the OpenGL ARB multisample extension if available
      if (!simplegraphics && (!vi || (vi->c_class != TrueColor))) {
        // Non-Stereo, multisample antialising, stencil buffer
        for (ns=16; ns>1; ns--) {
          int conf[]  = {GLX_DOUBLEBUFFER, GLX_RGBA, GLX_DEPTH_SIZE, dsize, 
                         GLX_STENCIL_SIZE, 1, 
                         GLX_SAMPLE_BUFFERS_ARB, 1, GLX_SAMPLES_ARB, ns, None};
          vi = glXChooseVisual(glxsrv->dpy, glxsrv->dpyScreen, conf);
    
          if (vi && (vi->c_class == TrueColor)) {
            *numsamples = ns;
            *msamp = TRUE;
            *stereo = FALSE; 
            *hasdvr = FALSE;
            break; // exit loop if we got a good visual
          } 
        }
      }
#endif

#if defined(GLX_SGIS_multisample)
      // Try SGI multisample extension if available
      if (!simplegraphics && (!vi || (vi->c_class != TrueColor))) {
        // Non-Stereo, SGI multisample antialising, stencil buffer
        for (ns=16; ns>0; ns--) {
          int conf[]  = {GLX_DOUBLEBUFFER, GLX_RGBA, GLX_DEPTH_SIZE, dsize,
                         GLX_STENCIL_SIZE, 1, 
                         GLX_SAMPLE_BUFFERS_SGIS, 1, GLX_SAMPLES_SGIS, ns, None};
          vi = glXChooseVisual(glxsrv->dpy, glxsrv->dpyScreen, conf);
    
          if (vi && (vi->c_class == TrueColor)) {
            *numsamples = ns;
            *msamp = TRUE;
            *stereo = FALSE;
            *hasdvr = FALSE;
            break; // exit loop if we got a good visual
          } 
        }
      }
#endif
    }

  } // end of loop over a wide range of depth buffer sizes

  // Ideally we should fall back to accumulation buffer based antialiasing
  // here, but not currently implemented.  At this point no multisample
  // antialiasing mode is available.

  // The preferred 24-bit color, quad buffered stereo mode
  // that high-end SGI and Sun machines normally provide.
  if (!simplegraphics && (!vi || (vi->c_class != TrueColor))) {
    int conf[] = {GLX_DOUBLEBUFFER, GLX_RGBA, GLX_DEPTH_SIZE, 16, GLX_STEREO,
                  GLX_STENCIL_SIZE, 1, 
                  GLX_RED_SIZE, 8, GLX_GREEN_SIZE, 8, GLX_BLUE_SIZE, 8, None};
    vi = glXChooseVisual(glxsrv->dpy, glxsrv->dpyScreen, conf);
    ns = 0; // no multisample antialiasing
    *numsamples = ns;
    *msamp = FALSE; 
    *stereo = TRUE; 
    *hasdvr = FALSE;
  }

  // Mode for machines that provide stereo only in
  // modes with less than 24-bit color.  Examples of this are
  // The SGI O2, and the HP Visualize-FX series boards.
  // Without this configuration attempt, these machines won't get stereo.
  if (!simplegraphics && (!vi || (vi->c_class != TrueColor))) {
    int conf[] = {GLX_DOUBLEBUFFER, GLX_RGBA, GLX_DEPTH_SIZE, 16, GLX_STEREO,
                  GLX_STENCIL_SIZE, 1, 
                  GLX_RED_SIZE, 1, GLX_GREEN_SIZE, 1, GLX_BLUE_SIZE, 1, None};
    vi = glXChooseVisual(glxsrv->dpy, glxsrv->dpyScreen, conf);
    ns = 0; // no multisample antialiasing
    *numsamples = ns;
    *msamp = FALSE; 
    *stereo = TRUE; 
    *hasdvr = FALSE;
  }

  // Mode for machines that provide stereo only without a stencil buffer, 
  // and with reduced color precision.  Examples of this are the SGI Octane2
  // machines with V6 graphics, with recent IRIX patch levels.
  // Without this configuration attempt, these machines won't get stereo.
  if (!simplegraphics && (!vi || (vi->c_class != TrueColor))) {
    int conf[] = {GLX_DOUBLEBUFFER, GLX_RGBA, GLX_DEPTH_SIZE, 16, GLX_STEREO,
                  GLX_RED_SIZE, 1, GLX_GREEN_SIZE, 1, GLX_BLUE_SIZE, 1, None};
    vi = glXChooseVisual(glxsrv->dpy, glxsrv->dpyScreen, conf);
    ns = 0; // no multisample antialiasing
    *numsamples = ns;
    *msamp = FALSE; 
    *stereo = TRUE; 
    *hasdvr = FALSE;
  }

  // This mode gives up on trying to get stereo, and goes back to trying
  // to get a high quality non-stereo visual.
  if (!simplegraphics && (!vi || (vi->c_class != TrueColor))) {
    int conf[] = {GLX_DOUBLEBUFFER, GLX_RGBA, GLX_DEPTH_SIZE, 16, 
                  GLX_STENCIL_SIZE, 1, 
                  GLX_RED_SIZE, 8, GLX_GREEN_SIZE, 8, GLX_BLUE_SIZE, 8, None};
    vi = glXChooseVisual(glxsrv->dpy, glxsrv->dpyScreen, conf);
    ns = 0; // no multisample antialiasing
    *numsamples = ns;
    *msamp = FALSE; 
    *stereo = FALSE;
  }
  
  // check if we have a TrueColor visual.
  if(!vi || (vi->c_class != TrueColor) ) {
    // still no TrueColor.  Try again, with a very basic request ...
    // This is a catch all, we're desperate for any truecolor
    // visual by this point.  We've given up hoping for 24-bit
    // color or stereo by this time.
    int conf[] = {GLX_DOUBLEBUFFER, GLX_RGBA, GLX_DEPTH_SIZE, 16, 
                  GLX_RED_SIZE, 1, GLX_GREEN_SIZE, 1, GLX_BLUE_SIZE, 1, None};
    vi = glXChooseVisual(glxsrv->dpy, glxsrv->dpyScreen, conf);
    ns = 0; // no multisample antialiasing
    *numsamples = ns;
    *msamp = FALSE; 
    *stereo = FALSE;
    *hasdvr = FALSE;
  }

  if (!vi || (vi->c_class != TrueColor)) {
    // complete failure
    ns = 0; // no multisample antialiasing
    *numsamples = ns;
    *msamp = FALSE; 
    *stereo = FALSE;
    *hasdvr = FALSE;
  }

  return vi;
}


/////////////////////////  constructor and destructor  

OpenGLDisplayDevice::OpenGLDisplayDevice()
: OpenGLRenderer((char *) "VMD " VMDVERSION " OpenGL Display") {

  // set up data possible before opening window
  stereoNames = glStereoNameStr;
  stereoModes = OPENGL_STEREO_MODES;

  renderNames = glRenderNameStr;
  renderModes = OPENGL_RENDER_MODES;

  cacheNames = glCacheNameStr;
  cacheModes = OPENGL_CACHE_MODES;

  memset(&glxsrv, 0, sizeof(glxsrv));
  glxsrv.dpy = NULL;
  glxsrv.dpyScreen = 0;
  have_window = FALSE;
  screenX = screenY = 0;
  vmdapp = NULL;
}

int OpenGLDisplayDevice::init(int argc, char **argv, VMDApp *app, int *size, int *loc) {
  vmdapp = app; // save VMDApp handle for use by drag-and-drop handlers

  // open the window
  glxsrv.windowID = open_window(name, size, loc, argc, argv);
  if (!have_window) return FALSE;

  // set flags for the capabilities of this display
  // whether we can do antialiasing or not.
  if (ext->hasmultisample) 
    aaAvailable = TRUE;  // we use multisampling over other methods
  else
    aaAvailable = FALSE; // no non-multisample implementation yet

  cueingAvailable = TRUE;
  cullingAvailable = TRUE;
  cullingEnabled = FALSE;

  // set default settings
  if (ext->hasmultisample) {
    aa_on();  // enable fast multisample based antialiasing by default
              // other antialiasing techniques are slow, so only multisample
              // makes sense to enable by default.
  } 
  cueing_off(); // leave depth cueing off by default, since its a speed hit.

  set_sphere_mode(sphereMode);
  set_sphere_res(sphereRes);
  set_line_width(lineWidth);
  set_line_style(lineStyle);

  // reshape and clear the display, which initializes some other variables
  reshape();
  normal();
  clear();
  update();

  // We have a window, return success.
  return TRUE;
}

// destructor ... close the window
OpenGLDisplayDevice::~OpenGLDisplayDevice(void) {
  if (have_window) {
    free_opengl_ctx(); // free display lists, textures, etc
 
    // close and delete windows, contexts, and display connections
    XUnmapWindow(glxsrv.dpy, glxsrv.windowID);
    glXDestroyContext(glxsrv.dpy, glxsrv.cx);
    XDestroyWindow(glxsrv.dpy, glxsrv.windowID);
    XCloseDisplay(glxsrv.dpy);
  }
}


/////////////////////////  protected nonvirtual routines  


// create a new window and set it's characteristics
Window OpenGLDisplayDevice::open_window(char *nm, int *size, int *loc,
					int argc, char** argv
) {
  Window win;
  int i, SX = 100, SY = 100, W, H;
 
  char *dispname;
  if ((dispname = getenv("VMDGDISPLAY")) == NULL)
    dispname = getenv("DISPLAY");

  if(!(glxsrv.dpy = XOpenDisplay(dispname))) {
    msgErr << "Exiting due to X-Windows OpenGL window creation failure." << sendmsg;
    if (dispname != NULL) {
      msgErr << "Failed to open display: " << dispname << sendmsg;
    }
    return (Window)0; 
  }
  // get info about root window
  glxsrv.dpyScreen = DefaultScreen(glxsrv.dpy);
  glxsrv.rootWindowID = RootWindow(glxsrv.dpy, glxsrv.dpyScreen);
  screenX = DisplayWidth(glxsrv.dpy, glxsrv.dpyScreen);
  screenY = DisplayHeight(glxsrv.dpy, glxsrv.dpyScreen);
  W = size[0];
  H = size[1];
  if (loc) {
    SX = loc[0];
    // The X11 screen uses Y increasing from upper-left corner down; this is
    // opposite to what GL does, which is the way VMD was set up originally
    SY = (screenY - loc[1]) - H;
  }

  // (3) make sure the GLX extension is available
  if (!glXQueryExtension(glxsrv.dpy, NULL, NULL)) {
    msgErr << "The X server does not support the OpenGL GLX extension." 
           << "   Exiting ..." << sendmsg;
    XCloseDisplay(glxsrv.dpy);
    return (Window)0;
  }

  ext->hasstereo = TRUE;         // stereo on until we find out otherwise.
  ext->stereodrawforced = FALSE; // no need for force stereo draws initially
  ext->hasmultisample = TRUE;    // multisample on until we find out otherwise.

  // (4) find an appropriate X-Windows GLX-capable visual and colormap ...
  XVisualInfo *vi;
  vi =  vmd_get_visual(&glxsrv, &ext->hasstereo, &ext->hasmultisample, &ext->nummultisamples, &ext->hasglvideoresizesun);
  // make sure we have what we want, darnit ...
  if (!vi) {
    msgErr << "A TrueColor visual is required, but not available." << sendmsg;
    msgErr << "The X server is not capable of displaying double-buffered," << sendmsg;
    msgErr << "RGB images with a Z buffer.   Exiting ..." << sendmsg;
    XCloseDisplay(glxsrv.dpy);
    return (Window)0;
  }

  Atom wmDeleteWindow = XInternAtom(glxsrv.dpy, "WM_DELETE_WINDOW", False);

  // (5) create an OpenGL rendering context
  if(!(glxsrv.cx = glXCreateContext(glxsrv.dpy, vi, None, GL_TRUE))) {
    msgErr << "Could not create OpenGL rendering context-> Exiting..." 
           << sendmsg;
    return (Window)0;
  }

  // (6) setup cursors, icons, iconized mode title, etc.
  glxsrv.cursor[0] = XCreateFontCursor(glxsrv.dpy, XC_left_ptr);
  glxsrv.cursor[1] = XCreateFontCursor(glxsrv.dpy, XC_fleur);
  glxsrv.cursor[2] = XCreateFontCursor(glxsrv.dpy, XC_sb_h_double_arrow);
  glxsrv.cursor[3] = XCreateFontCursor(glxsrv.dpy, XC_crosshair);
  glxsrv.cursor[4] = XCreateFontCursor(glxsrv.dpy, XC_watch);
  for(i=0; i < 5; i++)
    XRecolorCursor(glxsrv.dpy, glxsrv.cursor[i], &cursorFG, &cursorBG);


  //
  // Create the window
  //
  XSetWindowAttributes swa;

  //   For StaticGray , StaticColor, and TrueColor,
  //   alloc must be AllocNone , or a BadMatch error results
  swa.colormap = XCreateColormap(glxsrv.dpy, glxsrv.rootWindowID, 
                                 vi->visual, AllocNone);

  swa.background_pixmap = None;
  swa.border_pixel=0;
  swa.event_mask = ExposureMask;
  swa.cursor = glxsrv.cursor[0];

  win = XCreateWindow(glxsrv.dpy, glxsrv.rootWindowID, SX, SY, W, H, 0,
                      vi->depth, InputOutput, vi->visual,
                      CWBorderPixel | CWColormap | CWEventMask, &swa);
  XInstallColormap(glxsrv.dpy, swa.colormap);

  XFree(vi); // free visual info

  //
  // create size hints for new window
  //
  memset((void *) &(glxsrv.sizeHints), 0, sizeof(glxsrv.sizeHints));
  glxsrv.sizeHints.flags |= USSize;
  glxsrv.sizeHints.flags |= USPosition;
  glxsrv.sizeHints.width = W;
  glxsrv.sizeHints.height = H;
  glxsrv.sizeHints.x = SX;
  glxsrv.sizeHints.y = SY;

  XSetStandardProperties(glxsrv.dpy, win, nm, "VMD", None, argv, argc, &glxsrv.sizeHints);
  XWMHints *wmHints = XAllocWMHints();
  wmHints->initial_state = NormalState;
  wmHints->flags = StateHint;
  XSetWMHints(glxsrv.dpy, win, wmHints);
  XFree(wmHints);
  XSetWMProtocols(glxsrv.dpy, win, &wmDeleteWindow, 1);


  // (7) bind the rendering context to the window
  glXMakeCurrent(glxsrv.dpy, win, glxsrv.cx);


  // (8) actually request the window to be displayed
  XSelectInput(glxsrv.dpy, win, KeyPressMask | ButtonPressMask | 
               ButtonReleaseMask | StructureNotifyMask | ExposureMask);
  XMapRaised(glxsrv.dpy, win);

  // If we have acquired a multisample buffer with GLX, we
  // still need to test to see if we can actually use it.
  if (ext->hasmultisample) {
    int msampeext = 0;

    // check for ARB multisampling
    if (ext->vmdQueryExtension("GL_ARB_multisample")) {
      msampeext = 1;
    }

    // check for SGI multisampling
    if (ext->vmdQueryExtension("GL_SGIS_multisample")) {
      msampeext = 1;
    } 

    if (!msampeext) {
      ext->hasmultisample = FALSE;
      ext->nummultisamples = 0;
    }
  }

  // (9) configure the rendering properly
  setup_initial_opengl_state();  // setup initial OpenGL state

  // normal return: window was successfully created
  have_window = TRUE;
  // return window id
  return win;
}

void OpenGLDisplayDevice::do_resize_window(int w, int h) {
  XResizeWindow(glxsrv.dpy, glxsrv.windowID, w, h);
}
void OpenGLDisplayDevice::do_reposition_window(int xpos, int ypos) {
  XMoveWindow(glxsrv.dpy, glxsrv.windowID, xpos, ypos);
}

/////////////////////////  public virtual routines  

//
// get the current state of the device's pointer (i.e. cursor if it has one)
//

// abs pos of cursor from lower-left corner of display
int OpenGLDisplayDevice::x(void) {
  Window rw, cw;
  int rx, ry, wx, wy;
  unsigned int keymask;

  // get pointer info
  XQueryPointer(glxsrv.dpy, glxsrv.windowID, &rw, &cw, &rx, &ry, &wx, &wy, &keymask);

  // return value
  return rx;
}


// same, for y direction
int OpenGLDisplayDevice::y(void) {
  Window rw, cw;
  int rx, ry, wx, wy;
  unsigned int keymask;

  // get pointer info
  XQueryPointer(glxsrv.dpy, glxsrv.windowID, &rw, &cw, &rx, &ry, &wx, &wy, &keymask);

  // return value
  // return value ... must subtract position from total size since
  // X is opposite to GL in sizing the screen
  return screenY - ry;
}

// return the current state of the shift, control, and alt keys
int OpenGLDisplayDevice::shift_state(void) {
  int retval = 0;

  // get pointer info
  Window rw, cw;
  int rx, ry, wx, wy;
  unsigned int keymask;
  XQueryPointer(glxsrv.dpy, glxsrv.windowID, &rw, &cw, &rx, &ry, &wx, &wy, &keymask);

  // determine state of keys, and OR results together
  if ((keymask & ShiftMask) != 0)
    retval |= SHIFT;

  if ((keymask & ControlMask) != 0)
    retval |= CONTROL;

  if ((keymask & Mod1Mask) != 0)
    retval |= ALT;

  // return the result
  return retval;
}

// return the spaceball state, if any
int OpenGLDisplayDevice::spaceball(int *rx, int *ry, int *rz, int *tx, int *ty,
int *tz, int *buttons) {
  // not implemented yet
  return 0;
}


// set the Nth cursor shape as the current one.  If no arg given, the
// default shape (n=0) is used.
void OpenGLDisplayDevice::set_cursor(int n) {
  int cursorindex;

  switch (n) {
    default:
    case DisplayDevice::NORMAL_CURSOR: cursorindex = 0; break;
    case DisplayDevice::TRANS_CURSOR:  cursorindex = 1; break;
    case DisplayDevice::SCALE_CURSOR:  cursorindex = 2; break;
    case DisplayDevice::PICK_CURSOR:   cursorindex = 3; break;
    case DisplayDevice::WAIT_CURSOR:   cursorindex = 4; break;
  }

  XDefineCursor(glxsrv.dpy, glxsrv.windowID, glxsrv.cursor[cursorindex]);
}


//
// event handling routines
//

// queue the standard events (need only be called once ... but this is
// not done automatically by the window because it may not be necessary or
// even wanted)
void OpenGLDisplayDevice::queue_events(void) {
  XSelectInput(glxsrv.dpy, glxsrv.windowID,
	       KeyPressMask | ButtonPressMask | ButtonReleaseMask |
	       StructureNotifyMask | ExposureMask);
}


// This version of read_event flushes the entire queue before returning the
// last event to the caller.  It fixes buggy window resizing behavior on 
// Linux when using the Nvidia OpenGL drivers.  
int OpenGLDisplayDevice::read_event(long &retdev, long &retval) {
  XEvent xev;
  char keybuf[10];
  int keybuflen = 9;
  KeySym keysym;
  XComposeStatus comp;

  retdev = WIN_NOEVENT;
  // read all events, handling the ones that need to be handled internally,
  // and returning the last one for processing.
  int need_reshape = FALSE;
  while (XPending(glxsrv.dpy)) {
    XNextEvent(glxsrv.dpy, &xev);

    // find what kind of event it was
    switch(xev.type) {
    case Expose:
    case ConfigureNotify:
    case ReparentNotify:
    case MapNotify:
      need_reshape = TRUE; // Probably not needed for Expose or Map
      _needRedraw = 1;
      // retdev not set; we handle this ourselves.
      break;
    case KeyPress:
      {
        int k = XLookupString(&(xev.xkey), keybuf, keybuflen,  &keysym, &comp);
        if(k > 0 && *keybuf != '\0') {
          retdev = WIN_KEYBD;
          retval = *keybuf;
        }
        break;
      }
    case ButtonPress:
    case ButtonRelease:
      {
        unsigned int button = xev.xbutton.button;
        retval = (xev.type == ButtonPress);
        switch (button) {
          case Button1:
            retdev = WIN_LEFT;
            break;
          case Button2:
            retdev = WIN_MIDDLE;
            break;
          case Button3:
            retdev = WIN_RIGHT;
            break;
          case Button4:
            retdev = WIN_WHEELUP;
            break;
          case Button5:
            retdev = WIN_WHEELDOWN;
            break;
        }
        break;
      }
    } 
  } 
  if (need_reshape) reshape();
  return (retdev != WIN_NOEVENT);
}

//
// virtual routines for preparing to draw, drawing, and finishing drawing
//

// reshape the display after a shape change
void OpenGLDisplayDevice::reshape(void) {

  // get and store size of window
  XWindowAttributes xwa;
  Window childwin;                  // not used, just needed for X call
  int rx, ry;

  // 
  // XXX WireGL notes: 
  //   WireGL doesn't have a variable window size like normal 
  // OpenGL windows do.  Not only that, but the size values reported
  // by X11 will be widly different from those reported by 
  // the glGetIntegerv(GL_VIEWPORT) call, and cause schizophrenic
  // behavior.  For now, we don't do anything about this, but 
  // the default window that comes up on the tiled display is not
  // locked to the same size and aspect ratio as the host display,
  // so spheres can look rather egg shaped if the X window on the 
  // host display isn't adjusted. 
  //

  XGetWindowAttributes(glxsrv.dpy, glxsrv.windowID, &xwa);
  XTranslateCoordinates(glxsrv.dpy, glxsrv.windowID, glxsrv.rootWindowID, -xwa.border_width,
			-xwa.border_width, &rx, &ry, &childwin);

  xSize = xwa.width;
  ySize = xwa.height;
  xOrig = rx;
  yOrig = screenY - ry - ySize;
  
  switch (inStereo) {
    case OPENGL_STEREO_SIDE:
    case OPENGL_STEREO_CROSSED:
      set_screen_pos(0.5f * (float)xSize / (float)ySize);
      break;

    case OPENGL_STEREO_STENCIL:
      enable_stencil_stereo();
      set_screen_pos((float)xSize / (float)ySize);
      break;
 
    default:
      set_screen_pos((float)xSize / (float)ySize);
      break;
  }
}

unsigned char * OpenGLDisplayDevice::readpixels(int &xs, int &ys) {
  unsigned char * img = NULL;
  xs = xSize;
  ys = ySize;

#if defined(GL_SUN_read_video_pixels)
  // read antialiased graphics back from the Sun Zulu video readback
  // path instead of from the "sample 0" pixel path that would be 
  // followed by the normal glReadPixels() code.
  if (ext->hasglreadvideopixelssun) {
    if ((img = (unsigned char *) malloc(xs * ys * 3)) != NULL) {
      GLenum err;
      GLint buffer;
 
      // change draw buffer to one of the front buffers for the
      // video readback 
      err = glGetError(); // clear any existing OpenGL error
      glGetIntegerv(GL_READ_BUFFER, &buffer);
      glReadBuffer(GL_FRONT);
      glXSwapBuffers(glxsrv.dpy, glxsrv.windowID);

      // Use the video readback extension for the Sun XVR-4000 in order
      // to take advantage of its 1-Tflop antialiasing engine.
#if 0
      // Read the gamma-corrected pixel values
      glReadVideoPixelsSUN(0, 0, xs, ys, GL_RGB, GL_UNSIGNED_BYTE, img);
#else
      // Use the special "degamma" pixel format which leaves pixels in 
      // linear intensity curves, as with normal glReadPixels().
      // Without the use of the special pixel format, we would get
      // gamma-corrected pixel values rather than linear.
      glReadVideoPixelsSUN(0, 0, xs, ys, GL_RGB_DEGAMMA_SUN, GL_UNSIGNED_BYTE, img);
#endif

      glXSwapBuffers(glxsrv.dpy, glxsrv.windowID);
      glReadBuffer(buffer); // return draw buffer to its previous state

      // check for an OpenGL error
      if ((err = glGetError()) != GL_NO_ERROR) {
        msgErr << (const char *) gluErrorString(err) << sendmsg;
        free(img);
        xs = 0;
        ys = 0;
        return NULL;
      }

      return img; 
    }
  }
#endif

  // fall back to normal glReadPixels() if better methods fail
  if ((img = (unsigned char *) malloc(xs * ys * 3)) != NULL) {
    glPixelStorei(GL_PACK_ALIGNMENT, 1);
    glReadPixels(0, 0, xs, ys, GL_RGB, GL_UNSIGNED_BYTE, img);
    return img; 
  }

  // else bail out
  xs = 0;
  ys = 0;
  return NULL;
}


// update after drawing
void OpenGLDisplayDevice::update(int do_update) {
  if (wiregl) {
    glFinish(); // force cluster to synchronize before buffer swap, 
                // this gives much better results than if the 
                // synchronization is done implicitly by glXSwapBuffers.
  }

  if(do_update)
    glXSwapBuffers(glxsrv.dpy, glxsrv.windowID);

  glDrawBuffer(GL_BACK);
}

