/***************************************************************************
 *cr
 *cr            (C) Copyright 1995-2007 The Board of Trustees of the
 *cr                        University of Illinois
 *cr                         All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: MolFilePlugin.C,v $
 *      $Author: johns $        $Locker:  $             $State: Exp $
 *      $Revision: 1.75 $      $Date: 2007/02/27 17:23:32 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *   VMD interface to 'molfile' plugins.  Molfile plugins read coordinate
 *   files, structure files, volumetric data, and graphics data.  The data
 *   is loaded into a new or potentially preexisting molecule in VMD.
 *   Some molefile plugins can also export data from VMD to files.
 ***************************************************************************/

#include <stdio.h>

#include "MolFilePlugin.h"
#include "Molecule.h"
#include "AtomSel.h"
#include "Timestep.h"
#include "Inform.h"
#include "Scene.h"
#include "molfile_plugin.h"
#include "PeriodicTable.h"
#include "VolumetricData.h"

MolFilePlugin::MolFilePlugin(vmdplugin_t *p)
: plugin((molfile_plugin_t *)p) {
  rv = wv = NULL; 
  numatoms = 0;
  _filename = NULL;
}

MolFilePlugin::~MolFilePlugin() {
  close();
  delete [] _filename;
}

int MolFilePlugin::read_structure(Molecule *m, int filebonds, int autobonds) {
  if (!rv) return MOLFILE_ERROR; 
  if (!can_read_structure()) return MOLFILE_ERROR;
  if (!m->init_atoms(numatoms)) return MOLFILE_ERROR;
  molfile_atom_t *atoms = (molfile_atom_t *)malloc(
      numatoms*sizeof(molfile_atom_t));

  int rc, i;
  int optflags = MOLFILE_BADOPTIONS; /* plugin must reset this correctly */
  if ((rc = plugin->read_structure(rv, &optflags, atoms))) {
    free(atoms);
    return rc; // propagate error to caller
  }
  if (optflags == MOLFILE_BADOPTIONS) {
    free(atoms);
    msgErr << "MolFilePlugin: plugin didn't initialize optional data flags" << sendmsg;
    msgErr << "MolFilePlugin: file load aborted" << sendmsg;
    return MOLFILE_ERROR; /* abort load, data can't be trusted */
  }


  float *charge = m->charge();
  float *mass = m->mass();
  float *radius = m->radius();
  float *beta = m->beta();
  float *occupancy = m->occupancy();

  for (i=0; i<numatoms; i++) {
    molfile_atom_t &atom = atoms[i];
    int atomicnumber = (optflags & MOLFILE_ATOMICNUMBER) ? 
      atom.atomicnumber : -1;
    charge[i] = (optflags & MOLFILE_CHARGE) ?
      atom.charge : m->default_charge(atom.name);
 
    // If we're given an explicit mass value, use it.
    // Failing that, try doing a periodic table lookup if we have
    // a valid atomicnumber.  If that fails also, then we use a crude
    // guess based on the atom name string. 
    mass[i] = (optflags & MOLFILE_MASS) ?
      atom.mass : ((atomicnumber > 0) ?
        get_pte_mass(atomicnumber) : m->default_mass(atom.name));

    // If we're given an explicit VDW radius value, use it.
    // Failing that, try doing a periodic table lookup if we have
    // a valid atomicnumber.  If that fails also, then we use a crude
    // guess based on the atom name string. 
    radius[i] = (optflags & MOLFILE_RADIUS) ?
      atom.radius : ((atomicnumber > 0) ?
        get_pte_vdw_radius(atomicnumber) : m->default_radius(atom.name));

    beta[i] = (optflags & MOLFILE_BFACTOR) ?
      atom.bfactor : m->default_beta();
    occupancy[i] = (optflags & MOLFILE_OCCUPANCY) ?
      atom.occupancy : m->default_occup();
    const char *insertion = (optflags & MOLFILE_INSERTION) ? 
      atom.insertion : " ";
    const char *altloc = (optflags & MOLFILE_ALTLOC) ?
      atom.altloc : "";
    if (0 > m->add_atom(atom.name, atom.type, atomicnumber, 
                        atom.resname, atom.resid, 
                        atom.chain, atom.segid, (char *)insertion, altloc)) {
      // if an error occured while adding an atom, we should delete
      // the offending molecule since the data is presumably inconsistent,
      // or at least not representative of what we tried to load
      msgErr << "MolFilePlugin: file load aborted" << sendmsg;
      free(atoms);
      return MOLFILE_ERROR; // abort load, data can't be trusted
    }
  }
  free(atoms);

  if (filebonds && can_read_bonds()) {
    int nbonds, *from, *to; 
    float *bondorder;
    if (plugin->read_bonds(rv, &nbonds, &from, &to, &bondorder)) {
      msgErr << "Error reading bond information." << sendmsg;
      if (autobonds)
        m->find_bonds_from_timestep();
    } else {
      // if we didn't get an error, but the plugin didn't define bonds
      // for some reason (i.e. no usable PDB CONECT records found)
      // fall back to automatic bond search
      if (nbonds == 0 && from == NULL && to == NULL) {
        if (autobonds)
          m->find_bonds_from_timestep();
      } else if (nbonds > 0) {
        // If the bonds provided by the plugin were only the for
        // non-standard residues (i.e. PDB CONECT records) then we'll
        // also do our regular bond search, combining all together, but 
        // preventing duplicates
        if (optflags & MOLFILE_BONDSSPECIAL) {
          if (autobonds) {
            m->find_unique_bonds_from_timestep(); // checking for duplicates
          }

          // Now add in all of the special bonds provided by the plugin
          if (bondorder != NULL) {
            for (i=0; i<nbonds; i++)
              m->add_bond_dupcheck(from[i]-1, to[i]-1, bondorder[i]); // real bond order
          } else {
            for (i=0; i<nbonds; i++)
              m->add_bond_dupcheck(from[i]-1, to[i]-1, 1); // default to bond order of 1
          }
        } else {
          if (bondorder != NULL) {
            for (i=0; i<nbonds; i++)
              m->add_bond(from[i]-1, to[i]-1, bondorder[i]); // real bond order
          } else {
            for (i=0; i<nbonds; i++)
              m->add_bond(from[i]-1, to[i]-1, 1); // default to bond order of 1
          }
        }
      }
    }
  } else {
    if (autobonds)
      m->find_bonds_from_timestep();
  }

  // if the plugin can read angles, dihedrals, impropers, cross-terms, 
  // do it here...
#if vmdplugin_ABIVERSION > 9
  // XXX not implemented yet
#endif

  return MOLFILE_SUCCESS;
}
  
int MolFilePlugin::read_optional_structure(Molecule *m, int filebonds) {
  if (!rv) return MOLFILE_ERROR; 
  if (!can_read_structure()) return MOLFILE_ERROR;
  if (numatoms != m->nAtoms) return MOLFILE_ERROR;

  molfile_atom_t *atoms = (molfile_atom_t *)malloc(
      numatoms*sizeof(molfile_atom_t));

  int rc, i;
  int optflags = MOLFILE_BADOPTIONS; /* plugin must reset this correctly */
  if ((rc = plugin->read_structure(rv, &optflags, atoms))) {
    free(atoms);
    return rc; // propagate error to caller
  }
  if (optflags == MOLFILE_BADOPTIONS) {
    free(atoms);
    msgErr << "MolFilePlugin: plugin didn't initialize optional data flags" << sendmsg;
    msgErr << "MolFilePlugin: file load aborted" << sendmsg;
    return MOLFILE_ERROR; /* abort load, data can't be trusted */
  }


  float *charge = m->charge();
  float *mass = m->mass();
  float *radius = m->radius();
  float *beta = m->beta();
  float *occupancy = m->occupancy();
  for (i=0; i<numatoms; i++) {
    if (optflags & MOLFILE_CHARGE)
      charge[i] = atoms[i].charge;
    if (optflags & MOLFILE_MASS)
      mass[i] = atoms[i].mass;
    if (optflags & MOLFILE_RADIUS)
      radius[i] = atoms[i].radius;
    if (optflags & MOLFILE_OCCUPANCY)
      occupancy[i] = atoms[i].occupancy;
    if (optflags & MOLFILE_BFACTOR)
      beta[i] = atoms[i].bfactor;
    if (optflags & MOLFILE_ATOMICNUMBER)
      m->atom(i)->atomicnumber = atoms[i].atomicnumber;
  }
  free(atoms);

  // if no bonds are added, then we do not re-analyze the structure
  if (!can_read_bonds()) 
    return MOLFILE_SUCCESS;

  // When tacking on trajectory frames from PDB files with CONECT records,
  // we have to prevent partial CONECT record bonding information from
  // blowing away existing connectivity information derived from 
  // automatic bond search results or from a previously loaded file with
  // complete bond information such as a PSF.  We only accept 
  // complete bonding connectivity updates, no partial updates are allowed.
  // Also bonding information updates can be disabled by the user with the
  // filebonds flag.
  if (!(optflags & MOLFILE_BONDSSPECIAL) && filebonds) {
    int nbonds, *from, *to; 
    float *bondorder;
    if (plugin->read_bonds(rv, &nbonds, &from, &to, &bondorder)) 
      return MOLFILE_SUCCESS;

    if (nbonds == 0) 
      return MOLFILE_SUCCESS;

    for (i=0; i<numatoms; i++) m->atom(i)->bonds = 0;
    if (bondorder != NULL) {
      for (i=0; i<nbonds; i++)
        m->add_bond(from[i]-1, to[i]-1, bondorder[i]); // real bond order
    } else {
      for (i=0; i<nbonds; i++)
        m->add_bond(from[i]-1, to[i]-1, 1); // default to bond order of 1
    }
  } else {
    // if no bonds are added, then we do not re-analyze the structure
    return MOLFILE_SUCCESS;
  }

  // (re)analyze the molecular structure, since bonds may have been changed
  m->analyze();

  // force all reps and selections to be recalculated
  m->force_recalc(DrawMolItem::COL_REGEN | DrawMolItem::SEL_REGEN);

  // force secondary structure to be recalculated
  m->invalidate_ss(); 
 
  return MOLFILE_SUCCESS;
}

void MolFilePlugin::set_natoms(int n) {
  numatoms = n;
}

int MolFilePlugin::init_read(const char *file) {
  rv = NULL;
  if (can_read_structure() || can_read_timesteps() || can_read_graphics() ||
      can_read_volumetric() || can_read_metadata()) { 
    rv = plugin->open_file_read(file, plugin->name, &numatoms);
  }
  if (!rv) return MOLFILE_ERROR;
  delete [] _filename;
  _filename = stringdup(file);
  return MOLFILE_SUCCESS;
}

Timestep *MolFilePlugin::next() {
  if (!rv) return NULL;
  if (numatoms <= 0) return NULL;
  if (!can_read_timesteps()) return NULL;
  molfile_timestep_t timestep;
  memset(&timestep, 0, sizeof(molfile_timestep_t));

  // set useful defaults for unit cell information
  // a non-periodic structure has cell lengths of zero
  // if it's periodic in less than 3 dimensions, then only the
  // periodic directions will be non-zero.
  timestep.A = timestep.B = timestep.C = 0.0f;

  // cells are rectangular until told otherwise
  timestep.alpha = timestep.beta = timestep.gamma = 90.0f;

  Timestep *ts = new Timestep(numatoms);
  timestep.coords = ts->pos; 
  int rc = plugin->read_next_timestep(rv, numatoms, &timestep);
  if (rc) {
    delete ts;
    ts = NULL; 
  } else {
    ts->a_length = timestep.A;
    ts->b_length = timestep.B;
    ts->c_length = timestep.C;
    ts->alpha = timestep.alpha;
    ts->beta = timestep.beta;
    ts->gamma = timestep.gamma;
  }
  return ts;
}

int MolFilePlugin::skip() {
  if (!rv) return MOLFILE_ERROR;
  if (!can_read_timesteps()) return MOLFILE_ERROR;
  return plugin->read_next_timestep(rv, numatoms, 0);
}

void MolFilePlugin::close() {
  if (rv && (can_read_structure() || can_read_timesteps() || 
             can_read_graphics() || can_read_volumetric() ||
             can_read_metadata() || can_read_bonds())) { 
    plugin->close_file_read(rv);
    rv = NULL;
  }
  if (wv && (can_write_structure() || can_write_timesteps() ||
             can_write_bonds())) { 
    plugin->close_file_write(wv);
    wv = NULL;
  }
}

int MolFilePlugin::init_write(const char *file, int natoms) {
  wv = NULL;
  if (can_write_structure() || can_write_timesteps() 
#if vmdplugin_ABIVERSION > 9
          || can_write_volumetric()
#endif
          ) { 
    wv = plugin->open_file_write(file, plugin->name, natoms);
  } 
  if (!wv) return MOLFILE_ERROR;
  delete [] _filename;
  _filename = stringdup(file);

  // Cache the number of atoms to be written.  It's not strictly necessary,
  // but it lets us allocate only the necessary space in write_structure
  // and write_timestep.
  numatoms = natoms;

  return MOLFILE_SUCCESS;
}

int MolFilePlugin::write_structure(Molecule *m, const int *on) {
  if (!wv) return MOLFILE_ERROR;
  if (!can_write_structure()) return MOLFILE_ERROR;
  if (!m->has_structure()) {
    msgErr << "Molecule's structure has not been initialized." << sendmsg;
    return MOLFILE_ERROR;
  }
  
  int i, j, k;
  molfile_atom_t *atoms = (molfile_atom_t *)malloc(numatoms*sizeof(molfile_atom_t));
  int *atomindexmap = (int *) malloc(m->nAtoms*sizeof(int));

  // initialize the atom index map to an invalid atom index value, so that
  // we can use this to eliminate bonds to atoms that aren't selected.
  for (i=0; i<m->nAtoms; i++) 
    atomindexmap[i] = -1;

  const float *charge = m->charge();
  const float *mass = m->mass();
  const float *radius = m->radius();
  const float *beta = m->beta();
  const float *occupancy = m->occupancy();

  // build the array of selected atoms to be written out
  for (i=0, j=0; i<m->nAtoms; i++) {
    // skip atoms that aren't 'on', if we've been given an 'on' array
    if (on && !on[i]) 
      continue;

    // Check that the number of atoms specified in init_write is no smaller
    // than the number of atoms in the selection; otherwise we would 
    // corrupt memory.
    if (j >= numatoms) {
      msgErr << 
        "MolFilePlugin: Internal error, selection size exceeds numatoms ("
        << numatoms << ")" << sendmsg;
      free(atoms);
      free(atomindexmap);
      return MOLFILE_ERROR;
    }

    const MolAtom *atom = m->atom(i);

    // Try to restore the spacing on the name since VMD destroys it when it
    // reads it in.
    // XXX this is PDB-centric thinking, need to reconsider this
    char name[6], *nameptr;
    name[0] = ' ';
    strncpy(name+1, (m->atomNames).name(atom->nameindex), 4);
    name[5] = '\0';
    // the name must be left-justified
    if(strlen(name) == 5) {
      nameptr = name + 1;
    } else {
      nameptr = name;
      int p;
      while((p = strlen(name)) < 4) {
        name[p] = ' ';
        name[p+1] = '\0';
      }
    }

    molfile_atom_t &atm = atoms[j];
    strcpy(atm.name, nameptr);
    strcpy(atm.type, m->atomTypes.name(atom->typeindex));
    strcpy(atm.resname, m->resNames.name(atom->resnameindex));
    atm.resid = atom->resid;
    strcpy(atm.chain, m->chainNames.name(atom->chainindex));
    strcpy(atm.segid, m->segNames.name(atom->segnameindex));
    strcpy(atm.insertion, atom->insertionstr);
    strcpy(atm.altloc, m->altlocNames.name(atom->altlocindex));
    atm.atomicnumber = atom->atomicnumber;
    atm.occupancy = occupancy[i];
    atm.bfactor = beta[i];
    atm.mass = mass[i]; 
    atm.charge = charge[i]; 
    atm.radius = radius[i];

    atomindexmap[i] = j; // build index map for bond determination

    j++;
  }

  // check that the selection size matches numatoms
  if (j != numatoms) {
    msgErr 
      << "MolFilePlugin: Internal error, selection size (" << j 
      << ") doesn't match numatoms (" << numatoms << ")" << sendmsg;
    free (atoms);
    return MOLFILE_ERROR;
  }

  // set optional data fields we're providing
  int optflags = MOLFILE_INSERTION | MOLFILE_OCCUPANCY | 
    MOLFILE_BFACTOR | MOLFILE_MASS | MOLFILE_CHARGE |
    MOLFILE_RADIUS | MOLFILE_ALTLOC | MOLFILE_ATOMICNUMBER;

  // build and save a bond list if this plugin has a write_bonds() callback
  if (can_write_bonds()) {
    ResizeArray<int> bondfrom, bondto; 
    ResizeArray<float> bondorder;

    for (i=0; i<m->nAtoms; i++) {
      const MolAtom *atom = m->atom(i);
      int bfmap = atomindexmap[i] + 1; // 1-based mapped atom index

      for (k=0; k<atom->bonds; k++) {
        int bto = atom->bondTo[k];
        int btmap = atomindexmap[bto] + 1; // 1-based mapped atom index
 
        // add 1-based bonds to 'on' atoms to the bond list
        if (bfmap > 0 && btmap > bfmap) {
          bondfrom.append(bfmap);
          bondto.append(btmap);
          bondorder.append(m->getbondorder(i, k)); 
        }
      }
    } 

    if (plugin->write_bonds(wv, bondfrom.num(), &bondfrom[0], &bondto[0], &bondorder[0])) {
      free(atoms);
      free(atomindexmap);
      return MOLFILE_ERROR;
    }
  }

  // write the structure
  if (plugin->write_structure(wv, optflags, atoms)) {
    free(atoms);
    free(atomindexmap);
    return MOLFILE_ERROR;
  }

  free(atoms);
  free(atomindexmap);

  return MOLFILE_SUCCESS;
}
 
int MolFilePlugin::write_timestep(const Timestep *ts, const int *on) {
  // it isn't an error if this file format doesn't write timesteps
  if (!can_write_timesteps()) 
    return MOLFILE_SUCCESS; 

  molfile_timestep_t mol_ts;
  memset(&mol_ts, 0, sizeof(molfile_timestep_t));
  mol_ts.A = ts->a_length;
  mol_ts.B = ts->b_length;
  mol_ts.C = ts->c_length;
  mol_ts.alpha = ts->alpha;
  mol_ts.beta = ts->beta;
  mol_ts.gamma = ts->gamma;
  if (!on) {
    mol_ts.coords = ts->pos;
    return plugin->write_timestep(wv, &mol_ts);
  }
  float *coords = new float[3*numatoms];
  int j=0;
  for (int i=0; i<ts->num; i++) {
    if (on[i]) {
      if (on && !on[i]) continue;
      // check that the selection doesn't contain too many atoms
      if (j >= 3*numatoms) {
        msgErr << "MolFilePlugin::write_timestep: Internal error" << sendmsg;
        msgErr << "Selection size exceeds numatoms (" << numatoms << ")" 
               << sendmsg;
        delete [] coords;
        return MOLFILE_ERROR;
      }
      coords[j++] = ts->pos[3*i];
      coords[j++] = ts->pos[3*i+1];
      coords[j++] = ts->pos[3*i+2];
    }
  }
  // check that the selection size matches numatoms
  if (j != 3*numatoms) {
    msgErr << "MolFilePlugin::write_timestep: Internal error" << sendmsg;
    msgErr << "selection size (" << j << ") doesn't match numatoms (" 
           << numatoms << ")" << sendmsg;
    delete [] coords;
    return MOLFILE_ERROR;
  }
  mol_ts.coords = coords;
  int rc = plugin->write_timestep(wv, &mol_ts);
  delete [] coords;
  return rc;
}
 
int MolFilePlugin::read_rawgraphics(Molecule *m, Scene *sc) {
  if (!rv || !can_read_graphics()) return MOLFILE_ERROR;
  const molfile_graphics_t *graphics = NULL;
  int nelem = -1;
  if (plugin->read_rawgraphics(rv, &nelem, &graphics)) return MOLFILE_ERROR;
  msgInfo << "Reading " << nelem << " graphics elements..." << sendmsg;
  for (int i=0; i<nelem; i++) {
    const float *data = graphics[i].data;
    switch (graphics[i].type) {
      case MOLFILE_POINT: 
        m->add_point(data); 
        break;
      case MOLFILE_TRIANGLE:
        m->add_triangle(data, data+3, data+6);
        break;
      case MOLFILE_TRINORM: 
        {
          const float *ndata;
          // next element must be the norms
          if (i+1 >= nelem || graphics[i+1].type != MOLFILE_NORMS) {
            msgErr << "Invalid rawgraphics: NORMS must follow TRINORM."
              << sendmsg;
            return MOLFILE_ERROR;
          }
          ++i;
          ndata = graphics[i].data;
          m->add_trinorm(data, data+3, data+6, ndata, ndata+3, ndata+6);
        }
        break;
      case MOLFILE_TRICOLOR: 
        {
          const float *ndata, *cdata;
          // next element must be the norms
          if (i+1 >= nelem || graphics[i+1].type != MOLFILE_NORMS) {
            msgErr << "Invalid rawgraphics: NORMS must follow TRINORM."
              << sendmsg;
            return MOLFILE_ERROR;
          }
          ++i;
          ndata = graphics[i].data;
          // next element must be the vertex colors
          if (i+1 >= nelem || graphics[i+1].type != MOLFILE_COLOR) {
            msgErr << "Invalid rawgraphics: NORMS and COLOR must fullow TRICOLOR."
              << sendmsg;
            return MOLFILE_ERROR;
          }
          ++i;
          cdata = graphics[i].data;
          m->add_tricolor(data, data+3, data+6, ndata, ndata+3, ndata+6,
              sc->nearest_index(cdata[0], cdata[1], cdata[2]), 
              sc->nearest_index(cdata[3], cdata[4], cdata[5]), 
              sc->nearest_index(cdata[6], cdata[7], cdata[8]));
        }
        break;
      case MOLFILE_LINE:
        m->add_line(data, data+3, graphics[i].style, (int)graphics[i].size);
        break;
      case MOLFILE_CYLINDER:
        m->add_cylinder(data, data+3, graphics[i].size, graphics[i].style, 0);
        break;
      case MOLFILE_CAPCYL:
        m->add_cylinder(data, data+3, graphics[i].size, graphics[i].style, 1);
        break;
      case MOLFILE_CONE:
        m->add_cone(data, data+3, graphics[i].size, data[6], graphics[i].style);
        break;
      case MOLFILE_SPHERE:
        m->add_sphere(data, graphics[i].size, graphics[i].style);
        break;
      case MOLFILE_TEXT:
        {
          char text[24];
          strncpy(text, (char *)data+3, 24);
          text[23] = '\0';
          m->add_text(data, text, graphics[i].size);
        }
        break;
      case MOLFILE_COLOR:
        m->use_color(sc->nearest_index(data[0], data[1], data[2]));
        break;
      case MOLFILE_NORMS:
        msgErr << "Invalid rawgraphics: NORMS must follow TRINORM." << sendmsg;
        return MOLFILE_ERROR;
        break;
      default:
        msgErr << "Invalid rawgraphics: unknown type " << graphics[i].type
               << sendmsg;
    }
  }

  return MOLFILE_SUCCESS;
}


int MolFilePlugin::read_volumetric(Molecule *mol, int nsets, 
    const int *setids) {

  // Fetch metadata from file
  molfile_volumetric_t *metadata;
  int setsinfile = 0;
  plugin->read_volumetric_metadata(rv, &setsinfile, &metadata);

  // Get datasets specified in setids
  int n;
  int *sets;
  if (nsets < 0) {
    n = setsinfile;
    sets = new int[n];
    for (int i=0; i<n; i++) sets[i] = i;
  } else {
    n = nsets;
    sets = new int [n];
    for (int i=0; i<n; i++) sets[i] = setids[i];
  }

  for (int i=0; i< n; i++) {
    if (sets[i] < 0 || sets[i] >= setsinfile) {
      msgErr << "Bogus setid passed to read_volumetric: " << sets[i]
             << sendmsg;
      continue;
    }  
    const molfile_volumetric_t *v = metadata+sets[i];
    float *datablock = NULL, *colorblock = NULL;
    size_t size = v->xsize * v->ysize * v->zsize;
    datablock = new float[size];
    if (v->has_color) 
      colorblock = new float[3*size];
    if (plugin->read_volumetric_data(rv, sets[i], datablock, colorblock)) {
      msgErr << "Error reading volumetric data set " << sets[i]+1 << sendmsg;
      delete [] datablock;
      delete [] colorblock;
      continue;
    }
    char *dataname = stringdup(v->dataname);
    if (_filename) {
      // prepend the filename to the dataname; otherwise it's hard to tell
      // multiple data sets apart in the GUI.  This should be done here,
      // within VMD, rather than within each plugin because otherwise 
      // different plugins will end up exhibiting different behavior with
      // regard to naming their datasets.
      //
      // XXX The breakup_filename command uses forward slashes only and
      // is therefore Unix-specific.  Also, to avoid super-long dataset
      // names I'm going to use just the 'basename' part of the file.
      // It's easier just to code a correct version of what I want here.
      char sep = 
#ifdef WIN32
        '\\'
#else
        '/'
#endif
        ;
      const char *basename = strrchr(_filename, sep);
      if (!basename) {
        basename = _filename;
      } else {
        basename++; // skip the separator
      }
      char *tmp = new char[strlen(dataname)+5+strlen(basename)];
      sprintf(tmp, "%s : %s", basename, dataname);
      delete [] dataname;
      dataname = tmp;
    }
    mol->add_volume_data(dataname, v->origin,
         v->xaxis, v->yaxis, v->zaxis, v->xsize, v->ysize, v->zsize,
         datablock);
    delete [] dataname;
    delete [] colorblock;
  }
  delete [] sets;

  return MOLFILE_SUCCESS;
}


int MolFilePlugin::read_metadata(Molecule *mol) {
  // Fetch metadata from file
  molfile_metadata_t *metadata;

  plugin->read_molecule_metadata(rv, &metadata);

  mol->record_database(metadata->database, metadata->accession);
  if (metadata->remarks != NULL) 
    mol->record_remarks(metadata->remarks);
  else 
    mol->record_remarks("");

  return MOLFILE_SUCCESS;
}


#if vmdplugin_ABIVERSION > 9
int MolFilePlugin::read_qm_data(Molecule *mol) {
  // Fetch metadata from file
  molfile_qm_metadata_t metadata;
  plugin->read_qm_metadata(rv, &metadata);

  molfile_qm_t qmdata;  
  plugin->read_qm_rundata(rv, &qmdata);

  return MOLFILE_SUCCESS;
}


int MolFilePlugin::write_volumetric(Molecule *mol, int set) {
  if (set < 0 || set > mol->num_volume_data()) {
    msgErr << "Bogus setid passed to write_volumetric: " << set
           << sendmsg;
    return MOLFILE_SUCCESS;
  } 

  const VolumetricData *v = mol->get_volume_data(set); 

  molfile_volumetric_t volmeta;
  strncpy(volmeta.dataname, v->name, sizeof(volmeta.dataname)); 
  for (int i=0; i<3; i++) {
      volmeta.origin[i] = (float)v->origin[i];
      volmeta.xaxis[i] = (float)v->xaxis[i];
      volmeta.yaxis[i] = (float)v->yaxis[i];
      volmeta.zaxis[i] = (float)v->zaxis[i];
  }
  volmeta.xsize = v->xsize;
  volmeta.ysize = v->ysize;
  volmeta.zsize = v->zsize;
  volmeta.has_color = 0;
 
  float *datablock = v->data;
  float *colorblock = NULL;

  plugin->write_volumetric_data(wv, &volmeta, datablock, colorblock);
 
  return MOLFILE_SUCCESS;
}
#endif

