/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995-2007 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: DrawRingsUtils.C,v $
 *	$Author: johns $	$Locker:  $		$State: Exp $
 *	$Revision: 1.2 $	$Date: 2007/01/12 20:08:22 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Ulities for calculating ring axes, ring puckering and displacement of
 * atoms from the mean ring plane.
 *
 ***************************************************************************/

#include "DrawRingsUtils.h"

void vec_incr(double sum[3], double x, double y, double z) {
	sum[0] += x;
	sum[1] += y;
	sum[2] += z;
}

// c = a x b 
void crossp( double a[3], double b[3], double c[3]) {
	c[0] = a[1]*b[2] - a[2]*b[1];
	c[1] = a[2]*b[0] - a[0]*b[2];
	c[2] = a[0]*b[1] - a[1]*b[0];
}

// return a . b
double dotp(double a[3], double b[3]) {
	return a[0]*b[0] + a[1]*b[1] + a[2]*b[2];
}

// return (X,Y,Z) . b
double dotp(double X, double Y, double Z, double b[3]) {
	return X*b[0] + Y*b[1] + Z*b[2];
}

// return vec / sqrt( vec . vec)
void norm(double vec[3]) {
	double v0 = vec[0]; double v1 = vec[1]; double v2 = vec[2];
	double mag = sqrt(v0*v0 + v1*v1 + v2*v2);
	vec[0] /= mag; vec[1] /= mag; vec[2] /= mag;
}

// XXX: Write a useful description
void ring_axes(const double * X, const double * Y, const double * Z, int N, 
	           double x[3], double y[3], double z[3] ) {

	double Rp[3] = {0.0, 0.0, 0.0}; double Rpp[3] = {0.0, 0.0, 0.0};

	for ( int j=0; j<N; j++ ) {
		double ze_angle = 2.0 * M_PI * double(j-1) / double(N);
		double ze_sin = sin(ze_angle);
		double ze_cos = cos(ze_angle);
		vec_incr( Rp, X[j]*ze_sin, Y[j]*ze_sin, Z[j]*ze_sin );
		vec_incr( Rpp, X[j]*ze_cos, Y[j]*ze_cos, Z[j]*ze_cos );
	}

	crossp ( Rp, Rpp, z );
	norm ( z );
	
	/* 
	 * OK, now we have z, the norm to the central plane, we need
	 * to calculate y as the projection of Rp onto the plane
	 * and x as y x z
	 */
	double lambda = dotp( z, Rp );
	y[0] = Rp[0] - z[0]*lambda;
	y[1] = Rp[1] - z[1]*lambda;
	y[2] = Rp[2] - z[2]*lambda;
	norm ( y );
	crossp ( y, z, x ); 	// voila !
}


// Calculate distances of atoms from the mean ring plane
void atom_displ_from_mean_plane(double * X, double * Y, double * Z, 
				                double * displ, int N) {
	double cog[3] = {0.0,0.0,0.0};
	double x_axis[3], y_axis[3], z_axis[3];
    int i;
	
	// calculate centre of geometry
	for (i=0; i<N; i++) {
		cog[0] += X[i]; cog[1] += Y[i]; cog[2] += Z[i];
	}
	cog[0] /= double(N); 
	cog[1] /= double(N); 
	cog[2] /= double(N); 
	
	// centre the ring
	for (i=0; i<N; i++) {
		X[i] -= cog[0]; Y[i] -= cog[1]; Z[i] -= cog[2];
	}

	ring_axes( X, Y, Z, N, x_axis, y_axis, z_axis );

	//calculate displacement from mean plane
	for (i=0; i<N; i++) {
	  displ[i]=dotp(X[i],Y[i],Z[i],z_axis);
	}
	
}

// Calculate puckering parameters
int pucker_param(int N_ring_atoms, double * displ, double * q, 
				 double * phi, int  & m , double & Q)
{
  if(N_ring_atoms<3)
    return(-1);

  double N = double(N_ring_atoms);
  phi[0]=0;  q[0]=0;  //no puckering parameters for m=1

  //if even no ring atoms, first calculate unpaired q puck parameter
   if(fmod(N,2.0)==0)
     {
       double sum =0;
       m = N_ring_atoms/2 -1;
       for(int i= 0;i<N_ring_atoms; i++)         
	       sum += displ[i]*cos(i*M_PI);
       q[int(N_ring_atoms/2)-1]=sqrt(1.0/N)*sum;
     }
  else 
    {
      m = int(N_ring_atoms-1)/2;
    }

  //calculate paired puckering parameters
   for(int i=1;i<m;i++)
      {
		double q_cosphi=0, q_sinphi=0, tanphi=0;
		for(int j=0;j<N_ring_atoms;j++)
		{
			q_cosphi +=displ[j]*cos(2.0*M_PI*double((i+1)*j)/N);
			 q_sinphi +=displ[j]*sin(2.0*M_PI*double((i+1)*j)/N);
		}

		q_cosphi *= sqrt(2.0/N);
		q_sinphi *= -sqrt(2.0/N);
		tanphi = q_sinphi/q_cosphi;

		phi[i]=atan(tanphi);
			       
		if(q_cosphi<0)
			phi[i]+=M_PI;
		else if(q_sinphi <0) 
		  phi[i]+=2*M_PI;

		q[i]=q_cosphi/phi[i];
		//phi[i]*=180.0/M_PI;  //convert to degrees

     //calculate puckering amplitude
		Q=0.0;
		for(int i=0;i<N_ring_atoms;i++)
		  Q+=displ[i]*displ[i];
		Q=sqrt(Q);  
	  }
   return 1;
}
