/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995-2007 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: DrawMolecule.h,v $
 *	$Author: johns $	$Locker:  $		$State: Exp $
 *	$Revision: 1.69 $	$Date: 2007/01/12 20:08:22 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Displayable version of a molecule, derived from BaseMolecule and
 * Displayable.  This contains all the info for rendering the molecule.
 *
 ***************************************************************************/
#ifndef DRAWMOLECULE_H
#define DRAWMOLECULE_H

#include "BaseMolecule.h"
#include "Displayable.h"
#include "DrawMolItem.h"
#include "ResizeArray.h"

class AtomColor;
class AtomRep;
class AtomSel;
class VMDApp;
class MoleculeGraphics;
class DrawForce;

/// A monitor class that acts as a proxy for things like labels that
/// have to be notified when molecules change their state.  
class DrawMoleculeMonitor {
public:
  // called with id of molecule
  virtual void notify(int) = 0;
  DrawMoleculeMonitor() {}
  virtual ~DrawMoleculeMonitor() {}
};


/// Subclass of BaseMolecule and Displayable for drawing a molecule
class DrawMolecule : public BaseMolecule, public Displayable {
public:
  int active;     ///< is this molecule active?  Used by MoleculeList.

protected:
  VMDApp *app;

private:
  int repcounter; ///< counter for giving unique names to reps.

  /// a MoleculeGraphics instance, which handles custom graphics primitives
  /// added to this molecule.
  MoleculeGraphics *molgraphics;

  /// a DrawForce instance for drawing force arrows
  DrawForce *drawForce;

  /// representations
  ResizeArray<DrawMolItem *> repList;

  /// the Animation object for this molecule
  AnimationData<Timestep *> *mol_anim;

  /// calculation of the secondary structure is done on the fly, but
  /// only if I need it do I run STRIDE
  int did_secondary_structure;

  /// list of registered monitors
  ResizeArray<DrawMoleculeMonitor *> monitorlist;

  /// Tell reps that we added or removed a timestep.  
  void addremove_ts();

public:
  /// constructor ... pass in a VMDApp * and a parent displayable. 
  DrawMolecule(VMDApp *, Displayable *);
  virtual ~DrawMolecule(void);
  
  //
  // public utility routines
  //
  
  /// return whether the Nth atom is displayed.  This is true if ANY
  /// representation is displaying the given atom
  int atom_displayed(int);

  //
  // access routines for the drawn molecule components (which are the children)
  //
  
  /// total number of components
  int components(void) { return repList.num(); }
  
  /// return Nth component ... change to proper return type
  DrawMolItem *component(int);

  /// return the component that matches the given Pickable, or NULL if no match
  DrawMolItem *component_from_pickable(const Pickable *);

  /// retrieve the index of the component with the given name.  Returns -1 on 
  /// failure.
  int get_component_by_name(const char *);
 
  /// Get the name of the given component.  Names are initially "repN", where
  /// N starts at 0 and increases each time a rep is created.  Return NULL
  /// if the index is invalid.
  const char *get_component_name(int);
  

  /// delete the Nth representation ... return success
  int del_rep(int);

  /// Add a new representation (component).  This always succeeds,
  /// since all parameters must have already been verified as valid.
  /// The rep takes over ownership of the parameter objects (except Material).
  void add_rep(AtomColor *, AtomRep *, AtomSel *, const Material *);

  /// change the Nth representation ... return success.
  /// if any object is NULL, that characteristic is not changed.
  virtual int change_rep(int, AtomColor *, AtomRep *, const char *sel);

  /// force a recalc of all representations
  virtual void force_recalc(int);

  /// Tell reps that the periodic image parameters have been changed
  void change_pbc();

  /// Tell reps that the currently displayed timestep has changed
  void change_ts();

  /// Return the highlighted rep for this molecule.  Returns -1 if there is
  /// no such rep.
  int highlighted_rep() const;

  //
  // methods for dealing with frames
  //

  Animation *anim(void) { return mol_anim; }

  /// get the current frame
  Timestep *current(void) { return mol_anim->current(); }

  /// get the specifed frame
  Timestep *get_frame(int n) { return mol_anim->item(n);  }

  /// get the last frame
  Timestep *get_last_frame(void) {
    return mol_anim->item(mol_anim->num()-1);
  }

  /// delete the nth frame
  void delete_frame(int n);

  /// append the given frame
  void append_frame(Timestep *);

  /// duplicate the given frame
  /// passing NULL adds a 'null' frame (i.e. all zeros)
  void duplicate_frame(const Timestep *);

  /// scaling factor required to make the molecule fit within (-1 ... 1)
  virtual float scale_factor(void);

  /// center of volume of this molecule.  Return success.  Fails if there 
  /// aren't any coordinates, graphics, or volumetric data sets to compute
  /// the cov from.
  virtual int cov(float &, float &, float &);

  /// recalculate bonds via distance bond search based on current timestep
  int recalc_bonds(void);

  /// request ss calculation 
  int need_secondary_structure(int);

  /// invalidate current secondary structure when structure is changed
  void invalidate_ss();

  /// recalculate the secondary structure using current coordinates
  int recalc_ss();

  //
  // Add custom graphics to the molecule.   These all get delegated to the
  // internal MoleculeGraphics instance for now.
  //

  // return pointer to molgraphics so that MoleculeList::check_pickable
  // can test for it.
  MoleculeGraphics *moleculeGraphics() const { return molgraphics; }

  /// manipulate the data values
  int add_point(const float *x);
  int add_pickpoint(const float *x);
  int add_triangle(const float *x1, const float *x2, const float *x3);
  int add_trinorm(const float *x1, const float *x2, const float *x3,
                  const float *nx1, const float *nx2, const float *nx3);
  int add_tricolor(const float *x1, const float *x2, const float *x3,
                  const float *nx1, const float *nx2, const float *nx3,
                  int c1, int c2, int c3);
  int add_line(const float *x, const float *y, int line_style, int width);
  int add_cylinder(const float *x, const float *y,
                   float radius, int res, int filled);
  int add_cone(const float *x, const float *y,
               float radius, float radius2, int res);
  int add_sphere(const float *x, float r, int res);
  int add_text(const float *x, const char *text, float size);

  int use_materials(int yes_no);
  int use_color(int index);
  int use_material(const Material *);

  void delete_id(int id); // delete an entry
  void delete_all(void); // delete everything
  int replace_id(int id); // same as delete, but next addition goes here
  int index_id(int id);  // returns -1 if doesn't exist, else the index
  int num_elements(void);
  int element_id(int index);

  /// return data to make shape; static string space is always used
  /// return NULL if doesn't exist or is NONE
  const char *info_id(int id);
  
  virtual void prepare();

  /// register monitors
  void register_monitor(DrawMoleculeMonitor *);

  /// unregister monitors
  void unregister_monitor(DrawMoleculeMonitor *);

  void notify();
};

#endif

