/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995-2007 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: DrawMolecule.C,v $
 *	$Author: johns $	$Locker:  $		$State: Exp $
 *	$Revision: 1.123 $	$Date: 2007/01/12 20:08:22 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Displayable version of a DrawMolecule, derived from BaseMolecule and
 * Displayable.  This contains all the info for rendering
 * the molecule.
 *
 ***************************************************************************/

#include "DrawMolecule.h"
#include "AtomColor.h"
#include "AtomRep.h"
#include "AtomSel.h"
#include "utilities.h"
#include "VMDApp.h"
#include "MoleculeList.h"
#include "CommandQueue.h"
#include "CmdAnimate.h"
#include "Stride.h"
#include "Animation.h"
#include "PickList.h"
#include "MaterialList.h"
#include "Inform.h"
#include "TextEvent.h"
#include "DisplayDevice.h"
#include "MoleculeGraphics.h"
#include "BondSearch.h"
#include "DrawForce.h"
#include "VolumetricData.h"

///////////////////////  constructor and destructor

DrawMolecule::DrawMolecule(VMDApp *vmdapp, Displayable *par)
	: BaseMolecule(vmdapp->next_molid()), 
	  Displayable(par), app(vmdapp), repList(8) {
  repcounter = 0;
  active = TRUE;
  mol_anim = new AnimationData<Timestep *>;
  did_secondary_structure = 0;
  molgraphics = new MoleculeGraphics(this);
  vmdapp->pickList->add_pickable(molgraphics);
  drawForce = new DrawForce(this);

  need_find_bonds = 0;
}
  
// destructor ... free up any extra allocated space (the child Displayables
// will be deleted by the Displayable destructor)
DrawMolecule::~DrawMolecule(void) {
  // delete all molecule representations
  for(int i=0; i < components(); i++) {
    app->pickList->remove_pickable(component(i));
    delete component(i);  
  }

  app->pickList->remove_pickable(molgraphics);
  delete molgraphics;
  delete mol_anim;
}

///////////////////////  public routines

// return Nth component ... change to proper return type
DrawMolItem *DrawMolecule::component(int n) {
  if(n >= 0 && n < components())
    return repList[n];
  else
    return NULL;
}


// return the component corresponding to the pickable
DrawMolItem *DrawMolecule::component_from_pickable(const Pickable *p) {
  for (int i=0; i<components(); i++) 
    if (repList[i] == p) return repList[i];
  return NULL; // no matching component
}


// Return true if ANY representation is displaying atom n
int DrawMolecule::atom_displayed(int n) {
  if (displayed() && n >= 0 && n < nAtoms) {
    for (int i=(components() - 1); i >= 0; i--) {
      if ((repList[i])->atom_displayed(n))
        return TRUE; // atom is shown
    }
  }
  return FALSE; // atom is not shown
}


// delete the Nth representation ... return success
int DrawMolecule::del_rep(int n) {
  DrawMolItem *rep = component(n);
  if (rep) {
    app->pickList->remove_pickable(rep);
    delete rep;		// delete the object
    repList.remove(n);	// and it's slot in the representation list
  }

  return (rep != NULL);
}


void DrawMolecule::add_rep(AtomColor *ac, AtomRep *ar, AtomSel *as, 
                          const Material *am) {
  // Rep has unique name (unique within the molecule)
  char buf[50];
  sprintf(buf, "rep%d", repcounter++);
  DrawMolItem *rep = new DrawMolItem(buf, this, ac, ar, as);
  app->pickList->add_pickable(rep);
  rep->change_material(am);
  repList.append(rep);
}


// change the Nth representation ... return success.
// if any object is NULL, that characteristic is not changed.
int DrawMolecule::change_rep(int n, AtomColor *ac, AtomRep *ar, const char *sel) { 
  DrawMolItem *rep = component(n);
  if (rep) {
    rep->change_color(ac);
    rep->change_rep(ar);
    rep->change_sel(sel);  // returns TRUE if there was no problem, or if
                           // sel was NULL meaning no action is to be taken
    return TRUE;
  }

  return FALSE;
}


// redraw all the representations
void DrawMolecule::force_recalc(int reason) {
  int numcomp = components();
  for (int i=0; i<numcomp; i++) {
    component(i)->force_recalc(reason);
  }
  // The preceding loop updates all the DrawMolItem reps, but other children
  // of DrawMolecule (i.e. DrawForce) need to know about the update as well.
  // Calling need_matrix_recalc sets the _needUpdate flag for this purpose.
  need_matrix_recalc();
  app->commandQueue->runcommand(new CmdAnimNewFrame);
}


// tell the rep to update its PBC transformation matrices next prepare cycle
void DrawMolecule::change_pbc() {
  int numcomp = components();
  for (int i=0; i<numcomp; i++) 
    component(i)->change_pbc();
  // labels can be made between periodic images, so warn them that the
  // distance between images has changed.  Would be better to set a flag
  // so that notify can only get called once, inside of prepare().
  notify();
}


// tell the rep to update its timestep
void DrawMolecule::change_ts() {
  int numcomp = components();
  for (int i=0; i<numcomp; i++) 
    component(i)->change_ts();
  notify();
}


// query whether this molecule contains a highlighted rep
int DrawMolecule::highlighted_rep() const {
  if (app->highlighted_molid != id()) 
    return -1;
  return app->highlighted_rep;
}


// get the component by its string name
int DrawMolecule::get_component_by_name(const char *nm) {
  // XXX linear search for the name is slow
  int numreps = repList.num();
  for (int i=0; i<numreps; i++) {
    if (!strcmp(repList[i]->name, nm))
      return i; // return component
  }
  return -1; // failed to find a component with that name
}


// get the name of the specified component
const char *DrawMolecule::get_component_name(int ind) {
  DrawMolItem *rep = component(ind);
  if (!rep) 
    return FALSE;
  return rep->name;
}


// prepare the molecule for drawing
void DrawMolecule::prepare() {
  int need_notify = 0;
  if (anim()->anim_update()) {
    app->commandQueue->runcommand(new CmdAnimNewFrame);
    app->commandQueue->runcommand(new FrameEvent(id(), anim()->frame()));
    molgraphics->prepare();
    drawForce->prepare();
    change_ts();
    need_notify = 1;
  }
  if (need_notify || needUpdate()) {
    notify(); // notify monitors
  }
}


// notify monitors of an update
void DrawMolecule::notify() {
  int monnum = monitorlist.num();
  int nid = id();
  for (int i=0; i<monnum; i++) 
    monitorlist[i]->notify(nid);
}


// add a new frame
void DrawMolecule::append_frame(Timestep *ts) {
  // XXX we now delay timestep initialization until the first time
  //     the cov() or scale_factor() methods are called on the timestep.  
  //     This greatly speeds up load time for analysis script usage, 
  //     leaving the cov/minmax calculations to be done on-the-fly as needed.
  // ts->init();                  // initialize timestep cov/min/max etc

  mol_anim->append_frame(ts);  // add the timestep to the animation

  // recenter the molecule when the first coordinate frame is loaded
  if (anim()->num() == 1) {    
    app->scene_resetview_newmoldata();
  }

  // update bonds if needed, when any subsequent frame is loaded
  if (anim()->num() >= 1) {    
    // find bonds if necessary
    if (need_find_bonds == 1) {     
      need_find_bonds = 0;
      vmd_bond_search(this, ts, -1, 0); // just add bonds, no dup checking
    } else if (need_find_bonds == 2) {
      need_find_bonds = 0;
      vmd_bond_search(this, ts, -1, 1); // add bonds checking for dups
    }
  }

  addremove_ts();              // tell all reps to update themselves
  app->commandQueue->runcommand(new CmdAnimNewNumFrames); // update frame count
}


// duplicate an existing frame
void DrawMolecule::duplicate_frame(const Timestep *ts) {
  Timestep *newts;
  if (ts == NULL) { // append a 'null' frame
    newts = new Timestep(nAtoms);
    newts->zero_values();
  } else {
    newts = new Timestep(*ts);
  }
  append_frame(newts);
}


// delete a frame
void DrawMolecule::delete_frame(int n) {
  mol_anim->delete_frame(n);
  addremove_ts();
  app->commandQueue->runcommand(new CmdAnimNewNumFrames);
}


// add or remove a timestep
void DrawMolecule::addremove_ts() {
  int numcomp = components();
  for (int i=0; i<numcomp; i++) 
    component(i)->change_traj();
}


// return the norm, double-precision arguments
static float dnorm(const double *v) {
  return (float)sqrt(v[0]*v[0] + v[1]*v[1] + v[2]*v[2]);
}


// scaling factor required to make the molecule fit within (-1 ... 1)
float DrawMolecule::scale_factor(void) {
  if (mol_anim->num() > 0) {
    return mol_anim->item(0)->scale_factor();
  } else if (molgraphics->num_elements() > 0) {
    return molgraphics->scale_factor();
  } else if (volumeList.num() > 0) {
    // scale factor is 1.5/(maxrange), where maxrange is the largest range
    // of the data along any cardinal axis.  That's how Timestep does it, 
    // anyway.  The volumetric axes aren't necessarily orthogonal so I'll
    // just go with the largest value.
    const VolumetricData *data = volumeList[0];
    float x=dnorm(data->xaxis), y=dnorm(data->yaxis), z=dnorm(data->zaxis);
    float scale_factor = x > y ? x : y;
    scale_factor = scale_factor > z ? scale_factor : z;
    if (scale_factor > 0) return 1.5f/scale_factor;
  }
  return 1.0f;
}


// center of volume of this molecule
int DrawMolecule::cov(float& x, float& y, float& z) {
  if (mol_anim->num() > 0) {
    (mol_anim->item(0))->cov(x, y, z);
  } else if (molgraphics->num_elements() > 0) {
    molgraphics->cov(x, y, z);
  } else if (volumeList.num() > 0) {
    const VolumetricData *data = volumeList[0];
    x = (float) (data->origin[0] + 
        0.5*(data->xaxis[0] + data->yaxis[0] + data->zaxis[0]));
    y = (float) (data->origin[1] + 
        0.5*(data->xaxis[1] + data->yaxis[1] + data->zaxis[1]));
    z = (float) (data->origin[2] + 
        0.5*(data->xaxis[2] + data->yaxis[2] + data->zaxis[2]));
  } else {
    return FALSE;
  }
  return TRUE;
}


/// recompute molecule's bonds via distance bond search from current timestep
int DrawMolecule::recalc_bonds() {
  Timestep *ts = current();

  if (ts) {
    clear_bonds();                     // clear the existing bond list
    vmd_bond_search(this, ts, -1, 0);  // just add bonds, no dup checking
    msgInfo << "Bond count: " << count_bonds() << sendmsg;
    return 0;
  } 

  msgInfo << "No coordinates" << sendmsg;
  return -1;
}


// Calculate or update secondary structure when necessary
int DrawMolecule::need_secondary_structure(int calc_if_not_yet_done) {
  int rc = 0;

  if (did_secondary_structure) 
    return 0; // if already done, just return

  if (!current()) 
    return 0; // if no frame is available, just return

  if (calc_if_not_yet_done) {
    app->show_stride_message();
    int rc = ss_from_stride(this);
    if (rc) {
      msgErr << "Call to Stride program failed." << sendmsg;
    }
  }

  did_secondary_structure = 1;
  return rc;
}

void DrawMolecule::invalidate_ss() {
  did_secondary_structure = 0;
}

int DrawMolecule::recalc_ss() {
  did_secondary_structure = 0;
  int rc = need_secondary_structure(1);
  did_secondary_structure = 1;
  // XXX should check return code to see if secondary structure was changed;
  // need to audit all the functions in the call chain from 
  // need_secondary_structure to make sure they are returning consistent
  // error codes.
  for (int i=0; i<components(); i++) component(i)->change_ss();
  return rc;
}

int DrawMolecule::add_point(const float *x) {
  return molgraphics->add_point(x);
}
int DrawMolecule::add_pickpoint(const float *x) {
  return molgraphics->add_pickpoint(x);
}
int DrawMolecule::add_triangle(const float *x1, const float *x2, const float *x3) {
  return molgraphics->add_triangle(x1, x2, x3);
}
int DrawMolecule::add_trinorm(const float *x1, const float *x2, const float *x3,
                const float *nx1, const float *nx2, const float *nx3) {
  return molgraphics->add_trinorm(x1, x2, x3, nx1, nx2, nx3);
}
int DrawMolecule::add_tricolor(const float *x1, const float *x2, const float *x3,
                const float *nx1, const float *nx2, const float *nx3,
                int c1, int c2, int c3) {
  return molgraphics->add_tricolor(x1, x2, x3, nx1, nx2, nx3, c1, c2, c3);
}
int DrawMolecule::add_line(const float *x, const float *y, int line_style, int width) {
  return molgraphics->add_line(x, y, line_style, width);
}
int DrawMolecule::add_cylinder(const float *x, const float *y,
                 float radius, int res, int filled) {
  return molgraphics->add_cylinder(x, y, radius, res, filled);
}
int DrawMolecule::add_cone(const float *x, const float *y,
             float radius, float radius2, int res) {
  return molgraphics->add_cone(x, y, radius, radius2, res);
}
int DrawMolecule::add_sphere(const float *x, float r, int res) {
  return molgraphics->add_sphere(x, r, res);
}
int DrawMolecule::add_text(const float *x, const char *text, float size) {
  return molgraphics->add_text(x, text, size);
}
int DrawMolecule::use_materials(int yes_no) {
  return molgraphics->use_materials(yes_no);
}
int DrawMolecule::use_color(int index) {
  return molgraphics->use_color(index);
}
int DrawMolecule::use_material(const Material *m) {
  return molgraphics->use_material(m);
}
void DrawMolecule::delete_id(int id) {
  molgraphics->delete_id(id);
}
void DrawMolecule::delete_all(void) {
  molgraphics->delete_all();
}
int DrawMolecule::replace_id(int id) {
  return molgraphics->replace_id(id);
}
int DrawMolecule::index_id(int id) {
  return molgraphics->index_id(id);
}
int DrawMolecule::num_elements(void) {
  return molgraphics->num_elements();
}
int DrawMolecule::element_id(int index) {
  return molgraphics->element_id(index); 
}
const char *DrawMolecule::info_id(int id) {
  return molgraphics->info_id(id);
}

void DrawMolecule::register_monitor(DrawMoleculeMonitor *mon) {
  monitorlist.append(mon);
}
void DrawMolecule::unregister_monitor(DrawMoleculeMonitor *mon) {
  monitorlist.remove(monitorlist.find(mon));
}

