/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995-2007 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: DrawMolItemRings.C,v $
 *	$Author: johns $	$Locker:  $		$State: Exp $
 *	$Revision: 1.4 $	$Date: 2007/01/12 20:08:21 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Child Displayable component of a molecule; this is responsible for doing
 * the actual drawing of a molecule.  It contains an atom color, atom
 * selection, and atom representation object to specify how this component
 * should look.
 *
 ***************************************************************************/

#ifdef VMDWITHCARBS

#include "utilities.h"
#include "DrawMolItem.h"
#include "DrawMolecule.h"
#include "DrawRingsUtils.h"
#include <math.h>

void DrawMolItem::draw_rings_paperchain(float *framepos) {
  int i;

  sprintf (commentBuffer,"MoleculeID: %d ReprID: %d Beginning PaperChain Rings",
           mol->id(), repNumber);
  cmdCommentX.putdata(commentBuffer, cmdList);
  
  for (i=0; i < mol->smallringList.num(); i++) {
    paperchain_draw_ring(*(mol->smallringList[i]),framepos);
  }
}

void DrawMolItem::paperchain_get_ring_color(SmallRing &ring, float *framepos, float *rgb) {
  int N = ring.num(); //the number of atoms in the current ring
  double *xring = new double[N];
  double *yring = new double[N];
  double *zring = new double[N];
  double *displ = new double[N];
  double *q = new double[N];
  double *phi = new double[N];
  double Q;
  int m;
  float *atompos;
  int curatomid;

  vec_zero(rgb); // set default color to black

  for (int i=0; i<N; i++) {
    curatomid = ring[i];
    atompos = framepos + 3*curatomid; // pointer arithmetic is evil :)
    xring[i] = atompos[0];
    yring[i] = atompos[1];
    zring[i] = atompos[2];
  }     

  atom_displ_from_mean_plane(xring,yring,zring,displ,N);
         
  if (N==6) { //special case - pyranose rings
    if (pucker_param(N, displ, q, phi, m, Q)) {
      double cosTheta = q[2]/Q;
      double theta = acos(cosTheta); 
      double sinTheta = sin(theta);

      // Q is the puckering amplitude - i.e. the intensity of the pucker.
      // multiply by Q to show intensity, particularly for rings with 
      // little pucker (black)
      // NOTE -using abs - polar positions therefore equivalent
      double intensity = Q;
      
      rgb[0] = fabs(sinTheta)*intensity;
      rgb[1] = fabs(cosTheta)*intensity;
      rgb[2] = fabs(sin(3*phi[1])*sinTheta)*intensity;
    }
  } else if (N==5) { //special case - furanose rings
    if(pucker_param(N, displ, q, phi, m, Q)) {
      rgb[0] = 0;
      rgb[1] = 0;
      rgb[2] = Q;
    }
  }

  // clamp color values to legal range
  clamp_color(rgb);
 
  delete [] xring;
  delete [] yring;
  delete [] zring;
  delete [] displ;
  delete [] q;
  delete [] phi;
}

void DrawMolItem::paperchain_draw_ring(SmallRing &ring, float *framepos) {
  int N = ring.num(); //the number of atoms in the current ring
  int i, nexti;
  float centroid[3], normal[3], top[3], bottom[3], curvec[3], nextvec[3], x[3];
  float rgb[3];

  paperchain_get_ring_color(ring, framepos, rgb);

  // initialize centroid and centroid normal
  centroid[0] = centroid[1] = centroid[2] = 0.0;
  normal[0] = normal[1] = normal[2] = 0.0;

  // calculate centroid and normal
  for (i=0; i<N; i++) {
    // calculate next ring position (wrapping as necessary)
    nexti = i+1;
    if (nexti >= N) nexti = 0;
    
    vec_copy(curvec,framepos + 3*ring[i]);
    vec_copy(nextvec,framepos + 3*ring[nexti]);
        
    // update centroid
    vec_add(centroid,centroid,curvec);
    
    // update normal (this is Newell's method; see Carbohydra paper)
    normal[0] += (curvec[1] - nextvec[1]) * (curvec[2] + nextvec[2]); // (Y_i - Y_next_i) * (Z_i + Z_next_i)
    normal[1] += (curvec[2] - nextvec[2]) * (curvec[0] + nextvec[0]); // (Z_i - Z_next_i) * (X_i + X_next_i)
    normal[2] += (curvec[0] - nextvec[0]) * (curvec[1] + nextvec[1]); // (X_i - X_next_i) * (Y_i + Y_next_i)      
  }
  vec_scale(centroid,1.0/N,centroid);
  vec_normalize(normal);

  // calculate top and bottom points
  vec_scale(x,0.5*atomRep->get_data(AtomRep::LINETHICKNESS),normal);
  vec_add(top,centroid,x);
  vec_sub(bottom,centroid,x);

#if 1
  append(DMATERIALON); // turn on lighting
#else
  append(DMATERIALOFF); // turn off lighting
#endif

// XXX we should be looping over all of the rings
//     within this routine rather than doing them separately,
//     as we can generate one big triangle mesh from the results
//     eliminating various sources of rendering overhead if we do it right.

#if 1
  // draw triangles
  DispCmdTriMesh cmdTriMesh;
  ResizeArray<float> vertices;
  ResizeArray<float> colors;
  ResizeArray<float> normals;
  ResizeArray<int>   facets;

  // add top/bottom vertices first
  vertices.append(top[0]);
  vertices.append(top[1]);
  vertices.append(top[2]);
  normals.append(normal[0]);
  normals.append(normal[1]);
  normals.append(normal[2]);
  colors.append(rgb[0]);
  colors.append(rgb[1]);
  colors.append(rgb[2]);

  vertices.append(bottom[0]);
  vertices.append(bottom[1]);
  vertices.append(bottom[2]);
  normals.append(normal[0]);
  normals.append(normal[1]);
  normals.append(normal[2]);
  colors.append(rgb[0]);
  colors.append(rgb[1]);
  colors.append(rgb[2]);

  // draw top half
  for (i=0; i<N; i++) {
    // calculate next ring position (wrapping as necessary)
    nexti = i+1;
    if (nexti >= N) nexti = 0;
    
    vec_copy(curvec,framepos + 3*ring[i]);
    vec_copy(nextvec,framepos + 3*ring[nexti]);

    vertices.append(curvec[0]);
    vertices.append(curvec[1]);
    vertices.append(curvec[2]);

    float normtop[3], tmp0[3], tmp1[3];
    vec_sub(tmp0, curvec, nextvec);
    vec_sub(tmp1, nextvec, top);
    cross_prod(normtop, tmp0, tmp1);
    vec_normalize(normtop);
    normals.append(normtop[0]);
    normals.append(normtop[1]);
    normals.append(normtop[2]);

    colors.append(rgb[0]);
    colors.append(rgb[1]);
    colors.append(rgb[2]);

    facets.append(2+i);     // curvec
    facets.append(2+nexti); // nextvec
    facets.append(0);       // top
  }

  // draw bottom half
  for (i=0; i<N; i++) {
    // calculate next ring position (wrapping as necessary)
    nexti = i+1;
    if (nexti >= N) nexti = 0;
    
    vec_copy(curvec,framepos + 3*ring[i]);
    vec_copy(nextvec,framepos + 3*ring[nexti]);

    vertices.append(curvec[0]);
    vertices.append(curvec[1]);
    vertices.append(curvec[2]);

    float normbot[3], tmp0[3], tmp1[3];
    vec_sub(tmp0, curvec, nextvec);
    vec_sub(tmp1, nextvec, bottom);
    cross_prod(normbot, tmp0, tmp1);
    vec_normalize(normbot);
    normals.append(normbot[0]);
    normals.append(normbot[1]);
    normals.append(normbot[2]);

    colors.append(rgb[0]);
    colors.append(rgb[1]);
    colors.append(rgb[2]);

    facets.append(2+N+i);     // curvec
    facets.append(2+N+nexti); // nextvec
    facets.append(1);            // bottom
  }

  // printf("TriMesh N: %d nvert: %d nface: %d  rgb[]=%0.2f,%0.2f,%0.2f\n", N, vertices.num()/3, facets.num()/3, rgb[0], rgb[1], rgb[2]);

  // draw the resulting triangle mesh
  cmdTriMesh.putdata(&vertices[0], &normals[0], &colors[0], vertices.num()/3, 
                     &facets[0], facets.num()/3, 0, cmdList);
#else    
  // draw triangles
  for (i=0; i<N; i++) {
    // calculate next ring position (wrapping as necessary)
    nexti = i+1;
    if (nexti >= N) nexti = 0;
    
    vec_copy(curvec,framepos + 3*ring[i]);
    vec_copy(nextvec,framepos + 3*ring[nexti]);

    // Display command examples:
    //   cmdTriColor.putdata(...,cmdList);
    //   cmdTriangle.putdata(v1,v2,v3,cmdList)
    cmdTriangle.putdata(curvec, nextvec, top, cmdList);
    cmdTriangle.putdata(curvec, nextvec, bottom, cmdList);
  }
#endif

}


void DrawMolItem::draw_rings_twister(float *framepos) {
  sprintf (commentBuffer,"MoleculeID: %d ReprID: %d Beginning Twister Rings",
           mol->id(), repNumber);
  cmdCommentX.putdata(commentBuffer, cmdList);
}

#endif
