/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995-2007 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: CUDAAccel.C,v $
 *	$Author: johns $	$Locker:  $		$State: Exp $
 *	$Revision: 1.2 $	$Date: 2007/03/06 07:23:33 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *   Class to store and handle enumeration and initialization of 
 *   CUDA GPU accelerator devices.
 ***************************************************************************/
#include "config.h"     // rebuild on config changes
#include "Inform.h"
#include "ResizeArray.h"
#include "CUDAAccel.h"
#include "CUDAKernels.h"

#if defined(VMDTHREADS) // System-specific threads code
#include "VMDThreads.h"
#endif


#if 1
#define CUERR { cudaError_t err; \
  if ((err = cudaGetLastError()) != cudaSuccess) { \
  printf("CUDA error: %s, line %d\n", cudaGetErrorString(err), __LINE__); \
  return; }}
#else
#define CUERR
#endif


CUDAAccel::CUDAAccel(void) {
  cudaavail = 0;
  numdevices = 0;

#if defined(VMDCUDA)
  if (vmd_cuda_num_devices(&numdevices)) {
    numdevices = 0;
    return;
  }

  if (numdevices > 0) {
    cudaavail = 1;

    int i;
    for (i=0; i<numdevices; i++) {
      cudadevprops dp;
      vmd_cuda_device_props(i, dp.name, sizeof(dp.name),
                            &dp.major, &dp.minor, &dp.membytes);
      devprops.append(dp);
    }
  }

  print_cuda_devices();
#endif
}

// destructor
CUDAAccel::~CUDAAccel(void) {
}

void CUDAAccel::print_cuda_devices(void) {
  if (!cudaavail || numdevices == 0) {
    msgInfo << "No CUDA accelerator devices found." << sendmsg;
    return;
  }

  msgInfo << "Detected " << numdevices << " CUDA " 
          << ((numdevices > 1) ? "accelerators:" : "accelerator:") << sendmsg;
  int i;
  for (i=0; i<numdevices; i++) {
    msgInfo << "  Device[" << i << "]: '" << device_name(i) << "' Mem: "
            << (device_membytes(i) / (1024 * 1024)) << "MB" << sendmsg; 
  } 
}

int CUDAAccel::num_devices(void) {
  return numdevices;
}

const char *CUDAAccel::device_name(int dev) {
  if (!cudaavail || dev < 0 || dev >= numdevices)
    return NULL;
  return devprops[dev].name; 
}

int CUDAAccel::device_version_major(int dev) {
  if (!cudaavail || dev < 0 || dev >= numdevices)
    return 0; 
  return devprops[dev].major;
}

int CUDAAccel::device_version_minor(int dev) {
  if (!cudaavail || dev < 0 || dev >= numdevices)
    return 0; 
  return devprops[dev].minor;
}

unsigned long CUDAAccel::device_membytes(int dev) {
  if (!cudaavail || dev < 0 || dev >= numdevices)
    return 0; 
  return devprops[dev].membytes;
}





