/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995-2007 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: BondSearch.h,v $
 *	$Author: johns $	$Locker:  $		$State: Exp $
 *	$Revision: 1.21 $	$Date: 2007/01/12 20:08:17 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Distance based bond search code 
 *
 ***************************************************************************/

#include "ResizeArray.h"

/// linked list of atom index pairs generated by the vmd_gridsearch1 
/// and vmd_gridsearch2
/// XXX on 64-bit machines the next pointers are as large as the atom index
/// data in each node, so half of the memory is wasted in pointer chasing.
/// Calling malloc/free once per pair when building the linked list
/// is a severe multithreading scalability bottleneck due to OS-level 
/// mutex locks in malloc/free and/or the kernel VM.
/// On Linux, it is also problematic due to the threaded version of 
/// malloc allocating huge blocks and fragmenting memory tremendously.
/// This implementation should not be used for any threaded code.
struct GridSearchPair {
  int ind1, ind2;
  GridSearchPair *next;
};


/// Linked list of ResizeArrays containing pairlists, optimized for
/// multithreading.  Compared with the GridSearchPair linked list,
/// The list of ResizeArrays uses half as much memory per pair, reduces
/// the malloc calls so they are logarithmic rather than linear with
/// the number of bonds, and reduces free call count to one per-thread down
/// from one per-bond.  Since the ResizeArray results in a contiguous block
/// of memory, the traversal coherency is also significantly improved.
/// Ultimately we should convert all of the grid search routines to use
/// this type of data structure, or one like it.
struct GridSearchPairlist {
  ResizeArray<int> *pairlist;
  GridSearchPairlist *next;
};


/// Grid search for the case of a single set of atoms. It ignore pairs 
/// between atoms with identical coords.  The maxpairs parameter is 
/// set to -1 for no-limit pairlist calculation, or a maximum value otherwise.
GridSearchPair *vmd_gridsearch1(const float *pos, int n, const int *on, 
                               float dist, int allow_double_counting, int maxpairs);

/// Grid search for two different sets of atoms in same molecule.
/// (will eventually be obsoleted by the faster and more useful 
/// vmd_gridsearch3).
/// right now, is still needed by measure hbonds (until problems resolved)
/// The maxpairs parameter is set to -1 for no-limit pairlist calculation, 
/// or a maximum value otherwise.
GridSearchPair *vmd_gridsearch2(const float *pos, int natoms, const int *A, 
                                const int *B, float pairdist, int maxpairs);

/// Grid search for two different sets of atoms and/or molecules.
/// By default, if (posA == posB), all bonds are unique. Otherwise, 
/// double-counting is allowed. This can be overridden by setting
/// the allow_double_counting param (true=1, false=0, or default=-1).
/// The maxpairs parameter is set to -1 for no-limit pairlist calculation, 
/// or a maximum value otherwise.
GridSearchPair *vmd_gridsearch3(const float *posA, int natomsA, const int *A, 
                               const float *posB,int natomsB, const int *B,
                               float pairdist, int allow_double_counting, int maxpairs);

class Timestep;
class BaseMolecule;

/// Grid search for the case of a single set of atoms. It ignore pairs 
/// between atoms with identical coords.  The maxpairs parameter is 
/// set to -1 for no-limit pairlist calculation, or a maximum value otherwise.
/// This is the same code as gridsearch1(), but simplified and hopefully
/// a bit faster.
GridSearchPairlist *vmd_gridsearch_bonds(const float *pos, const float *radii,
                                         int n, float dist, int maxpairs);

/// Compute bonds for the molecule using the given timestep (which must
/// not be NULL) and adds them to the given molecule.  Return success.
/// The code currently calls gridsearch1 with a pairlist limit of 
/// 27 * natoms, which should easily be sufficient for any real structure.
int vmd_bond_search(BaseMolecule *mol, const Timestep *ts, 
                    float cutoff, int dupcheck);

/// Multithreaded bond search worker routine handles spawning and joining all
/// of the worker threads, and merging their results into a single list.
int vmd_bondsearch_thr(const float *pos, const float *radii,
                       GridSearchPairlist * cur, int totb, 
                       int **boxatom, int *numinbox, int **nbrlist, 
                       int maxpairs, float pairdist);
