/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995-2007 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: BondSearch.C,v $
 *	$Author: johns $	$Locker:  $		$State: Exp $
 *	$Revision: 1.54 $	$Date: 2007/03/29 21:13:42 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Distance based bond search code 
 *
 ***************************************************************************/

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include "BondSearch.h"
#include "Timestep.h"
#include "BaseMolecule.h"
#include "Molecule.h"
#include "Inform.h"
#include "VMDThreads.h"
#include <ctype.h>         // needed for isdigit()
#include <string.h>

static void add_link(GridSearchPair *link, int i, int j) {
  link->next = (GridSearchPair *) malloc(sizeof(GridSearchPair));
  link->next->ind1 = i;
  link->next->ind2 = j;
  link->next->next = NULL;
}

static void find_minmax_all(const float *pos, int n,
                        float *min, float *max) {
  float x1, x2, y1, y2, z1, z2;
  int i=0;

  // return immediately if there are no atoms, or no atoms are on.
  if (n < 1) return;

  // initialize min/max to first 'on' atom, and advance the counter.
  pos += 3*i;
  x1 = x2 = pos[0];
  y1 = y2 = pos[1];
  z1 = z2 = pos[2];
  pos += 3;
  i++;

  for (; i < n; i++) {
    if (pos[0] < x1) x1 = pos[0];
    if (pos[0] > x2) x2 = pos[0];
    if (pos[1] < y1) y1 = pos[1];
    if (pos[1] > y2) y2 = pos[1];
    if (pos[2] < z1) z1 = pos[2];
    if (pos[2] > z2) z2 = pos[2];
    pos += 3;
  }
  min[0] = x1; min[1] = y1; min[2] = z1;
  max[0] = x2; max[1] = y2; max[2] = z2;
}

static void find_minmax(const float *pos, int n, const int *on, 
                        float *min, float *max, int *oncount) {
  float x1, x2, y1, y2, z1, z2;
  int i, numon;

  // return immediately if there are no atoms, or no atoms are on.
  if (n < 1) return;

  // init on count
  numon = 0;

  // find first on atom
  for (i=0; i<n; i++) {
    if (on[i]) {
      numon++;
      break;
    }
  }
  if (i==n) {
    if (oncount != NULL) 
      *oncount = numon;
    return;
  }

  // initialize min/max to first 'on' atom, and advance the counter.
  pos += 3*i;
  x1 = x2 = pos[0];
  y1 = y2 = pos[1];
  z1 = z2 = pos[2];
  pos += 3;
  i++;

  for (; i < n; i++) {
    if (on[i]) {
      if (pos[0] < x1) x1 = pos[0];
      else if (pos[0] > x2) x2 = pos[0];
      if (pos[1] < y1) y1 = pos[1];
      else if (pos[1] > y2) y2 = pos[1];
      if (pos[2] < z1) z1 = pos[2];
      else if (pos[2] > z2) z2 = pos[2];
      numon++;
    }
    pos += 3;
  }
  min[0] = x1; min[1] = y1; min[2] = z1;
  max[0] = x2; max[1] = y2; max[2] = z2;

  if (oncount != NULL) 
    *oncount = numon;
}

static int make_neighborlist(int **nbrlist, int xb, int yb, int zb) {
  int xi, yi, zi, aindex, xytotb;

  if (nbrlist == NULL)
    return -1;

  xytotb = xb * yb;
  aindex = 0;
  for (zi=0; zi<zb; zi++) {
    for (yi=0; yi<yb; yi++) {
      for (xi=0; xi<xb; xi++) {
        int nbrs[14];           
        int n=0;                
        nbrs[n++] = aindex;     
        if (xi < xb-1) nbrs[n++] = aindex + 1;
        if (yi < yb-1) nbrs[n++] = aindex + xb;
        if (zi < zb-1) nbrs[n++] = aindex + xytotb;
        if (xi < (xb-1) && yi < (yb-1)) nbrs[n++] = aindex + xb + 1;
        if (xi < (xb-1) && zi < (zb-1)) nbrs[n++] = aindex + xytotb + 1;
        if (yi < (yb-1) && zi < (zb-1)) nbrs[n++] = aindex + xytotb + xb;
        if (xi < (xb-1) && yi > 0)      nbrs[n++] = aindex - xb + 1;
        if (xi > 0 && zi < (zb-1))     nbrs[n++] = aindex + xytotb - 1;
        if (yi > 0 && zi < (zb-1))     nbrs[n++] = aindex + xytotb - xb;
        if (xi < (xb-1) && yi < (yb-1) && zi < (zb-1))
                                       nbrs[n++] = aindex + xytotb + xb + 1;
        if (xi > 0 && yi < (yb-1) && zi < (zb-1))
                                       nbrs[n++] = aindex + xytotb + xb - 1;
        if (xi < (xb-1) && yi > 0 && zi < (zb-1))
                                       nbrs[n++] = aindex + xytotb - xb + 1;
        if (xi > 0 && yi > 0 && zi < (zb-1))
                                       nbrs[n++] = aindex + xytotb - xb - 1;

        int *lst = (int *) malloc((n+1)*sizeof(int));
        if (lst == NULL)
          return -1; // return on failed allocations
        nbrlist[aindex] = lst;
        memcpy(nbrlist[aindex], nbrs, n*sizeof(int));
        nbrlist[aindex][n] = -1;  
        aindex++;
      }
    }
  }

  return 0;
}


// symetrical version of neighbourlist (each cell has 27 neighbours, incl. itself, instead of just 14)
static int make_neighborlist_sym(int **nbrlist, int xb, int yb, int zb) {
  int xi, yi, zi, aindex, xytotb;

  if (nbrlist == NULL)
    return -1;

  xytotb = xb * yb;
  aindex = 0;
  for (zi=0; zi<zb; zi++) {
    for (yi=0; yi<yb; yi++) {
      for (xi=0; xi<xb; xi++) {
        int nbrs[27];           
        int n=0;                
        nbrs[n++] = aindex;     
        if (xi < xb-1) nbrs[n++] = aindex + 1;
        if (yi < yb-1) nbrs[n++] = aindex + xb;
        if (zi < zb-1) nbrs[n++] = aindex + xytotb;
        if (xi < (xb-1) && yi < (yb-1)) nbrs[n++] = aindex + xb + 1;
        if (xi < (xb-1) && zi < (zb-1)) nbrs[n++] = aindex + xytotb + 1;
        if (yi < (yb-1) && zi < (zb-1)) nbrs[n++] = aindex + xytotb + xb;
        if (xi < (xb-1) && yi) nbrs[n++] = aindex - xb + 1;
        if (xi && zi < (zb-1)) nbrs[n++] = aindex + xytotb - 1;
        if (yi && zi < (zb-1)) nbrs[n++] = aindex + xytotb - xb;
        if (xi < (xb-1) && yi < (yb-1) && zi < (zb-1)) nbrs[n++] = aindex + xytotb + xb + 1;
        if (xi && yi < (yb-1) && zi < (zb-1)) nbrs[n++] = aindex + xytotb + xb - 1;
        if (xi < (xb-1) && yi && zi < (zb-1)) nbrs[n++] = aindex + xytotb - xb + 1;
        if (xi && yi && zi < (zb-1)) nbrs[n++] = aindex + xytotb - xb - 1;
        
        if (xi) nbrs[n++] = aindex - 1;
        if (yi) nbrs[n++] = aindex - xb;
        if (zi) nbrs[n++] = aindex - xytotb;
        if (xi && yi) nbrs[n++] = aindex - xb - 1;
        if (xi && zi) nbrs[n++] = aindex - xytotb - 1;
        if (yi && zi) nbrs[n++] = aindex - xytotb - xb;
        if (xi && yi < (yb-1)) nbrs[n++] = aindex + xb - 1;        
        if (xi < (xb-1) && zi) nbrs[n++] = aindex - xytotb + 1;
        if (yi < (yb-1) && zi) nbrs[n++] = aindex - xytotb + xb;
        if (xi && yi && zi) nbrs[n++] = aindex - xytotb - xb - 1;   
        if (xi < (xb-1) && yi && zi) nbrs[n++] = aindex - xytotb - xb + 1;
        if (xi && yi < (yb-1) && zi) nbrs[n++] = aindex - xytotb + xb - 1;
        if (xi < (xb-1) && yi < (yb-1) && zi) nbrs[n++] = aindex - xytotb + xb + 1;

        int *lst = (int *) malloc((n+1)*sizeof(int));
        if (lst == NULL)
          return -1; // return on failed allocations
        nbrlist[aindex] = lst;
        memcpy(nbrlist[aindex], nbrs, n*sizeof(int));
        nbrlist[aindex][n] = -1;  
        aindex++;
      }
    }
  }

  return 0;
}


GridSearchPair *vmd_gridsearch1(const float *pos,int natoms, const int *on, 
                               float pairdist, int allow_double_counting, int maxpairs) {
  GridSearchPair *head, *cur;
  float min[3], max[3], sqdist;
  int i, j, xb, yb, zb, xytotb, totb, aindex;
  int **boxatom, *numinbox, *maxinbox, **nbrlist;
  int numon = 0;
  float sidelen[3], volume;
  const float *loc;
  int paircount = 0;
  int maxpairsreached = 0;
  msgtimer msgt;
  sqdist = pairdist * pairdist;

  // find bounding box for selected atoms, and number of atoms in selection.
  find_minmax(pos, natoms, on, min, max, &numon);

  // do sanity checks and complain if we've got bogus atom coordinates,
  // we shouldn't ever have density higher than 0.1 atom/A^3, but we'll
  // be generous and allow much higher densities.  
  if (maxpairs != -1) {
    vec_sub(sidelen, max, min);
    // include estimate for atom radius (1 Angstrom) in volume determination
    volume = fabsf((sidelen[0] + 2.0f) * (sidelen[1] + 2.0f) * (sidelen[2] + 2.0f));
    if ((numon / volume) > 1.0) {
      msgWarn << "vmd_gridsearch1: insane atom density" << sendmsg;
    }
  }

  // I don't want the grid to get too large, otherwise I could run out
  // of memory.  Octrees would be cool, but I'll just limit the grid size
  // and let the performance degrade a little for pathological systems.
  // Note that sqdist is what gets used for the actual distance checks;
  // from here on out pairdist is only used to set the grid size, so we 
  // can set it to anything larger than the original pairdist.
  const int MAXBOXES = 4000000;
  totb = MAXBOXES + 1;

  // setup head of pairlist
  head = (GridSearchPair *) malloc(sizeof(GridSearchPair));
  head->next = NULL;
  cur = head;

  float newpairdist = pairdist;
  do {
    pairdist = newpairdist;
    xb = (int)((max[0]-min[0])/pairdist)+1;
    yb = (int)((max[1]-min[1])/pairdist)+1;
    zb = (int)((max[2]-min[2])/pairdist)+1;
    xytotb = yb * xb;
    totb = xytotb * zb;
    newpairdist = pairdist * 1.26f; // cbrt(2) is about 1.26
  } while (totb > MAXBOXES || totb < 1); // check for integer wraparound too
 
  // 2. Sort each atom into appropriate bins
  boxatom = (int **) calloc(1, totb*sizeof(int *));
  numinbox = (int *) calloc(1, totb*sizeof(int));
  maxinbox = (int *) calloc(1, totb*sizeof(int));
  if (boxatom == NULL || numinbox == NULL || maxinbox == NULL) {
    if (boxatom != NULL)
      free(boxatom);
    if (numinbox != NULL)
      free(numinbox);
    if (maxinbox != NULL)
      free(maxinbox);
    msgErr << "Gridsearch memory allocation failed, bailing out" << sendmsg;
    return NULL; // ran out of memory, bail out!
  }

  loc = pos;
  for (i=0; i<natoms; i++) {
    if (on[i]) {
      int axb, ayb, azb, aindex, num;
      axb = (int)((loc[0] - min[0])/pairdist);
      ayb = (int)((loc[1] - min[1])/pairdist);
      azb = (int)((loc[2] - min[2])/pairdist);
      aindex = azb * xytotb + ayb * xb + axb;
      num = numinbox[aindex];
      if (maxinbox[aindex] == num) {
        void *tmp = realloc(boxatom[aindex], (num+4)*sizeof(int));
        boxatom[aindex] = (int *)tmp;
        maxinbox[aindex] += 4;
      }
      boxatom[aindex][num] = i;
      numinbox[aindex]++;
    }
    loc += 3;
  }
  free(maxinbox);
 
  nbrlist = (int **) calloc(1, totb*sizeof(int *));
  if (make_neighborlist(nbrlist, xb, yb, zb)) {
    if (boxatom != NULL) {
      for (i=0; i<totb; i++) {
        if (boxatom[i] != NULL) free(boxatom[i]);
      }
      free(boxatom);
    }
    if (nbrlist != NULL) {
      for (i=0; i<totb; i++) {
        if (nbrlist[i] != NULL) free(nbrlist[i]);
      }
      free(nbrlist);
    }
    free(numinbox);
    msgErr << "Gridsearch memory allocation failed, bailing out" << sendmsg;
    return NULL; // ran out of memory, bail out!
  }

  msg_timer_init(&msgt, 5);
  for (aindex = 0; (aindex < totb) && (!maxpairsreached); aindex++) {
    int *tmpbox, *tmpnbr, *nbr;
    tmpbox = boxatom[aindex];
    tmpnbr = nbrlist[aindex];

    if (msg_timer_timeout(&msgt)) {
      char tmpbuf[128];
      sprintf(tmpbuf, "%6.2f", (100.0f * aindex) / (float) totb);
      msgInfo << "vmd_gridsearch1: " << tmpbuf << "% complete" << sendmsg;
    }

    for (nbr = tmpnbr; (*nbr != -1) && (!maxpairsreached); nbr++) {
      int *nbrbox = boxatom[*nbr];
      for (i=0; (i<numinbox[aindex]) && (!maxpairsreached); i++) {
        int ind1 = tmpbox[i];
        if (!on[ind1]) 
          continue;
        const float *p1 = pos + 3*ind1;
        int startj = 0;
        if (aindex == *nbr) startj = i+1;
        for (j=startj; (j<numinbox[*nbr]) && (!maxpairsreached); j++) {
          int ind2 = nbrbox[j];
          if (on[ind2]) {
            const float *p2 = pos + 3*ind2;
            float ds2 = distance2(p1, p2);

            // ignore pairs between atoms with nearly identical coords
            if (ds2 < 0.001)
              continue;

            if (ds2 > sqdist) 
              continue;

            if (maxpairs > 0) {
              if (paircount >= maxpairs) {
                maxpairsreached = 1;
                continue; 
              }   
            }

            add_link(cur, ind1, ind2);
            paircount++;

            // XXX double-counting still ignores atoms with same coords...
            if (allow_double_counting) { 
              add_link(cur, ind2, ind1); 
              paircount++;
            }
            cur = cur->next;
            cur->next = NULL;
          }
        }
      }
    }
  } 

  for (i=0; i<totb; i++) {
    free(boxatom[i]);
    free(nbrlist[i]);
  }
  free(boxatom);
  free(nbrlist);
  free(numinbox);

  cur = head->next;
  free(head);

  if (maxpairsreached) 
    msgErr << "vmdgridsearch1: exceeded pairlist sanity check, aborted" << sendmsg;

  return cur;
}

GridSearchPair *vmd_gridsearch2(const float *pos,int natoms, 
                               const int *A,const int *B, float pairdist, int maxpairs) {
  GridSearchPair *head, *cur;
  float min[3], max[3], sqdist;
  int i, j, xb, yb, zb, xytotb, totb, aindex;
  int **boxatom, *numinbox, *maxinbox, **nbrlist;
  float sidelen[3], volume;
  int numon = 0;
  const float *loc;
  int paircount = 0;
  int maxpairsreached = 0;
  sqdist = pairdist * pairdist;

  // on = union(A,B).  We grid all atoms, then allow pairs only when one
  // atom is A and the other atom is B.  An alternative scheme would be to
  // to grid the A and B atoms separately, and/or bin them separately, so
  // there would be fewer rejected pairs. 

  int *on = (int *) malloc(natoms*sizeof(int)); 
  for (i=0; i<natoms; i++) {
    if (A[i] || B[i]) {
      numon++;
      on[i] = 1;
    } else {
      on[i] = 0;
    }
  }

  // find bounding box for selected atoms
  find_minmax(pos, natoms, on, min, max, NULL);

  // do sanity checks and complain if we've got bogus atom coordinates,
  // we shouldn't ever have density higher than 0.1 atom/A^3, but we'll
  // be generous and allow much higher densities.
  if (maxpairs != -1) {
    vec_sub(sidelen, max, min);
    // include estimate for atom radius (1 Angstrom) in volume determination
    volume = fabsf((sidelen[0] + 2.0f) * (sidelen[1] + 2.0f) * (sidelen[2] + 2.0f));
    if ((numon / volume) > 1.0) {
      msgWarn << "vmd_gridsearch2: insane atom density" << sendmsg;
    }
  }

  // I don't want the grid to get too large, otherwise I could run out
  // of memory.  Octrees would be cool, but I'll just limit the grid size
  // and let the performance degrade a little for pathological systems.
  // Note that sqdist is what gets used for the actual distance checks;
  // from here on out pairdist is only used to set the grid size, so we 
  // can set it to anything larger than the original pairdist.
  const int MAXBOXES = 4000000;
  totb = MAXBOXES + 1;

  // setup head of pairlist
  head = (GridSearchPair *) malloc(sizeof(GridSearchPair));
  head->next = NULL;
  cur = head;

  float newpairdist = pairdist;
  do {
    pairdist = newpairdist;
    xb = (int)((max[0]-min[0])/pairdist)+1;
    yb = (int)((max[1]-min[1])/pairdist)+1;
    zb = (int)((max[2]-min[2])/pairdist)+1;
    xytotb = yb * xb;
    totb = xytotb * zb;
    newpairdist = pairdist * 1.26f; // cbrt(2) is about 1.26
  } while (totb > MAXBOXES || totb < 1); // check for integer wraparound too
 
  // 2. Sort each atom into appropriate bins
  boxatom = (int **) calloc(1, totb*sizeof(int *));
  numinbox = (int *) calloc(1, totb*sizeof(int));
  maxinbox = (int *) calloc(1, totb*sizeof(int));
  if (boxatom == NULL || numinbox == NULL || maxinbox == NULL) {
    if (boxatom != NULL)
      free(boxatom);
    if (numinbox != NULL)
      free(numinbox);
    if (maxinbox != NULL)
      free(maxinbox);
    msgErr << "Gridsearch memory allocation failed, bailing out" << sendmsg;
    return NULL; // ran out of memory, bail out!
  }

  loc = pos;
  for (i=0; i<natoms; i++) {
    if (on[i]) {
      int axb, ayb, azb, aindex, num;
      axb = (int)((loc[0] - min[0])/pairdist);
      ayb = (int)((loc[1] - min[1])/pairdist);
      azb = (int)((loc[2] - min[2])/pairdist);
      aindex = azb * xytotb + ayb * xb + axb;
      num = numinbox[aindex];
      if (maxinbox[aindex] == num) {
        void *tmp = realloc(boxatom[aindex], (num+4)*sizeof(int));
        boxatom[aindex] = (int *)tmp;
        maxinbox[aindex] += 4;
      }
      boxatom[aindex][num] = i;
      numinbox[aindex]++;
    }
    loc += 3;
  }
  free(maxinbox);
  free(on);
 
  nbrlist = (int **) calloc(1, totb*sizeof(int *));
  if (make_neighborlist(nbrlist, xb, yb, zb)) {
    if (boxatom != NULL) {
      for (i=0; i<totb; i++) {
        if (boxatom[i] != NULL) free(boxatom[i]);
      }
      free(boxatom);
    }
    if (nbrlist != NULL) {
      for (i=0; i<totb; i++) {
        if (nbrlist[i] != NULL) free(nbrlist[i]);
      }
      free(nbrlist);
    }
    free(numinbox);
    msgErr << "Gridsearch memory allocation failed, bailing out" << sendmsg;
    return NULL; // ran out of memory, bail out!
  }

  for (aindex = 0; aindex < totb; aindex++) {
    int *tmpbox, *tmpnbr, *nbr;
    tmpbox = boxatom[aindex];
    tmpnbr = nbrlist[aindex];
    for (nbr = tmpnbr; (*nbr != -1) && (!maxpairsreached); nbr++) {
      int *nbrbox = boxatom[*nbr];
      for (i=0; (i<numinbox[aindex]) && (!maxpairsreached); i++) {
        const float *p1;
        int ind1, startj;
        ind1 = tmpbox[i];
        p1 = pos + 3*ind1;
        startj = 0;
        if (aindex == *nbr) startj = i+1;
        for (j=startj; (j<numinbox[*nbr]) && (!maxpairsreached); j++) {
          const float *p2;
          int ind2;
          ind2 = nbrbox[j];
          if ((A[ind1] && B[ind2]) || (A[ind2] && B[ind1])) {
            p2 = pos + 3*ind2;
            if (distance2(p1,p2) > sqdist) continue;

            if (maxpairs > 0) {
              if (paircount >= maxpairs) {
                maxpairsreached = 1;
                continue; 
              }   
            }

            if (A[ind1]) {
              add_link(cur, ind1, ind2);
              paircount++;
            } else {
              add_link(cur, ind2, ind1);
              paircount++;
            }
            cur = cur->next;
            cur->next = NULL;
          }
        }
      }
    }
  } 

  for (i=0; i<totb; i++) {
    free(boxatom[i]);
    free(nbrlist[i]);
  }
  free(boxatom);
  free(nbrlist);
  free(numinbox);

  cur = head->next;
  free(head);

  if (maxpairsreached) 
    msgErr << "vmdgridsearch2: exceeded pairlist sanity check, aborted" << sendmsg;

  return cur;
}



// Like vmd_gridsearch2, but pairs up atoms from different locations and molecules.
// By default, if (posA == posB), all bonds are unique. Otherwise, double-counting is allowed.
// This can be overridden by setting allow_double_counting (true, false, or default=-1).
GridSearchPair *vmd_gridsearch3(const float *posA, int natomsA, const int *A, 
                                const float *posB, int natomsB, const int *B, 
                                float pairdist, int allow_double_counting, int maxpairs) {
  if (allow_double_counting == -1) { //default
    if (posA == posB && natomsA == natomsB) 
      allow_double_counting = FALSE;
    else 
      allow_double_counting = TRUE;
  }
  
  // if same molecule and *A[] == *B[], it is a lot faster to use gridsearch1
  if (posA == posB && natomsA == natomsB) {
    bool is_equal = TRUE;
    for (int i=0; i<natomsA && is_equal; i++) {
      if (A[i] != B[i])
        is_equal = FALSE;
    }
    if (is_equal)
      return vmd_gridsearch1(posA, natomsA, A, pairdist, allow_double_counting, maxpairs);
  }
  
  GridSearchPair *head, *cur;
  float min[3], max[3], sqdist;
  float minB[3], maxB[3]; //tmp storage
  int i, j, xb, yb, zb, xytotb, totb, aindex;
  int **boxatomA, *numinboxA, *maxinboxA;
  int **boxatomB, *numinboxB, *maxinboxB;
  int **nbrlist;
  float sidelen[3], volume;
  const float *loc;
  int numonA = 0;   int numonB = 0;
  int paircount = 0;
  int maxpairsreached = 0;
  sqdist = pairdist * pairdist;

  // 1. Find grid size for binning
  // find bounding box for selected atoms
  find_minmax(posA, natomsA, A, min, max, &numonA);
  find_minmax(posB, natomsB, B, minB, maxB, &numonB);
  for (i=0; i<3; i++) {
    if (minB[i] < min[i]) min[i] = minB[i];
    if (maxB[i] > max[i]) max[i] = maxB[i];
  }

  // do sanity checks and complain if we've got bogus atom coordinates,
  // we shouldn't ever have density higher than 0.1 atom/A^3, but we'll
  // be generous and allow much higher densities.  
  if (maxpairs != -1) {
    vec_sub(sidelen, max, min);
    // include estimate for atom radius (1 Angstrom) in volume determination
    volume = fabsf((sidelen[0] + 2.0f) * (sidelen[1] + 2.0f) * (sidelen[2] + 2.0f));
    if (((numonA + numonB) / volume) > 1.0) {
      msgWarn << "vmd_gridsearch3: insane atom density" << sendmsg;
    }
  } 

  // I don't want the grid to get too large, otherwise I could run out
  // of memory.  Octrees would be cool, but I'll just limit the grid size
  // and let the performance degrade a little for pathological systems.
  // Note that sqdist is what gets used for the actual distance checks;
  // from here on out pairdist is only used to set the grid size, so we 
  // can set it to anything larger than the original pairdist.
  const int MAXBOXES = 4000000;
  totb = MAXBOXES + 1;

  // setup head of pairlist
  head = (GridSearchPair *) malloc(sizeof(GridSearchPair));
  head->next = NULL;
  cur = head;

  float newpairdist = pairdist;
  do {
    pairdist = newpairdist;
    xb = (int)((max[0]-min[0])/pairdist)+1;
    yb = (int)((max[1]-min[1])/pairdist)+1;
    zb = (int)((max[2]-min[2])/pairdist)+1;
    xytotb = yb * xb;
    totb = xytotb * zb;
    newpairdist = pairdist * 1.26f; // cbrt(2) is about 1.26
  } while (totb > MAXBOXES || totb < 1); // check for integer wraparound too
 
  // 2. Sort each atom into appropriate bins
  boxatomA = (int **) calloc(1, totb*sizeof(int *));
  numinboxA = (int *) calloc(1, totb*sizeof(int));
  maxinboxA = (int *) calloc(1, totb*sizeof(int));
  if (boxatomA == NULL || numinboxA == NULL || maxinboxA == NULL) {
    if (boxatomA != NULL)
      free(boxatomA);
    if (numinboxA != NULL)
      free(numinboxA);
    if (maxinboxA != NULL)
      free(maxinboxA);
    msgErr << "Gridsearch memory allocation failed, bailing out" << sendmsg;
    return NULL; // ran out of memory, bail out!
  }

  loc = posA;
  for (i=0; i<natomsA; i++) {
    if (A[i]) {
      int axb, ayb, azb, aindex, num;
      axb = (int)((loc[0] - min[0])/pairdist);
      ayb = (int)((loc[1] - min[1])/pairdist);
      azb = (int)((loc[2] - min[2])/pairdist);
      aindex = azb * xytotb + ayb * xb + axb;
      num = numinboxA[aindex];
      if (maxinboxA[aindex] == num) {
        void *tmp = realloc(boxatomA[aindex], (num+4)*sizeof(int));
        boxatomA[aindex] = (int *)tmp;
        maxinboxA[aindex] += 4;
      }
      boxatomA[aindex][num] = i;
      numinboxA[aindex]++;
    }
    loc += 3;
  }
  free(maxinboxA);

  boxatomB = (int **) calloc(1, totb*sizeof(int *));
  numinboxB = (int *) calloc(1, totb*sizeof(int));
  maxinboxB = (int *) calloc(1, totb*sizeof(int));
  if (boxatomB == NULL || numinboxB == NULL || maxinboxB == NULL) {
    if (boxatomB != NULL)
      free(boxatomB);
    if (numinboxB != NULL)
      free(numinboxB);
    if (maxinboxB != NULL)
      free(maxinboxB);
    msgErr << "Gridsearch memory allocation failed, bailing out" << sendmsg;
    return NULL; // ran out of memory, bail out!
  }

  loc = posB;
  for (i=0; i<natomsB; i++) {
    if (B[i]) {
      int axb, ayb, azb, aindex, num;
      axb = (int)((loc[0] - min[0])/pairdist);
      ayb = (int)((loc[1] - min[1])/pairdist);
      azb = (int)((loc[2] - min[2])/pairdist);
      aindex = azb * xytotb + ayb * xb + axb;
      num = numinboxB[aindex];
      if (maxinboxB[aindex] == num) {
        void *tmp = realloc(boxatomB[aindex], (num+4)*sizeof(int));
        boxatomB[aindex] = (int *)tmp;
        maxinboxB[aindex] += 4;
      }
      boxatomB[aindex][num] = i;
      numinboxB[aindex]++;
    }
    loc += 3;
  }
  free(maxinboxB);
  

  // 3. Build pairlists of atoms less than sqrtdist apart
  nbrlist = (int **) calloc(1, totb*sizeof(int *));
  if (make_neighborlist_sym(nbrlist, xb, yb, zb)) {
    if (boxatomA != NULL) {
      for (i=0; i<totb; i++) {
        if (boxatomA[i] != NULL) free(boxatomA[i]);
      }
      free(boxatomA);
    }
    if (boxatomB != NULL) {
      for (i=0; i<totb; i++) {
        if (boxatomB[i] != NULL) free(boxatomB[i]);
      }
      free(boxatomB);
    }
    if (nbrlist != NULL) {
      for (i=0; i<totb; i++) {
        if (nbrlist[i] != NULL) free(nbrlist[i]);
      }
      free(nbrlist);
    }
    free(numinboxA);
    free(numinboxB);
    msgErr << "Gridsearch memory allocation failed, bailing out" << sendmsg;
    return NULL; // ran out of memory, bail out!
  }

  for (aindex = 0; aindex < totb; aindex++) {
    int *tmpbox, *tmpnbr, *nbr;
    tmpbox = boxatomA[aindex];
    tmpnbr = nbrlist[aindex];
    for (nbr = tmpnbr; (*nbr != -1) && (!maxpairsreached); nbr++) {
      int *nbrboxB = boxatomB[*nbr];
      for (i=0; (i<numinboxA[aindex]) && (!maxpairsreached); i++) {
        const float *p1;
        int ind1 = tmpbox[i];
        p1 = posA + 3*ind1;
        for (j=0; (j<numinboxB[*nbr]) && (!maxpairsreached); j++) {  
          const float *p2;
          int ind2 = nbrboxB[j];
          p2 = posB + 3*ind2;
          if (!allow_double_counting && B[ind1] && A[ind2] && ind2<=ind1) continue; //don't double-count bonds XXX
          if (distance2(p1,p2) > sqdist) continue;

          if (maxpairs > 0) {
            if (paircount >= maxpairs) {
              maxpairsreached = 1;
              continue; 
            }   
          }

          add_link(cur, ind1, ind2);
          paircount++;
          cur = cur->next;
          cur->next = NULL;
        }
      }
    }
  } 

  for (i=0; i<totb; i++) {
    free(boxatomA[i]);
    free(boxatomB[i]);
    free(nbrlist[i]);
  }
  free(boxatomA);
  free(boxatomB);
  free(nbrlist);
  free(numinboxA);
  free(numinboxB);

  cur = head->next;
  free(head);

  if (maxpairsreached) 
    msgErr << "vmdgridsearch3: exceeded pairlist sanity check, aborted" << sendmsg;

  return cur;
}



GridSearchPairlist *vmd_gridsearch_bonds(const float *pos, const float *radii,
                                   int natoms, float pairdist, int maxpairs) {
  GridSearchPairlist *head, *cur;
  float min[3], max[3];
  int i, xb, yb, zb, xytotb, totb;
  int **boxatom, *numinbox, *maxinbox, **nbrlist;
  int numon = 0;
  float sidelen[3], volume;
  const float *loc;
  int paircount = 0;

  // find bounding box for selected atoms, and number of atoms in selection.
  find_minmax_all(pos, natoms, min, max);

  // do sanity checks and complain if we've got bogus atom coordinates,
  // we shouldn't ever have density higher than 0.1 atom/A^3, but we'll
  // be generous and allow much higher densities.  
  if (maxpairs != -1) {
    vec_sub(sidelen, max, min);
    // include estimate for atom radius (1 Angstrom) in volume determination
    volume = fabsf((sidelen[0] + 2.0f) * (sidelen[1] + 2.0f) * (sidelen[2] + 2.0f));
    if ((numon / volume) > 1.0) {
      msgWarn << "vmd_gridsearch_bonds: insane atom density" << sendmsg;
    }
  }

  // I don't want the grid to get too large, otherwise I could run out
  // of memory.  Octrees would be cool, but I'll just limit the grid size
  // and let the performance degrade a little for pathological systems.
  // Note that pairdist^2 is what gets used for the actual distance checks;
  // from here on out pairdist is only used to set the grid size, so we 
  // can set it to anything larger than the original pairdist.
  const int MAXBOXES = 4000000;
  totb = MAXBOXES + 1;

  float newpairdist = pairdist;
  do {
    pairdist = newpairdist;
    xb = (int)((max[0]-min[0])/pairdist)+1;
    yb = (int)((max[1]-min[1])/pairdist)+1;
    zb = (int)((max[2]-min[2])/pairdist)+1;
    xytotb = yb * xb;
    totb = xytotb * zb;
    newpairdist = pairdist * 1.26f; // cbrt(2) is about 1.26
  } while (totb > MAXBOXES || totb < 1); // check for integer wraparound too
 
  // 2. Sort each atom into appropriate bins
  boxatom = (int **) calloc(1, totb*sizeof(int *));
  numinbox = (int *) calloc(1, totb*sizeof(int));
  maxinbox = (int *) calloc(1, totb*sizeof(int));
  if (boxatom == NULL || numinbox == NULL || maxinbox == NULL) {
    if (boxatom != NULL)
      free(boxatom);
    if (numinbox != NULL)
      free(numinbox);
    if (maxinbox != NULL)
      free(maxinbox);
    msgErr << "Gridsearch memory allocation failed, bailing out" << sendmsg;
    return NULL; // ran out of memory, bail out!
  }

  loc = pos;
  const float invpairdist = 1.0f / pairdist; 
  for (i=0; i<natoms; i++) {
    int axb, ayb, azb, aindex, num;

    axb = (int)((loc[0] - min[0])*invpairdist);
    ayb = (int)((loc[1] - min[1])*invpairdist);
    azb = (int)((loc[2] - min[2])*invpairdist);
    aindex = azb * xytotb + ayb * xb + axb;
    num = numinbox[aindex];
    if (maxinbox[aindex] == num) {
      void *tmp = realloc(boxatom[aindex], (num+4)*sizeof(int));
      boxatom[aindex] = (int *)tmp;
      maxinbox[aindex] += 4;
    }
    boxatom[aindex][num] = i;
    numinbox[aindex]++;

    loc += 3;
  }
  free(maxinbox);
 
  nbrlist = (int **) calloc(1, totb*sizeof(int *));
  if (make_neighborlist(nbrlist, xb, yb, zb)) {
    if (boxatom != NULL) {
      for (i=0; i<totb; i++) {
        if (boxatom[i] != NULL) free(boxatom[i]);
      }
      free(boxatom);
    }
    if (nbrlist != NULL) {
      for (i=0; i<totb; i++) {
        if (nbrlist[i] != NULL) free(nbrlist[i]);
      }
      free(nbrlist);
    }
    free(numinbox);
    msgErr << "Gridsearch memory allocation failed, bailing out" << sendmsg;
    return NULL; // ran out of memory, bail out!
  }

  // if maxpairs is "unlimited", set it to the biggest positive int
  if (maxpairs < 0) {
    maxpairs = 2147483647;
  }

  // setup head of pairlist
  head = (GridSearchPairlist *) malloc(sizeof(GridSearchPairlist));
  head->next = NULL;
  paircount = vmd_bondsearch_thr(pos, radii, head, totb, 
                                 boxatom, numinbox, nbrlist, 
                                 maxpairs, pairdist);

  for (i=0; i<totb; i++) {
    free(boxatom[i]);
    free(nbrlist[i]);
  }
  free(boxatom);
  free(nbrlist);
  free(numinbox);

  cur = head->next;
  free(head);

  if (paircount > maxpairs) 
    msgErr << "vmdgridsearch_bonds: exceeded pairlist sanity check, aborted" << sendmsg;

  return cur;
}



// bond search thread parameter structure
typedef struct {
  int threadid;
  int threadcount;
  vmd_mutex_t *pairlistmutex;
  GridSearchPairlist * head;
  float *pos;
  float *radii;
  int totb;
  int **boxatom;
  int *numinbox;
  int **nbrlist;  
  int maxpairs;
  float pairdist;
} bondsearchthrparms;

// thread prototype 
static void * bondsearchthread(void *);

// setup and launch bond search threads
int vmd_bondsearch_thr(const float *pos, const float *radii,
                       GridSearchPairlist * head, 
                       int totb, int **boxatom, 
                       int *numinbox, int **nbrlist, int maxpairs, 
                       float pairdist) {
  int i;
  bondsearchthrparms *parms;
  vmd_thread_t * threads;
  vmd_mutex_t pairlistmutex; ///< guards pairlist
  vmd_mutex_init(&pairlistmutex); // init mutex before use

  int numprocs = vmd_thread_numprocessors();

  /* allocate array of threads */
  threads = (vmd_thread_t *) calloc(numprocs * sizeof(vmd_thread_t), 1);

  /* allocate and initialize array of thread parameters */
  parms = (bondsearchthrparms *) malloc(numprocs * sizeof(bondsearchthrparms));
  for (i=0; i<numprocs; i++) {
    parms[i].threadid = i;
    parms[i].threadcount = numprocs;
    parms[i].pairlistmutex = &pairlistmutex;
    parms[i].head = NULL;
    parms[i].pos = (float *) pos;
    parms[i].radii = (float *) radii;
    parms[i].totb = totb;
    parms[i].boxatom = boxatom;
    parms[i].numinbox = numinbox;
    parms[i].nbrlist = nbrlist;  
    parms[i].maxpairs = maxpairs;
    parms[i].pairdist = pairdist;
  }

#if defined(VMDTHREADS)
  /* spawn child threads to do the work */
  for (i=0; i<numprocs; i++) {
    vmd_thread_create(&threads[i], bondsearchthread, &parms[i]);
  }

  /* join the threads after work is done */
  for (i=0; i<numprocs; i++) {
    vmd_thread_join(threads[i], NULL);
  }
#else
  bondsearchthread(&parms[0]); // single-threaded code
#endif

  // assemble final pairlist from sublists
  for (i=0; i<numprocs; i++) {
    if (parms[i].head != NULL) {
      GridSearchPairlist *tmp = head->next;
      head->next = parms[i].head;
      parms[i].head->next = tmp;
    }
  }

  vmd_mutex_destroy(&pairlistmutex); // destroy mutex when finished

  /* free thread parms */
  free(parms);
  free(threads);

  return 0;
}

static void * bondsearchthread(void *voidparms) {
  int i, j, aindex;
  int paircount = 0;

  bondsearchthrparms *parms = (bondsearchthrparms *) voidparms;

  const int threadid = parms->threadid;
  const int threadcount = parms->threadcount;
  vmd_mutex_t *pairlistmutex = parms->pairlistmutex;
  const float *pos = parms->pos;
  const float *radii = parms->radii;
  const int totb = parms->totb;
  const int **boxatom = (const int **) parms->boxatom;
  const int *numinbox = parms->numinbox;
  const int **nbrlist = (const int **) parms->nbrlist; 
  const int maxpairs = parms->maxpairs;
  const float pairdist = parms->pairdist;

  ResizeArray<int> *pairs = new ResizeArray<int>;
  float sqdist = pairdist * pairdist;

  msgtimer msgt;
  msg_timer_init(&msgt, 5);
  for (aindex = threadid; (aindex < totb) && (paircount < maxpairs); aindex+=threadcount) {
    const int *tmpbox, *nbr;
    tmpbox = boxatom[aindex];
    int anbox = numinbox[aindex];

    if (((aindex & 255) == 0) && msg_timer_timeout(&msgt)) {
      char tmpbuf[128];
      sprintf(tmpbuf, "%6.2f", (100.0f * aindex) / (float) totb);
//  XXX: we have to use printf here as msgInfo is not thread-safe.
//  msgInfo << "vmd_gridsearch_bonds (thread " << threadid << "): " 
//          << tmpbuf << "% complete" << sendmsg;
      printf("vmd_gridsearch_bonds (thread %d): %s %% complete\n", 
        threadid, tmpbuf); 
    }

    for (nbr = nbrlist[aindex]; (*nbr != -1) && (paircount < maxpairs); nbr++) {
      int nnbr=*nbr;
      const int *nbrbox = boxatom[nnbr];
      int nbox=numinbox[nnbr];
      int self = (aindex == nnbr) ? 1 : 0;

      for (i=0; (i<anbox) && (paircount < maxpairs); i++) {
        int ind1 = tmpbox[i];
        const float *p1 = pos + 3*ind1;

        // skip over self and already-tested atoms
        int startj = (self) ? i+1 : 0;

        for (j=startj; (j<nbox) && (paircount < maxpairs); j++) {
          int ind2 = nbrbox[j];
          const float *p2 = pos + 3*ind2;
          float dx = p1[0] - p2[0];
          float dy = p1[1] - p2[1];
          float dz = p1[2] - p2[2];
          float ds2 = dx*dx + dy*dy + dz*dz;

          // perform distance test, but ignore pairs between atoms
          // with nearly identical coords
          if ((ds2 > sqdist) || (ds2 < 0.001))
            continue;

          if (radii) { // Do atom-specific distance check
            float cut = 0.6f * (radii[ind1] + radii[ind2]);
            if (ds2 > cut*cut)
              continue;
          }

          pairs->append(ind1);
          pairs->append(ind2);
          paircount++;
        }
      }
    }
  }

  // setup results pairlist node
  GridSearchPairlist *head;
  head = (GridSearchPairlist *) malloc(sizeof(GridSearchPairlist));
  head->next = NULL;
  head->pairlist = pairs;

  vmd_mutex_lock(pairlistmutex);   // lock pairlist before update
  parms->head = head;
  vmd_mutex_unlock(pairlistmutex); // unlock pairlist after update

  return NULL;
}





// determine bonds from position of atoms previously read.
// If cutoff < 0, use vdw radius to determine if bonded.
int vmd_bond_search(BaseMolecule *mol, const Timestep *ts, 
                    float cutoff, int dupcheck) {
  const float *pos;
  int natoms;
  int i;
  const float *radius = mol->radius();
 
  if (ts == NULL) {
    msgErr << "Internal Error: NULL Timestep in vmd_bond_search" << sendmsg;
    return 0;
  }

  natoms = mol->nAtoms; 
  if (natoms == 0 || cutoff == 0.0)
    return 1;

  msgInfo << "Determining bond structure from distance search ..." << sendmsg;

  if (dupcheck)
    msgInfo << "Eliminating bonds duplicated from existing structure..." << sendmsg;

  // Set box distance to either the cutoff, or 1.2 times the largest VDW radius
  float dist = cutoff; 
  if (cutoff < 0.0) {
    for (i=0; i<natoms; i++) {  
      float rad = radius[i];
      if (rad > dist) dist = rad;
    }
    dist = 1.2f * dist;
  }
  
  pos = ts->pos; 
 
  // Call the bond search to get all atom pairs within the specified distance
  // XXX set maxpairs to 27 bonds per atom, which ought to be ridiculously high
  //     for any real structure someone would load, but well below N^2
  GridSearchPairlist *pairlist = vmd_gridsearch_bonds(pos, 
                                                 (cutoff < 0) ? radius : NULL,
                                                 natoms, dist, natoms * 27);

  // Go through the pairlist adding validated bonds freeing nodes as we go. 
  GridSearchPairlist *p, *tmp; 
  for (p = pairlist; p != NULL; p = tmp) {
    int numpairs = p->pairlist->num() / 2;
    
    for (i=0; i<numpairs; i++) {
      int ind1 = (*p->pairlist)[i*2  ]; 
      int ind2 = (*p->pairlist)[i*2+1];

      MolAtom *atom1 = mol->atom(ind1);
      MolAtom *atom2 = mol->atom(ind2);

      // don't bond atoms that aren't part of the same conformation
      // or that aren't in the all-conformations part of the structure
      if ((atom1->altlocindex != atom2->altlocindex) &&
          ((mol->altlocNames.name(atom1->altlocindex)[0] != '\0') && 
           (mol->altlocNames.name(atom2->altlocindex)[0] != '\0'))) {
        continue;
      }

      // Prevent hydrogens from bonding with each other.
#if 1
      // XXX must use the atom name strings presently because the
      // hydrogen flags aren't necessarily set by the time the bond search
      // code executes.  It may soon be time to do something a different
      // with per-atom flag storage so that the atom types can be setup
      // during the earliest phase of structure analysis, eliminating this
      // and other potential gotchas.
      if (!IS_HYDROGEN(mol->atomNames.name(atom1->nameindex)) ||
          !IS_HYDROGEN(mol->atomNames.name(atom2->nameindex)) ) {
#else
      // Use atomType info derived during initial molecule analysis for speed.
      if (!(atom1->atomType == ATOMHYDROGEN) ||
          !(atom2->atomType == ATOMHYDROGEN)) {
#endif
        // Add a bond, bondorder defaults to 1
        if (dupcheck)
          mol->add_bond_dupcheck(ind1, ind2, 1);
        else
          mol->add_bond(ind1, ind2, 1);
      }
    }

    // free this pairlist node and its ResizeArray of pairs
    tmp = p->next;
    delete p->pairlist;
    free(p);
  }

  return 1;
}
