/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995-2007 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Animation.h,v $
 *	$Author: johns $	$Locker:  $		$State: Exp $
 *	$Revision: 1.38 $	$Date: 2007/01/12 20:08:16 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The Animation class, which stores a list of pointers to Timestep objects
 * that contain 3D coordinates and other data that varies with time.
 *
 ***************************************************************************/
#ifndef ANIMATION_H
#define ANIMATION_H

#include "ResizeArray.h"     // need ResizeArray for the AnimationData template

#define SPEED_FACTOR	0.5f // max fraction of a sec between redraws


/// Animation defines and implements a set of controls for how the current 
/// frame can change with time and with user input.  
class Animation {
public:
  // enums for Animation options
  enum AnimDir 	{ ANIM_FORWARD, ANIM_FORWARD1, ANIM_REVERSE, ANIM_REVERSE1, 
                  ANIM_PAUSE, ANIM_TOTAL_DIRS };
  enum AnimStyle  { ANIM_ONCE, ANIM_LOOP, ANIM_ROCK, ANIM_TOTAL_STYLES };

protected:
  /// child classes call this with the new frame number after appending a frame 
  void append_frame(int);

  void delete_frame(int);  ///< child classes call this after deleting a frame

private:
  /// last time the image was drawn, for use with determining speed
  double lastTime;

  /// flag saying whether we should set the frameChanged setting
  int needSetFrameChanged;

  /// number of frames I'm keeping track of; this is changed only by subclasses
  int numFrames;           

  /// set the current frame position to a new value
  void change_current_frame(int n, int setChangeFlag = 1);

  int currFrame;       ///< current frame number (0 ... # frames - 1)
  int frameSkip;       ///< frames to skip to next position when animating
  float Speed;         ///< animation speed, from 0.0 (slowest) to 1.0 (fastest)
  AnimDir animDir;     ///< current animation direction
  AnimStyle animStyle; ///< what to do when you get to the end of the loop
  int moveTo;          ///< frame to jump to (no jump if moveTo < 0)
  float currTime;      ///< current 'time' elapsed since beginning of animation

public:
  Animation(void);           
  virtual ~Animation(void);
  
  /// total number of frames currently stored
  int num(void) { return numFrames; }

  /// return current time
  float time(void) { return currTime; }

  /// return the current frame number (frames 0...(frames -1); -1 => no frames)
  int frame(void) { return currFrame; }

  /// return whether there is a 'current' frame (i.e. are any frames present)
  int is_current(void) { return (num() > 0 && currFrame >= 0); }

  /// explicitely move the current frame to the given one.  This does NOT
  /// update anything else, so it should only be used when temporarily
  /// changing the value to be returned by current().
  /// There are more error message here than change_current_frame since
  /// this is the one available to the Tcl 'molinfo' command
  void override_current_frame(int);

  /// signal that the current frame has changed somehow
  void curr_frame_changed(void);

  /// move to the specified frame, if possible
  void goto_frame(int fr) { moveTo = fr; }

  /// update the animation list based on current mode; return current frame.
  virtual int anim_update(void);

  void skip(int newsk);                            ///< set # frames to skip
  int skip(void) { return frameSkip; }             ///< get # frames to skip
  void anim_dir(AnimDir ad) { animDir = ad; }      ///< set animation direction
  AnimDir anim_dir(void) { return animDir; }       ///< get animation direction
  void anim_style(AnimStyle as);                   ///< set animation style
  AnimStyle anim_style(void) { return animStyle; } ///< get animation style


  /// animation speed methods:
  ///	newsp should be from 0 (min speed) to 1 (max speed)
  float speed(float newsp);
  float speed(void) { return (float) (Speed / SPEED_FACTOR); }
};

/// static storage for strings describing animation styles
extern const char *animationStyleName[Animation::ANIM_TOTAL_STYLES];

/// static storage for strings describing animation directions
extern const char *animationDirName[Animation::ANIM_TOTAL_DIRS];



/// AnimationData provides methods for setting, duplicating, and removing items
/// to be animated. 
template<class T>
class AnimationData : public Animation {
public:
  AnimationData(void) : tsList(2) {}
  ~AnimationData(void) {
    for(int i=(tsList.num() - 1); i >= 0; i--) {
      delete tsList[i];
      tsList.remove(i);
    }
  }

  /// delete a specific frame 
  void delete_frame(int n) {
    if(n >= 0 && n < tsList.num()) {
      delete tsList[n];           // delete object stored at Nth position
      tsList.remove(n);           // delete Nth position
      Animation::delete_frame(n);
    }
  }

  /// append a new frame and return its number
  void append_frame(T ts) {
    tsList.append(ts);
    Animation::append_frame(tsList.num()-1);
  }

  /// return 'current' frame, if available; NULL otherwise
  T current(void) {
    T retval = NULL;
    if(frame() >= 0)
      retval = tsList[frame()];
    return retval;
  }

  /// return Nth frame, referred to here as an 'item'
  T item(int n) {
    T retval = NULL;
    if(n >= 0 && n < tsList.num())
      retval = tsList[n];
    return retval;
  }
 
private:
  ResizeArray<T> tsList;
};
 

#endif

