/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995-2007 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Animation.C,v $
 *	$Author: johns $	$Locker:  $		$State: Exp $
 *	$Revision: 1.49 $	$Date: 2007/01/12 20:08:16 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The Animation class, which stores a list of pointers to Timestep objects
 * that contain 3D coordinates and other data that varies with time.
 *
 ***************************************************************************/

#include "Animation.h"
#include "utilities.h"  // for time_of_day

// strings describing animation styles
const char *animationStyleName[Animation::ANIM_TOTAL_STYLES] = {
	"Once", "Loop", "Rock" 
};
const char *animationDirName[Animation::ANIM_TOTAL_DIRS] = {
  "forward", "next", "reverse", "prev", "pause"
};

// constructor
Animation::Animation() {

  lastTime = time_of_day();	// get the current time for determining speed
  currFrame = (-1);		// no frames yet
  moveTo = (-1);		// we do not want to jump to a frame
  speed(1.0);			// set initial speed to maximum
  skip(1);			// don't skip any frames yet
  anim_dir(ANIM_PAUSE);		// not yet animating
  anim_style(ANIM_LOOP);	// loop back to beginning when end reached
  currTime = 0.0;		// at beginning of animation
  needSetFrameChanged = TRUE; // data has changed; now exists
  numFrames = 0;
}


//// destructor

Animation::~Animation(void) { }

void Animation::curr_frame_changed(void) { 
  needSetFrameChanged = TRUE; 
}

void Animation::override_current_frame(int n) {
  if (numFrames == 0) return; 
  if (n < 0) {
    currFrame = 0;
  } else if (n >= numFrames) {
    currFrame = numFrames-1;
  } else {
    currFrame = n;
  }
}

void Animation::change_current_frame(int n, int setChangeFlag) {
  currFrame = n;
  if (setChangeFlag)
    curr_frame_changed();
}
 
// update the animation list based on current mode; return if curr frame change
int Animation::anim_update(void) {
  double cur_time, dt;
  int i;

  // If we don't have more than one frame, there is nothing to do.
  // Must check needSetFrameChanged since adding/deleting frames may 
  // have changed the frame we need to display (or removed all frames).
  if (num() < 2) {
    int frameChanged = needSetFrameChanged;
    needSetFrameChanged = FALSE;
    return frameChanged;
  }  

  // If moveTo is greater than our number of frames, go to the last frame.
  if (moveTo >= num())
    moveTo = num() - 1;

  // if a jump has been requested, move to that frame and just return
  if (moveTo >= 0 && moveTo < num()) {
    if (moveTo != currFrame)
      change_current_frame(moveTo);
    moveTo = (-1);
    anim_dir(ANIM_PAUSE);            // pause after jumping
    lastTime = time_of_day();        // reset timing
  } else if (animDir == ANIM_PAUSE || num() <= 0) {
    animDir = ANIM_PAUSE;            // if just paused, do nothing
  } else if (num() > 0) {
    // check the clock to see if the current frame should be updated
    if (Speed  < 1.0) {
      cur_time = time_of_day();
      dt = cur_time - lastTime;
      if (dt <= (SPEED_FACTOR - Speed)) {
        int frameChanged = needSetFrameChanged;
        needSetFrameChanged = FALSE;
        return frameChanged;
      } else {
        lastTime = cur_time;
      }
    }

    // skip the current frame ahead the proper amount
    for (i=0; i < frameSkip; i++) {
      if (animDir == ANIM_REVERSE || animDir == ANIM_REVERSE1) {
        if (currFrame <= 0) {
          if (animStyle == ANIM_LOOP || animDir == ANIM_REVERSE1) {
	    change_current_frame(num() - 1);
          } else if (animStyle == ANIM_ROCK) {
            animDir = ANIM_FORWARD;
          } else if (animStyle == ANIM_ONCE) {
            animDir = ANIM_PAUSE;
          }
        } else {
          change_current_frame(currFrame - 1);
        }

      } else if (animDir == ANIM_FORWARD || animDir == ANIM_FORWARD1) {
        if (currFrame >= (num() - 1)) {
          if (animStyle == ANIM_LOOP || animDir == ANIM_FORWARD1) {
	    change_current_frame(0);
          } else if (animStyle == ANIM_ROCK) {
            animDir = ANIM_REVERSE;
          } else if (animStyle == ANIM_ONCE) {
            animDir = ANIM_PAUSE;
          }
        } else {
          change_current_frame(currFrame + 1);
        }
      }
    }

    if (animDir == ANIM_FORWARD1 || animDir == ANIM_REVERSE1)
      animDir = ANIM_PAUSE;
  }
  
  // indicate whether the current frame has changed
  int frameChanged = needSetFrameChanged;
  needSetFrameChanged = FALSE;
  return frameChanged;
}

void Animation::skip(int newsk) { 
  frameSkip = ( newsk >= 1 ? newsk : 1);
}

float Animation::speed(float newsp)  {
  if (newsp < 0.0)
    Speed = 0.0;
  else if (newsp > 1.0)
    Speed = SPEED_FACTOR;
  else
    Speed = newsp*SPEED_FACTOR;

  return Speed;
}

void Animation::anim_style(AnimStyle as) 
{ 
  animStyle = as; 
}

void Animation::delete_frame(int n) {
  numFrames--;
  if (num() == 0) {
    change_current_frame(-1);
    moveTo = (-1);
  } else {
    if (moveTo > n)
      moveTo--;
    if (currFrame >= num()) {
      change_current_frame(num() - 1);
    }
  }
}

void Animation::append_frame(int newfr) {
  numFrames++;
  if (currFrame < 0) {
    // must set current frame because this frame is the first one
    change_current_frame(0);
  }
  else if (currFrame >= newfr)
    // inserted frame before current one
    change_current_frame(currFrame + 1);

  // move to new position ONLY if currently paused
  if (animDir == ANIM_PAUSE)
    change_current_frame(newfr);
}

