## NAME: scale_bar
## 
## $Id: scalebar.tcl,v 1.5 2006/05/04 03:15:13 johns Exp $
##
## SYNOPSIS:
##   scale_bar draws a scale bar on the screen to show a size scale
##
## AUTHOR:
##      John Stone
##

# This function draws a color bar to show the color scale
# length = the length of the color bar
# width = the width of the color bar

package provide scalebar 1.0

namespace eval ::ScaleBar:: {
  namespace export scale_bar delete_scale_bar
  variable w
  variable bar_mol          -1
  variable lengthsetting    10.0
  variable widthsetting     1.0
  variable fixedsetting     1
  variable textcolor        white
}

proc ::ScaleBar::scale_bar {{length 10.0} {width 1.0} {fixed 1} {text white}} {
  variable bar_mol
  display update off

  set old_top [molinfo top]
  if { $old_top == -1 } {
    puts "Scale Bar Plugin: No molecules loaded"
    return -1;
  }

  if { [display get projection] != "Orthographic" } {
    puts "Scale Bar Plugin: display must be set to orthographic projection mode"
    display projection Orthographic
  }

  # Create a seperate molid to draw in, so it's possible for the user to 
  # delete the bar.
  set bar_mol [mol new]
  mol top $bar_mol
  mol rename top "Scale Bar"


  # We want to draw relative to the location of the top mol so that the bar 
  # will always show up nicely.
  set center [molinfo $old_top get center]
  set center [regsub -all {[{}]} $center ""]
  set center [split $center]

  puts "[lindex $center 0]"
  
  # draw the bar
  set start_y [expr [lindex $center 1] - (0.5 * $length)]
  set use_x [expr [lindex $center 0]]
  set use_z [lindex $center 2]
  set step [expr $length / 1024.0]

  puts "x: $use_x y: $start_y z: $use_z"

  draw color $text
  for { set lv 0 } { $lv < 1024 } { incr lv 1 } {
    set cur_y [ expr $start_y + $lv * $step ]
    draw line "$use_x $cur_y $use_z"  "[expr $use_x+$width] $cur_y $use_z"
  }

  # draw the labels
  set label_num 2
  set max $length
  set min 0
  set coord_x [expr $width + $use_x];
  set step_size [expr $length / $label_num]
  set color_step [expr 1024.0/$label_num]
  set value_step [expr ($max - $min ) / double ($label_num)]
  
  draw color $text
  for {set i 0} {$i <= $label_num } { incr i 1} {
    set coord_y [expr $start_y + ($i * $step_size) ]
    set cur_text [expr $min + $i * $value_step ]
    draw text  " $coord_x $coord_y $use_z"  [format %6.2f  $cur_text]
  }

  # re-set top
  mol top $old_top
  display resetview
  display update on

  # If a fixed bar was requested...
  if {$fixed == 1} {
    mol fix $bar_mol
  }

  return 0
}

proc ::ScaleBar::delete_scale_bar { } {
  variable bar_mol
  mol delete $bar_mol
}


proc ::ScaleBar::gui { } {
  variable w
  variable lengthsetting
  variable widthsetting
  variable minvalue
  variable maxvalue
  variable axislabels
  variable textcolor

  # If already initialized, just turn on
  if { [winfo exists .scalebargui] } {
    wm deiconify $w
    return
  }

  set w [toplevel ".scalebargui"]
  wm title $w "Scale Bar"
  wm resizable $w 0 0

  ##
  ## make the menu bar
  ##
  frame $w.menubar -relief raised -bd 2 ;# frame for menubar
  pack $w.menubar -padx 1 -fill x
  menubutton $w.menubar.help -text "Help   " -underline 0 -menu $w.menubar.help.menu

  ##
  ## help menu
  ##
  menu $w.menubar.help.menu -tearoff no
  $w.menubar.help.menu add command -label "Help..." -command "vmd_open_url [string trimright [vmdinfo www] /]/plugins/scalebar"
  pack $w.menubar.help

  frame $w.length
  label $w.length.label -text "Bar length"
  entry $w.length.entry -width 4 -relief sunken -bd 2 \
    -textvariable ::ScaleBar::lengthsetting
  pack $w.length.label $w.length.entry -side left -anchor w

  frame $w.width
  label $w.width.label -text "Bar width"
  entry $w.width.entry -width 4 -relief sunken -bd 2 \
    -textvariable ::ScaleBar::widthsetting
  pack $w.width.label $w.width.entry -side left -anchor w

  frame $w.textcolor
  label $w.textcolor.label -text "Color of labels"
  tk_optionMenu $w.textcolor.chooser ::ScaleBar::textcolor \
    "blue" \
    "red" \
    "gray" \
    "orange" \
    "yellow" \
    "tan" \
    "silver" \
    "green" \
    "white" \
    "pink" \
    "cyan" \
    "purple" \
    "lime" \
    "mauve" \
    "ochre" \
    "iceblue" \
    "black" 

  pack $w.textcolor.label $w.textcolor.chooser -side left -anchor w

  button $w.drawcolorscale -text "Draw Scale Bar" \
    -command {::ScaleBar::scale_bar $::ScaleBar::lengthsetting $::ScaleBar::widthsetting $::ScaleBar::fixedsetting $::ScaleBar::textcolor}

  pack $w.menubar $w.length $w.width \
    $w.textcolor $w.drawcolorscale -anchor w -fill x

  return $w
}

proc scalebar_tk_cb { } {
  ::ScaleBar::gui
  return $::ScaleBar::w
}

