
#include "rmsdTools.h"


// Constructor
//
RmsdTools::RmsdTools(StructureAlignment* sa)
  : alignment(sa) {
  
  int structCount = alignment->getStructureCount();
  int i;

  rmsdScores = new float*[structCount];
  for (i=0; i<structCount; i++) {
    rmsdScores[i] = new float[structCount];
    for (int j=0; j<structCount; j++) {
      rmsdScores[i][j] = 0;
    }
  }
  rmsdPerRes = new float*[structCount];
  for (i=0; i<structCount; i++) {
    rmsdPerRes[i] = new float[alignment->getLength()];
    for (int j=0; j<alignment->getLength(); j++) {
      rmsdPerRes[i][j] = 0.0;
    }
  }
}


// Destructor
//
RmsdTools::~RmsdTools() {
  
  delete rmsdScores;
  delete rmsdPerRes;

  return;
}


// rmsd
//
int RmsdTools::rmsd() {

  Coordinate3D* coord1 = 0;
  Coordinate3D* coord2 = 0;
  int structCount = alignment->getStructureCount();
  float distance = 0.0;

  float** norms = new float*[structCount];
  int i=0;
  for (i=0; i<structCount; i++) {
    norms[i] = new float[structCount];
    for (int j=0; j<structCount; j++) {
      norms[i][j] = 0;
    }
  }
  
  for (int row1=0; row1<structCount; row1++) {
    for (int row2=row1+1; row2<structCount; row2++) {
      for (int col=0; col<alignment->getLength(); col++) {
	if ( !(alignment->getAlphabet()->isGap(alignment->getSymbol(row1,col))) &&
	     !(alignment->getAlphabet()->isGap(alignment->getSymbol(row2,col))) ) {
	  coord1 = alignment->getCoordinate(row1,col);
	  coord2 = alignment->getCoordinate(row2,col);
	  distance = coord1->getDistanceTo(coord2);
	  rmsdScores[row1][row2] += pow(distance,2);
	  rmsdScores[row2][row1] += pow(distance,2);
	  norms[row1][row2]++;
	  norms[row2][row1]++;
	}
      }
    }
  }

  for (i=0; i<structCount; i++) {
    for (int j=0; j<structCount; j++) {
      if (i != j) {
	if (norms[i][j] > 0) {
	  rmsdScores[i][j] /= norms[i][j];
	  rmsdScores[i][j] = sqrt(rmsdScores[i][j]);
	}
	else {
	  rmsdScores[i][j] = 0;
	  printf("Error - RmsdTools::rmsd, divide by zero\n");
	  printf("   norms[%d][%d]\n",i,j);
	}
      }
    }
  }

  delete norms;

  return 1;
}


// rmsdPerResidue
//   Calculate RMSD between aligned residues of two structures;
//   -1 if there is at least one gap in the column
int RmsdTools::rmsdPerResidue() {

  for (int row=1; row<alignment->getStructureCount(); row++) {
      for (int col=0; col<alignment->getLength(); col++) {
        if ( !(alignment->getAlphabet()->isGap(alignment->getSymbol(0,col))) && !(alignment->getAlphabet()->isGap(alignment->getSymbol(row,col))) ) {
          Coordinate3D* coord1 = alignment->getCoordinate(0,col);
          Coordinate3D* coord2 = alignment->getCoordinate(row,col);
          rmsdPerRes[row][col] = coord1->getDistanceTo(coord2);
        }
        else {
          rmsdPerRes[row][col] = -1.0;
        }
      }
  }

  return 1;
}


// printRmsd
//
int RmsdTools::printRmsd(FILE* outfile) {

  if (rmsdScores == 0) {
    printf("Error: RmsdTools::printRmsd\n");
    return 0;
  }

  int structCount = alignment->getStructureCount();
  
  for (int i=0; i<structCount; i++) {
    int j=0;
    fprintf(outfile,"%6.4f", rmsdScores[i][j]);
    for (j=1; j<structCount; j++) {
      fprintf(outfile," %6.4f", rmsdScores[i][j]);
    }
    fprintf(outfile,"\n");
  }

  return 1;
}


// printRmsdPerResidue
//
int RmsdTools::printRmsdPerResidue(FILE* outfile) {

  if (rmsdPerRes == 0) {
    printf("Error: RmsdTools::printRmsdPerResidue\n");
    return 0;
  }

  for (int i=0; i<alignment->getStructureCount(); i++) {
      for (int j=0; j<alignment->getLength(); j++) {
        fprintf(outfile,"%6.4f ", rmsdPerRes[i][j]);
      }
      fprintf(outfile,"\n");
  }

  return 1;
}
