/************************************************************************
 * This program takes time-data and calculates the
 * powerspectrum/fourier transform of the autocorrelation function.
 * This is a re-write of the fourier.x code written by volker kleinschmidt
 * and harald forbert as a tcl plugin for VMD by axel kohlmeyer.
 * (c) 2002-2005 Harald Forbert, Volker Kleinschmidt (c) 2002-2006 Axel Kohlmeyer.
 *
 * usage: calc_specden(<ndata>,<input>,<deltat>,<maxfreq>,<temp>,<specr>);
 * <ndata>   number of data sets
 * <input>   time series data.
 * <deltat>  time difference between data sets (in atomic units).
 * <maxfreq> max fequency (in wavenumbers).
 * <temp>    temperature (in kelvin)
 * <specr>   resolution of spectrum (1 gives maximal resolution and noise).
 *
 * The output will contain the power-spectrum up to a maximum frequency.
 * The format of this file is:
 *
 * <frequency> <value1> <value2> <value3> <value4> <value5>
 * 
 * <frequency> is the frequency in wave numbers.
 * <value1>    is the plain power spectrum of the input data (normalized to
 *             unity in the output frequency range.
 * <value2>    is the power spectrum with a prefactor of 
 *             \omega ( 1 - \exp(-\beta \hbar \omega) )
 *             corresponding to the classical/Gordon limit.
 * <value3>    is the power spectrum with a prefactor of
 *             \omega \tanh(\beta \hbar \omega/2)
 *             corresponding to the Kubo correction
 * <value4>    is the power spectrum with a prefactor of
 *             \omega \beta \hbar \omega
 *             corresponding to the high temperature / harmonic limit
 *             NOTE: this is the _recommended_ correction factor.
 * <value5>    is the power spectrum with a prefactor of
 *             \omega ( 1 - \exp(-\beta \hbar \omega) ) *
 *                                              \exp(\beta \hbar \omega /2)
 *             corresponding to Schofield's correction
 *
 * All spectra with their corresponding prefactor are separately normalized
 * in the output range to sum up to unity.
 *
 * Note: the index of refraction of the medium is set to unity.
 *************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "specden.h"

/* helper function */
static void fourier_sum (const int n, const double *input, const double omega, 
                         double *cos_sum, double *sin_sum);

int calc_specden(const int ndat, double *input, double *output, const int normtype,
            const int specr, const double maxfreq, const double deltat, const double temp) 
{
  int    nn, i, j, k;
  double wave_fac, bh, dt, t, c, f, s, e;

  double *ftrans, *wtrans;
  double norm_fourier, norm_classic, norm_kubo, norm_harmonic, norm_schofield;
   
  wave_fac = 219474.0/deltat;
  bh       = 1.05459e-34/1.38066e-23/2.41889e-17/deltat/temp;
  
  if (specr < 1) {
    fprintf(stderr, "\nspecden spectrum resolution factor must be bigger or equal 1.\n");
    return -20;
  }

  /* number of frequencies */
  nn = (int) ((double)ndat)*maxfreq/wave_fac/(2.0*M_PI);
  if (nn+1 > ndat) {
    fprintf(stderr, "Maximum frequency too large\n");
    return -40;
  }
  nn = nn/specr;
  
  ftrans = malloc((nn+2)*sizeof(double));
  if (ftrans == NULL) {
    fprintf(stderr, "Out of memory, while trying to allocate array 'ftrans'.\n");
    return -50;
  }
  wtrans = malloc((nn+2)*sizeof(double));
  if (ftrans == NULL) {
    fprintf(stderr, "Out of memory, while trying to allocate array 'wtrans'.\n");
    return -60;
  }

  /* read data and apply windowing function */
  for (i=1; i<ndat+1; ++i) {
    double win;
    
    win=((double)(2*i-ndat-1))/((double)(ndat+1));
    win=1.0-win*win;
    input[3*i]   *=win;
    input[3*i+1] *=win;
    input[3*i+2] *=win;
  }
  input[3*ndat+3] = 0.0;
  input[3*ndat+4] = 0.0;
  input[3*ndat+5] = 0.0;
  
  dt = 2.0*specr*M_PI/((ndat+1)*specr);
  for (i=0; i<nn+1; ++i) {
    
    t = 2.0*((double)(i*specr))*M_PI/((double)(ndat+1));
    c = 0.0;
    
    for (k=0; k < specr; ++k) {
      f = 0.0;

      /* sum over all three dimensions */
      for (j=0; j<3; ++j) {
        fourier_sum(ndat,(input+j), t+(double)k*dt, &e, &s);
        f = f + e*e+s*s;
      }
      
      /* input data should have zero mean... */
      if (i+k == 0) {
        f=0.0;
      }
        
      /* apply cubic spline correction for input data */
      s=0.5*(t+k*dt);
        
      if (s>0.1) {
        e=pow(sin(s)/s,4.0);
      } else {
        e=pow(1.0-(s*s)/6.0+(s*s*s*s)/120.0,4.0);
      }
      e = e*3.0/(1.0+2.0*cos(s)*cos(s));
      c = c+e*e*f;
    }
    
    wtrans[1+i] = t+0.5*dt*((double)(specr-1));
    ftrans[1+i] = c;
  }

  /* compute norm */
  norm_fourier=norm_classic=norm_kubo=norm_harmonic=norm_schofield=0.0;
  for (i=0; i<=nn; ++i) {
    t = wtrans[1+i];
    f = ftrans[1+i];
    e = t*(1.0 - exp(-bh*t));
    
    norm_fourier  += f;
    norm_classic  += f*e;
    norm_kubo     += f*e/(1.0+exp(-bh*t));
    norm_harmonic += f*t*t;
    norm_schofield += f*e*exp(0.5*bh*t);
  }
  norm_fourier  = 1.0/norm_fourier;
  norm_classic  = 1.0/norm_classic;
  norm_kubo     = 1.0/norm_kubo;
  norm_harmonic = 1.0/norm_harmonic;
  norm_schofield = 1.0/norm_schofield;

  /* output */
  for (i=0; i<=nn; ++i) {
    t = wtrans[1+i];
    f = ftrans[1+i];
    e = t*(1.0 - exp(-bh*t));

    output[2*i] = wave_fac*t;
    switch (normtype) {
      case NORM_FOURIER:
         output[2*i+1] = norm_fourier*f;
         break;
      case NORM_CLASSIC:
         output[2*i+1] = norm_classic *f*e;
         break;
      case NORM_KUBO:
         output[2*i+1] = norm_kubo*f*e/(1.0+exp(-bh*t));
         break;
      case NORM_HARMONIC:
         output[2*i+1] = norm_harmonic*f*t*t;
         break;
      case NORM_SCHOFIELD:
         output[2*i+1] = norm_schofield*f*e*exp(0.5*bh*t);
         break;
      default:
         fprintf(stderr, "specden: unknown normalization. %d\n", normtype);
         return -200;
    }
  }
  return nn;
}


/*
 * calculate cos_sum =  sum_j cos(j*w) input_j
 *       and sin_sum =  sum_j sin(j*w) input_j
 *
 * sums start at 0, but indices of input start at 1, e.g.
 * cos_sum = sum_{j=1}^n cos((j-1)*w) input_j
 */

void fourier_sum (const int n, const double *input, const double omega, double *cos_sum, double *sin_sum)
{
  int k;
  double  lambda, duk, uk, cs, ss, cf, sf;
  
  /*
   * in order to be able to sum up the input_j in ascending order
   * use above algorithm with inverse data ordering and correct
   * omega -> - omega and the new origin at the end
   */

  uk = 0.0;
  duk = 0.0;
  if (cos(omega) > 0.0) {
    lambda = -4.0*sin(0.5*omega)*sin(0.5*omega);
    for (k=1; k <= n; ++k) {
      uk  = uk + duk;
      duk = lambda*uk + duk + input[3*k];
    }
  } else { /* cos(omega) <= 0.0_dbl */
    lambda = 4.0*cos(0.5*omega)*cos(0.5*omega);
    for (k=1; k <= n; ++k) {
      uk  = duk - uk;
      duk = lambda*uk - duk + input[3*k];
    }
  }
  cs = duk - 0.5 * lambda * uk;
  ss = uk * sin(omega);

  /* now correct for ordering: */
  cf = cos(omega*(n-1));
  sf = sin(omega*(n-1));

  *cos_sum = cf*cs+sf*ss;
  *sin_sum = sf*cs-cf*ss;
}

