/*
 * Copyright (C) 2007 by David J. Hardy.  All rights reserved.
 *
 * mgpot_shortrng.c - compute short-range contribution to potentials
 */

#include <math.h>
#include "mgpot_defn.h"


void mgpot_shortrng(Mgpot *mg, float *atoms, float *grideners,
    long int numplane, long int numcol, long int numpt, long int natoms,
    float gridspacing, unsigned char *excludepos) {

  const int split = mg->split;
  const float a_1 = mg->a_1;
  const float a2 = mg->a * mg->a;
  const float inv_a2 = 1/a2;
  const float inv_gridspacing = 1/gridspacing;
  const long int radius = (long int) (mg->a * inv_gridspacing);
    /* grid point radius about each atom */

  long int n;
  long int i, j, k;
  long int ia, ib, ic;
  long int ja, jb, jc;
  long int ka, kb, kc;
  long int index;

  float x, y, z, q;
  float dx, dy, dz;
  float dz2, dydz2, r2;
  float s, gs;

if (TAYLOR2==split) {
  /* inline softened potential directly rather than using switch stmt */
  DEBUG( printf("(executing with inlined softened potential)\n"); );

  for (n = 0;  n < natoms;  n++) {
    q = atoms[ INDEX_Q(n) ];
    if (0==q) continue;

    x = atoms[ INDEX_X(n) ];
    y = atoms[ INDEX_Y(n) ];
    z = atoms[ INDEX_Z(n) ];

    /* find closest grid point with position less than or equal to atom */
    ic = (long int) (x * inv_gridspacing);
    jc = (long int) (y * inv_gridspacing);
    kc = (long int) (z * inv_gridspacing);

    /* find extent of surrounding box of grid points */
    ia = ic - radius;
    ib = ic + radius + 1;
    ja = jc - radius;
    jb = jc + radius + 1;
    ka = kc - radius;
    kb = kc + radius + 1;

    /* trim box edges so that they are within grid point lattice */
    if (ia < 0) ia = 0;
    if (ib >= numpt) ib = numpt-1;
    if (ja < 0) ja = 0;
    if (jb >= numcol) jb = numcol-1;
    if (ka < 0) ka = 0;
    if (kb >= numplane) kb = numplane-1;

    /* loop over surrounding grid points */
    for (k = ka;  k <= kb;  k++) {
      dz = z - k*gridspacing;
      dz2 = dz*dz;
      for (j = ja;  j <= jb;  j++) {
        dy = y - j*gridspacing;
	dydz2 = dy*dy + dz2;
	for (i = ia;  i <= ib;  i++) {
	  index = (k*numcol + j)*numpt + i;
	  if (excludepos[index]) continue;
	  dx = x - i*gridspacing;
	  r2 = dx*dx + dydz2;
	  if (r2 >= a2) continue;
	  s = r2 * inv_a2;
          gs = 1 + (s-1)*(-1./2 + (s-1)*(3./8));  /* TAYLOR2 */
	  grideners[index] += q * (1/sqrtf(r2) - a_1 * gs);
	}
      }
    } /* end loop over surrounding grid points */

  } /* end loop over atoms */

}
else {
  /* fall back on mgpot_split() macro for other choices */
  DEBUG( printf("(executing with macro expanded switch stmt)\n"); );

  for (n = 0;  n < natoms;  n++) {
    q = atoms[ INDEX_Q(n) ];
    if (0==q) continue;

    x = atoms[ INDEX_X(n) ];
    y = atoms[ INDEX_Y(n) ];
    z = atoms[ INDEX_Z(n) ];

    /* find closest grid point with position less than or equal to atom */
    ic = (long int) (x * inv_gridspacing);
    jc = (long int) (y * inv_gridspacing);
    kc = (long int) (z * inv_gridspacing);

    /* find extent of surrounding box of grid points */
    ia = ic - radius;
    ib = ic + radius + 1;
    ja = jc - radius;
    jb = jc + radius + 1;
    ka = kc - radius;
    kb = kc + radius + 1;

    /* trim box edges so that they are within grid point lattice */
    if (ia < 0) ia = 0;
    if (ib >= numpt) ib = numpt-1;
    if (ja < 0) ja = 0;
    if (jb >= numcol) jb = numcol-1;
    if (ka < 0) ka = 0;
    if (kb >= numplane) kb = numplane-1;

    /* loop over surrounding grid points */
    for (k = ka;  k <= kb;  k++) {
      dz = z - k*gridspacing;
      dz2 = dz*dz;
      for (j = ja;  j <= jb;  j++) {
        dy = y - j*gridspacing;
	dydz2 = dy*dy + dz2;
	for (i = ia;  i <= ib;  i++) {
	  index = (k*numcol + j)*numpt + i;
	  if (excludepos[index]) continue;
	  dx = x - i*gridspacing;
	  r2 = dx*dx + dydz2;
	  if (r2 >= a2) continue;
	  s = r2 * inv_a2;
	  mgpot_split(&gs, s, split);  /* macro expands into switch */
	  grideners[index] += q * (1/sqrtf(r2) - a_1 * gs);
	}
      }
    } /* end loop over surrounding grid points */

  } /* end loop over atoms */

}

}
