/*
 * Copyright (C) 2007 by David J. Hardy.  All rights reserved.
 *
 * mgpot_defn.h - definitions for main mgpot data structure
 */

#ifndef MGPOT_DEFN_H
#define MGPOT_DEFN_H

#include "mgpot_lattice.h"
#include "mgpot_split.h"

#ifdef __cplusplus
extern "C" {
#endif

  /* for accessing atom[] components */
#define INDEX_X(n)  (4*(n))
#define INDEX_Y(n)  (4*(n)+1)
#define INDEX_Z(n)  (4*(n)+2)
#define INDEX_Q(n)  (4*(n)+3)

  /* identifies different interpolants */
  enum MgpotInterp_t {
    INTERP_NONE=0,
    CUBIC,
    QUINTIC1,
    INTERP_MAX
  };

  typedef struct Mgpot_t {
    float h;      /* finest grid spacing */
    float h_1;    /* 1/h */
    float a;      /* cutoff distance for short-range part */
    float a_1;    /* 1/a */
    int interp;   /* identify grid interpolant */
    int split;    /* identify 1/r splitting */
    int scalexp;  /* scaling exponent from cionize to finest mgpot grid,
		   * where:  (2**scalexp)*h_cionize = h_mgpot  */
    int nlevels;  /* number of grid levels */
    floatLattice **qgrid;  /* nlevels of charge grids */
    floatLattice **egrid;  /* nlevels of potential grids */
    floatLattice **gdsum;  /* g(r) direct sum weights for each level */
    floatLattice **potinterp;  /* interp weights to cionize grid potentials,
				* length is (2**scalexp)**3 */
  } Mgpot;

  /* routines with external linkage */
  int mgpot_setup(Mgpot *mg, float h, float a,
      long int nx, long int ny, long int nz,
      int scalexp, int interp, int split);

  void mgpot_cleanup(Mgpot *mg);

  void mgpot_shortrng(Mgpot *mg, float *atoms, float *grideners,
      long int numplane, long int numcol, long int numpt, long int natoms,
      float gridspacing, unsigned char *excludepos);

  void mgpot_longrng(Mgpot *mg, float *atoms, float *grideners,
      long int numplane, long int numcol, long int numpt, long int natoms,
      unsigned char *excludepos);

  void mgpot_longrng_cubic(Mgpot *mg, float *atoms, float *grideners,
      long int numplane, long int numcol, long int numpt, long int natoms,
      unsigned char *excludepos);

  void mgpot_longrng_quintic1(Mgpot *mg, float *atoms, float *grideners,
      long int numplane, long int numcol, long int numpt, long int natoms,
      unsigned char *excludepos);


#ifdef __cplusplus
}
#endif

#endif /* MGPOT_DEFN_H */
