/*
 * Copyright (C) 2007 by David J. Hardy.  All rights reserved.
 *
 * mgpot.c - entry point for computing grid energies using mgpot
 */

#include <stdio.h>
#include "mgpot.h"
#include "mgpot_defn.h"

#include "util.h"    /* timer code taken from Tachyon */

#define CUTOFF 12.0


int calc_grid_energies_excl_mgpot(float* atoms, float* grideners,
    long int numplane, long int numcol, long int numpt, long int natoms,
    float gridspacing, unsigned char* excludepos, int maxnumprocs) {

  float h;
  float a;
  int interp;
  int split;
  int scalexp, scaling;
  long int nx, ny, nz;
  float xmax, ymax, zmax;
  Mgpot mg;
  long int i;

  rt_timerhandle timer = rt_timer_create();
  float totaltime=0, lasttime, elapsedtime;

  /* start timer for setup */
  rt_timer_start(timer);
  lasttime = rt_timer_timenow(timer);

  /* reset grideners */
  memset(grideners, 0, numplane*numcol*numpt*sizeof(float));

  /*
   * grid used by mgpot needs spacing h >= 2,
   * determine grid scaling factor:  scaling * h_ionize = h_mgpot
   */
  scalexp = 0;
  h = gridspacing;
  while (h < 2) {
    h *= 2;
    scalexp++;
  }
  scaling = (1 << scalexp);

  DEBUG( printf("h=%g\n" "scalexp=%d\n", h, scalexp); )

  /*
   * need enough mgpot grid points to cover space:
   *   nx = ceiling(numpt / scaling), etc.
   */
  nx = numpt / scaling + (numpt % scaling ? 1 : 0);
  ny = numcol / scaling + (numcol % scaling ? 1 : 0);
  nz = numplane / scaling + (numplane % scaling ? 1 : 0);
  DEBUG( printf("nx=%ld  ny=%ld  nz=%ld\n", nx, ny, nz); );

  /*
   * make sure that atoms are contained within mgpot grid
   * (otherwise mgpot will crash)
   */
  xmax = nx * h;
  ymax = ny * h;
  zmax = nz * h;
  for (i = 0;  i < natoms;  i++) {
    if (atoms[ INDEX_X(i) ] < 0 || atoms[ INDEX_X(i) ] >= xmax
        || atoms[ INDEX_Y(i) ] < 0 || atoms[ INDEX_Y(i) ] >= ymax
        || atoms[ INDEX_Z(i) ] < 0 || atoms[ INDEX_Z(i) ] >= zmax) {
      ERROR("atom %d is outside of grid\n", i);
      abort();
    }
  }

  /* use these defaults for other values (for now) */
  a = CUTOFF;
  interp = CUBIC;
  split = TAYLOR2;

  mgpot_setup(&mg, h, a, nx, ny, nz, scalexp, interp, split);
  elapsedtime = rt_timer_timenow(timer) - lasttime;
  printf("time for setup:             %.2f\n", elapsedtime);
  totaltime += elapsedtime;

  lasttime = rt_timer_timenow(timer);
  mgpot_longrng(&mg, atoms, grideners, numplane, numcol, numpt, natoms,
      excludepos);
  elapsedtime = rt_timer_timenow(timer) - lasttime;
  printf("time for long-range part:   %.2f\n", elapsedtime);
  totaltime += elapsedtime;

  lasttime = rt_timer_timenow(timer);
  mgpot_shortrng(&mg, atoms, grideners, numplane, numcol, numpt, natoms,
      gridspacing, excludepos);
  elapsedtime = rt_timer_timenow(timer) - lasttime;
  printf("time for short-range part:  %.2f\n", elapsedtime);
  totaltime += elapsedtime;

  printf("total time:                 %.2f\n", totaltime);

  mgpot_cleanup(&mg);

  return 0;
}
