/***************************************************************************
 *cr
 *cr            (C) Copyright 2007 The Board of Trustees of the
 *cr                        University of Illinois
 *cr                         All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: cionize.c,v $
 *      $Author: petefred $        $Locker:  $             $State: Exp $
 *      $Revision: 1.30 $      $Date: 2007/03/24 21:48:09 $
 *
 ***************************************************************************/

#if defined(_AIX)
/* Define to enable large file extensions on AIX */
#define _LARGE_FILE
#define _LARGE_FILES
#else
/* Defines which enable LFS I/O interfaces for large (>2GB) file support
 * on 32-bit machines.  These must be defined before inclusion of any
 * system headers.
 */
#define _LARGEFILE_SOURCE
#define _FILE_OFFSET_BITS 64
#endif

#define IONIZE_MAJOR_VERSION 1
#define IONIZE_MINOR_VERSION 0
#define MAX_PLUGINS 200

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <unistd.h>
#include <string.h>
#include <sys/time.h>

#include "util.h"
#include "threads.h"
#include "energythr.h"
#if defined(MGRID)
#include "energymgrid.h"
#endif

/* DH: always included, but not used by default */
#include "mgpot.h"

#if defined(CUDA)
#include "cudaenergythr.h"
#endif

/* Molfile plugin headers */
#include "libmolfile_plugin.h"
#include "molfile_plugin.h"
#include "getplugins.h"

#include "cionize_structs.h"
#include "cionize_grid.h"
#include "cionize_gridio.h"
#include "cionize_molfileio.h"
#include "cionize_userio.h"
#include "cionize_internals.h"
#include "cionize_enermethods.h"


int main(int argc, char* argv[]) {
  /* Set up the structs used for all the internal information */
  cionize_params params;
  cionize_molecule molecule;
  cionize_grid grid;
  cionize_api molfiles;

  /* Initialize everything that needs it */
  init_defaults(&params, &grid);
  setup_apis(&molfiles);

  /* Parse the command line */
  if (get_opts(&params, &molfiles, argc, argv) != 0) return 1;

  /* Set up the input api */
  if (setup_inapi(&params, &molfiles) != 0) return 1;

  /* Open the input file and go through the initial parameter setup phase */
  if (open_input(&params) != 0) return 1;

  /* Now go through the initial input loop where we set all the parameters */
  if (settings_inputloop(&params, &grid) != 0) return 1;


  /* Read the input molecule and set up memory for the molecule and grid */
  printf("\nRunning cionize on input file %s\n\tIon-solute distance: %f\n\tIon-Ion distance: %f\n\tGrid spacing: %f\n\tBoundary size: %f\n\tMax. Processors: %i\n", params.pdbin, params.r_ion_prot, params.r_ion_ion, grid.gridspacing, params.bordersize, params.maxnumprocs);
  if (params.ddd != 0) {
    printf("\tDistance dependent dielectric constant: %f * r\n", params.ddd);
  }

  read_input(&params, &molfiles, &molecule, &grid);

  printf("\nMaximum extents of the considered system (including boundary) are:\n\tX: %8.3f to %8.3f\tY: %8.3f to %8.3f\tZ: %8.3f to %-8.3f\nUsing a grid spacing of %f angstroms, the grid is %6ld x %6ld x %6ld\n", grid.minx, grid.maxx, grid.miny, grid.maxy, grid.minz, grid.maxz, grid.gridspacing, grid.numpt, grid.numcol, grid.numplane);

  /* Now allocate the arrays we need and pack the appropriate data into them*/
  printf("\nAllocating memory for data arrays...\n");
  if (alloc_mainarrays(&grid, &molecule) != 0) return 1;
  printf( "Successfully allocated data arrays\n");

  /* Stick all the data we need from the molfile type into ionize arrays */
  unpack_molfile_arrays(&params, &molfiles, &molecule, &grid);

  /*Exclude grid points too close to the protein */
  printf("\nExcluding grid points too close to protein\n");
  if (exclude_solute(&params, &molecule, &grid) != 0) return 1;
  printf("Finished with exclusion\n");

  /* Read or calculate the energy grid */

  if (params.useoldgrid == 0) {
    int i;

    /* Initialize our grid to all zeros so we can just have the energy
     * function add to it */
    for (i=0; i<(grid.numpt * grid.numcol * grid.numplane); i++) {
      grid.eners[i] = 0.0;
    }

    /*Now that all our input is done, calculate the initial grid energies*/
    printf( "\nCalculating grid energies...\n");
#if defined(CUDA)
    calc_grid_energies_cuda_thr(molecule.atoms, grid.eners, grid.numplane, grid.numcol, grid.numpt, molecule.natom, grid.gridspacing, grid.excludepos, params.maxnumprocs);
#elif defined(MGRID)
    calc_grid_energies_excl_mgrid(molecule.atoms, grid.eners, grid.numplane, grid.numcol, grid.numpt, molecule.natom, grid.gridspacing, grid.excludepos, params.maxnumprocs);
#else
    if (params.enermethod == MULTIGRID) {  /* DH: if requested, use mgpot for initial grid */
      calc_grid_energies_excl_mgpot(molecule.atoms, grid.eners,
	  grid.numplane, grid.numcol, grid.numpt, molecule.natom,
	  grid.gridspacing, grid.excludepos, params.maxnumprocs);
    } else {  /* DH: else method defaults to brute force */
      calc_grid_energies(molecule.atoms, grid.eners,
	  grid.numplane, grid.numcol, grid.numpt, molecule.natom,
	  grid.gridspacing, grid.excludepos, params.maxnumprocs, params.enermethod, 
    params.ddd);
    }
#endif
    printf( "Done calculating grid energies\n");
  } else {
    printf( "\nReading grid energies from %s\n", params.oldgridfile);
    if (read_grid(params.oldgridfile, grid.eners, grid.gridspacing, grid.numplane, grid.numcol, grid.numpt, grid.minx, grid.miny, grid.minz) != 0) return 1;
    printf("Finished reading energy grid\n");
  }

  /* Now, begin a new input loop reading statements to either place ions, save ions, or write the grid */
  if (do_placements(&params, &grid, &molfiles) != 0) return 1;

  printf("\ncionize: normal exit\n");
  return 0;
}







