############################################################################
#cr
#cr            (C) Copyright 1995-2006 The Board of Trustees of the
#cr                        University of Illinois
#cr                         All Rights Reserved
#cr
############################################################################

############################################################################
#
# multiseqdialog
#
# provides an API to get paths where databases are stored on the user's system
#
############################################################################

package provide multiseqdialog 1.0

namespace eval ::MultiSeqDialog:: {

    global env
    variable dialogWindow
    variable dialogWindowName ".multiseqDialogWindow"
    #variable baseURL "http://vidar.scs.uiuc.edu/~erobert3"
    variable baseURL "http://www.scs.uiuc.edu/multiseq"
    variable infoURL "$baseURL/versions.txt"
    variable platformInfoURL "$baseURL/versions_[vmdinfo arch].txt"
    variable infoFile [file join $env(VERSIONDIR) "versions.txt"]
    variable platformInfoFile [file join $env(VERSIONDIR) "versions_[vmdinfo arch].txt"]
    variable tmpdir
    variable xmlValueList ""
    variable rcFileName
    variable defaultDir
    variable changeNotifier ""
}

############################################################################
#
# openDialog
#
# Opens the dialog window that allows a user to change the locations
# where databases are stored. Also allows the user to download the latest
# version
#
# Parameters: message - a message that appears at the top of the dialog,
#                       which could provide an error message, tell the user
#                       which database is needed, etc.
#
# Returns: none
#
############################################################################

proc ::MultiSeqDialog::openDialog {message arg_changeNotifier} {

    global env

    variable dialogWindow
    variable dialogWindowName
    variable xmlValueList
    variable rcFileName
    variable defaultDir
    variable changeNotifier
    set changeNotifier $arg_changeNotifier

    set helpURL "http://www.scs.uiuc.edu/multiseq/versions.txt"
    
    # Read the preferences.
    readRCFile
    
    # Make sure a default directory has been set.
    set defaultDir [getDirectory "default"]
    if {$defaultDir == ""} {
        set defaultDir [file join [file dirname $rcFileName] ".multiseqdb"]
        tk_messageBox -title "Choose Default Directory" -icon info -message "You must now select a directory to use for storing databases. If you choose \"Cancel\", a default directory will be used." -type ok
        set dir [tk_chooseDirectory -title "Choose default directory"]
        if {$dir != ""} {
            set defaultDir $dir
        }
    }
    file mkdir $defaultDir
    setDirectory "default" $defaultDir
    puts "Info) Default multiseq db directory: $defaultDir"

    # Build the window.    
    if { [winfo exists $dialogWindowName] } {
        foreach child [winfo children $dialogWindowName ] {
            destroy $child
        }
    } else {
        set dialogWindow [toplevel $dialogWindowName]
        wm title $dialogWindowName $message
    }

    set dbFrame [frame $dialogWindowName.dbFrame -relief sunken -borderwidth 1]
    set dbScrollFrame [::MultiSeqDialog::scrolledframe::scrolledframe $dbFrame.panel -fill both -yscrollcommand "$dbFrame.scroll set"]
    set dbScrollBar [scrollbar $dbFrame.scroll -command "$dbScrollFrame yview"]
    set swFrame [frame $dialogWindowName.swFrame -relief sunken -borderwidth 1]
    set swScrollFrame [::MultiSeqDialog::scrolledframe::scrolledframe $swFrame.panel -height 400 -width 200 -fill x -yscrollcommand "$swFrame.scroll set"]
    set swScrollBar [scrollbar $swFrame.scroll -command "$swScrollFrame yview"]

    set switchFrame [frame $dialogWindowName.switchFrame]
    set dbButton [button $switchFrame.dbButton -text "Databases" -command "pack forget $swFrame; pack $dbFrame -side top -fill both -expand true"]
    set swButton [button $switchFrame.swButton -text "Software" -command "pack forget $dbFrame; pack $swFrame -side top -fill both -expand true"]

    pack $switchFrame -side top -anchor nw -pady 10
    pack $dbButton -side left -anchor w
    pack $swButton -side left -anchor w

    grid $dbScrollFrame -row 0 -column 0 -sticky nsew
    grid $dbScrollBar   -row 0 -column 1 -sticky ns
    grid rowconfigure $dbFrame 0 -weight 1
    grid columnconfigure $dbFrame 0 -weight 1
    
    grid $swScrollFrame -row 0 -column 0 -sticky nsew
    grid $swScrollBar   -row 0 -column 1 -sticky ns
    grid rowconfigure $swFrame 0 -weight 1
    grid columnconfigure $swFrame 0 -weight 1
    
    if { [readInfoFile] == -1 } {
	wm withdraw $dialogWindowName
	return
    }

    pack $dbFrame -side top -fill both -expand true
    
    set newList ""

    set command {
	set newList "";
	foreach typeFrame [winfo children $::MultiSeqDialog::dialogWindowName ] {

        foreach panel [ winfo children $typeFrame ] {
            
            foreach scrolledChildFrame [ winfo children $panel ] {
                
                foreach fieldFrame [ winfo children $scrolledChildFrame ] {
                    
                    set childFrames [winfo children $fieldFrame]
                    
                    set varFrameIndex [lsearch -regexp $childFrames .*variables]
                    if { $varFrameIndex != -1 } {
                        set varFrame [lindex $childFrames $varFrameIndex]
                        foreach var [winfo children $varFrame] {
            
                        set varChildren [winfo children $var]
                        set keyIndex [lsearch -regexp $varChildren .*_label]
                        set keyName [lindex $varChildren $keyIndex]
            
                        set valueIndex [lsearch -regexp $varChildren .*_text]
                        set valueName [lindex $varChildren $valueIndex]
            
                        set key [$keyName cget -text]
                        set value [$valueName get]
            
                        set widgets [split $keyName "."];
                        set widgetName [lindex $widgets 6];
                        regsub "_variables" $widgetName "" fieldName;
            
                        ::MultiSeqDialog::setVariable $fieldName $key $value
                        }
                    }
            
                    set textFrameIndex [lsearch -regexp $childFrames .*dirFrame]
                    if { $textFrameIndex == -1 } {
                        continue
                    }
                    set fieldFrames [winfo children [lindex $childFrames $textFrameIndex]]
                    set textField [lindex $fieldFrames [lsearch -regexp $fieldFrames .*_text]]
                    
                    set widgets [split $textField "."];
                    set widgetName [lindex $widgets 7];
                    regsub "_text" $widgetName "" fieldName;
                    ::MultiSeqDialog::setDirectory $fieldName [$textField get];
                }
            }
        }
	}

	wm withdraw $::MultiSeqDialog::dialogWindowName;

	::MultiSeqDialog::writeRCFile;
    if {$::MultiSeqDialog::changeNotifier != ""} {
        $::MultiSeqDialog::changeNotifier
    }
    }
    
    set bottomFrame [frame $dialogWindowName.bottom]
    set buttonFrame [frame $bottomFrame.buttonFrame]
    set okayButton [button $buttonFrame.okayButton -text "Close" -command $command ]    
    set helpButton [button $buttonFrame.helpButton -text "Help" -command "vmd_open_url $helpURL"]
    bind $dialogWindow <Return> $command
    bind $dialogWindow <Escape> $command
    
    pack $bottomFrame  -fill x -side bottom
    pack $buttonFrame  -side bottom
    pack $okayButton   -side left -padx 5 -pady 5
    pack $helpButton   -side left -padx 5 -pady 5
    
    centerDialog
    wm state $dialogWindowName normal
}

    # Centers the dialog.
    proc ::MultiSeqDialog::centerDialog {{parent ""}} {
        
        # Import global variables.        
        variable dialogWindow
        
        # Set the width and height, since calculating doesn't work properly.
        set width 600
        set height [expr 400+22]
        
        # Figure out the x and y position.
        if {$parent != ""} {
            set cx [expr {int ([winfo rootx $parent] + [winfo width $parent] / 2)}]
            set cy [expr {int ([winfo rooty $parent] + [winfo height $parent] / 2)}]
            set x [expr {$cx - int ($width / 2)}]
            set y [expr {$cy - int ($height / 2)}]
            
        } else {
            set x [expr {int (([winfo screenwidth $dialogWindow] - $width) / 2)}]
            set y [expr {int (([winfo screenheight $dialogWindow] - $height) / 2)}]
        }
        
        # Make sure we are within the screen bounds.
        if {$x < 0} {
            set x 0
        } elseif {[expr $x+$width] > [winfo screenwidth $dialogWindow]} {
            set x [expr [winfo screenwidth $dialogWindow]-$width]
        }
        if {$y < 22} {
            set y 22
        } elseif {[expr $y+$height] > [winfo screenheight $dialogWindow]} {
            set y [expr [winfo screenheight $dialogWindow]-$height]
        }
            
        wm geometry $dialogWindow ${width}x${height}+${x}+${y}
        wm positionfrom $dialogWindow user
    }
    
############################################################################
#
# addFieldToDialog
#
# Adds a new database field to the dialog box, with its name, path,
#  and download button
#
# parameters:
#  key - a unique key used by MultiSeq to identify this database
#  description - a text description; MUST be in quotes (ie, "desc")
#  url - location to download the file from
#  backupurl - location to download the file from if URL is unavailable
#  version - most recent version number of the database
#  size - size in bytes of the downloaded file
#  type - download type; "db" for database or "sw" for software
#  required - is this download required for proper operation of MultiSeq?  Y or N
#  envVariables - a list of name/value pairs of environment variables. Used for software
#
# returns:
#  none
#
############################################################################

proc ::MultiSeqDialog::addFieldToDialog { key description url backupurl version size type required { envVariables } } {

    global env
    variable dialogWindow
    variable dialogWindowName
    variable defaultDir
    variable xmlValueList
    
    if { ![winfo exists $dialogWindowName] } {
	set dialogWindow [toplevel $dialogWindowName]
    }

    set fullDialogWindow ""

    if { [string match $type "db"] } {
        
        append fullDialogWindow $dialogWindowName . "dbFrame.panel.scrolled"
        append frameHandle $key _frame
        append frameName $fullDialogWindow . $frameHandle
    
        append dirFrameHandle $key _dirFrame
        append dirFrameName $frameName . $dirFrameHandle
    
        append labelFrameHandle $key _labelFrame
        append labelFrameName $frameName . $labelFrameHandle
    
        append labelHandle $key _label
        append labelName $labelFrameName . $labelHandle
    
        append textHandle $key _text
        append textName $dirFrameName . $textHandle
    
        append versionHandle $key _version
        append versionName $labelFrameName . $versionHandle
    
        set oldVersion [getVersion $key]
        if { [string match $oldVersion ""] } {
        set oldVersion "n/a"
        }
    
        append downloadButtonHandle $key _downloadButton
        append downloadButtonName $frameName . $downloadButtonHandle
        set pathList [split $url "/"]
        set fileName [lindex $pathList end]
    
        append chooseDirButtonHandle $key _chooseDirButton
        append chooseDirButtonName $dirFrameName . $chooseDirButtonHandle
    
        set text $description
        if { [string match $required "Y" ] } {
        append text " (Required)"
        }
    
        set $frameHandle [frame $frameName -borderwidth 1 -highlightbackground black -relief solid -padx 5 -pady 5]
    
        set $labelFrameHandle [frame $labelFrameName]
        set $dirFrameHandle [frame $dirFrameName]
    
        #set $versionFrameHandle [frame $versionFrameName]
    
        set $versionHandle [label $versionName  -text "Current version: $oldVersion\nLatest version: $version" -justify left]
    
        set $labelHandle [label $labelName -text $text -font "-family Helvetica -size 9 -weight bold" ]
        set $textHandle [entry $textName -width 50]
    
        eval [append q $ $textHandle] insert 0 [list [getDirectory $key]]
    
        set downloadCommand "::MultiSeqDialog::download $url \[$textName get\] $fileName;$versionName configure -text \"Current version: $version\nLatest version: $version\";::MultiSeqDialog::setVersion $key $version;"
    
        set $downloadButtonHandle [button $downloadButtonName -text "Download $fileName from $backupurl ($size KB)" -command $downloadCommand]
    
        set $chooseDirButtonHandle [button $chooseDirButtonName -text "Browse..." -command "::MultiSeqDialog::setDownloadDir \[$textName get\] $textName" ]
    
    
    
        pack $frameName -fill x -anchor nw
        pack $labelFrameName -side top -fill x -anchor nw
        pack $dirFrameName -side top -fill x -anchor nw
    
    
        #pack $versionFrameName -side bottom -anchor sw -pady 5
        pack $labelName -side left -fill x -anchor nw
        pack $versionName -side left -fill x -anchor nw
        pack $textName  -side left -fill x -anchor nw
        pack $downloadButtonName -side top -anchor sw -pady 5
        pack $chooseDirButtonName -anchor w
    
    
        if { ![string match $envVariables ""] } {
        append variableFrameHandle $key _variables
        append variableFrameName $frameName . $variableFrameHandle
    
        set $variableFrameHandle [frame $variableFrameName]
        pack $variableFrameName -side top -anchor sw
        
        foreach { name value } $envVariables {
            set varFrameHandle [append varFrameHandle$name  var_ $name _frame]
            set varFrameName [append varFrame$name $variableFrameName . $varFrameHandle]
    
            set varLabelHandle [append varLabelHandle$name var_ $name _label]
            set varLabelName [append varLabel$name $varFrameName . $varLabelHandle]
    
            set varTextHandle [append varTextHandle$name var_ $name _text]
            set varTextName [append varText$name $varFrameName . $varTextHandle]
    
            set $varFrameHandle [frame $varFrameName]
    
            set $varLabelHandle [label $varLabelName -text $name]
            set $varTextHandle [entry $varTextName -width 50]
    
            if { [set savedValue [getVariable $key $name]] != "" } {
            
            eval [append x $ $varTextHandle] insert 0 $savedValue
            } else {
            eval [append x $ $varTextHandle] insert 0 $value
            }
            set x ""
    
            pack $varFrameName -side top -fill x
            pack $varLabelName -side left -fill x
            pack $varTextName -side left -fill x
        }
        }
    
    } elseif { [string match $type "sw"] } {
        
        append fullDialogWindow $dialogWindowName . "swFrame.panel.scrolled"
        append frameHandle $key _frame
        append frameName $fullDialogWindow . $frameHandle
    
        append dirFrameHandle $key _dirFrame
        append dirFrameName $frameName . $dirFrameHandle
    
        append labelFrameHandle $key _labelFrame
        append labelFrameName $frameName . $labelFrameHandle
    
        append labelHandle $key _label
        append labelName $labelFrameName . $labelHandle
    
        append textHandle $key _text
        append textName $dirFrameName . $textHandle
    
        append chooseDirButtonHandle $key _chooseDirButton
        append chooseDirButtonName $dirFrameName . $chooseDirButtonHandle
    
        set text $description
        if { [string match $required "Y" ] } {
        append text " (Required)"
        }
    
        set $frameHandle [frame $frameName -borderwidth 1 -highlightbackground black -relief solid -padx 5 -pady 5]
    
        set $labelFrameHandle [frame $labelFrameName]
        set $dirFrameHandle [frame $dirFrameName]
    
        set $labelHandle [label $labelName -text $text -font "-family Helvetica -size 9 -weight bold" ]
        set $textHandle [entry $textName -width 50]
    
        eval [append q $ $textHandle] insert 0 [list [getDirectory $key]]
    
        set $chooseDirButtonHandle [button $chooseDirButtonName -text "Browse..." -command "::MultiSeqDialog::setDownloadDir \[$textName get\] $textName" ]
    
        pack $frameName -fill x -anchor nw
        pack $labelFrameName -side top -fill x -anchor nw
        pack $dirFrameName -side top -fill x -anchor nw
    
    
        pack $labelName -side left -fill x -anchor nw
        pack $textName  -side left -fill x -anchor nw
        pack $chooseDirButtonName -anchor w
    
    
        if { ![string match $envVariables ""] } {
        append variableFrameHandle $key _variables
        append variableFrameName $frameName . $variableFrameHandle
    
        set $variableFrameHandle [frame $variableFrameName]
        pack $variableFrameName -side top -anchor sw
        
        foreach { name value } $envVariables {
            set varFrameHandle [append varFrameHandle$name  var_ $name _frame]
            set varFrameName [append varFrame$name $variableFrameName . $varFrameHandle]
    
            set varLabelHandle [append varLabelHandle$name var_ $name _label]
            set varLabelName [append varLabel$name $varFrameName . $varLabelHandle]
    
            set varTextHandle [append varTextHandle$name var_ $name _text]
            set varTextName [append varText$name $varFrameName . $varTextHandle]
    
            set $varFrameHandle [frame $varFrameName]
    
            set $varLabelHandle [label $varLabelName -text $name]
            set $varTextHandle [entry $varTextName -width 50]
    
            set x ""
            if { [getVariable $key $name] != "" } {            
                eval [append x $ $varTextHandle] insert 0 [list [getVariable $key $name]]
            } else {
                eval [append x $ $varTextHandle] insert 0 [list $value]
            }
    
            pack $varFrameName -side top -fill x
            pack $varLabelName -side left -fill x
            pack $varTextName -side left -fill x
        }
        }
    }
}


############################################################################
#
# writeRCFile
#
# writes the .multiseqrc file to the user's hard drive
#
# parameters: data - the data to write
#
# returns: none
#
############################################################################

proc ::MultiSeqDialog::writeRCFile {} {
    
    variable rcFileName
    variable xmlValueList

    puts "Info) Saving multiseq resource file: $rcFileName"
    file delete $rcFileName
    set fp [open $rcFileName w+]
    puts $fp $xmlValueList
    close $fp
}


############################################################################
#
# readRCFile
#
# reads the .multiseqrc file from the user's hard drive and puts the results
# into xmlValueList
#
# parameters: none
#
# returns: 1 if the file exists, otherwise 0
#
############################################################################

proc ::MultiSeqDialog::readRCFile {} {
    global env
    variable rcFileName
    variable xmlValueList

    # Fgure out the home directory.
    set homeDir "~/"
    if {[catch {
        set foo ""
        set os [vmdinfo arch]
        regexp WIN32 $os foo
        if { [string match $foo WIN32] } {
            set homeDir $env(HOME)
        } else {
            set homeDir $env(HOME)
        }
    } errorMessage] != 0} {
        global errorInfo
        set callStack [list "Error) Could not find home directory: $errorInfo"]
        puts [join $callStack "\n"]
    }
    
    # Construct the resource file name, if we don't have it already.
    if { ![info exists rcFileName] } {
        set rcFileName [file join $homeDir ".multiseqrc"]
    }

    # If the file exists, read it in.
    if {[file exists $rcFileName]} {
        set data ""
        if {[catch {
            puts "Info) Loading multiseq resource file: $rcFileName"
            set fp [open $rcFileName r+]
            set data [read $fp]
            close $fp
        } errorMessage] != 0} {
            global errorInfo
            set callStack [list "Error) Could not load multiseq resource file: $errorInfo"]
            puts [join $callStack "\n"]
        }
        if {$data != ""} {
            puts "Info) Loaded multiseq preferences."
            set xmlValueList $data
            return 1
        }
    }
    
    # Otherwise return that we don't have a resource file yet.
    return 0
}

############################################################################
#
# readInfoFile
#
# reads the web page with the list of latest databases and versions, and adds
#  corresponding fields to the dialog box
#
# parameters: none
#
# returns: -1 if the user cancels the operation, else none
#
############################################################################

proc ::MultiSeqDialog::readInfoFile { } {

    variable xmlValueList
    variable infoURL
    variable platformInfoURL
    variable infoFile
    variable platformInfoFile
    variable defaultDbs
    variable defaultDir

    global env
    
    set foo ""
    set os [vmdinfo arch]
    regexp WIN32 $os foo
    if { [string match $foo WIN32] } {
        catch { set tmpdir $env(TEMP)}
        catch { set tmpdir $env(TMP) }
    } else {
        set tmpdir $env(TMPDIR)
    }

    set tempFileName [file join $tmpdir "versions.txt"]
    
    if { [file exists $tempFileName] } {
        file delete $tempFileName
    }

    # Try to get either the platform or generic version of the info file from the web.
    set data ""
    foreach location [list $platformInfoURL $infoURL $platformInfoFile $infoFile] {
        
        if {[catch {
            puts "Info) Trying to retrieve multiseq db version from: $location"
            if {[string first "http" $location] == 0} {
                vmdhttpcopy $location $tempFileName
                set fp [open $tempFileName]
                regexp {\%VERS\s1\.0\s*([^%]*)\%EVERS\s1\.0} [read $fp] unused data
                close $fp
            } else {
                if {[file exists $location]} {
                    set fp [open $location]	
                    regexp {\%VERS\s1\.0\s*([^%]*)\%EVERS\s1\.0} [read $fp] unused data
                    close $fp
                }
            }
        } errorMessage] != 0} {
            puts "Error message: $errorMessage"
            global errorInfo
            set callStack [list "Error) " $errorInfo]
            puts [join $callStack "\n"]
        }
        
        # If we got data, we are finished.
        if {$data != ""} {
            break
        }
    }
    
    # Parse the data.
    if {$data != ""} {
        foreach line [split $data "\n"] {
            if { [string first "\#" $line] == 0 || [string match $line ""] } {
                continue
            } else {
        
                # Format of data is as follows:
                # Blank lines are ignored
                # Each line consists of space-seperated fields in the following order:
                #  Key - a unique key used by MultiSeq to identify this database
                #  Description - a text description; MUST be in quotes (ie, "desc")
                #  URL - location to download the file from
                #  BackupURL - location to download the file from if URL is unavailable
                #  Version - most recent version number of the database
                #  Size - size in bytes of the downloaded file
                #  Type - download type; "db" for database or "sw" for software
                #  Required - is this download required for proper operation of MultiSeq?  Y or N
                #  envVariables - additional variables. A list of name/value pairs
        
                # Split up the line.
                set columns [regexp -inline -all {\"[^\"]*\"|\S+} $line]
                for {set i 0} {$i < [llength $columns]} {incr i} {
                    set columns [lreplace $columns $i $i [join [regexp -inline {[^\"]+} [lindex $columns $i]] " "]]
                }
                
                # Process the line.
                set key [lindex $columns 0]
                set description [lindex $columns 1]
                set url [lindex $columns 2]
                set backupurl [lindex $columns 3]
                set version [lindex $columns 4]
                set size [lindex $columns 5]
                set type [lindex $columns 6]
                set required [lindex $columns 7]
                set envVariables [lrange $columns 8 end]
                
                # If this is a database.
                if {$type == "db"} {
                    
                    # Set the default directory, if none exists.
                    if { [string match [getDirectory $key] "" ] } {
                        setDirectory $key $defaultDir
                    }
                    
                    # Ask the user to update, if needed.
                    checkNewVersion $key $type [getVersion $key] $version [getDirectory $key] [lindex [split $url "/"] end] $url $size
                }
                
                # Create the fialog entry.
                addFieldToDialog $key $description $url $backupurl $version $size $type $required $envVariables
            }
        }
    }
}

########################
#
# toXML
#
# creates an XML string from name/value pairs
# 
# parameters:
#  tag: The tag name for the XML entry
#  entries: a list of name, value entries
#
# returns:
#  an XML String representing the name/value pairs

proc toXML { tag entries } {
    
    set xml "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
    append xml "<$tag>\n"
    foreach { name value } $entries {
       # character substitutions for valid XML
       regsub -all ">" $value "ampersandgt;" value
       regsub -all "<" $value "ampersandlt;" value
       regsub -all {\&} $value "ampersand" value
       regsub -all {\?} $value "questionmark" value
       regsub -all {\}} $value "ampersand\#123;" value
       regsub -all {\{} $value "ampersand\#125;" value

       append xml " <$name>\n  $value\n </$name>\n"
    }
    append xml "</$tag>"

    return $xml
}

proc toXMLMultiple { mainTag values } {

    set xml ""
    set xml "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
    append xml "<$mainTag>\n"
    

    foreach { tag fields } $values {
	
	append xml " <$tag>\n"
	
	foreach { name value } $fields {

	    append xml "  <$name>\n"
	    append xml "   $value\n"
	    append xml "  </$name>\n"
	}

	append xml " </$tag>\n"
	
    }
    append xml "</$mainTag>"

    return $xml
}




############################################################################
#
# getXMLValue
#
# gets the value of a key stored in an XML string
#
# parameters: tag - the tag to get the field of (such as "ncbi_tax")
#             field - the field to get the value of (such as "directory" or "version"
#             xml - the xml text to check
#
# returns: the value of the field, or an empty string if it doesn't exist.
#
############################################################################

proc ::MultiSeqDialog::getXMLValue { tag field xml } {
    set startIndex [string first "<$tag>" $xml]
    set endIndex [string first "</$tag>" $xml]

    if { $startIndex == -1 } {
        return ""
    }

    set substr [string range $xml $startIndex $endIndex]

    set startIndex [string first "<$field>" $substr]
    set endIndex [string first "</$field>" $substr]
    
    if { $startIndex == -1 } {
        return ""
    }

    set data [string range $substr [expr $startIndex + [string length "<$field>"]] [ expr $endIndex - 1] ]

    return [string trim $data]
}

############################################################################
#
# setXMLValue
#
# sets the value of a key stored in an XML string
#
# parameters: tag - the tag to get the field of (such as "ncbi_tax")
#             field - the field to get the value of (such as "directory" or "version"
#             value - the new value for the field
#             xml - the xml text to add the data to
#
# returns: the new value of the xml string with the updated value. If "tag"
#          does not exist, it is added, along with the new value. If "field"
#          does not exist within the given tag, it is added.
#
############################################################################

proc ::MultiSeqDialog::setXMLValue { tag field value xml } {

    if { [string match $xml ""] } {
	set xml "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n<multiseq>\n</multiseq>"
    }

    set startIndex [string first "<$tag>" $xml]
    set endIndex [string first "</$tag>" $xml]

    if { $startIndex == -1 } {
	# insert it
	set lastIndex [string last "</" $xml]
	incr lastIndex -1
	set newXML [string replace $xml $lastIndex $lastIndex "\n <$tag>\n  <$field>\n   $value\n  </$field>\n </$tag>\n"]
	return $newXML
    }

    set fieldStart [string first "<$field>" $xml $startIndex]
    set fieldEnd [string first "</$field>" $xml $startIndex]
    if { $fieldStart == -1 || $fieldStart > $endIndex } {
        # insert it
        incr endIndex -1
        set newXML [string replace $xml $endIndex $endIndex "\n  <$field>\n   $value\n  </$field>\n"]
        return $newXML
    } else {
        
        # Replace it.
        set newXML [string range $xml 0 [expr $fieldStart+[string length "<$field>"]]]
        append newXML "   $value\n  "
        append newXML [string range $xml $fieldEnd end]
        return $newXML
    }
}


############################################################################
#
# setDownloadDir
#
# sets the directory to download a database
#
# parameters: oldDir - the previous directory
#             textName - the name of the tcl entry widget from which to get
#                        the new value
#
# Note: this changes the name in the text box, but does not change the 
#       xml in the xmlValueList data member
#
############################################################################

proc ::MultiSeqDialog::setDownloadDir {oldDir textName} {

    if {$oldDir != ""} {
        set newDirectory [tk_chooseDirectory -initialdir $oldDir -title "Choose Directory"]
    } else {
        set newDirectory [tk_chooseDirectory -title "Choose Directory"]
    }
    if { ![string match $newDirectory ""] } {
        $textName delete 0 end
        $textName insert 0 $newDirectory
    }
}

############################################################################
#
# checkNewVersion
#
# checks the version stored in the .multiseqrc file vs. the version on the 
#  web page to see if a more recent version is available, and prompt the user
#  to download it.
#
# parameters: dbName - name of the database (such as "nbci_tax"
#             oldVersion - current version of the user's database
#             newVersion - new version number to check
#             path - directory to place the database
#             file - File name of the database
#             url - URL from which to download the database
#
# returns: none
############################################################################

proc ::MultiSeqDialog::checkNewVersion { dbName type oldVersion newVersion path file url size} {

    variable xmlValueList
    variable defaultDir
    variable dialogWindowName

    # If the file is present and is the correct size, use it.
    if {[file exists [file join $path $file]] && [file size [file join $path $file]] == $size} {
		setDirectory $dbName $path
		setVersion $dbName $newVersion
    
    # Otherwise, download the database.
    } else {
        set downloadVer [tk_dialog ".newVersionAvailable" "New Version" "Version $newVersion is available for the $dbName database. Do you wish to download the new version to $path?\n\nIf you already have this version installed elsewhere, select the \"Find directory\" button and choose the directory it is currently in. The version will be set to $newVersion."  "" 0 "Yes" "No" "Find directory"]
        if { $downloadVer == 0 } {   
            download $url $path $file
            setDirectory $dbName $path
            setVersion $dbName $newVersion
        } elseif { $downloadVer == 2 } {
            set newDir [tk_chooseDirectory -initialdir $defaultDir -title "Select Database Directory"]
            if { ![string match $newDir ""] } {
            setDirectory $dbName $newDir
            setVersion $dbName $newVersion
            append textLabel $dialogWindowName . $type Frame . $dbName _frame. $dbName _labelFrame. $dbName _version
            if { [winfo exists $textLabel] } {
                $textLabel configure -text "Current version: $newVersion\nLatest version: $newVersion"
            }
            }	
        }
    }
}

############################################################################
#
# download
#
# downloads a database file
#
# parameters: url - the URL from which to download
#             path - the directory in which to put the file
#             fileName - the name of the database file
#
# returns: none
#
############################################################################

proc ::MultiSeqDialog::download { url path fileName } {

    # Create the directory.
    file mkdir $path
    
    if { ![file exists $path] } {
	set error [tk_dialog ".error" "Error" "The directory:\n\n$path\n\ncould not be found. Please check that the path exists and is spelled correctly." "" 0 "Okay"]
	return
    }
    set fullPath [file join $path $fileName]

    catch {
	if { [file exists $fullPath] } {
	    if { [tk_dialog ".multiSeqDownloadOverwrite" "File exists" "The file $fileName already exists in directory $path. Do you wish to overwrite it?" "" 0 "Yes" "No"] != 0 } {
	    return
	    }
	}
	
	vmdhttpcopy $url $fullPath
	if { [string first ".gz" $fileName] != -1  } {
	    set arch [vmdinfo arch]
	    if { [string match $arch WIN32] } {
            tk_dialog ".unzip" "Unzip necessary" "The file $fullPath needs to be unzipped. Please use a program such as WinZip to decompress the file in the directory in which you downloaded it." "" 0 "Okay"
	    } else {
            if {[regexp {^(.+)\.tar\.gz$} $fullPath unused prefix]} {
                puts "Info) Untaring $fullPath"
                exec gunzip -c $fullPath > $prefix.tar
                exec tar -xf $prefix.tar -C $path
                file delete $prefix.tar
            } elseif {[regexp {^(.+)\.gz$} $fullPath unused prefix]} {
                puts "Info) Unzipping $fullPath"
                exec gunzip -c $fullPath > $prefix
            }
	    }
	}
	
    } download_error
    
    if { [string first "couldn't open socket" $download_error] != -1 } {
	set error [tk_dialog ".error" "Error" "There was an error in performing the task you requested.\nPlease be sure the server name you specified is correct, and that you have a working internet\nconnection and try again." "" 0 "Okay"]
    } elseif { [string first "couldn't open \"$fullPath" $download_error] != -1 } {
	set error [tk_dialog ".error" "Error" "There was an error in downloading the file to:\n\n$fullPath\n\nPlease check that the path exists and is spelled correctly, and that you have permissions to write to the directory." "" 0 "Okay"]
    } else {
	puts $download_error
    }
}

############################################################################
#
# getDirectory
#
# shortcut to getting a directory for a given database
#
# parameters: dbName - name of the database
#
# returns: the path to the database, or an empty string if it can't be found
#
############################################################################

proc ::MultiSeqDialog::getDirectory { dbName } {

    variable xmlValueList

    set dir [getXMLValue $dbName directory $xmlValueList]
    if {$dir == ""} {
        set dir [getXMLValue "$dbName\_1" directory $xmlValueList]
    }
    return $dir

}

############################################################################
#
# setDirectory
#
# shortcut to setting a directory for a given database
#
# parameters: dbName - name of the database
#             newDirectory - the new value for the directory
#
# returns: the XML of xmlValueList with the new directory
#
############################################################################

proc ::MultiSeqDialog::setDirectory { dbName newDirectory } {

    variable xmlValueList

    return [set xmlValueList [setXMLValue $dbName directory $newDirectory $xmlValueList]]

}


############################################################################
#
# getVersion
#
# shortcut to getting a version for a given database
#
# parameters: dbName - name of the database
#
# returns: the version of the database, or an empty string if it can't be found
#
############################################################################

proc ::MultiSeqDialog::getVersion { dbName } {

    variable xmlValueList
    set version [getXMLValue $dbName version $xmlValueList]
    if {$version == ""} {
        set version [getXMLValue "$dbName\_1" version $xmlValueList]
    }
    return $version
}

############################################################################
#
# setVersion
#
# shortcut to setting a version for a given database
#
# parameters: dbName - name of the database
#             newVersion - the new value for the version
#
# returns: the XML of xmlValueList with the new version
#
############################################################################

proc ::MultiSeqDialog::setVersion { dbName newVersion } {

    variable xmlValueList

    return [set xmlValueList [setXMLValue $dbName version $newVersion $xmlValueList]]

}

############################################################################
#
# getVersion
#
# shortcut to getting an environment variable for a given software package
#
# parameters: swName - name of the software package
#             key - the name of the variable to get the value of
#
# returns: the value of the variable, or an empty string if it can't be found
#
############################################################################

proc ::MultiSeqDialog::getVariable { swName key } {

    variable xmlValueList
    set value [getXMLValue $swName $key $xmlValueList]
    if {$value == ""} {
        set value [getXMLValue "$swName\_1" $key $xmlValueList]
    }
    return $value
}

############################################################################
#
# setVariable
#
# shortcut to setting an environment variable for a given software
#
# parameters: swName - name of the software package
#             key - variable name
#             value - new value
#
# returns: the XML of xmlValueList with the new version
#
############################################################################

proc ::MultiSeqDialog::setVariable { swName key value } {

    variable xmlValueList

    return [set xmlValueList [setXMLValue $swName $key $value $xmlValueList]]

}