######################################
# The measure routines

proc measure {option args} {
    # if the subfunction exists, call it
    if [string length [info commands vmd_measure_$option]] {
	return [eval vmd_measure_$option $args]
    }
    # otherwise, complain
    error "measure: option '$option' doesn't exist"
}

###########
# The following call binaries from the LASSP (Laboratory of Atomic 
# and Solid State Physics) Tools at Cornell; http://www.lassp.cornell.edu/
# to compute the eigen-values and -vectors, determinate, and inverse of
# a NxN matrix.


# Function: vmd_measure_eigen {real|complex} {symmetric|hermitian} matrix
#             default is real an neither symmetric or hermitian
#  Returns: the eigenvalues and corresponding eigenvectors
#  This implementation calls the program Eigen from LASSP
proc vmd_measure_eigen {args} {
    global env
    # options are real/complex/symmetric/hermitian -- default is
    # real, non-symmetric
    set type real
    
    while 1 {
	set option [lvarpop args]
	switch -exact -- $option {
	    real {set type real}
	    complex {set type complex}
	    symmetric {set type symmetric}
	    hermitian {set type hermitian}
	    default {lvarpush args $option ; break }
	}
    }
    set mat_filename [unique_file $env(TMPDIR)/eigen_matrix]
    set mat_file [open $mat_filename w]
    puts $mat_file $args
    close $mat_file
    set arch_name [vmdinfo arch]
    set eigen [open "|$env(VMDDIR)/Eigen_$arch_name -$type -file_in $mat_filename | sed s/,//g" r]
    
    # get the eigen values
    gets $eigen line
    set line [list $line]
    # get the eigen vectors
    while {![eof $mat_file]} {
	gets $eigen data
	if [string length $data] {lappend line $data}
    }
    close $mat_file
    catch {exec /bin/rm -f $mat_filename}
    return $line
}


# Function: vmd_measure_determinate {real|complex} matrix
#  Returns: the determinate of the given matrix
#  This implementation calls the program Det from LASSP
proc vmd_measure_determinate {args} {
    global env
    # assumes a real matrix
    set type real
    while 1 {
	set option [lvarpop args]
	switch -exact -- $option {
	    real {set type real}
	    complex {set type complex}
	    default {lvarpush args $option ; break }
	}
    }
    set mat_filename [unique_file $env(TMPDIR)/det_matrix]
    set mat_file [open $mat_filename w]
    puts $mat_file $args
    close $mat_file
    set arch_name [vmdinfo arch]
    set det_file [open "|$env(VMDDIR)/Det_$arch_name -$type -file_in $mat_filename" r]
    # get the determinate
    gets $det_file det
    close $det_file
    catch {exec /bin/rm -f $mat_filename}
    return $det
}


# Function: vmd_measure_inverse {real|complex} matrix
#  Returns: the inverse of the given matrix
#  This implementation calls the program Invert from LASSP
proc vmd_measure_inverse {args} {
    global env
    # assumes a real matrix
    set type real
    while 1 {
	set option [lvarpop args]
	switch -exact -- $option {
	    real {set type real}
	    complex {set type complex}
	    default {lvarpush args $option ; break }
	}
    }
    set mat_filename [unique_file $env(TMPDIR)/invert_matrix]
    set mat_file [open $mat_filename w]
    puts $mat_file $args
    close $mat_file
    set arch_name [vmdinfo arch]
    set inv_file [open "|$env(VMDDIR)/Invert_$arch_name -$type -file_in $mat_filename" r]
    # get the inverse
    set inv {}
    while {![eof $inv_file]} {
	gets $inv_file data
	if [string length $data] {lappend inv $data}
    }
    close $inv_file
    catch {exec /bin/rm -f $mat_filename}
    return $inv
}
##################

# These commands are in C++
#Function:  vmd_measure_center <selection>
#                              <selection> weight <keyword>
#                              <selection> weight <list of weights>
# Returns: the center of the selection where the weights are either
#         1 (if no weights given)
#         a selection variable  (like mass, charge, etc.)
#         a tcl list of length either the total number of atoms OR
#                   the number of atoms in the selection

# compute the min and max coordinates of the selection
#  vmd_measure_minmax <selection> [weight <keyword|list> 

# compute the selection's radius of gyration 
#  vmd_measure_gyration <selection> [weight <keyword|list> 
# returns: {minx miny minz} {maxx maxy maxz}

# compute the transformation matrix needed to move selection1 
#     to selection2
# using Kabsch method for least rmsd fit
#  vmd_measure_fit <selection1> <selection2> [weight <keyword|list>]

# some helper functions for the degenerate case of 2 and 3 atoms
# with the Kabsch method (1 atom is handled internally to VMD)

proc vmd_measure_fit_2points {x y comx comy} {
    set t1 [transoffset [vecinvert $comx]]
    set t2 [transoffset $comy]
    set v1 [vecsub $x $comx]
    set v2 [vecsub $y $comy]
    set m1 [transvecinv $v1]
    set m2 [transvec $v2]
    # compose the result (comx to origin, xcoord to x axis,
    # x axis to ycoord, and ycoord to comy)
    return [transmult $t2 $m2 $m1 $t1]
}

proc vmd_measure_fit_3points {x1 x2 y1 y2 comx comy} {
    # center of mass to origin
    set tx [transoffset [vecinvert $comx]]
    # x1 to x axis
    set rx1 [transvecinv [vecsub $x1 $comx]]
    ## need to find the rotation to bring x2 along x-y plane
    #  with a rotation around x
    # converted x2
    set x2t [coordtrans $rx1 [vecsub $x2 $comx]]
    lassign $x2t a1 a2 a3
    # This is just atan2 of y,z components
    set angle [atan2 $a3,$a2]
    # make the rotation
    set rx2 [transaxis x -$angle rad]
    # the combined matrix
    set mx [transmult $rx2 $rx1 $tx]

    # do the same operations on y
    # center of mass to origin
    set ty [transoffset [vecinvert $comy]]
    # y1 to y axis
    set ry1 [transvecinv [vecsub $y1 $comy]]
    ## need to find the rotation to bring y2 along x-y plane
    #  with a rotation around x
    # converted y2
    set y2t [coordtrans $ry1 [vecsub $y2 $comy]]
    lassign $y2t a1 a2 a3
    # This is just atan2 of y,z components
    set angle [atan2 $a3,$a2]
    # make the rotation
    set ry2 [transaxis x -$angle rad]
    # the combined matrix
    set my [transmult $ry2 $ry1 $ty]

    # and get the inverse
    set myinv [measure inverse $my]

    return [transmult $myinv $mx]
}

