/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: UIText.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.52 $	$Date: 96/05/14 04:52:15 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * This is the User Interface for text commands.  It reads characters from
 * the console, and executes the commands.
 *
 * This will use the Tcl library for general script interpretation, which
 * allows for general script capabilities such as variable substitution,
 * loops, etc.  If Tcl cannot be used, text commands will still be available
 * in the program, but the general script capabilities will be lost.
 ***************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>

#if defined(ARCH_IRIX4) || defined(ARCH_IRIX5) || defined(ARCH_IRIX6)
#include <bstring.h>
#endif

#if defined(ARCH_HPUX9)
#include <time.h>
#else
#include <sys/select.h>
#endif

#include "UIText.h"
#include "CommandQueue.h"
#include "Inform.h"
#include "config.h"
#include "utilities.h"

// standard VMD command objects
#include "CmdAnimate.h"
#include "CmdColor.h"
#include "CmdDisplay.h"
#include "CmdLabel.h"
#include "CmdMenu.h"
#include "CmdMol.h"
#include "CmdMouse.h"
#include "CmdRender.h"
#include "CmdTool.h"
#include "CmdTracker.h"
#include "CmdTrans.h"
#include "CmdUser.h"
#include "CmdUtil.h"
#include "Measure.h"

// optional VMD command objects

#ifdef VMDREMOTE
#include "CmdRemote.h"
#endif

#ifdef VMDSIGMA
#include "CmdSigma.h"
#endif

///////////////////  Tcl variables and functions  //////////////////
#ifdef VMDTCL

#include <tcl.h>
#include <tclExtend.h>
#include "Global.h"


// instantiate the tclInterpreter
Tcl_Interp *UIText::tclInterp = NULL;

// number of TextEvent's which are using tclInterp; used to determine when
// to create and destroy the interpreter.  Since we only have one UIText,
// this hasn't been tested
static int numUsingTcl = 0;


// function called when a VMD command is encountered; this will then call
// the VMD callback function, and set the return codes, etc.
int UIText_process_Tcl(ClientData clientData, Tcl_Interp *,
			int argc, char *argv[]) {
  // clientData holds the pointer to the UIText object that invoked this
  UIText *uitxt = (UIText *)clientData;

  // now call the text processor callback function for the word
  // and return proper code
  if(uitxt->process_command(argc,argv) == 0) {
    // update the screen
    VMDupdate(FALSE);
  }

  // since we print out an error message if necessary we just always return OK
  return TCL_OK;
}

#include "TclCommands.h"

#endif

// check that EXTERNAL was defined correctly
#ifdef VMDEXTERNAL 
#ifndef VMDTCL
  Error: EXTERNAL can only be defined with TCL
#endif
  extern "C" {
#include "dp.h"
  }
#endif



///////////////////  special text callback functions  //////////////////

// text callback routine for commands which are optional, and have not been
// included in this executable
int text_cmd_unsupported(int, char **argv, CommandQueue *, int) {
  msgWarn << "'" << argv[0] << "' commands are not supported in this version";
  msgWarn << " of the program.  Please recompile with the option enabled.";
  msgWarn << sendmsg;
  return TRUE;
}


////////////////////////////  TextEvent routines  ///////////////////////////

// constructor
TextEvent::TextEvent(char *str, int delcmd, int newUIid)
	: Command(Command::TEXT_EVENT, newUIid) {
  deleteCmd = delcmd;
  cmd = str;
  // get rid of newline at end, if any
  if(cmd[strlen(cmd) - 1] == '\n')
    cmd[strlen(cmd) - 1] = '\0';
}

// destructor; frees up command string memory if necessary
TextEvent::~TextEvent(void) {
  if(deleteCmd && cmd)
    delete [] cmd;
}


////////////////////////////  UIText routines  ///////////////////////////

// This emulates the "vmdinfo_tcl" function when not using Tcl
#define SIMPLE_NOTCL_OPT(string, result) \
if (!strcmp(argv[1], string)) {          \
  msgInfo << result << sendmsg;          \
  return FALSE;                          \
}

#ifndef VMDTCL
// def'ed out since I don't like compiler warnings issued when this isn't used
static int vmdinfo_notcl(int argc, char *argv[], CommandQueue *, int )
{
  if (argc == 2) {
    SIMPLE_NOTCL_OPT("version", VMDVERSION);
    SIMPLE_NOTCL_OPT("versionmsg", VERSION_MSG);
    SIMPLE_NOTCL_OPT("authors", VMD_AUTHORS);
    SIMPLE_NOTCL_OPT("arch", VMD_ARCH);
    SIMPLE_NOTCL_OPT("options", VMD_OPTIONS);
    SIMPLE_NOTCL_OPT("www", VMD_HOMEPAGE);
    SIMPLE_NOTCL_OPT("wwwhelp", VMD_HELPPAGE);
  }
  msgErr << 
"vmdinfo: version | versionmsg | authors | arch | options | www | wwwhelp";
  return TRUE;
}
#endif

// constructor
UIText::UIText(UIList *uil, CommandQueue *cq)
  : UIObject("Text Console",uil, cq), input_files(64), txtMsgEcho("Echo") {

  MSGDEBUG(1,"Creating UIText ..." << sendmsg);

  // record which commands we want
  command_wanted(Command::TEXT_EVENT);

  // display system prompt
  need_prompt();
  
  // don't wait for anything
  delay = 0;
  mytimer.clear();

  // echoing is off by default
  doEcho = FALSE;

  // initialize Tcl, if necessary
#ifdef VMDTCL
  if(!(numUsingTcl++)) {
    tclInterp = Tcl_CreateInterp();
    if (TclX_Init(tclInterp) == TCL_ERROR) {
      msgErr << "Tcl-X startup error" << sendmsg;
    }
#ifdef VMDEXTERNAL
    if (Tdp_Init(tclInterp) == TCL_ERROR) {
      msgErr << "Tcl-DP startup error" << sendmsg;
    }
#endif
    // set tcl_interactive, which lets us run unix commands as from a shell
    Tcl_SetVar(tclInterp, "tcl_interactive", "1", 0);

    // add commands for analysis and manipulation
    Tcl_CreateCommand(tclInterp, "vmdinfo", vmdinfo_tcl,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand(tclInterp, "atomselect", make_tcl_atomsel,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand(tclInterp, "molinfo", molecule_tcl, 
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand(tclInterp, "graphics", graphics_tcl, 
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand(tclInterp, "colorinfo", tcl_colorinfo, 
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand(tclInterp, "feedback", feedback_tcl,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand(tclInterp, "edm", make_tcl_edm,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    // use the Tcl 7.5 foreach commmand
    Tcl_CreateCommand(tclInterp, "foreach", Tcl_ForeachCmd,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    // read the startup script
    if (Tcl_Eval(tclInterp, "if {![info exists env(VMDDIR)]} {"
	 "source /usr/local/lib/vmd/scripts/vmd/vmdinit.tcl} else {"
	 "source $env(VMDDIR)/scripts/vmd/vmdinit.tcl}") != TCL_OK) {
      // some sort of error, oh my!
      msgErr << "Could not read the vmd initialization file -" << sendmsg;
      msgErr << " $env(VMDDIR)/scripts/vmd/vmdinit.tcl" << sendmsg;
      msgErr << tclInterp -> result << sendmsg;
      msgErr << "The VMDDIR environment variable is defined in the vmd "
	     << "startup script and" << sendmsg;
      msgErr << "should point to the top of the VMD hierarchy.  VMD will "
	     << "continue with " << sendmsg;
      msgErr << "limited functionality." << sendmsg;
    }

    // set up the 'measure' and override some of the vector/matrix commands
    init_measure_commands(tclInterp);

    Tcl_CreateCommand(tclInterp, "vecadd", proc_vecadd,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(tclInterp, "vecsub", proc_vecsub,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(tclInterp, "vecscale", proc_vecscale,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(tclInterp, "transoffset", proc_transoffset,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(tclInterp, "transmult", proc_transmult,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(tclInterp, "vectrans", proc_vectrans,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(tclInterp, "vmd_atomselect_move", 
		      proc_vmd_atomselect_move,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(tclInterp, "vmd_atomselect_moveby", 
		      proc_vmd_atomselect_moveby,
		      (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
  }
#else
  // Tcl is not defined, but I still want a (weaker) version of vmdinfo
  add_command("vmdinfo", vmdinfo_notcl);
#endif // VMDTCL


  // register all the text command here.  Enter them with the 'add_command'
  // routine, in alphabetical order.
  // NOTE: for command which are optional, either add the command with the
  //	the proper callback function, or add it with the callback function
  //	'text_cmd_unsupported', via an ifdef/else/endif sequence.

  add_command("animate",	text_cmd_animate);
  add_command("axes",		text_cmd_axes);
  add_command("color",		text_cmd_color);
  add_command("debug",		text_cmd_debug);
  add_command("display",	text_cmd_display);
  add_command("echo",		text_cmd_echo);
  add_command("help",		text_cmd_help);
  add_command("journal",	text_cmd_log);
  add_command("label",		text_cmd_label);
  add_command("light",		text_cmd_light);
  add_command("log",		text_cmd_log);
  add_command("menu",		text_cmd_menu);
  add_command("mol",		text_cmd_mol);
  add_command("molecule",	text_cmd_mol);
  add_command("mouse",		text_cmd_mouse);
  add_command("play",		text_cmd_play);
  add_command("run",		text_cmd_play);
  add_command("render",		text_cmd_render);
  add_command("rock",		text_cmd_rock);
  add_command("rot",		text_cmd_rotate);
  add_command("rotate",		text_cmd_rotate);
  add_command("rotation",	text_cmd_rotate);
  add_command("rotmat",		text_cmd_rotmat);
#ifdef VMDREMOTE
  add_command("remote",		text_cmd_remote);
  add_command("sim",		text_cmd_sim);
  add_command("simulation",	text_cmd_sim);
#else
  add_command("remote",		text_cmd_unsupported);
  add_command("sim",		text_cmd_unsupported);
  add_command("simulation",	text_cmd_unsupported);
#endif
  add_command("scale",		text_cmd_scale);
#ifdef VMDSIGMA
  add_command("sigma",		text_cmd_sigma);
#else
  add_command("sigma",		text_cmd_unsupported);
#endif
  add_command("stage",		text_cmd_stage);
  add_command("tool",		text_cmd_tool);
  add_command("tracker",	text_cmd_tracker);
  //  add_command("trans",		text_cmd_translate);
  add_command("translate",	text_cmd_translate);
  add_command("user",		text_cmd_user);
  add_command("wait",		text_cmd_wait);

  add_command("quit",		text_cmd_quit);
  add_command("exit",		text_cmd_quit);
  add_command("stop",		text_cmd_quit);
}


// destructor
UIText::~UIText(void) {
#ifdef VMDTCL
  if(--numUsingTcl < 1) {
    Tcl_DeleteInterp(tclInterp);
    tclInterp = NULL;
  }
#endif
}


// display the prompt for the user
void UIText::prompt(void) {

  printf(VMD_CMD_PROMPT,myName);
  fflush(stdout);
  
  // tell Message objects they need a newline before next output message
  msgInfo.need_newline(TRUE);
  msgWarn.need_newline(TRUE);
  msgErr.need_newline(TRUE);
  msgDebug.need_newline(TRUE);
}


// return number of text commands currently understood
int UIText::num_commands(void) { return textProcessors.num(); }


// return the Nth word understood; NULL if error
char *UIText::word(int n) {
  if(n < 0 || n >= num_commands())
    return NULL;
  else
    return textProcessors.name(n);
}


// add a new text command
void UIText::add_command(char *nm, TextCallback *cb) {
  textProcessors.add_name(nm, cb);

  // if using Tcl, register Tcl callback as well
#ifdef VMDTCL
  Tcl_CreateCommand(tclInterp, nm, UIText_process_Tcl,
	(ClientData)this, (Tcl_CmdDeleteProc *) NULL);
#endif
}


// process an argc, argv command, and return < 0 if the command is not
// known, 0 if it was successful, or > 0 if an error occcurred
int UIText::process_command(int argc, char **argv) {

  // make sure we have at least one word
  if(argc < 1)
    return 1;

  // search for the word, and when found execute it, otherwise an error
  int wordindx = textProcessors.typecode(argv[0], CMDLEN);
  if(wordindx < 0) {
    msgErr << "Unknown command '" << argv[0] << "'."
           << sendmsg;
    return (-1);
  }

  // now call the text processor callback function for the word
  int retval=(*(textProcessors.data(wordindx))) (argc, argv, cmdQueue, id());

  // print if an error occurred
  if(retval)
    msgErr << "Invalid format for command '" << argv[0] << "'." << sendmsg;

  return retval;
}


// specify new file to read commands from
void UIText::read_from_file(char *fname) {
  FILE *f;
  if((f = fopen(fname,"r")) != NULL) {
    msgInfo << "Reading commands from '" << fname << "'." << sendmsg;
    input_files.push(f);
  } else {
    msgErr << "Cannot open file '" << fname << "' for reading." << sendmsg;
  }
}


// set the text processor to wait for the given number of seconds before
// reading another text command
void UIText::wait(float wd) {
  mytimer.stop();
  mytimer.reset();
  delay = wd;
  if (delay > 0.0)
    mytimer.start();
  else
    delay = 0.0;
}


// check for an event; return TRUE if we found an event; FALSE otherwise
int UIText::check_event(void) {
  static char cmdbuffer[1024];
  fd_set readvec;
  int ret, stdin_fd;
  struct timeval timeout;

#ifdef VMDEXTERNAL
  // read any Tcl-DP events that may have occured
  Tcl_Eval(tclInterp, "dp_update");
#endif

  // check to see if I've waited long enough
  if (delay > 0.0)
    if (delay < mytimer.clock_time())
      wait(-1.0);
    else
      return FALSE;  // gotta wait some more

  // check for text from input device
  if(input_files.stack_size() < 1) {
    if(needPrompt) {
      prompt();
      needPrompt = FALSE;
    }
    timeout.tv_sec = 0;
    timeout.tv_usec = 0;
    stdin_fd = 0;
    FD_ZERO(&readvec);
    FD_SET(stdin_fd, &readvec);
#if defined(ARCH_IRIX4) || defined(ARCH_IRIX5) || defined(ARCH_IRIX6)
    ret = select(16,&readvec,NULL,NULL,&timeout);
#else
    ret = select(16,(int *)(&readvec),NULL,NULL,&timeout);
#endif
  
    if (ret == -1) {
      // got an error
      msgErr << "Error from select, while attempting to read text input."
             << sendmsg;
      return FALSE;
    } else if (ret == 0) {
      // time out
      return FALSE;
    }

    if (fgets(cmdbuffer,1024,stdin) == NULL) {
      addcommand(new CmdQuit(FALSE,id()));
      return FALSE;
    }
    
  } else {
    // just read in next line from current file, if available
    if(!fgets(cmdbuffer, 1024, input_files.top())) {
      msgInfo << "EOF encountered for current input file." << sendmsg;
      input_files.pop();	// now go back to reading previous file
      return FALSE;
    }
  }

  MSGDEBUG(3, "Read command: " << cmdbuffer << sendmsg);

  // create a text event, and queue it; the event object must deallocate the
  // string space allocated to hold the text command.
#ifdef VMDTCL
  static Tcl_DString cmd;
  static int tclInit = 0;
  
  if(!tclInit) {
    Tcl_DStringInit(&cmd);
    tclInit = 1;
  }

  Tcl_DStringAppend(&cmd, cmdbuffer, -1);
  
  if(Tcl_CommandComplete(Tcl_DStringValue(&cmd))) {
    addcommand(new TextEvent(stringdup(Tcl_DStringValue(&cmd)), TRUE, id()));
    Tcl_DStringFree(&cmd);
    return TRUE;
  } else {
    // print out a special prompt to indicate we're in the middle of a
    // multi-line command
    if(input_files.stack_size() < 1) {
      printf("? ");
      fflush(stdout);
    }
    return FALSE;
  }
#else
  // strip off trailing newline
  int cmdstrlen = strlen(cmdbuffer);
  if(cmdstrlen > 0 && cmdbuffer[cmdstrlen - 1] == '\n')
    cmdbuffer[cmdstrlen - 1] = '\0';

  addcommand(new TextEvent(stringdup(cmdbuffer), TRUE, id()));
  return TRUE;
#endif
}


// update the display due to a command being executed.  Return whether
// any action was taken on this command.
// Arguments are the command type, command object, and the 
// success of the command (T or F).
int UIText::act_on_command(int type, Command *cmd, int) {

  MSGDEBUG(3,"UIText: acting on command " << type << sendmsg);

  // check all the possible commands that we look for ...
  if(type == Command::TEXT_EVENT) {

    // echo the command to the console, if necessary
    if(doEcho)
      txtMsgEcho << ((TextEvent *)cmd)->cmd << sendmsg;

#ifdef VMDTCL
    // we're using Tcl, so ask the interpreter to parse it and possibly
    // call a callback function when a VMD command is encountered
    Tcl_RecordAndEval(tclInterp, ((TextEvent *)cmd)->cmd, 0);

    // print out result if necessary
    if(*(tclInterp->result) != NULL)
      msgInfo << tclInterp->result << sendmsg;
#else
    // it's a text event, and we're not using Tcl, so tokenize and process
    // the command
    int argc;
    char *argv[256];

    if(!str_tokenize(((TextEvent *)cmd)->cmd, &argc, argv)) {
      need_prompt();
      return FALSE;
    }

    // make sure there is at least ONE token in the command, and skip comments
    if(argc < 1 || argv[0][0] == '#')
      ;		// we processed it, but did nothing 
    else
      // search for the word, and when found execute it, otherwise an error
      process_command(argc, argv);
#endif

    // display the prompt again, if the command was read from the text
    if(cmd->getUIid() == id())
      need_prompt();
    
  } else
    // unknown command type
    return FALSE;

  return TRUE;
}


